; (function ($) {
  "use strict";

  const App = {
    config: {
      headroom: {
        enabled: true,
        options: {
          classes: {
            initial: "headroom",
            pinned: "is-pinned",
            unpinned: "is-unpinned",
            top: "is-top",
            notTop: "is-not-top",
            bottom: "is-bottom",
            notBottom: "is-not-bottom",
            frozen: "is-frozen",
          },
        },
      },
      ajax: {
        enabled: true,
      },
      cursorFollower: {
        enabled: true,
        disableBreakpoint: 992,
      },
    },
    html: document.documentElement,
    body: document.body,
  };

  window.App = App;

  window.onload = () => {
    if (App.config.cursorFollower.enabled) {
      Cursor.init();
    }
  };

  const Cursor = (() => {
    const cursor = document.querySelector(".pxl-js-cursor");

    let state = false;
    let clientX = -100, clientY = -100;
    let cursorWidth = 0, cursorHeight = 0;
    let isBackTopHover = false;
    let backTopIconCenter = { x: 0, y: 0 };

    let follower, label, drap, icon;
    let cursor_arrow_prev, cursor_arrow_next;

    const getAttr = (el, attr) => el?.getAttribute(attr);

    const variables = () => {
      follower = cursor.querySelector(".pxl-js-follower");
      label = cursor.querySelector(".pxl-js-label");
      drap = cursor.querySelector(".pxl-js-drap");
      icon = cursor.querySelector(".pxl-js-icon");
      cursor_arrow_prev = cursor.querySelector(".pxl-cursor-arrow-prev");
      cursor_arrow_next = cursor.querySelector(".pxl-cursor-arrow-next");

      cursorWidth = cursor.offsetWidth / 2;
      cursorHeight = cursor.offsetHeight / 2;
    };

    const render = () => {
      if (isBackTopHover) {
        // Center cursor around the back-to-top icon
        cursor.style.transform = `translate(${backTopIconCenter.x - cursorWidth}px, ${backTopIconCenter.y - cursorHeight}px)`;
      } else {
        // Normal cursor following mouse
        cursor.style.transform = `translate(${clientX - cursorWidth}px, ${clientY - cursorHeight}px)`;
      }
      requestAnimationFrame(render);
    };

    const update = () => {
      const triggers = document.querySelectorAll([
        ".pxl-cursor--cta",
        ".pxl-cursor-remove",
        ".pxl-close",
        "button",
        "a",
        "input",
        "textarea",
        "[data-cursor]",
        "[data-cursor-label]",
        "[data-cursor-drap]",
        "[data-drap-style]",
        "[data-cursor-icon]",
        "[data-cursor-icon-left]",
        "[data-cursor-icon-right]",
      ].join(","));

      triggers.forEach(el => {
        el.addEventListener("mouseenter", enterHandler);
        el.addEventListener("mouseleave", leaveHandler);
      });

      // Add specific handler for back-to-top button
      addBackTopHoverEffect();
    };

    const clear = () => {
      const triggers = document.querySelectorAll([
        ".pxl-cursor--cta",
        ".pxl-cursor-remove",
        ".pxl-close",
        "button",
        "a",
        "input",
        "textarea",
        "[data-cursor]",
        "[data-cursor-label]",
        "[data-cursor-drap]",
        "[data-drap-style]",
        "[data-cursor-icon]",
        "[data-cursor-icon-left]",
        "[data-cursor-icon-right]",
      ].join(","));

      triggers.forEach(el => {
        el.removeEventListener("mouseenter", enterHandler);
        el.removeEventListener("mouseleave", leaveHandler);
      });

      // Clear back-to-top handlers
      clearBackTopHoverEffect();
    };

    const enterHandler = ({ target }) => {
      cursor.classList.add('is-active');

      const labelText = getAttr(target, 'data-cursor-label');
      const drapText = getAttr(target, 'data-cursor-drap');
      const drapStyle = getAttr(target, 'data-drap-style');

      if (labelText) {
        App.body.classList.add('is-cursor-active');
        cursor.classList.add('has-label');
        label.innerHTML = labelText;
      }

      if (drapText) {
        App.body.classList.add('is-cursor-active');
        cursor.classList.add('has-drap');
        drap.innerHTML = drapText;
      }

      if (drapStyle) {
        cursor.classList.add(drapStyle);
        drap.innerHTML = drapStyle;
      }

      if (getAttr(target, 'data-cursor-icon')) {
        App.body.classList.add('is-cursor-active');
        cursor.classList.add('has-icon');
      }

      if (getAttr(target, 'data-cursor-icon-left')) {
        App.body.classList.add('is-cursor-active');
        cursor.classList.add('has-icon-left');
      }

      if (getAttr(target, 'data-cursor-icon-right')) {
        App.body.classList.add('is-cursor-active');
        cursor.classList.add('has-icon-right');
      }

      if (getAttr(target, 'data-has-remove')) {
        cursor.classList.add('has-remove');
      }
    };

    const leaveHandler = () => {
      App.body.classList.remove('is-cursor-active');
      cursor.classList.remove(
        'is-active',
        'has-label',
        'has-drap',
        'has-icon',
        'has-icon-left',
        'has-icon-right',
        'has-remove'
      );
      label.innerHTML = '';
      drap.innerHTML = '';
      icon.innerHTML = '';
    };

    // New function to handle back-to-top button hover effect
    const addBackTopHoverEffect = () => {
      const backTopButtons = document.querySelectorAll('.parallax-wrap');

      backTopButtons.forEach(button => {
        button.addEventListener('mouseenter', (e) => {
          const iconElement = button.querySelector('svg, i, .icon, [class*="icon"]') || button;
          const rect = iconElement.getBoundingClientRect();

          // Calculate icon center position
          backTopIconCenter.x = rect.left + rect.width / 2;
          backTopIconCenter.y = rect.top + rect.height / 2;

          isBackTopHover = true;
          cursor.classList.add('is-parallax-element');
          App.body.classList.add('is-cursor-paralax');
        });

        button.addEventListener('mouseleave', () => {
          isBackTopHover = false;
          cursor.classList.remove('is-parallax-element');
          App.body.classList.remove('is-cursor-paralax');
        });
      });
    };

    const clearBackTopHoverEffect = () => {
      const backTopButtons = document.querySelectorAll('.pxl-language-switch');

      backTopButtons.forEach(button => {
        const newButton = button.cloneNode(true);
        button.parentNode.replaceChild(newButton, button);
      });
    };

    const addHoverEffects = () => {
      const hideCursorOnHover = [
        '.pxl-icon-list.style-box-paralax a',
        // '.pxl-swiper-slider a',
        '.pxl-swiper-arrow',
        '.pxl-swiper-slider .pxl-swiper-arrow-wrap.style-3 .cursor-arrow-next',
        '.pxl-swiper-slider .pxl-swiper-arrow-wrap.style-3 .cursor-arrow-prev',
      ];

      const hideNextArrow = [
        '.pxl-swiper-slider .pxl-swiper-arrow-wrap.style-3 .cursor-arrow-next',
      ];

      const hidePrevArrow = [
        '.pxl-swiper-slider .pxl-swiper-arrow-wrap.style-3 .cursor-arrow-prev',
      ];

      hideCursorOnHover.forEach(selector => {
        document.querySelectorAll(selector).forEach(el => {
          el.addEventListener('mouseenter', hide);
          el.addEventListener('mouseleave', show);
        });
      });

      hideNextArrow.forEach(selector => {
        document.querySelectorAll(selector).forEach(el => {
          el.addEventListener('mouseenter', () => {
            hide();
            hide_arrow();
          });
          el.addEventListener('mouseleave', () => {
            show();
            show_arrow();
          });
        });
      });

      hidePrevArrow.forEach(selector => {
        document.querySelectorAll(selector).forEach(el => {
          el.addEventListener('mouseenter', () => {
            hide();
            hide_arrow_prev();
          });
          el.addEventListener('mouseleave', () => {
            show();
            show_arrow_prev();
          });
        });
      });

      // ✅ Hover vào .test thì follower hiện ra
      document.querySelectorAll('.pxl-text-carousel9').forEach(el => {
        el.addEventListener('mouseenter', () => {
          if (follower) {
            follower.style.opacity = '1';
            follower.style.transform = 'scale(1)';
          }
        });
        el.addEventListener('mouseleave', () => {
          if (follower) {
            follower.style.opacity = '';
            follower.style.transform = '';
          }
        });
      });
    };

    const breakpoint = () => {
      const resizeHandler = () => {
        const width = window.innerWidth;
        if (width < App.config.cursorFollower.disableBreakpoint) {
          state = false;
          cursor.classList.remove('is-enabled');
          clear();
        } else {
          state = true;
          cursor.classList.add('is-enabled');
          update();
        }
      };

      resizeHandler();
      window.addEventListener('resize', resizeHandler);
    };

    const init = () => {
      if (!cursor) return;

      variables();
      state = true;
      cursor.classList.add('is-enabled');

      document.addEventListener("mousedown", () => cursor.classList.add('is-mouse-down'));
      document.addEventListener("mouseup", () => cursor.classList.remove('is-mouse-down'));
      document.addEventListener("mousemove", e => {
        clientX = e.clientX;
        clientY = e.clientY;
      });

      requestAnimationFrame(render);
      update();
      addHoverEffects();
      breakpoint();
    };

    const hide = () => cursor?.classList.add('is-hidden');
    const show = () => cursor?.classList.remove('is-hidden');
    const hide_arrow = () => cursor?.classList.add('is-hidden-arrow');
    const show_arrow = () => cursor?.classList.remove('is-hidden-arrow');
    const hide_arrow_prev = () => cursor?.classList.add('is-hidden-arrow-prev');
    const show_arrow_prev = () => cursor?.classList.remove('is-hidden-arrow-prev');

    return {
      init,
      update,
      clear,
      hide,
      show,
      show_arrow,
      hide_arrow,
      show_arrow_prev,
      hide_arrow_prev,
    };
  })();
})(jQuery);