class WebGLHoverManager {
    constructor(container) {
        this.canvas = document.createElement('canvas');
        this.canvas.className = 'webgl-hover-canvas';
        this.lastMouseMoveTime = performance.now();
        container.querySelector('.wrap-distort-canvas')?.appendChild(this.canvas);

        this.gl = this.canvas.getContext('webgl');
        if (!this.gl) {
            console.error('WebGL not supported');
            return;
        }

        this.images = [];
        this.textures = {};
        this.time = 0;

        this.initWebGL();
        if (!this.createShaders()) return;
        this.createBuffers();
        this.startRender();

        window.addEventListener('resize', this.resize.bind(this));
        document.addEventListener('mousemove', this.onMouseMove.bind(this));
    }

    initWebGL() {
        this.resize();
    }

    resize() {
        const wrapper = this.canvas.parentElement;
        if (!wrapper) return;

        const rect = wrapper.getBoundingClientRect();
        this.canvas.width = rect.width * window.devicePixelRatio;
        this.canvas.height = rect.height * window.devicePixelRatio;

        this.gl.viewport(0, 0, this.canvas.width, this.canvas.height);
    }

    createShaders() {
        const gl = this.gl;

        const vertexSource = `
        attribute vec2 position;
        attribute vec2 aTexCoord;
        varying vec2 vTexCoord;
        void main() {
            vTexCoord = aTexCoord;
            gl_Position = vec4(position, 0.0, 1.0);
        }`;

        const fragmentSource = `
        precision highp float;
        varying vec2 vTexCoord;
        uniform sampler2D uTexture;
        uniform float uTime;
        uniform float uStrength;
        uniform vec2 uMouse;
        uniform bool uHover;

        void main() {
            vec2 uv = vTexCoord;
            if (uHover) {
                vec2 mouseDir = uv - uMouse;
                float dist = length(mouseDir);
                float angle = atan(mouseDir.y, mouseDir.x);
                float wave = sin(dist * 15.0 - angle) * 0.02;
                float effect = uStrength * smoothstep(0.45, 0.1, dist);
                vec2 offset = normalize(mouseDir) * wave * effect;
                vec2 distortedPos = uv + offset;

                vec2 rgbDir = normalize(mouseDir);
                float rgbAmt = 0.02 * effect;

                float r = texture2D(uTexture, distortedPos + rgbDir * rgbAmt).r;
                float g = texture2D(uTexture, distortedPos).g;
                float b = texture2D(uTexture, distortedPos - rgbDir * rgbAmt).b;

                gl_FragColor = vec4(r, g, b, 1.0);
            } else {
                gl_FragColor = texture2D(uTexture, uv);
            }
        }`;

        const vertexShader = this.compileShader(gl.VERTEX_SHADER, vertexSource);
        const fragmentShader = this.compileShader(gl.FRAGMENT_SHADER, fragmentSource);
        if (!vertexShader || !fragmentShader) return false;

        this.program = gl.createProgram();
        gl.attachShader(this.program, vertexShader);
        gl.attachShader(this.program, fragmentShader);
        gl.linkProgram(this.program);

        if (!gl.getProgramParameter(this.program, gl.LINK_STATUS)) {
            console.error('Shader link error:', gl.getProgramInfoLog(this.program));
            return false;
        }

        gl.useProgram(this.program);

        this.attribLocations = {
            position: gl.getAttribLocation(this.program, 'position'),
            aTexCoord: gl.getAttribLocation(this.program, 'aTexCoord'),
        };
        this.uniformLocations = {
            uTexture: gl.getUniformLocation(this.program, 'uTexture'),
            uTime: gl.getUniformLocation(this.program, 'uTime'),
            uStrength: gl.getUniformLocation(this.program, 'uStrength'),
            uMouse: gl.getUniformLocation(this.program, 'uMouse'),
            uHover: gl.getUniformLocation(this.program, 'uHover'),
        };

        return true;
    }

    compileShader(type, source) {
        const shader = this.gl.createShader(type);
        this.gl.shaderSource(shader, source);
        this.gl.compileShader(shader);

        if (!this.gl.getShaderParameter(shader, this.gl.COMPILE_STATUS)) {
            console.error('Shader compile failed:', this.gl.getShaderInfoLog(shader));
            this.gl.deleteShader(shader);
            return null;
        }
        return shader;
    }

    createBuffers() {
        const gl = this.gl;
        this.vertexBuffer = gl.createBuffer();
        this.texCoordBuffer = gl.createBuffer();

        // Static full-screen quad data (uploaded once)
        const vertices = new Float32Array([-1, -1, 1, -1, -1, 1, 1, 1]);
        const texCoords = new Float32Array([0, 0, 1, 0, 0, 1, 1, 1]);

        gl.bindBuffer(gl.ARRAY_BUFFER, this.vertexBuffer);
        gl.bufferData(gl.ARRAY_BUFFER, vertices, gl.STATIC_DRAW);
        gl.vertexAttribPointer(this.attribLocations.position, 2, gl.FLOAT, false, 0, 0);
        gl.enableVertexAttribArray(this.attribLocations.position);

        gl.bindBuffer(gl.ARRAY_BUFFER, this.texCoordBuffer);
        gl.bufferData(gl.ARRAY_BUFFER, texCoords, gl.STATIC_DRAW);
        gl.vertexAttribPointer(this.attribLocations.aTexCoord, 2, gl.FLOAT, false, 0, 0);
        gl.enableVertexAttribArray(this.attribLocations.aTexCoord);
    }

    addImage(imgElement) {
        const src = imgElement.src;
        if (!this.textures[src]) {
            const image = new Image();
            image.crossOrigin = 'anonymous';
            image.src = src;
            image.onload = () => {
                this.textures[src] = this.createTexture(image);
                this.storeImageInfo(imgElement, this.textures[src]);
            };
        } else {
            this.storeImageInfo(imgElement, this.textures[src]);
        }
    }

    createTexture(image) {
        const gl = this.gl;
        const texture = gl.createTexture();
        gl.bindTexture(gl.TEXTURE_2D, texture);
        gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
        gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, image);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
        return texture;
    }

    storeImageInfo(element, texture) {
        this.images.push({
            element,
            texture,
            strength: 0.0,
            targetStrength: 0.0,
            mouseX: 0.5,
            mouseY: 0.5,
        });
    }

    onMouseMove(e) {
        const x = e.clientX;
        const y = e.clientY;
        this.lastMouseMoveTime = performance.now();

        this.images.forEach(img => {
            const rect = img.element.getBoundingClientRect();
            const inside = x >= rect.left && x <= rect.right && y >= rect.top && y <= rect.bottom;
            img.targetStrength = inside ? 1.0 : 0.0;
            if (inside) {
                img.mouseX = (x - rect.left) / rect.width;
                img.mouseY = 1.0 - (y - rect.top) / rect.height;
            }
        });
    }

    startRender() {
        this.lastTime = performance.now();
        requestAnimationFrame(this.render);
    }

    render = () => {
        const gl = this.gl;
        const current = performance.now();
        const dt = Math.min((current - this.lastTime) / 1000, 1 / 60);
        this.lastTime = current;
        this.time += dt;

        gl.clearColor(0, 0, 0, 0);
        gl.clear(gl.COLOR_BUFFER_BIT);
        gl.useProgram(this.program);

        this.images.forEach(img => {
            // Smoothly decay when mouse stops moving near the element
            if (img.targetStrength > 0 && current - this.lastMouseMoveTime > 180) {
                img.targetStrength = Math.max(0, img.targetStrength - dt * 2.0);
            }

            img.strength += (img.targetStrength - img.strength) * dt * 4.0;
            img.strength = Math.max(0, Math.min(1, img.strength));
            // Geometry/attributes are static; no per-frame uploads

            gl.activeTexture(gl.TEXTURE0);
            gl.bindTexture(gl.TEXTURE_2D, img.texture);
            gl.uniform1i(this.uniformLocations.uTexture, 0);
            gl.uniform1f(this.uniformLocations.uTime, this.time);
            gl.uniform1f(this.uniformLocations.uStrength, img.strength);
            gl.uniform2f(this.uniformLocations.uMouse, img.mouseX, img.mouseY);
            gl.uniform1i(this.uniformLocations.uHover, img.strength > 0.0 ? 1 : 0);

            gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);
        });

        requestAnimationFrame(this.render);
    }
}

function initializeWebGLEffect(container, manager) {
    const images = container.querySelectorAll('.wrap-img-distortion img');

    images.forEach(img => {
        if (!img.getAttribute('crossorigin')) {
            img.setAttribute('crossorigin', 'anonymous');
            img.src = img.src;
        }

        if (!img.complete || !img.naturalWidth) {
            img.onload = () => manager.addImage(img);
        } else {
            manager.addImage(img);
        }
    });
}

function initWebGLEffect() {
    const items = document.querySelectorAll('.wrap-img-distortion');
    if (!items.length || window.innerWidth < 1024) return;

    imagesLoaded(items, () => {
        setTimeout(() => {
            items.forEach(item => {
                if (!item.closest('.pxl-swiper-container') && !item.classList.contains('pxl-swiper-container')) {
                    if (!item.webGLHoverManager) {
                        const manager = new WebGLHoverManager(item);
                        item.webGLHoverManager = manager;
                        initializeWebGLEffect(item, manager);
                    }
                }
            });
        }, 300);
    });
}

setTimeout(() => {
    initWebGLEffect();
}, 10);

if (window.barba) {
    barba.hooks.after(() => {
        setTimeout(() => {
            initWebGLEffect();
        }, 10);
    });
}
