; (function ($) {
    "use strict";

    // Performance optimization utilities
    function throttle(func, limit) {
        let inThrottle;
        return function () {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }

    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Device capability detection
    const deviceCapabilities = {
        isLowEnd: () => {
            const memory = navigator.deviceMemory || 4;
            const cores = navigator.hardwareConcurrency || 4;
            const connection = navigator.connection;

            return memory < 4 || cores < 4 ||
                (connection && connection.effectiveType === 'slow-2g');
        },
        isSafari: () => {
            const ua = navigator.userAgent.toLowerCase();
            return ua.includes('safari') && !ua.includes('chrome');
        },
        isMobileSafari: () => {
            const ua = navigator.userAgent.toLowerCase();
            return ua.includes('safari') && ua.includes('mobile') && !ua.includes('chrome');
        },
        supportsSmoothScrolling: () => {
            return 'scrollBehavior' in document.documentElement.style;
        },
        supportsIntersectionObserver: () => {
            return 'IntersectionObserver' in window;
        },
        supportsRequestAnimationFrame: () => {
            return 'requestAnimationFrame' in window;
        }
    };

    // Performance settings based on device capabilities
    const performanceSettings = {
        scrollThrottle: deviceCapabilities.isLowEnd() || deviceCapabilities.isSafari() ? 150 : 100,
        resizeThrottle: deviceCapabilities.isLowEnd() || deviceCapabilities.isSafari() ? 500 : 300,
        animationDuration: deviceCapabilities.isLowEnd() || deviceCapabilities.isSafari() ? 0.6 : 0.4,
        enableParallax: !deviceCapabilities.isLowEnd() && !deviceCapabilities.isSafari(),
        enableSmoothAnimations: !deviceCapabilities.isLowEnd() && !deviceCapabilities.isMobileSafari(),
        useTransform3d: !deviceCapabilities.isSafari(),
        enableScrollEffects: !deviceCapabilities.isLowEnd() && !deviceCapabilities.isSafari()
    };

    // Production check - remove console logs in production
    const isProduction = window.location.hostname !== 'localhost' && window.location.hostname !== '127.0.0.1';

    const safeLog = function (message) {
        if (!isProduction && typeof console !== 'undefined' && console.log) {
            console.log(message);
        }
    };

    const safeWarn = function (message) {
        if (!isProduction && typeof console !== 'undefined' && console.warn) {
            console.warn(message);
        }
    };

    // Global variables
    let pxl_scroll_top;
    let pxl_window_height;
    let pxl_window_width;
    let pxl_scroll_status = '';
    let pxl_last_scroll_top = 0;
    let pxl_post_slip = false;
    let scrollAnimationFrameId = null;
    let resizeAnimationFrameId = null;

    $(window).on('load', function () {
        dreamslab_loading();
        dreamslab_loading_overlay();

        setTimeout(function () {
            $(".pxl-loader").addClass("is-loaded");
        }, 60);

        $('.pxl-swiper-slider, .pxl-header-mobile-elementor').css('opacity', '1');
        $('.pxl-gallery-scroll').parents('body').addClass('body-overflow').addClass('body-visible-sm');

        pxl_scroll_top = $(window).scrollTop();
        pxl_window_width = $(window).width();
        pxl_window_height = $(window).height();

        dreamslab_header_sticky();
        dreamslab_header_mobile();
        dreamslab_menu_divider_move();
        dreamslab_scroll_to_top();
        dreamslab_footer_fixed();
        dreamslab_shop_quantity();
        dreamslab_check_scroll();
        dreamslab_submenu_responsive();
        dreamslab_panel_anchor_toggle();
        dreamslab_slider_column_offset();
        dreamslab_height_ct_grid();

        if (performanceSettings.enableParallax) {
            dreamslab_bgr_parallax();
        }

        dreamslab_shop_view_layout();
        dreamslab_fit_to_screen();

        if (performanceSettings.enableParallax) {
            dreamslab_el_parallax();
        }
    });

    // Optimized scroll handler for fast scrolling
    let isScrolling = false;
    let scrollTimeout;

    $(window).on('scroll', function () {
        pxl_scroll_top = $(window).scrollTop();
        pxl_window_height = $(window).height();
        pxl_window_width = $(window).width();

        if (pxl_scroll_top < pxl_last_scroll_top) {
            pxl_scroll_status = 'up';
        } else {
            pxl_scroll_status = 'down';
        }
        pxl_last_scroll_top = pxl_scroll_top;

        isScrolling = true;

        if (scrollTimeout) {
            clearTimeout(scrollTimeout);
        }

        if (scrollAnimationFrameId) {
            cancelAnimationFrame(scrollAnimationFrameId);
        }

        scrollAnimationFrameId = requestAnimationFrame(function () {
            dreamslab_header_sticky();
            dreamslab_scroll_to_top();

            if (pxl_scroll_top > 100) {
                dreamslab_footer_fixed();
                dreamslab_check_scroll();

                if (performanceSettings.enableScrollEffects) {
                    dreamslab_ptitle_scroll_opacity();
                }
            }

            if (pxl_scroll_top < 100) {
                $('.pxl-header-elementor-sticky').removeClass('pxl-header-fixed');
                $('#pxl-header-mobile').removeClass('pxl-header-mobile-fixed');
                $('.elementor > .pin-spacer').removeClass('scroll-top-active');
            }

            scrollAnimationFrameId = null;
        });

        scrollTimeout = setTimeout(function () {
            isScrolling = false;
        }, 150);
    });

    // Optimized resize handler
    $(window).on('resize', throttle(function () {
        pxl_window_height = $(window).height();
        pxl_window_width = $(window).width();

        if (resizeAnimationFrameId) {
            cancelAnimationFrame(resizeAnimationFrameId);
        }

        resizeAnimationFrameId = requestAnimationFrame(function () {
            dreamslab_submenu_responsive();
            dreamslab_header_mobile();
            dreamslab_header_sticky();

            setTimeout(function () {
                dreamslab_height_ct_grid();
                dreamslab_slider_column_offset();
                dreamslab_fit_to_screen();
                dreamslab_menu_divider_move();
            }, 300);

            resizeAnimationFrameId = null;
        });
    }, performanceSettings.resizeThrottle));

    $(document).ready(function () {
        pxl_window_width = $(window).width();

        dreamslab_backtotop_progess_bar();
        dreamslab_button_progess_bar();
        dreamslab_bgr_hv();
        dreamslab_type_file_upload();
        dreamslab_zoom_point();
        // dreamslab_scroll_checkp_blog();
        dreamslab_pagination_cf();

        if (pxl_window_width > 767) {
            dreamslab_button_parallax();
            HeightTitles();
        }

        dreamslab_cursor_hover();
        dreamslab_accordion_hover();

        let runningColumnAnimations = 0;
        const maxColumnAnimations = 2;

        function animateColumn(colId, speed) {
            if (runningColumnAnimations >= maxColumnAnimations) return;

            const $col = $('#' + colId);
            if (!$col.length) return;

            const slideHeight = $col.outerHeight() / 2;
            if (slideHeight <= 0) return;

            runningColumnAnimations++;

            gsap.to($col[0], {
                y: -slideHeight,
                ease: "none",
                duration: speed * 1.5,
                repeat: -1,
                modifiers: {
                    y: gsap.utils.unitize(y => parseFloat(y) % slideHeight)
                },
                onComplete: () => runningColumnAnimations--
            });
        }

        if (performanceSettings.enableSmoothAnimations) {
            setTimeout(() => {
                if ($('#col1').length) animateColumn("col1", 34);
            }, 100);

            setTimeout(() => {
                if ($('#col2').length) animateColumn("col2", 32);
            }, 200);

            setTimeout(() => {
                if ($('#col3').length) animateColumn("col3", 30);
            }, 300);
        }
        // --------------------------------------------- //

        // Split text animation
        $('[data-split]').each(function () {
            const $el = $(this);
            const text = $el.data('split');
            let spans = '';

            for (let i = 0; i < text.length; i++) {
                spans += '<span style="display:inline-block">' + text[i] + '</span>';
            }

            $el.html(spans);
        });

        // Animate text elements
        $('.animate-text').each(function () {
            const $el = $(this);

            if (performanceSettings.enableSmoothAnimations) {
                gsap.fromTo($el.find('span'),
                    { opacity: 0, y: 50 },
                    {
                        opacity: 1,
                        y: 0,
                        duration: performanceSettings.animationDuration * 2,
                        stagger: 0.05,
                        ease: "back.out(1.7)",
                        scrollTrigger: {
                            trigger: $el[0],
                            start: "top 85%"
                        }
                    }
                );
            } else {
                // Fallback for low-end devices
                $el.find('span').css({ opacity: 1, transform: 'translateY(0)' });
            }
        });

        // Menu Text Animation
        if (performanceSettings.enableSmoothAnimations) {
            $('.fr-style-hidden .pxl-menu-primary > li > a').each(function () {
                const text = $(this).text();
                const spans = [...text].map(char => {
                    return `<span>${char === ' ' ? '&nbsp;' : char}</span>`;
                }).join('');

                $(this).html(`<div class="menu-text">${spans}</div>`);
            });
        }

        // Menu toggle functionality
        $('#menuToggle').on('click', function () {
            $(this).toggleClass('active');

            const menu = $(this).closest('.fr-style-hidden')
                .find('.menu-menu-main-container');

            if ($(this).hasClass('active')) {
                menu.css('display', 'flex').hide().stop(true, true).slideDown(300);
            } else {
                menu.stop(true, true).slideUp(300);
            }
        });

        // Smooth scroll to anchor links
        $('a[href^="#"]:not(.tabs a)').on('click', function (e) {
            e.preventDefault();

            const target = $(this.getAttribute('href'));

            if (target.length) {
                if (deviceCapabilities.supportsSmoothScrolling()) {
                    // Use native smooth scrolling if available
                    target[0].scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                } else {
                    // Fallback to jQuery animation
                    $('html, body').animate(
                        {
                            scrollTop: target.offset().top
                        },
                        600
                    );
                }
            }
        });

        // 
        $(".pxl-process5 .pxl-images .pxl-last--image").on("mouseenter", function () {
            $(".pxl-process5 .pxl-images .pxl-last--image").removeClass("active");
            $(this).addClass("active");
        });

        // 
        var svgPaths = $('.pxl-icon-list.style-5 svg line, .pxl-icon-list.style-5 svg circle,.pxl-icon-box7 .pxl-item--icon svg path,.pxl-icon-box7 .pxl-item--icon svg line,.pxl-icon-box7 .pxl-item--icon svg rect,.pxl-icon-box7 .pxl-item--icon svg circle');
        var svgPaths_load = $('.pxl-icon-list.style-5 .animated svg path, .pxl-icon-list.style-5 .animated svg line, .pxl-icon-list.style-5 .animated svg circle');
        svgPaths.each(function () {
            var totalLength = this.getTotalLength();

            $(this).attr({
                'stroke-dashoffset': totalLength,
                'stroke-dasharray': totalLength,
            });
        });


        // Button Image Scroll
        var isSlidUp = false;

        $("#scrollButton").on('click', function () {

            var $targetImage = $(".targetImage");
            var imgHeight = $targetImage.height();
            var containerHeight = $(".pxl-image-single.button-scroll").height();

            var translateYValue = - (imgHeight - containerHeight);

            var newYValue = isSlidUp ? "0px" : translateYValue + "px";
            var buttonText = isSlidUp ? "Slide Image Up" : "Slide Image Down";
            var buttonClassRemove = isSlidUp ? "up" : "down";
            var buttonClassAdd = isSlidUp ? "down" : "up";

            gsap.to($targetImage, {
                y: newYValue,
                duration: 1,
                ease: "power2.out"
            });

            $("#scrollButton")
                .text(buttonText)
                .removeClass(buttonClassRemove)
                .addClass(buttonClassAdd);

            isSlidUp = !isSlidUp;
        });

        /* Custom One Page by theme */
        if ($('.pxl-link-scroll1').length) {
            $('.pxl-item--onepage').on('click', function (e) {
                var _this = $(this);
                var _link = $(this).attr('href');
                var _id_data = e.currentTarget.hash;
                var _offset;
                var _data_offset = $(this).attr('data-onepage-offset');
                if (_data_offset) {
                    _offset = _data_offset;
                } else {
                    _offset = 0;
                }
                if ($(_id_data).length === 1) {
                    var _target = $(_id_data);
                    $('.pxl-onepage-active').removeClass('pxl-onepage-active');
                    _this.addClass('pxl-onepage-active');
                    $('html, body').stop().animate({ scrollTop: _target.offset().top - _offset }, 1000);
                    return false;
                } else {
                    window.location.href = _link;
                }
                return false;
            });
            $.each($('.pxl-item--onepage'), function (index, item) {
                var target = $(item).attr('href');
                var el = $(target);
                var _data_offset = $(item).attr('data-onepage-offset');
                var waypoint = new Waypoint({
                    element: el[0],
                    handler: function (direction) {
                        if (direction === 'down') {
                            $('.pxl-onepage-active').removeClass('pxl-onepage-active');
                            $(item).addClass('pxl-onepage-active');
                        }
                        else if (direction === 'up') {
                            var prev = $(item).parent().prev().find('.pxl-item--onepage');
                            $(item).removeClass('pxl-onepage-active');
                            if (prev.length > 0)
                                prev.addClass('pxl-onepage-active');
                        }
                    },
                    offset: _data_offset,
                });
            });
        }

        // 
        $('.pxl-contact-form .cm-divider.your-service').on('click', function () {
            $('.pxl-contact-form .cm-divider').removeClass('active');
            $(this).addClass('active');
        });
        $('.pxl-counter7 .pxl-counter--holder').on('click', function () {
            $('.pxl-counter7 .pxl-counter--holder').removeClass('active');
            $(this).addClass('active');
        });
        // 
        $('.cm-divider input,.cm-divider textarea').on('focus', function () {
            $(this).closest('.cm-divider').addClass('focused');
        });
        $('.cm-divider input,.cm-divider textarea').on('blur', function () {
            $(this).closest('.cm-divider').removeClass('focused');
        });

        /* Start Menu Mobile */
        $('.pxl-header-menu li.menu-item-has-children').append('<span class="pxl-menu-toggle"></span>');
        $('.pxl-menu-toggle').on('click', function () {
            if ($(this).hasClass('active')) {
                $(this).closest('ul').find('.pxl-menu-toggle.active').toggleClass('active');
                $(this).closest('ul').find('.sub-menu.active').toggleClass('active').slideToggle();
            } else {
                $(this).closest('ul').find('.pxl-menu-toggle.active').toggleClass('active');
                $(this).closest('ul').find('.sub-menu.active').toggleClass('active').slideToggle();
                $(this).toggleClass('active');
                $(this).parent().find('> .sub-menu').toggleClass('active');
                $(this).parent().find('> .sub-menu').slideToggle();
            }
        });

        $("#pxl-nav-mobile, .pxl-anchor-mobile-menu").on('click', function () {
            $(this).toggleClass('active');
            $('body').toggleClass('body-overflow');
            $('.pxl-header-menu').toggleClass('active');
        });

        $(".pxl-menu-close, .pxl-header-menu-backdrop, #pxl-header-mobile .pxl-menu-primary a.is-one-page").on('click', function () {
            $(this).parents('.pxl-header-main').find('.pxl-header-menu').removeClass('active');
            $('#pxl-nav-mobile').removeClass('active');
            $('body').toggleClass('body-overflow');
        });
        /* End Menu Mobile */

        /* Custom Grid Filter Moving Border */
        if (performanceSettings.enableSmoothAnimations) {
            $('.pxl-grid-filter,.pxl-tabs--inner').each(function () {
                var marker = $(this).find('.filter-marker'),
                    item = $(this).find('.filter-item,.pxl-item--title'),
                    current = $(this).find('.filter-item.active,.pxl-item--title.active');

                var offsetTop = current.length ? current.position().top : 0;

                marker.css({
                    top: offsetTop + (current.length ? current.outerHeight() : 0),
                    left: current.length ? current.position().left : 0,
                    width: current.length ? current.outerWidth() : 0,
                    display: "block"
                });

                item.on('mouseover', function () {
                    var self = $(this),
                        offsetacTop = self.position().top,
                        offsetLeft = self.position().left,
                        width = self.outerWidth() || current.outerWidth(),
                        top = offsetacTop == 0 ? 0 : offsetacTop || offsetTop,
                        left = offsetLeft == 0 ? 0 : offsetLeft || current.position().left;

                    marker.stop().animate({
                        top: top + (current.length ? current.outerHeight() : 0),
                        left: left,
                        width: width,
                    }, 300);
                });

                item.on('click', function () {
                    current = $(this);
                });

                item.on('mouseleave', function () {
                    var offsetlvTop = current.length ? current.position().top : 0;
                    marker.stop().animate({
                        top: offsetlvTop + (current.length ? current.outerHeight() : 0),
                        left: current.length ? current.position().left : 0,
                        width: current.length ? current.outerWidth() : 0
                    }, 300);
                });
            });
        }


        $('.pxl-menu-hidden-sidebar .pxl-menu-hidden > li').on('mouseenter', function () {
            $(this).removeClass('hoverd').siblings().addClass('hoverd');
        });

        $('.pxl-menu-hidden-sidebar .pxl-menu-hidden > li').on('mouseleave', function () {
            $(this).removeClass('hoverd').siblings().removeClass('hoverd');
        });

        /* Menu Hidden Sidebar Popup */
        const $menuItems = $('.pxl-menu-hidden-sidebar li.menu-item-has-children > a');
        $menuItems.append('<span class="pxl-arrow-toggle"><i class="fas fa-chevron-right"></i></span>');

        $menuItems.on('mouseenter', function () {
            const $this = $(this);
            const $subMenu = $this.next('.sub-menu');
            const $closestUl = $this.closest('ul');
            const $activeSubMenu = $closestUl.find('.sub-menu.active');
            const $activeLink = $closestUl.find('a.active');

            if ($this.hasClass('active')) {
                return;
            }

            $activeSubMenu.removeClass('active').slideUp(300);
            $activeLink.removeClass('active');

            $this.addClass('active');
            $subMenu.addClass('active').slideDown(300);
        });

        $('.pxl-menu-hidden-sidebar li > a').on('mouseenter', function () {
            const $this = $(this);
            const $closestUl = $this.closest('ul');

            if (!$this.next('.sub-menu').length) {
                const $activeSubMenu = $closestUl.find('.sub-menu.active');
                const $activeLink = $closestUl.find('a.active');

                $activeSubMenu.removeClass('active').slideUp(300);
                $activeLink.removeClass('active');
            }
        });

        /*Start Menu popup */
        $('.pxl-menu-hidden-sidebar .pxl-menu-button').on('click', function () {
            const $menuSidebar = $(this).parents('.pxl-menu-hidden-sidebar');
            $menuSidebar.toggleClass('active');

            if ($menuSidebar.hasClass('active')) {
                const timeline = gsap.timeline({ ease: "power4.inOut" });
                // timeline.to('.pxl-menu-hidden > li', {  duration: 0.25, opacity: 1, y: -25, ease: Power2.out});
                timeline.to('.pxl-menu-popup-wrap .pxl-item--image', { duration: 0.25, opacity: 1, x: 0, ease: Power2.out });
                timeline.to('.pxl-menu-popup-wrap .pxl-item-content', { duration: 0.25, opacity: 1, x: -25, ease: Power2.out });
                $('body').toggleClass('body-overflow');
            }
        });

        $('.pxl-menu-popup-close').on('click', function () {
            const $menuSidebar = $(this).parents('.pxl-menu-hidden-sidebar');
            const timeline = gsap.timeline({ ease: "power2.inOut" });

            $menuSidebar.removeClass('active');
            $('body').removeClass('body-overflow');

            // timeline.to('.pxl-menu-hidden > li', {  duration: 0.25, opacity: 0, y: 25, ease: Power2.out});
            timeline.to('.pxl-menu-popup-wrap .pxl-item--image', { duration: 0.25, opacity: 0, x: 25, rotate: '0deg', ease: Power2.out });
            timeline.to('.pxl-menu-popup-wrap .pxl-item-content', { duration: 0.25, opacity: 0, x: 25, ease: Power2.out });
        });

        /*End Menu popup */

        $('.pxl-menu-popup-overlay').on('click', function () {
            const $parent = $(this).parent();
            $parent.removeClass('active').addClass('boxOut');
            $('body').removeClass('body-overflow');
        });

        $('.pxl-menu-hidden-sidebar .pxl-menu-hidden a.is-one-page').on('click', function () {
            const $menuSidebar = $(this).parents('.pxl-menu-hidden-sidebar');
            $menuSidebar.removeClass('active').addClass('boxOut');
            $('body').removeClass('body-overflow');
        });

        const $firstMenuItem = $('.pxl-menu-hidden-sidebar li.menu-item-has-children:first-child > a');
        $firstMenuItem.addClass('active');
        $firstMenuItem.next('.sub-menu').addClass('active').slideDown();

        /* Custom Effects */
        function setupPxlTransitions() {
            $('.pxl-parent-transition').each(function () {
                const $parent = $(this);
                const $transitions = $parent.find('.pxl-transtion');

                $transitions.addClass('pxl-hover-transition');

                $parent.on('mouseenter', function () {
                    $transitions.addClass('pxl-hover-transition');
                });

                $parent.find('.pxl-switch-button').on('mouseover', function () {
                    $transitions.removeClass('pxl-hover-transition');
                });
            });
        }

        /* Menu Vertical */
        $('.pxl-nav-vertical li.menu-item-has-children > a').append('<span class="pxl-arrow-toggle"><i class="fas fa-chevron-up"></i></span>');
        $('.pxl-nav-vertical li.menu-item-has-children > a').on('click', function () {
            if ($(this).hasClass('active')) {
                $(this).next().toggleClass('active').slideToggle();
            } else {
                $(this).closest('ul').find('.sub-menu.active').toggleClass('active').slideToggle();
                $(this).closest('ul').find('a.active').toggleClass('active');
                $(this).find('.pxl-menu-toggle.active').toggleClass('active');
                $(this).toggleClass('active');
                $(this).next().toggleClass('active').slideToggle();
            }
        });

        $(".comments-area .btn-submit").append('<i class="fas fa-comment"></i>');
        /* Mega Menu Max Height */
        var m_h_mega = $('li.pxl-megamenu > .sub-menu > .pxl-mega-menu-elementor').outerHeight();
        var w_h_mega = $(window).height();
        var w_h_mega_css = w_h_mega - 120;
        if (m_h_mega > w_h_mega) {
            $('li.pxl-megamenu > .sub-menu > .pxl-mega-menu-elementor').css('max-height', w_h_mega_css + 'px');
            $('li.pxl-megamenu > .sub-menu > .pxl-mega-menu-elementor').css('overflow-x', 'scroll');
        }
        // Active Mega Menu Hover
        $(document).on({
            mouseenter: function () {
                $(this).parents('.elementor-element').addClass('section-mega-active');
            },
            mouseleave: function () {
                $(this).parents('.elementor-element').removeClass('section-mega-active');
            }
        }, 'li.pxl-megamenu');
        /* End Mega Menu Max Height */
        /* Search Popup */
        var $search_wrap_init = $("#pxl-search-popup");
        var search_field = $('#pxl-search-popup .search-field');
        var $body = $('body');

        $(".pxl-search-popup-button").on('click', function (e) {
            if (!$search_wrap_init.hasClass('active')) {
                $search_wrap_init.addClass('active');
                setTimeout(function () { search_field.get(0).focus(); }, 500);
            } else if (search_field.val() === '') {
                $search_wrap_init.removeClass('active');
                search_field.get(0).focus();
            }
            e.preventDefault();
            return false;
        });

        $(".pxl-subscribe-popup .pxl-item--overlay, .pxl-subscribe-popup .pxl-item--close").on('click', function (e) {
            $(this).parents('.pxl-subscribe-popup').removeClass('pxl-active');
            e.preventDefault();
            return false;
        });

        $("#pxl-search-popup .pxl-item--overlay, #pxl-search-popup .pxl-item--close").on('click', function (e) {
            $body.addClass('pxl-search-out-anim');
            setTimeout(function () {
                $body.removeClass('pxl-search-out-anim');
            }, 800);
            setTimeout(function () {
                $search_wrap_init.removeClass('active');
            }, 800);
            e.preventDefault();
            return false;
        });

        /* Scroll To Top */

        $('.pxl-scroll-top').on('click', function () {
            $('html, body').animate({ scrollTop: 0 }, 1200);
            $(this).parents('.pxl-wapper').find('.elementor > .pin-spacer').addClass('scroll-top-active');
            return false;
        });

        /* Animate Time Delay */

        $('.pxl-grid-masonry').each(function () {
            var eltime = 80;
            var elt_inner = $(this).children().length;
            var _elt = elt_inner - 1;
            $(this).find('> .pxl-grid-item > .wow').each(function (index, obj) {
                $(this).css('animation-delay', eltime + 'ms');
                if (_elt === index) {
                    eltime = 80;
                    _elt = _elt + elt_inner;
                } else {
                    eltime = eltime + 80;
                }
            });
        });

        $('.btn-text-nina').each(function () {
            var eltime = 0.045;
            var elt_inner = $(this).children().length;
            var _elt = elt_inner - 1;
            $(this).find('> .pxl--btn-text > span').each(function (index, obj) {
                $(this).css('transition-delay', eltime + 's');
                eltime = eltime + 0.045;
            });
        });

        $('.btn-text-nanuk').each(function () {
            var eltime = 0.05;
            var elt_inner = $(this).children().length;
            var _elt = elt_inner - 1;
            $(this).find('> .pxl--btn-text > span').each(function (index, obj) {
                $(this).css('animation-delay', eltime + 's');
                eltime = eltime + 0.05;
            });
        });

        $('.btn-text-smoke').each(function () {
            var eltime = 0.05;
            var elt_inner = $(this).children().length;
            var _elt = elt_inner - 1;
            $(this).find('> .pxl--btn-text > span > span > span').each(function (index, obj) {
                $(this).css('--d', eltime + 's');
                eltime = eltime + 0.05;
            });
        });

        $('.btn-text-reverse .pxl-text--front, .btn-text-reverse .pxl-text--back').each(function () {
            var eltime = 0.05;
            var elt_inner = $(this).children().length;
            var _elt = elt_inner - 1;
            $(this).find('.pxl-text--inner > span').each(function (index, obj) {
                $(this).css('transition-delay', eltime + 's');
                eltime = eltime + 0.05;
            });
        });

        /* End Animate Time Delay */

        /* Lightbox Popup */
        $('.pxl-action-popup').magnificPopup({
            type: 'iframe',
            mainClass: 'mfp-fade',
            removalDelay: 160,
            preloader: false,
            fixedContentPos: false
        });

        /* Page Title Parallax */
        if (pxl_window_width > 767) {
            if ($('#pxl-page-title-default').hasClass('pxl--parallax')) {
                $(this).stellar();
            }
        }

        /* Cart Sidebar Popup */
        $(".pxl-cart-sidebar-button").on('click', function () {
            $('body').addClass('body-overflow');
            $('#pxl-cart-sidebar').addClass('active');
        });
        $("#pxl-cart-sidebar .pxl-popup--overlay, #pxl-cart-sidebar .pxl-item--close").on('click', function () {
            $('body').removeClass('body-overflow');
            $('#pxl-cart-sidebar').removeClass('active');
        });
        $(".pxl-accordion1.style2 .pxl-accordion--content").find("br").remove();

        /* Start Icon Bounce */
        var boxEls = $('.el-bounce, .pxl-image-effect1, .el-effect-zigzag');
        $.each(boxEls, function (boxIndex, boxEl) {
            loopToggleClass(boxEl, 'active');
        });

        function loopToggleClass(el, toggleClass) {
            el = $(el);
            let counter = 0;
            if (el.hasClass(toggleClass)) {
                waitFor(function () {
                    counter++;
                    return counter == 2;
                }, function () {
                    counter = 0;
                    el.removeClass(toggleClass);
                    loopToggleClass(el, toggleClass);
                }, 'Deactivate', 1000);
            } else {
                waitFor(function () {
                    counter++;
                    return counter == 3;
                }, function () {
                    counter = 0;
                    el.addClass(toggleClass);
                    loopToggleClass(el, toggleClass);
                }, 'Activate', 1000);
            }
        }

        function closestEdge(x, y, w, h) {
            const topEdgeDist = distMetric(x, y, w / 2, 0);
            const bottomEdgeDist = distMetric(x, y, w / 2, h);
            const min = Math.min(topEdgeDist, bottomEdgeDist);
            return min === topEdgeDist ? 'top' : 'bottom';
        }

        function distMetric(x, y, x2, y2) {
            const xDiff = x - x2;
            const yDiff = y - y2;
            return (xDiff * xDiff) + (yDiff * yDiff);
        }

        function waitFor(condition, callback, message, time) {
            if (message == null || message == '' || typeof message == 'undefined') {
                message = 'Timeout';
            }
            if (time == null || time == '' || typeof time == 'undefined') {
                time = 100;
            }
            var cond = condition();
            if (cond) {
                callback();
            } else {
                setTimeout(function () {
                    waitFor(condition, callback, message, time);
                }, time);
            }
        }
        /* End Icon Bounce */

        /* Image Effect */
        if ($('.pxl-image-tilt').length) {
            $('.pxl-image-tilt').parents('.elementor-top-section').addClass('pxl-image-tilt-active');
            $('.pxl-image-tilt').each(function () {
                var pxl_maxtilt = $(this).data('maxtilt'),
                    pxl_speedtilt = $(this).data('speedtilt'),
                    pxl_perspectivetilt = $(this).data('perspectivetilt');
                VanillaTilt.init(this, {
                    max: pxl_maxtilt,
                    speed: pxl_speedtilt,
                    perspective: pxl_perspectivetilt
                });
            });
        }

        /* Select Theme Style */
        $('.wpcf7-select').each(function () {
            var $this = $(this), numberOfOptions = $(this).children('option').length;

            $this.addClass('pxl-select-hidden');
            $this.wrap('<div class="pxl-select"></div>');
            $this.after('<div class="pxl-select-higthlight"></div>');

            var $styledSelect = $this.next('div.pxl-select-higthlight');
            $styledSelect.text($this.children('option').eq(0).text());

            var $list = $('<ul />', {
                'class': 'pxl-select-options'
            }).insertAfter($styledSelect);

            for (var i = 0; i < numberOfOptions; i++) {
                $('<li />', {
                    text: $this.children('option').eq(i).text(),
                    rel: $this.children('option').eq(i).val()
                }).appendTo($list);
            }

            var $listItems = $list.children('li');

            $styledSelect.on('click', function (e) {
                e.stopPropagation();
                $('div.pxl-select-higthlight.active').not(this).each(function () {
                    $(this).removeClass('active').next('ul.pxl-select-options').addClass('pxl-select-lists-hide');
                });
                $(this).toggleClass('active');
            });

            $listItems.on('click', function (e) {
                e.stopPropagation();
                $styledSelect.text($(this).text()).removeClass('active');
                $this.val($(this).attr('rel'));
            });

            $(document).on('click', function () {
                $styledSelect.removeClass('active');
            });

        });

        /* Nice Select */
        $('.woocommerce-ordering .orderby, #pxl-sidebar-area select, .variations_form.cart .variations select, .pxl-open-table select, .pxl-nice-select').each(function () {
            $(this).niceSelect();
        });

        $('.pxl-post-list .nice-select').each(function () {
            $(this).niceSelect();
        });

        /* Typewriter */
        if ($('.pxl-title--typewriter').length) {
            function typewriterOut(elements, callback) {
                if (elements.length) {
                    elements.eq(0).addClass('is-active');
                    elements.eq(0).delay(3000);
                    elements.eq(0).removeClass('is-active');
                    typewriterOut(elements.slice(1), callback);
                }
                else {
                    callback();
                }
            }

            function typewriterIn(elements, callback) {
                if (elements.length) {
                    elements.eq(0).addClass('is-active');
                    elements.eq(0).delay(3000).slideDown(3000, function () {
                        elements.eq(0).removeClass('is-active');
                        typewriterIn(elements.slice(1), callback);
                    });
                }
                else {
                    callback();
                }
            }

            function typewriterInfinite() {
                typewriterOut($('.pxl-title--typewriter .pxl-item--text'), function () {
                    typewriterIn($('.pxl-title--typewriter .pxl-item--text'), function () {
                        typewriterInfinite();
                    });
                });
            }
            $(function () {
                typewriterInfinite();
            });
        }
        /* End Typewriter */

        /* Section Particles */
        let particlesLoaded = false;
        setTimeout(function () {
            if (particlesLoaded) return;

            const $particleRows = $(".pxl-row-particles");
            if (!$particleRows.length) return;

            $particleRows.each(function () {
                const $this = $(this);
                const particleId = $this.attr('id');
                if (!particleId) return;

                const particleData = {
                    "particles": {
                        "number": {
                            "value": $this.data('number'),
                        },
                        "color": {
                            "value": $this.data('color')
                        },
                        "shape": {
                            "type": "circle",
                        },
                        "size": {
                            "value": $this.data('size'),
                            "random": $this.data('size-random'),
                        },
                        "line_linked": {
                            "enable": false,
                        },
                        "move": {
                            "enable": true,
                            "speed": 2,
                            "direction": $this.data('move-direction'),
                            "random": true,
                            "out_mode": "out",
                        }
                    },
                    "retina_detect": true
                };

                try {
                    if (typeof particlesJS === 'function') {
                        particlesJS(particleId, particleData);
                    }
                } catch (e) {
                    safeWarn('Particles.js failed to load:', e);
                }
            });

            particlesLoaded = true;
        }, 400);

        /* Get checked input - Mailchimpp */
        $('.mc4wp-form input:checkbox').on('change', function () {
            if ($(this).is(":checked")) {
                $('.mc4wp-form').addClass("pxl-input-checked");
            } else {
                $('.mc4wp-form').removeClass("pxl-input-checked");
            }
        });

        /* Scroll to content */
        $('.pxl-link-to-section .btn').on('click', function (e) {
            var id_scroll = $(this).attr('href');
            var offsetScroll = $('.pxl-header-elementor-sticky').outerHeight();
            e.preventDefault();
            $("html, body").animate({ scrollTop: $(id_scroll).offset().top - offsetScroll }, 600);
        });

        // Hover Item Active
        $(".pxl-post-modern1 .pxl-post--content .pxl-post--item")
            .on("mouseenter", function () {
                $(this).addClass("active");
                $(".pxl-post-modern1 .pxl-post--images .pxl-post--featured").removeClass('active');
                var selected_item = $(this).find(".pxl-content--inner").attr("data-image");
                $(selected_item).addClass('active').removeClass('non-active');
            })
            .on("mouseleave", function () {
                $(".pxl-post-modern1 .pxl-post--content .pxl-post--item").removeClass('active');
                $(".pxl-post-modern1 .pxl-post--images .pxl-post--featured").removeClass('non-active');
                var selected_item = $(this).find(".pxl-content--inner").attr("data-image");
                $(selected_item).removeClass('active').addClass('non-active');
            }
            );

        // Hover Overlay Effect
        $('.pxl-overlay-shake').on('mousemove', function (event) {
            var offset = $(this).offset();
            var W = $(this).outerWidth();
            var X = (event.pageX - offset.left);
            var Y = (event.pageY - offset.top);
            $(this).find('.pxl-overlay--color').css({
                'top': + Y + 'px',
                'left': + X + 'px'
            });
        });

        /* Social Button Click */
        $('.pxl-social--button').on('click', function () {
            $(this).toggleClass('active');
        });
        $(document).on('click', function (e) {
            if (e.target.className == 'pxl-social--button active')
                $('.pxl-social--button').removeClass('active');
        });

        // Header Home 2
        $('#home-2-header').append('<span class="pxl-header-divider1"></span><span class="pxl-header-divider2"></span><span class="pxl-header-divider3"></span><span class="pxl-header-divider4"></span>');
        $('#home-2-header-sticky').append('<span class="pxl-header-divider2"></span><span class="pxl-header-divider4"></span>');

    });

    jQuery(document).ajaxComplete(function (event, xhr, settings) {
        dreamslab_shop_quantity();
        dreamslab_height_ct_grid();
        dreamslab_bgr_hv();
    });

    jQuery(document).on('updated_wc_div', function () {
        dreamslab_shop_quantity();
    });

    /**
     * Header Sticky Function
     * Handles sticky header behavior with performance optimizations
     */
    function dreamslab_header_sticky() {
        let isInViewport = false;

        // Check if portfolio carousel is in viewport
        $('.pxl-portfolio-carousel5').each(function () {
            const rect = this.getBoundingClientRect();
            if (rect.bottom >= 0 && rect.top <= window.innerHeight) {
                isInViewport = true;
                return false;
            }
        });

        // Check if slip elements are in viewport
        $('.pxl-tabs-slip, .pxl-images-slip, .pxl-texts-slip').each(function () {
            const rect = this.getBoundingClientRect();
            if (rect.bottom >= 0 && rect.top <= window.innerHeight) {
                isInViewport = true;
                return false;
            }
        });

        if (isInViewport) {
            return;
        }

        if ($('#pxl-header-elementor').hasClass('is-sticky')) {
            // For Safari, use CSS transform instead of adding/removing classes to avoid jitter
            if (deviceCapabilities.isSafari()) {
                const $stickyHeader = $('.pxl-header-elementor-sticky.pxl-sticky-stb');
                const $mobileHeader = $('#pxl-header-mobile');

                if (pxl_scroll_top > 100) {
                    if (!$stickyHeader.hasClass('pxl-header-fixed')) {
                        $stickyHeader.addClass('pxl-header-fixed');
                        $mobileHeader.addClass('pxl-header-mobile-fixed');
                    }
                } else {
                    if ($stickyHeader.hasClass('pxl-header-fixed')) {
                        $stickyHeader.removeClass('pxl-header-fixed');
                        $mobileHeader.removeClass('pxl-header-mobile-fixed');
                    }
                }

                if (pxl_scroll_status === 'up' && pxl_scroll_top > 100) {
                    $('.pxl-header-elementor-sticky.pxl-sticky-stt').addClass('pxl-header-fixed');
                } else {
                    $('.pxl-header-elementor-sticky.pxl-sticky-stt').removeClass('pxl-header-fixed');
                }
            } else {
                // Original logic for other browsers
                if (pxl_scroll_top > 100) {
                    $('.pxl-header-elementor-sticky.pxl-sticky-stb').addClass('pxl-header-fixed');
                    $('#pxl-header-mobile').addClass('pxl-header-mobile-fixed');
                } else {
                    $('.pxl-header-elementor-sticky.pxl-sticky-stb').removeClass('pxl-header-fixed');
                    $('#pxl-header-mobile').removeClass('pxl-header-mobile-fixed');
                }

                if (pxl_scroll_status === 'up' && pxl_scroll_top > 100) {
                    $('.pxl-header-elementor-sticky.pxl-sticky-stt').addClass('pxl-header-fixed');
                } else {
                    $('.pxl-header-elementor-sticky.pxl-sticky-stt').removeClass('pxl-header-fixed');
                }
            }
        }

        $('.pxl-header-elementor-sticky').parents('body').addClass('pxl-header-sticky');
    }

    /**
     * Header Mobile Function
     * Handles mobile header height adjustments
     */
    function dreamslab_header_mobile() {
        const h_header_mobile = $('#pxl-header-elementor').outerHeight();

        if (pxl_window_width < 1199) {
            $('#pxl-header-elementor').css('min-height', h_header_mobile + 'px');
        }
    }

    /**
     * Scroll To Top Function
     * Controls scroll-to-top button visibility
     */
    function dreamslab_scroll_to_top() {
        if (pxl_scroll_top < pxl_window_height) {
            $('.pxl-scroll-top,.pxl-link-scroll1').addClass('pxl-off').removeClass('pxl-on');
        }

        if (pxl_scroll_top > pxl_window_height) {
            $('.pxl-scroll-top,.pxl-link-scroll1').addClass('pxl-on').removeClass('pxl-off');
        }
    }

    /**
     * Footer Fixed Function
     * Adjusts main content margin for fixed footer
     */
    function dreamslab_footer_fixed() {
        setTimeout(function () {
            const h_footer = $('.pxl-footer-fixed #pxl-footer-elementor').outerHeight() - 1;
            $('.pxl-footer-fixed #pxl-main').css('margin-bottom', h_footer + 'px');
        }, 600);
    }

    /**
     * Custom Check Scroll Function
     * Handles scroll-based visibility for grid items
     */
    function dreamslab_check_scroll() {
        const $gridItems = $('.pxl-check-scroll .list-item,.pxl-check-scroll .pxl-grid-item');
        if (!$gridItems.length) return;

        // Use Intersection Observer if available for better performance
        if ('IntersectionObserver' in window) {
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    const $item = $(entry.target);
                    if (entry.isIntersecting) {
                        $item.addClass('visible');
                    } else {
                        $item.removeClass('visible');
                    }
                });
            }, {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            });

            $gridItems.each(function () {
                observer.observe(this);
            });
        } else {
            // Fallback for older browsers
            const viewportBottom = pxl_scroll_top + $(window).height();

            $gridItems.each(function () {
                const $gridItem = $(this);
                const elementTop = $gridItem.offset().top;
                const elementBottom = elementTop + $gridItem.outerHeight();

                if (elementTop < viewportBottom && elementBottom > pxl_scroll_top) {
                    $gridItem.addClass('visible');
                } else {
                    $gridItem.removeClass('visible');
                }
            });
        }
    }


    /* WooComerce Quantity */
    function dreamslab_shop_quantity() {
        "use strict";
        $('#pxl-wapper .quantity').append('<span class="quantity-icon quantity-down pxl-icon--minus"></span><span class="quantity-icon quantity-up pxl-icon--plus"></span>');
        $('.quantity-up').on('click', function () {
            $(this).parents('.quantity').find('input[type="number"]').get(0).stepUp();
            $(this).parents('.woocommerce-cart-form').find('.actions .button').removeAttr('disabled');
        });
        $('.quantity-down').on('click', function () {
            $(this).parents('.quantity').find('input[type="number"]').get(0).stepDown();
            $(this).parents('.woocommerce-cart-form').find('.actions .button').removeAttr('disabled');
        });
        $('.quantity-icon').on('click', function () {
            var quantity_number = $(this).parents('.quantity').find('input[type="number"]').val();
            var add_to_cart_button = $(this).parents(".product, .woocommerce-product-inner").find(".add_to_cart_button");
            add_to_cart_button.attr('data-quantity', quantity_number);
            add_to_cart_button.attr("href", "?add-to-cart=" + add_to_cart_button.attr("data-product_id") + "&quantity=" + quantity_number);
        });
        $('.woocommerce-cart-form .actions .button').removeAttr('disabled');
    }

    /* Menu Responsive Dropdown */
    function dreamslab_submenu_responsive() {
        var $dreamslab_menu = $('.pxl-header-elementor-main, .pxl-header-elementor-sticky');
        $dreamslab_menu.find('.pxl-menu-primary li').each(function () {
            var $dreamslab_submenu = $(this).find('> ul.sub-menu');
            if ($dreamslab_submenu.length == 1) {
                if (($dreamslab_submenu.offset().left + $dreamslab_submenu.width() + 0) > $(window).width()) {
                    $dreamslab_submenu.addClass('pxl-sub-reverse');
                }
            }
        });
    }

    function dreamslab_panel_anchor_toggle() {
        'use strict';
        $(document).on('click', '.pxl-anchor-button', function (e) {
            e.preventDefault();
            e.stopPropagation();
            var target = $(this).attr('data-target');
            $(target).toggleClass('active');
            $('body').addClass('body-overflow');
            $('.pxl-popup--conent .wow').addClass('animated').removeClass('aniOut');
            $('.pxl-popup--conent .fadeInPopup').removeClass('aniOut');
            if ($(target).find('.pxl-search-form').length > 0) {
                setTimeout(function () {
                    $(target).find('.pxl-search-form .pxl-search-field').focus();
                }, 1000);
            }
        });

        $('.pxl-post-taxonomy .pxl-count').each(function () {
            var content = $(this).html();
            if (content) {
                var newContent = content.replace('(', '');
                var newContent2 = newContent.replace(')', '');
                $(this).html(newContent2);
            }
        });

        $(document).on('click', '.pxl-button.pxl-atc-popup > .btn ,.pxl-text-carousel8 .pxl-atc-popup .btn-overlay ,.pxl-icon1 .pxl-icon--popup,.pxl-info-box3 .btn-atc-popup.btn-overlay, .pxl-team-grid .pxl-item--image > a', function (e) {
            $('.pxl-page-popup').removeClass('active');
            e.preventDefault();
            e.stopPropagation();
            var target = $(this).attr('data-target');
            $(target).toggleClass('active');
            $('body').addClass('body-overflow');
        });

        $('.pxl-anchor-button').each(function () {
            var t_target = $(this).attr('data-target');
            var t_delay = $(this).attr('data-delay-hover');
            $(t_target).find('.pxl-popup--conent').css('transition-delay', t_delay + 'ms');
            $(t_target).find('.pxl-popup--overlay').css('transition-delay', t_delay + 'ms');
        });

        $(".pxl-hidden-panel-popup .pxl-popup--overlay, .pxl-hidden-panel-popup .pxl-close-popup").on('click', function () {
            $('body').removeClass('body-overflow');
            $('.pxl-hidden-panel-popup').removeClass('active');
            $('.pxl-popup--conent .wow').addClass('aniOut').removeClass('animated');
            $('.pxl-popup--conent .fadeInPopup').addClass('aniOut');
        });

        $(".pxl-icon-box6 .btn-show-more").on('click', function () {
            $(this).parents('.pxl-icon-box6').addClass('active');
            $(this).parents('.pxl-icon-box6').find('.content-2').addClass('active');
        });


        $(".pxl-popup--close").on('click', function () {
            $('body').removeClass('body-overflow');
            $(this).parent().removeClass('active');
        });
        $(".pxl-close-popup").on('click', function () {
            $('body').removeClass('body-overflow');
            $('.pxl-page-popup').removeClass('active');
        });
    }

    /* Page Title Scroll Opacity */
    function dreamslab_ptitle_scroll_opacity() {
        var divs = $('#pxl-page-title-elementor.pxl-scroll-opacity .elementor-widget'),
            limit = $('#pxl-page-title-elementor.pxl-scroll-opacity').outerHeight();
        if (pxl_scroll_top <= limit) {
            divs.css({ 'opacity': (1 - pxl_scroll_top / limit) });
        }
    }

    function HeightTitles() {
        const titleSelector = '.pxl-heading .pxl-item--title.style-caption-timeline';
        if (document.querySelectorAll(titleSelector).length == 0) {
            return;
        }

        function generateSpans(selector) {
            document.querySelectorAll(selector).forEach((el) => {
                const words = el.textContent.trim().split(' ');
                el.innerHTML = words.map((word, wi) => {
                    const chars = [...word].map(c => `<span>${c}</span>`).join('');
                    const space = wi < words.length - 1 ? '<div><span></span></div>' : '';
                    return `<div>${chars}</div>${space}`;
                }).join('');
            });
        }

        function applyHoverEffect(selector) {
            const spans = document.querySelectorAll(selector);
            let ticking = false;

            spans.forEach(span => {
                span.originalScaleY = 1;
                span.addEventListener('mousemove', (e) => {
                    if (!ticking) {
                        window.requestAnimationFrame(() => handleHoverEffect(e, spans));
                        ticking = true;
                    }
                });

                span.addEventListener('mouseleave', () => {
                    spans.forEach(s => {
                        gsap.to(s, {
                            scaleY: s.originalScaleY,
                            duration: 0.4,
                            ease: 'power4.out'
                        });
                    });
                });
            });

            function handleHoverEffect(e, allSpans) {
                ticking = false;
                const hovered = e.target;
                const rect = hovered.getBoundingClientRect();
                const mouseX = e.clientX - rect.left;
                const center = rect.width / 2;
                const scaleFactor = 0.2;

                const scale = (scaleFactor + 1) + (scaleFactor * (center - Math.abs(center - mouseX))) / center;

                gsap.to(hovered, {
                    scaleY: scale,
                    duration: 0.4,
                    ease: 'power4.out'
                });

                const index = Array.from(allSpans).indexOf(hovered);
                [index - 1, index + 1].forEach(i => {
                    const neighbor = allSpans[i];
                    if (neighbor) {
                        const d = Math.min(Math.abs(e.clientX - neighbor.getBoundingClientRect().left), center);
                        const neighborScale = 1 + (scaleFactor * (center - d)) / center;
                        gsap.to(neighbor, {
                            scaleY: neighborScale,
                            duration: 0.4,
                            ease: 'power4.out'
                        });
                    }
                });
            }
        }

        // 3. Animation xuất hiện ban đầu
        function animateIntro(selector) {
            const spans = document.querySelectorAll(selector);
            if (!spans.length) return;

            gsap.set(spans, { scaleY: 0.3, opacity: 0, y: 0 });

            const shuffled = [...spans].sort(() => Math.random() - 0.5);
            const tl = gsap.timeline({ delay: 0.1 });

            gsap.set(".ani-load", {
                scale: 1.09,
                filter: "blur(5px)",
                opacity: 1
            });

            tl.to(".ani-load", {
                duration: 1,
                scale: 1,
                filter: "blur(0px)",
                ease: "expo.out"
            }, 0.3);

            shuffled.forEach((span, i) => {
                tl.to(span, {
                    scaleY: 1,
                    opacity: 1,
                    y: 0,
                    duration: 0.32,
                    ease: 'power2.out',
                }, "-=0.3" + `+${i * 0.015}`);
            });
        }

        generateSpans(titleSelector);
        applyHoverEffect(`${titleSelector} span`);
        animateIntro(`${titleSelector} span`);
    }



    /* Slider Column Offset */
    function dreamslab_slider_column_offset() {
        var content_w = ($('#pxl-main').width() - 1200) / 2;
        if (pxl_window_width > 1200) {
            $('.pxl-slider2 .pxl-item--left').css('padding-left', content_w + 'px');
        }
    }

    /* Preloader Default */
    $.fn.extend({
        jQueryImagesLoaded: function () {
            var $imgs = this.find('img[src!=""]')

            if (!$imgs.length) {
                return $.Deferred()
                    .resolve()
                    .promise()
            }

            var dfds = []

            $imgs.each(function () {
                var dfd = $.Deferred()
                dfds.push(dfd)
                var img = new Image()
                img.onload = function () {
                    dfd.resolve()
                }
                img.onerror = function () {
                    dfd.resolve()
                }
                img.src = this.src
            })

            return $.when.apply($, dfds)
        }
    })

    function dreamslab_bgr_parallax() {
        if (!performanceSettings.enableParallax) return;

        setTimeout(function () {
            $('.pxl-section-bg-parallax').each(function () {
                if (!$(this).hasClass('pinned-zoom-clipped') && !$(this).hasClass('pinned-circle-zoom-clipped') && !$(this).hasClass('mask-parallax')) {
                    try {
                        jarallax(this, {
                            speed: deviceCapabilities.isLowEnd() ? 0.1 : 0.2,
                        });
                    } catch (e) {
                        safeWarn('Jarallax failed to initialize:', e);
                    }
                }
            });
        }, 300);
    }

    function dreamslab_el_parallax() {
        if (!performanceSettings.enableParallax) return;

        const $parallaxWraps = $('.el-parallax-wrap');
        if (!$parallaxWraps.length) return;

        $parallaxWraps.on({
            mouseenter: function () {
                const $this = $(this);
                $this.addClass('hovered');
                $this.find('.el-parallax-item').css({
                    transition: 'none'
                });
            },
            mouseleave: function () {
                const $this = $(this);
                $this.removeClass('hovered');

                const transformValue = performanceSettings.useTransform3d ?
                    'translate3d(0px, 0px, 0px)' : 'translate(0px, 0px)';

                $this.find('.el-parallax-item').css({
                    transition: `transform ${performanceSettings.animationDuration}s ease`,
                    transform: transformValue
                });
            },
            mousemove: throttle(function (e) {
                const $this = $(this);
                const bounds = this.getBoundingClientRect();
                const centerX = bounds.left + bounds.width / 2;
                const centerY = bounds.top + bounds.height / 2;
                const deltaX = (centerX - e.clientX) * (deviceCapabilities.isLowEnd() ? 0.035 : 0.07104);
                const deltaY = (centerY - e.clientY) * (deviceCapabilities.isLowEnd() ? 0.053 : 0.10656);

                const transformValue = performanceSettings.useTransform3d ?
                    `translate3d(${deltaX}px, ${deltaY}px, 0px)` : `translate(${deltaX}px, ${deltaY}px)`;

                if (deviceCapabilities.isSafari()) {
                    $this.find('.el-parallax-item').css({
                        transform: transformValue
                    });
                } else {
                    requestAnimationFrame(() => {
                        $this.find('.el-parallax-item').css({
                            transform: transformValue
                        });
                    });
                }
            }, deviceCapabilities.isLowEnd() ? 32 : 16)
        });
    }

    function dreamslab_accordion_hover() {
        const $nav = $('.pxl-accordion1');

        $nav.each(function () {
            const $elementor = $(this).parents('.elementor-element').last();
            const $slides = $elementor.find('.pxl-image-carousel8.default .pxl-swiper-thumbs .pxl-swiper-slide');
            const $slides_on = $elementor.find('.pxl-image-carousel8.hover-image .pxl-swiper-slide .pxl-item--inner');
            const $accordion = $(this);
            const $items = $accordion.find('.pxl--item');

            if ($slides.length > 0) {
                $items.on('click', function () {
                    const navId = $(this).attr('id');
                    $items.removeClass('active').parent().removeClass('active');
                    $(this).addClass('active').parent().addClass('active');

                    $slides.each(function () {
                        if ($(this).attr('id') === navId) {
                            $(this).trigger('click');
                        }
                    });
                });

            } else if ($slides_on.length > 0) {
                $items.on('click', function () {
                    const navId = $(this).attr('id');
                    $items.removeClass('active').parent().removeClass('active');
                    $(this).addClass('active').parent().addClass('active');
                    $slides_on.removeClass('active');
                    $slides_on.filter('#' + navId).addClass('active');
                });
            }
        });
    }

    /* Button Parallax */
    function dreamslab_button_parallax() {
        // Disable button parallax on Safari to avoid jitter
        if (deviceCapabilities.isSafari()) return;

        const $buttons = $(
            '.btn:not(.btn-stroke).btn-icon-box-hover, ' +
            '.pxl-icon1.style-box-paralax a, ' +
            '.pxl-search-popup-button, ' +
            '.pxl-menu-button, ' +
            '.pxl-menu-hidden-sidebar .pxl-item--social > a'
        );
        if ($buttons.length === 0) return;

        const textSelector = 'i, svg:not(.pxl-scroll-progress-circle), span, .pxl-counter--title, .language, .pxl-anchor-divider, .pxl-anchor--title';

        $buttons.on('mouseenter', function () {
            const $text = $(this).find(textSelector);
            if ($text.length) {
                gsap.set($text, { transformOrigin: "50% 50%" });
            }
            // Cache bounds to avoid calling getBoundingClientRect multiple times
            this._bounds = this.getBoundingClientRect();
        });

        $buttons.on('mousemove', throttle(function (e) {
            const $btn = $(this);
            const $text = $btn.find(textSelector);
            if ($text.length === 0) return;

            const bounds = this._bounds || this.getBoundingClientRect();
            const centerX = bounds.left + bounds.width / 2;
            const centerY = bounds.top + bounds.height / 2;

            const deltaX = (e.clientX - centerX) * 0.444;
            const deltaY = (e.clientY - centerY) * 0.444;

            gsap.killTweensOf([$btn, $text]);
            gsap.to($btn.add($text), {
                duration: performanceSettings.animationDuration * 1.6,
                x: deltaX,
                y: deltaY,
                ease: "power3.out"
            });
        }, deviceCapabilities.isLowEnd() ? 32 : 16)); // Throttle based on device capabilities

        $buttons.on('mouseleave', function () {
            const $btn = $(this);
            const $text = $btn.find(textSelector);
            gsap.killTweensOf([$btn, $text]);
            gsap.to($btn.add($text), {
                duration: performanceSettings.animationDuration * 1.6,
                x: 0,
                y: 0,
                ease: "elastic.out(1, 0.3)"
            });
            // Clear cached bounds
            delete this._bounds;
        });
    }


    function dreamslab_bgr_hv() {
        const $slides = $('.pxl-portfolio-carousel2 .pxl-swiper-slide .pxl-post--inner');
        const $backgroundChanger = $('.bgr-change');

        $slides.each(function () {
            const bg = extractBackgroundImage($(this).css('background-image'));
            $backgroundChanger.css('background-image', `url(${bg})`);
        });

        $(document).on("mouseenter", ".pxl-portfolio-carousel2 .pxl-swiper-slide .pxl-post--inner", function () {
            const bg = extractBackgroundImage($(this).css('background-image'));
            $backgroundChanger.css('background-image', `url(${bg})`);
            $backgroundChanger.addClass('flicker');

            setTimeout(() => {
                $backgroundChanger.removeClass('flicker');
            }, 600);
        });
    }

    function extractBackgroundImage(backgroundImage) {
        return backgroundImage.replace(/url\(["']?|["']?\)/g, '');
    }

    /* Menu Divider Move */
    function dreamslab_menu_divider_move() {
        $('.pxl-nav-menu1.fr-style-box, .pxl-nav-menu1.fr-style-box2, .pxl-menu-show-line').each(function () {
            var current = $(this).find('.pxl-menu-primary > .current-menu-item, .pxl-menu-primary > .current-menu-parent, .pxl-menu-primary > .current-menu-ancestor');
            if (current.length > 0) {
                var marker = $(this).find('.pxl-divider-move');
                marker.css({
                    left: current.position().left,
                    width: current.outerWidth(),
                    display: "block"
                });
                marker.addClass('active');
                current.addClass('pxl-shape-active');
                if (Modernizr.csstransitions) {
                    $(this).find('.pxl-menu-primary > li').on('mouseover', function () {
                        var self = $(this),
                            offsetLeft = self.position().left,
                            width = self.outerWidth() || current.outerWidth(),
                            left = offsetLeft == 0 ? 0 : offsetLeft || current.position().left;
                        marker.css({
                            left: left,
                            width: width,
                        });
                        marker.addClass('active');
                        current.removeClass('pxl-shape-active');
                    });
                    $(this).find('.pxl-menu-primary').on('mouseleave', function () {
                        marker.css({
                            left: current.position().left,
                            width: current.outerWidth()
                        });
                        current.addClass('pxl-shape-active');
                    });
                }
            } else {
                var marker = $(this).find('.pxl-divider-move');
                var current = $(this).find('.pxl-menu-primary > li:nth-child(1)');
                marker.css({
                    left: current.position().left,
                    width: current.outerWidth(),
                    display: "block"
                });
                if (Modernizr.csstransitions) {
                    $(this).find('.pxl-menu-primary > li').on('mouseover', function () {
                        var self = $(this),
                            offsetLeft = self.position().left,
                            width = self.outerWidth() || current.outerWidth(),
                            left = offsetLeft == 0 ? 0 : offsetLeft || current.position().left;
                        marker.css({
                            left: left,
                            width: width,
                        });
                        marker.addClass('active');
                    });
                    $(this).find('.pxl-menu-primary').on('mouseleave', function () {
                        marker.css({
                            left: current.position().left,
                            width: current.outerWidth()
                        });
                        marker.removeClass('active');
                    });
                }
            }
        });
    }

    /**
     * Loading Function - Optimized to reduce jitter
     * Handles preloader animation with smooth transitions
     */
    function dreamslab_loading() {
        const $load = $(".preloader-wrap");
        if (!$load.length) return;

        if (deviceCapabilities.supportsSmoothScrolling()) {
            setTimeout(function () {
                window.scrollTo({ top: 0, behavior: 'smooth' });
            }, 100);
        } else {
            setTimeout(function () {
                window.scrollTo(0, 0);
            }, 200);
        }

        const time = 1000;
        let animationCompleted = false;

        function animateValue(id, start, end, duration) {
            const $obj = $(id);
            if (!$obj.length) return;

            const range = end - start;
            const increment = end > start ? 1 : -1;
            const stepTime = Math.max(50, Math.abs(Math.floor(duration / range)));
            let current = start;

            const timer = setInterval(function () {
                current += increment;
                $obj.text(current);

                if (current === end) {
                    clearInterval(timer);
                }
            }, stepTime);
        }

        const $percentageID = $("#precent");
        if ($percentageID.length) {
            animateValue($percentageID, 0, 100, time);
        }

        imagesLoaded('body', function () {
            $(".loadbar").stop().animate({ width: "100%" }, time);
        });

        setTimeout(function () {
            imagesLoaded('body', function () {
                if (animationCompleted) return;
                animationCompleted = true;

                const clipPathValue = window.innerWidth <= 1024 ?
                    'inset(46% 30% 46% 30%)' : 'inset(45% 40% 45% 40%)';

                const tl = gsap.timeline({
                    defaults: {
                        ease: "power2.out",
                        overwrite: "auto"
                    }
                });

                gsap.set($(".trackbar"), { clipPath: clipPathValue });

                tl.to($(".percentage-wrapper"), {
                    duration: 1.0,
                    x: $(".trackbar").width() * 0.5 - $(".percentage-wrapper").width() * 0.5,
                    delay: 0.5,
                    ease: "power2.out"
                }, 0);

                tl.to($(".percentage"), {
                    duration: 1.0,
                    opacity: 0,
                    y: -30,
                    delay: 1.2,
                    ease: "power2.out"
                }, 0.3);

                tl.to($(".percentage-intro"), {
                    duration: 0.8,
                    opacity: 0,
                    ease: "power2.out"
                }, 0.2);

                tl.to($(".preloader-intro"), {
                    duration: 0.8,
                    opacity: 0,
                    ease: "power2.out"
                }, 0.2);

                tl.to($(".trackbar"), {
                    duration: 1.0,
                    clipPath: 'inset(0% 0%)',
                    ease: "power2.out"
                }, 0.4);

                tl.to($(".preloader-wrap"), {
                    duration: 0.6,
                    opacity: 0,
                    ease: "power2.out"
                }, 1.2);

                tl.set($(".preloader-wrap"), {
                    visibility: 'hidden',
                    yPercent: -50
                }, 1.8);

                setTimeout(function () {
                    if (pxl_window_width > 767) {
                        HeightTitles();
                    }
                }, 1000);
            });
        }, time);
    }

    /* Back To Top Progress Bar */
    function dreamslab_backtotop_progess_bar() {
        if ($('.pxl-scroll-top').length > 0) {
            var progressPaths = document.querySelectorAll('.pxl-scroll-top path');
            progressPaths.forEach(function (progressPath) {
                var pathLength = progressPath.getTotalLength();
                progressPath.style.transition = progressPath.style.WebkitTransition = 'none';
                progressPath.style.strokeDasharray = pathLength + ' ' + pathLength;
                progressPath.style.strokeDashoffset = pathLength;
                progressPath.getBoundingClientRect();
                progressPath.style.transition = progressPath.style.WebkitTransition = 'stroke-dashoffset 10ms linear';
            });

            var updateProgress = function () {
                var scroll = $(window).scrollTop();
                var height = $(document).height() - $(window).height();
                progressPaths.forEach(function (progressPath) {
                    var pathLength = progressPath.getTotalLength();
                    var progress = pathLength - (scroll * pathLength / height);
                    progressPath.style.strokeDashoffset = progress;
                });
            };

            updateProgress();
            $(window).scroll(throttle(updateProgress, 100));

            var offset = 50;
            var duration = 550;
            $(window).on('scroll', function () {
                if ($(this).scrollTop() > offset) {
                    $('.pxl-scroll-top').addClass('active-progress');
                } else {
                    $('.pxl-scroll-top').removeClass('active-progress');
                }
            });
        }

    }

    function dreamslab_cursor_hover() {
        const $ttPhMask = $(".pxl-bg-prx-effect-circle-parallax .pxl-text-carousel");
        let cursorX = 0;
        let cursorY = 0;
        if ($ttPhMask.length) {
            $(".pxl-bg-prx-effect-circle-parallax").parent(".elementor-element").addClass("e-circle-effect");
            $("body").addClass("pxl-bg-prx-effect-circle-parallax-on");
            window.addEventListener("mousemove", (e) => {
                cursorX = e.pageX;
                cursorY = e.pageY - window.scrollY;
                updateMaskPosition();
            }
            );
            function updateMaskPosition() {
                const maskRect = $ttPhMask[0].getBoundingClientRect();
                const xPercent = ((cursorX - maskRect.left) / maskRect.width) * 100;
                const yPercent = ((cursorY - maskRect.top) / maskRect.height) * 100;
                gsap.to($ttPhMask, {
                    "--x": `${xPercent}%`,
                    "--y": `${yPercent}%`,
                    duration: 0.3,
                    ease: "sine.out"
                });
            }
            window.addEventListener("scroll", updateMaskPosition);
            window.addEventListener("resize", updateMaskPosition);

            $("body.pxl-bg-prx-effect-circle-parallax-on .e-circle-effect .pxl-carousel-inner").filter(function () {
                return !$(this).closest(".elementor-element").hasClass("pxl-bg-prx-effect-circle-parallax");
            }).on("mouseover", function () {
                $("body").addClass("pxl-bg-prx-effect-circle-parallax-active");
            }).on("mouseleave", function () {
                $("body").removeClass("pxl-bg-prx-effect-circle-parallax-active");
            });
        }
    }

    function dreamslab_button_progess_bar() {
        if ($('.btn-icon-box-dot').length > 0) {
            var progressPaths = document.querySelectorAll('.btn-icon-box-dot path');

            progressPaths.forEach(function (progressPath) {
                var pathLength = progressPath.getTotalLength();
                $(progressPath).attr({
                    'stroke-dashoffset': pathLength,
                    'stroke-dasharray': pathLength,
                });
                progressPath.getBoundingClientRect();
            });
        }
    }

    function dreamslab_pagination_cf() {
        const $form = $("form[data-count-fieldset]");
        if (!$form.length) return;

        const totalSteps = parseInt($form.attr("data-count-fieldset"), 10);
        const $wrapper = $form.find(".fieldset-cf7mls-wrapper");
        const $paginationContainer = $("<div>", { class: "cf7mls-pagination" });

        for (let i = 0; i < totalSteps; i++) {
            const $dot = $("<span>").addClass("dot");
            if (i === 0) $dot.addClass("active");
            $paginationContainer.append($dot);
        }

        $wrapper.after($paginationContainer);

        const $fieldsets = $wrapper.find("fieldset");

        function updatePagination() {
            const $current = $wrapper.find(".cf7mls_current_fs");
            const currentIndex = parseInt($current.attr("data-cf7mls-order"), 10) || 0;

            $paginationContainer.find(".dot").each(function (index) {
                $(this).toggleClass("active", index === currentIndex);
            });
        }

        const observer = new MutationObserver(updatePagination);

        $fieldsets.each(function () {
            observer.observe(this, { attributes: true, attributeFilter: ['class'] });
        });
    }


    /* Custom Type File Upload*/
    function dreamslab_type_file_upload() {

        var multipleSupport = typeof $('<input/>')[0].multiple !== 'undefined',
            isIE = /msie/i.test(navigator.userAgent);

        $.fn.pxl_custom_type_file = function () {

            return this.each(function () {

                var $file = $(this).addClass('pxl-file-upload-hidden'),
                    $wrap = $('<div class="pxl-file-upload-wrapper">'),
                    $button = $('<button type="button" class="pxl-file-upload-button">Choose File</button>'),
                    $input = $('<input type="text" class="pxl-file-upload-input" placeholder="No File Choose" />'),
                    $label = $('<label class="pxl-file-upload-button" for="' + $file[0].id + '">Choose File</label>');
                $file.css({
                    position: 'absolute',
                    opacity: '0',
                    visibility: 'hidden'
                });

                $wrap.insertAfter($file)
                    .append($file, $input, (isIE ? $label : $button));

                $file.attr('tabIndex', -1);
                $button.attr('tabIndex', -1);

                $button.on('click', function () {
                    $file.focus().trigger('click');
                });

                $file.on('change', function () {

                    var files = [], fileArr, filename;

                    if (multipleSupport) {
                        fileArr = $file[0].files;
                        for (var i = 0, len = fileArr.length; i < len; i++) {
                            files.push(fileArr[i].name);
                        }
                        filename = files.join(', ');
                    } else {
                        filename = $file.val().split('\\').pop();
                    }

                    $input.val(filename)
                        .attr('title', filename)
                        .focus();
                });

                $input.on({
                    blur: function () { $file.trigger('blur'); },
                    keydown: function (e) {
                        if (e.which === 13) {
                            if (!isIE) {
                                $file.trigger('click');
                            }
                        } else if (e.which === 8 || e.which === 46) {
                            $file.replaceWith($file = $file.clone(true));
                            $file.trigger('change');
                            $input.val('');
                        } else if (e.which === 9) {
                            return;
                        } else {
                            return false;
                        }
                    }
                });

            });

        };
        $('.wpcf7-file[type=file]').pxl_custom_type_file();
    }

    //divider blog
    function dreamslab_scroll_checkp_blog($scope) {
        $('.pxl-service-carousel2 .pxl-post--icon, .pxl-service-list .pxl-post--icon,.pxl-icon-list.style-5,.pxl-icon-box7').each(function () {
            var wcont1 = $(this);

            function checkScrollPosition() {
                var pxl_scroll_top = $(window).scrollTop(),
                    viewportBottom = pxl_scroll_top + $(window).height(),
                    elementTop = wcont1.offset().top,
                    elementBottom = elementTop + wcont1.outerHeight();

                if (elementTop < viewportBottom && elementBottom > pxl_scroll_top) {
                    wcont1.addClass('animated');
                }
            }

            checkScrollPosition();

            $(window).on('scroll', function () {
                checkScrollPosition();
            });

        });
    }

    //Shop View Grid/List
    function dreamslab_shop_view_layout() {

        $(document).on('click', '.pxl-view-layout .view-icon a', function (e) {
            e.preventDefault();
            if (!$(this).parent('li').hasClass('active')) {
                $('.pxl-view-layout .view-icon').removeClass('active');
                $(this).parent('li').addClass('active');
                $(this).parents('.pxl-content-area').find('ul.products').removeAttr('class').addClass($(this).attr('data-cls'));
            }
        });
    }

    function dreamslab_height_ct_grid($scope) {
        $('.pxl-icon-box7').each(function () {
            var elementHeight2 = $(this).find(".pxl-item--description").height();
            $(this).find(".pxl-item--description").css("margin-bottom", "-" + elementHeight2 + "px");
        });
    }
    // Zoom Point
    function dreamslab_zoom_point() {
        $(".pxl-zoom-point").each(function () {

            let scaleOffset = $(this).data('offset');
            let scaleAmount = $(this).data('scale-mount');

            function scrollZoom() {
                const images = document.querySelectorAll("[data-scroll-zoom]");
                let scrollPosY = 0;
                scaleAmount = scaleAmount / 100;

                const observerConfig = {
                    rootMargin: "0% 0% 0% 0%",
                    threshold: 0
                };

                images.forEach(image => {
                    let isVisible = false;
                    const observer = new IntersectionObserver((elements, self) => {
                        elements.forEach(element => {
                            isVisible = element.isIntersecting;
                        });
                    }, observerConfig);

                    observer.observe(image);

                    image.style.transform = `scale(${1 + scaleAmount * percentageSeen(image)})`;

                    window.addEventListener("scroll", () => {
                        if (isVisible) {
                            scrollPosY = window.pageYOffset;
                            image.style.transform = `scale(${1 +
                                scaleAmount * percentageSeen(image)})`;
                        }
                    });
                });

                function percentageSeen(element) {
                    const parent = element.parentNode;
                    const viewportHeight = window.innerHeight;
                    const scrollY = window.scrollY;
                    const elPosY = parent.getBoundingClientRect().top + scrollY + scaleOffset;
                    const borderHeight = parseFloat(getComputedStyle(parent).getPropertyValue('border-bottom-width')) + parseFloat(getComputedStyle(element).getPropertyValue('border-top-width'));
                    const elHeight = parent.offsetHeight + borderHeight;

                    if (elPosY > scrollY + viewportHeight) {
                        return 0;
                    } else if (elPosY + elHeight < scrollY) {
                        return 100;
                    } else {
                        const distance = scrollY + viewportHeight - elPosY;
                        let percentage = distance / ((viewportHeight + elHeight) / 100);
                        percentage = Math.round(percentage);

                        return percentage;
                    }
                }
            }

            scrollZoom();

        });
    }

    function dreamslab_loading_overlay() {
        const $tt_pageTransition = $(".pxl-loader--transition");
        const $tt_ptrOverlayTop = $(".loader-overlay-top");
        const $tt_ptrOverlayBottom = $(".loader-overlay-bottom");
        const $tt_ptrContentWrap = $("#pxl-main");
        const $headerLogo = $(".pxl-loader--transition ~ header .pxl-logo img");
        const $loaderLogoContainer = $tt_pageTransition.find(".loader-logo");

        const isMobile = window.matchMedia("(max-width: 767px)").matches;

        // --------- MOBILE ---------
        if (isMobile) {
            $headerLogo.css("opacity", 1);
            $loaderLogoContainer.css("opacity", 0);
            $tt_pageTransition.fadeOut(400, function () {
                $(this).remove();
            });
            return;
        }

        // --------- DESKTOP ---------
        const duration = 0.7;
        $headerLogo.css("opacity", 0);

        // Wrap text captions
        const $phCaptionAppear = $(".pxl-caption-title, .pxl-caption-subtitle, .pxl-caption-description");
        if ($phCaptionAppear.length) {
            $phCaptionAppear.each(function () {
                $(this).contents().each(function () {
                    if (this.nodeType === 3 && this.textContent.trim().length) {
                        const text = this.textContent;
                        const wrappedText = text.replace(/([^\s]+)/g,
                            '<span class="tt-cap-word-wrap"><span class="tt-cap-word">$1</span></span>');
                        $(this).replaceWith(wrappedText);
                    } else if (this.nodeType === 1) {
                        const $el = $(this);
                        if ($el.is('br, hr')) return;
                        const $wrapped = $('<span class="tt-cap-word-wrap"><span class="tt-cap-word"></span></span>');
                        $wrapped.find('.tt-cap-word').append($el.clone(true));
                        $el.replaceWith($wrapped);
                    }
                });
            });
            $(".tt-cap-word-wrap").css({ display: "inline-flex", overflow: "hidden" });
            $(".tt-cap-word").css({ display: "inline-block", willChange: "transform" });
        }

        // --------- TIMELINE ---------
        function ttAnimateTransitionOut() {
            const tl = gsap.timeline({ defaults: { duration: 0.7, ease: "expo.inOut" } });

            if ($tt_pageTransition.length) {
                tl.to($tt_ptrOverlayTop, { scaleX: 0, transformOrigin: "center left" }, 0.5);
                tl.to($tt_ptrOverlayBottom, { scaleX: 0, transformOrigin: "center right" }, 0.5);
                tl.from($tt_ptrContentWrap, { autoAlpha: 0, clearProps: "all" }, 0.7);

                if ($headerLogo.length && $loaderLogoContainer.length) {
                    const fromRect = $loaderLogoContainer[0].getBoundingClientRect();
                    const toRect = $headerLogo[0].getBoundingClientRect();
                    const xOffset = toRect.left - fromRect.left;
                    const yOffset = (toRect.top + window.scrollY) - (fromRect.top + window.scrollY);
                    const scaleX = toRect.width / fromRect.width;
                    const scaleY = toRect.height / fromRect.height;

                    tl.to($loaderLogoContainer, { scale: 1.05, duration: 0.3, ease: "power1.out" }, 0.2);
                    tl.to($loaderLogoContainer, {
                        x: xOffset, y: yOffset, scaleX, scaleY,
                        duration: 1, ease: "power3.inOut",
                        onComplete: () => {
                            $loaderLogoContainer.css("opacity", 0);
                            $headerLogo.css("opacity", 1);
                        }
                    }, 0.5);
                } else if ($loaderLogoContainer.length) {
                    tl.to($loaderLogoContainer, {
                        scale: 1.1, autoAlpha: 0, duration: 0.6, ease: "power2.out"
                    }, 0.3);
                }
            }

            // Gom các caption animation lại thành 1 cấu hình
            const captionAnimations = [
                { selector: ".pxl-caption-title .tt-cap-word", vars: { xPercent: 101, ease: "power2.out", clearProps: "xPercent" }, at: 1.3 },
                { selector: ".pxl-caption-subtitle .tt-cap-word", vars: { xPercent: -101, ease: "power2.out", clearProps: "xPercent" }, at: 1.3 },
                { selector: ".pxl-caption-description .tt-cap-word", vars: { yPercent: 101, ease: "power2.out", clearProps: "yPercent" }, at: 1.8 },
                { selector: ".pxl-caption-btn", vars: { y: 50, autoAlpha: 0, ease: "power2.out", clearProps: "all" }, at: 2 },
                { selector: ".pxl-caption-image, .pxl-video video", vars: { duration: 1.2, scale: 1.2, autoAlpha: 0, ease: "power2.out", clearProps: "all" }, at: 1 },
                { selector: ".pxl-caption-custom .pxl-item--link, .pxl-caption-custom .pxl--item", vars: { yPercent: 101, autoAlpha: 0, ease: "power2.out", clearProps: "yPercent" }, at: 2 },
            ];

            captionAnimations.forEach(anim => {
                if ($(anim.selector).length) {
                    tl.from(anim.selector, anim.vars, anim.at);
                }
            });
        }

        window.addEventListener("pageshow", function (event) {
            if (event.persisted) window.location.reload();
        });

        setTimeout(ttAnimateTransitionOut, 500);
    }

    // Fit to Screen
    function dreamslab_fit_to_screen() {
        $('.pxl-gallery-scroll.h-fit-to-screen').each(function () {
            var h_adminbar = 0;
            var h_section_header = 0;
            var h_section_footer = 0;
            if ($('#wpadminbar').length == 1) {
                h_adminbar = $('#wpadminbar').outerHeight();
            }
            if ($('#pxl-header-elementor').length == 1) {
                h_section_header = $('#pxl-header-elementor').outerHeight();
            }
            if ($('#pxl-footer-elementor').length == 1) {
                h_section_footer = $('#pxl-footer-elementor').outerHeight();
            }
            var h_total = pxl_window_height - (h_adminbar + h_section_header + h_section_footer);
            $(this).css('height', h_total + 'px');
        });
    }

    $(document).on('click', function (event) {
        var clickedElement = $(event.target);
        var divWithClass = $('.pxl-icon-box6.active');
        var divWithClass2 = $('.pxl-icon-box6 .content-2');

        if (clickedElement.hasClass('pxl-icon-box6') && clickedElement.hasClass('active')) {
            return;
        }

        var isClickInsideDiv = divWithClass.has(clickedElement).length > 0;

        if (!isClickInsideDiv) {
            divWithClass.removeClass('active');
            divWithClass2.removeClass('active');
        }
    });

    /**
     * Cleanup function to prevent memory leaks
     */
    function cleanupTheme() {
        // Clear any remaining timers
        if (window.themeTimers && Array.isArray(window.themeTimers)) {
            window.themeTimers.forEach(timer => {
                if (timer) clearTimeout(timer);
            });
            window.themeTimers = [];
        }

        // Cancel animation frames
        if (scrollAnimationFrameId) {
            cancelAnimationFrame(scrollAnimationFrameId);
            scrollAnimationFrameId = null;
        }

        if (resizeAnimationFrameId) {
            cancelAnimationFrame(resizeAnimationFrameId);
            resizeAnimationFrameId = null;
        }

        // Kill GSAP animations if available
        if (typeof gsap !== 'undefined') {
            gsap.killTweensOf('*');
        }

        // Remove event listeners
        $(window).off('scroll.theme');
        $(window).off('resize.theme');

        // Clear particles if loaded
        if (particlesLoaded && typeof pJSDom !== 'undefined') {
            pJSDom.forEach(particle => {
                if (particle && particle.pJS && particle.pJS.fn) {
                    particle.pJS.fn.vendors.destroypJS();
                }
            });
        }

        // Clear any remaining jQuery animations
        $('*').stop(true, true);
    }

    // Call cleanup when page unloads
    $(window).on('beforeunload', cleanupTheme);

    /**
     * Performance monitoring (development only)
     */
    if (!isProduction) {
        let frameCount = 0;
        let lastTime = performance.now();

        function checkPerformance() {
            frameCount++;
            const currentTime = performance.now();

            if (currentTime >= lastTime + 1000) {
                const fps = Math.round((frameCount * 1000) / (currentTime - lastTime));
                if (fps < 30) {
                    safeWarn(`Low FPS detected: ${fps}. Consider disabling some animations.`);
                }
                frameCount = 0;
                lastTime = currentTime;
            }

            requestAnimationFrame(checkPerformance);
        }

        if (performanceSettings.enableSmoothAnimations) {
            requestAnimationFrame(checkPerformance);
        }
    }

    /**
     * Safari-specific CSS optimizations
     */
    if (deviceCapabilities.isSafari()) {
        const style = document.createElement('style');
        style.textContent = `
            /* Safari-specific optimizations to reduce jitter */
            .pxl-header-elementor-sticky {
                -webkit-transform: translateZ(0);
                -webkit-backface-visibility: hidden;
                -webkit-perspective: 1000px;
                will-change: transform;
            }
            
            .pxl-header-elementor-sticky.pxl-header-fixed {
                -webkit-transform: translateZ(0);
                -webkit-backface-visibility: hidden;
            }
            
            /* Disable hardware acceleration for problematic elements on Safari */
            .el-parallax-wrap,
            .pxl-portfolio-carousel5,
            .pxl-tabs-slip,
            .pxl-images-slip,
            .pxl-texts-slip {
                -webkit-transform: translateZ(0);
                -webkit-backface-visibility: hidden;
                -webkit-perspective: 1000px;
            }
            
            /* Optimize scroll performance on Safari */
            html {
                -webkit-overflow-scrolling: touch;
            }
            
            /* Reduce animation complexity on Safari */
            * {
                -webkit-animation-duration: 0.001ms;
                animation-duration: 0.001ms;
                -webkit-animation-iteration-count: 1;
                animation-iteration-count: 1;
                -webkit-animation-delay: 0s;
                animation-delay: 0s;
            }
        `;
        document.head.appendChild(style);
    }

    // Add GSAP performance optimizations
    if (typeof gsap !== 'undefined') {
        gsap.config({
            nullTargetWarn: false,
            autoSleep: 60,
            force3D: deviceCapabilities.isSafari() ? false : true
        });

        // Reduce GSAP ticker frequency on low-end devices
        if (deviceCapabilities.isLowEnd()) {
            gsap.ticker.lagSmoothing(1000, 33);
        }
    }

})(jQuery);