<?php
defined( 'ABSPATH' ) || die();

class WCLM_Generators {
	private $order_by;
	private $order;
	private $columns;
	private $per_page;

	private $entity = 'generators';

	public function __construct() {
		$this->columns = [
			'rule_id'             => 'ID',
			'product_id'          => 'Product',
			'prefix'              => 'Prefix',
			'chunks_number'       => 'Number of Chunks',
			'chunks_length'       => 'Chunk Length',
			'suffix'              => 'Suffix',
			'max_instance_number' => 'Activation Limit',
			'valid'               => 'Validity',
			'active'              => 'Status',
		];

		$this->per_page = (int) get_option( 'fslm_nb_rows_by_page', 15 );

		add_action( 'admin_menu', array( $this, 'add_admin_page' ) );
		add_action( 'admin_init', array( $this, 'save_hidden_columns' ) );
		add_action( 'wp_ajax_fetch_products_variations', array( $this, 'fetch_products_variations_callback' ) );

		if ( isset( $_POST['wclm_bulk_action'] ) == $this->entity ) {
			if ( $_POST['wclm_bulk_action'] == 'delete' ) {
				wclm_delete_generator( $_POST['wclm-bulk-generator-ids'] );
			} elseif ( in_array( $_POST['wclm_bulk_action'], [ 'activate', 'deactivate' ] ) ) {
				wclm_set_generator_status( $_POST['wclm-bulk-generator-ids'], $_POST['wclm_bulk_action'] );
			}
		}
	}

	function fetch_products_variations_callback() {
		global $wpdb;

		// Get the search term from the AJAX request
		$search = isset( $_GET['search'] ) ? sanitize_text_field( $_GET['search'] ) : '';

		// Query to get products and variations based on search term
		$query = "
	        SELECT p.ID as product_id, p.post_title as product_name, v.ID as variation_id, v.post_title as variation_name
	        FROM {$wpdb->posts} p
	        LEFT JOIN {$wpdb->posts} v ON v.post_parent = p.ID AND v.post_type = 'product_variation'
	        WHERE p.post_type = 'product' AND p.post_status = 'publish' AND p.post_title LIKE %s
	        ORDER BY p.post_title
	        LIMIT 30
	    ";

		$results = $wpdb->get_results( $wpdb->prepare( $query, '%' . $wpdb->esc_like( $search ) . '%' ) );

		$data = array();
		foreach ( $results as $row ) {
			// Combine product and variation into one option
			$data[] = array(
				'id'   => $row->variation_id ? $row->variation_id : $row->product_id,
				'text' => $row->variation_id ? $row->product_name . ' - ' . $row->variation_name . ' (#' . $row->variation_id . ')' : $row->product_name . ' (#' . $row->product_id . ')'
			);
		}

		// Return the results
		wp_send_json( array(
			'results'     => $data,
			'total_count' => count( $data )
		) );
	}

	public function add_admin_page() {
		$user_capability = fslm_vendors_permission() ? 'manage_product' : 'manage_woocommerce';

		if ( FS_WC_licenses_Manager::is_active() ) {
			add_submenu_page(
				'license-manager',
				esc_html__( 'Generators', 'fslm' ),
				esc_html__( 'Generators', 'fslm' ),
				$user_capability,
				'wclm-generators',
				[ $this, 'render_page' ]
			);
		}
	}

	public function save_hidden_columns() {
		if ( isset( $_POST['wclm_screen_meta_group'] ) && $_POST['wclm_screen_meta_group'] == $this->entity ) {
			check_admin_referer( 'wclm_save_screen_options', 'wclm_save_screen_options_nonce' );

			$hidden = [];
			if ( isset( $_POST['wclm_hidden_columns'] ) ) {
				$hidden = array_diff( array_keys( $this->columns ), $_POST['wclm_hidden_columns'] );
			}

			update_user_meta( get_current_user_id(), 'wclm_generators_hidden_columns', $hidden );
		}
	}

	public function render_page() {
		require_once WCLM_PLUGIN_DIR . 'includes/functions.php';

		if ( ( isset( $_GET['function'] ) && $_GET['function'] == 'edit-rule' ) && isset( $_GET['rule_id'] ) ) {
			require WCLM_PLUGIN_DIR . 'includes/edit_generator_rule.php';

			die();
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'wc_fs_product_licenses_keys_generator_rules';

		$this->order_by = isset( $_GET['orderby'] ) ? sanitize_text_field( $_GET['orderby'] ) : 'active != 1, rule_id';
		$this->order    = isset( $_GET['order'] ) && strtolower( $_GET['order'] ) === 'asc' ? 'ASC' : 'DESC';

		// Pagination setup
		$paged  = isset( $_GET['paged'] ) ? (int) $_GET['paged'] : 1;
		$offset = ( $paged - 1 ) * $this->per_page;

		$hidden_columns = get_user_meta( get_current_user_id(), 'wclm_generators_hidden_columns', true );
		if ( ! is_array( $hidden_columns ) ) {
			$hidden_columns = [];
		}

		$sortable_columns = array_keys( $this->columns );
		if ( ! in_array( $this->order_by, $sortable_columns ) ) {
			$this->order_by = 'active != 1, rule_id';
		}

		// Filter inputs
		$prefix            = isset( $_GET['prefix'] ) ? sanitize_text_field( $_GET['prefix'] ) : '';
		$suffix            = isset( $_GET['suffix'] ) ? sanitize_text_field( $_GET['suffix'] ) : '';
		$product_variation = isset( $_GET['product_variation'] ) ? sanitize_text_field( $_GET['product_variation'] ) : '';

		// Filter query
		$where = '1=1'; // Default condition

		if ( $prefix ) {
			$where .= $wpdb->prepare( " AND prefix LIKE %s", '%' . $wpdb->esc_like( $prefix ) . '%' );
		}
		if ( $suffix ) {
			$where .= $wpdb->prepare( " AND suffix LIKE %s", '%' . $wpdb->esc_like( $suffix ) . '%' );
		}
		if ( $product_variation ) {
			$where .= $wpdb->prepare( " AND (product_id = %d OR variation_id = %d)", $wpdb->esc_like( $product_variation ), $wpdb->esc_like( $product_variation ) );
		}


		// Get total count for pagination
		$count_query = "SELECT COUNT(*) FROM $table_name WHERE $where";
		$total_count = $wpdb->get_var( $count_query );
		$total_pages = ceil( $total_count / $this->per_page );

		// Get the paginated results
		$query   = "SELECT * FROM $table_name WHERE $where ORDER BY {$this->order_by} {$this->order} LIMIT %d OFFSET %d";
		$results = $wpdb->get_results( $wpdb->prepare( $query, $this->per_page, $offset ) );

		include WCLM_PLUGIN_DIR . 'new-ui/app/includes/parts/screen-meta.php';

		echo '<div class="wrap fslm wclm">';
		echo '<h1 class="wp-heading-inline">' . esc_html__( 'Generators', 'fslm' ) . '</h1>&nbsp;' . '<a class="page-title-action" href="' . admin_url( 'admin.php?page=license-manager-license-generator' ) . '">' . esc_attr__( 'Add New Generator', 'fslm' ) . ' </a>' . '<hr class="wp-header-end">';

		include WCLM_PLUGIN_DIR . 'new-ui/app/includes/parts/generators-table-nav.php';
		include WCLM_PLUGIN_DIR . 'new-ui/app/includes/parts/generators-filter-form.php';

		echo '<div class="wclm-table-responsive">';
		echo '<table class="wp-list-table widefat fixed striped table-view-list wclm-generators">
            <thead>
                <tr>';

		echo "<th id='wclm-generators-table-checkbox'><input id='wclm-bulk-select-all' type='checkbox'></th>";

		foreach ( $this->columns as $key => $label ) {
			if ( ! in_array( $key, $hidden_columns ) ) {
				$order = ( $this->order_by === $key && $this->order === 'ASC' ) ? 'DESC' : 'ASC';
				$link  = "<a href='?page=wclm-generators&orderby={$key}&order={$order}'>{$label}</a>";
				echo "<th id='wclm-generators-table-{$key}'>{$link}</th>";
			}
		}

		echo '<th>' . esc_html__( 'Actions', 'fslm' ) . '</th>';

		echo '</tr></thead><tbody>';

		if ( ! empty( $results ) ) {
			foreach ( $results as $row ) {
				echo "<tr>";
				echo '<td><input class="wclm-bulk-select" id="wclm-bulk-select-' . $row->rule_id . '" type="checkbox" value="' . $row->rule_id . '"></td>';
				foreach ( $this->columns as $key => $label ) {
					if ( ! in_array( $key, $hidden_columns ) ) {
						if ( $key === 'rule_id' ) {
							echo '<td>' . esc_html( $row->rule_id ) . '</td>';
						} elseif ( $key === 'product_id' ) {
							echo '<td>';
							if ( $row->variation_id == 0 ) {
								$product_query = $wpdb->prepare(
									"SELECT p.post_title AS product_name
                                FROM {$wpdb->posts} AS p
                                WHERE p.ID = %d AND p.post_type = 'product'",
									$row->product_id
								);
								$result        = $wpdb->get_row( $product_query );

								if ( $result ) {
									echo '<a href="' . admin_url( 'post.php?post=' . $row->product_id . '&action=edit' ) . '" target="_blank">' . $result->product_name . ' (#' . $row->product_id . ')' . '</a>';
								} else {
									echo 'Product not found';
								}
							} else {
								$product_query = $wpdb->prepare(
									"SELECT p.post_title AS product_name
                                FROM {$wpdb->posts} AS p
                                WHERE p.ID = %d AND p.post_type = 'product_variation'",
									$row->variation_id
								);
								$result        = $wpdb->get_row( $product_query );

								if ( $result ) {
									echo '<a href="' . admin_url( 'post.php?post=' . $row->product_id . '&action=edit' ) . '" target="_blank">' . $result->product_name  . $result->product_name . ' (#' . $row->variation_id . ')' . '</a>';
								} else {
									echo 'Variation or Product not found';
								}
							}
							echo '</td>';

						} elseif ( $key === 'prefix' ) {
							echo '<td>' . esc_html( $row->prefix ) . '</td>';
						} elseif ( $key === 'chunks_number' ) {
							echo '<td>' . esc_html( $row->chunks_number ) . '</td>';
						} elseif ( $key === 'chunks_length' ) {
							echo '<td>' . esc_html( $row->chunks_length ) . '</td>';
						} elseif ( $key === 'suffix' ) {
							echo '<td>' . esc_html( $row->suffix ) . '</td>';
						} elseif ( $key === 'max_instance_number' ) {
							echo '<td>' . esc_html( $row->max_instance_number ) . '</td>';
						} elseif ( $key === 'valid' ) {
							if ( (int) $row->valid > 0 ) {
								echo '<td>' . $row->valid . esc_html__( 'Days', 'fslm' ) . '</td>';
							} else {
								echo '<td>-</td>';
							}
						} elseif ( $key === 'active' ) {
							echo '<td>' . ( (int) $row->active == 1 ? esc_html__( 'Active', 'fslm' ) : esc_html__( 'Inactive', 'fslm' ) ) . '</td>';
						}
					}
				}

				include WCLM_PLUGIN_DIR . 'new-ui/app/includes/parts/generator-actions.php';

				echo "</tr>";
			}
		} else {
			echo '<tr><td colspan="' . count( $this->columns ) . '">' . esc_html__( 'No generators found.', 'fslm' ) . '</td></tr>';
		}

		echo '</tbody></table></div>';

		echo '<div class="tablenav bottom">';
		include WCLM_PLUGIN_DIR . 'new-ui/app/includes/parts/generators-pagination.php';
		echo '</div>';

		echo '</div>';
	}

}

new WCLM_Generators();
