<?php

namespace AgileStoreLocator\Admin;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use AgileStoreLocator\Admin\Base;

/**
 * Handling all the assets to migrate from the previous version
 *
 * @link       https://agilestorelocator.com
 * @since      4.8.1
 *
 * @package    AgileStoreLocator
 * @subpackage AgileStoreLocator/Admin/Analytics
 */

class Analytics extends Base
{
    /**
     * [__construct description]
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * [get_stats Get the Stats of the Analytics]
     * @return [type] [description]
     */
    public function get_stats()
    {
        global $wpdb;

        $start_date   = isset($_REQUEST['sl-start']) ? ($_REQUEST['sl-start']) : date('Y-m-d');
        $end_date     = isset($_REQUEST['sl-end']) ? ($_REQUEST['sl-end']) : date('Y-m-d');

        // Trim dates
        $start_date = date('Y-m-d', strtotime(trim($start_date))) . ' 00:00:00';
        $end_date   = date('Y-m-d', strtotime(trim($end_date))) . ' 23:59:00';

        $days_count = $this->date_diff($start_date, $end_date);

        // Either month or day based
        $month_based = ($days_count > 31) ? true : false;

        $group_by    = ($month_based) ? 'MONTH' : 'DAY';
        $dur_format  = ($month_based) ? 'M-Y' : 'd-M';
        $data_key    = ($month_based) ? 'Y-m' : 'Y-m-d';

        // For a single day, show by hours
        if ($days_count == 1) {
            $group_by   = 'HOUR';
            $dur_format = 'H';
            $data_key   = 'Y-m-d H';
        }

        if ($days_count == 1) {
            $date_format = '%%Y-%%m-%%d %%H';
        } else {
            $date_format = $month_based ? '%%Y-%%m' : '%%Y-%%m-%%d';
        }

        ////////////////////
        //// Chart Values //
        ////////////////////
        $searches_results = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE_FORMAT(created_on, '{$date_format}') AS d, COUNT(*) AS c 
       FROM `" . ASL_PREFIX . "stores_view` 
       WHERE created_on BETWEEN %s AND %s AND is_search = 1 
       GROUP BY DATE_FORMAT(created_on, '{$date_format}')",
            $start_date,
            $end_date
        ));

        $interaction_results = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE_FORMAT(created_on, '{$date_format}') AS d, COUNT(*) AS c 
       FROM `" . ASL_PREFIX . "stores_view` 
       WHERE created_on BETWEEN %s AND %s AND is_search = 0
       GROUP BY DATE_FORMAT(created_on, '{$date_format}')",
            $start_date,
            $end_date
        ));

        // Chart data for the searches
        $searches_results    = $this->_create_chart_data($start_date, $end_date, $days_count, $group_by, $dur_format, $data_key, $searches_results);

        // Chart data for the interactions
        $interaction_results = $this->_create_chart_data($start_date, $end_date, $days_count, $group_by, $dur_format, $data_key, $interaction_results);

        /////////////
        //// Stats //
        /////////////
        $limit = (isset($_REQUEST['len']) && $_REQUEST['len']) ? intval($_REQUEST['len']) : null;

        // Top views
        $top_stores = $wpdb->get_results($wpdb->prepare(
            'SELECT COUNT(*) AS views, ' . ASL_PREFIX . 'stores_view.`store_id`, title, city 
        FROM `' . ASL_PREFIX . 'stores_view` 
        LEFT JOIN `' . ASL_PREFIX . 'stores` 
        ON ' . ASL_PREFIX . 'stores_view.`store_id` = ' . ASL_PREFIX . 'stores.`id` 
        WHERE store_id IS NOT NULL 
        AND ' . ASL_PREFIX . 'stores_view.created_on BETWEEN %s AND %s 
        GROUP BY store_id 
        ORDER BY views' . (($limit) ? ' DESC LIMIT ' . $limit : ''),
            $start_date,
            $end_date
        ));

        // Clean the store data
        foreach ($top_stores as $store) {
            $store->title = $store->title ? esc_attr($store->title) : '';
            $store->city  = $store->city ? esc_attr($store->city) : '';
        }

        // Top Searches
        $top_search = $wpdb->get_results($wpdb->prepare(
            'SELECT COUNT(*) AS views, search_str 
        FROM `' . ASL_PREFIX . 'stores_view` 
        WHERE store_id IS NULL AND is_search = 1 
        AND created_on BETWEEN %s AND %s 
        GROUP BY search_str 
        ORDER BY views' . (($limit) ? ' DESC LIMIT ' . $limit : ''),
            $start_date,
            $end_date
        ));

        // Return the response with the data
        return $this->send_response([
            'stores'      => $top_stores,
            'searches'    => $top_search,
            'chart_data'  => [$searches_results, $interaction_results],
            'granularity' => $group_by // "DAY", "MONTH", "HOUR"
        ]);
    }

    /**
     * [create_chart_data Create the chart data]
     * @param  [type] $start_date [description]
     * @param  [type] $end_date   [description]
     * @param  [type] $days_count [description]
     * @param  [type] $group_by   [description]
     * @param  [type] $dur_format [description]
     * @param  [type] $data_key   [description]
     * @param  [type] $results    [description]
     * @return [type]             [description]
     */
    private function _create_chart_data($start_date, $end_date, $days_count, $group_by, $dur_format, $data_key, $results)
    {
        $days_stats = [];

        $begin  = new \DateTime($start_date);
        $end    = new \DateTime($end_date);

        // Adjust end to include the last day or hour
        $end->modify('+1 ' . ($days_count == 1 ? 'hour' : $group_by));

        // Get all the labels
        $interval = \DateInterval::createFromDateString('1 ' . ($days_count == 1 ? 'hour' : $group_by));
        $period   = new \DatePeriod($begin, $interval, $end);

        $days_stats = [];

        // Initialize stats with 0 data
        foreach ($period as $dt) {
            $days_stats[(string)$dt->format($data_key)] = ['label' => $dt->format($dur_format), 'data' => 0];
        }

        // Fill the data using the appropriate key (full date or hour)
        foreach ($results as $row) {
            if (isset($days_stats[(string)$row->d])) {
                $days_stats[(string)$row->d]['data'] = $row->c;
            }
        }

        return $days_stats;
    }

    /**
     * [export_analytics Export the ASL Analytics CSV with better formatting]
     */
    public function export_analytics()
    {
        global $wpdb;

        // Dates
        $start_date_raw = isset($_REQUEST['sl-start']) ? trim($_REQUEST['sl-start']) : date('Y-m-d');
        $end_date_raw   = isset($_REQUEST['sl-end']) ? trim($_REQUEST['sl-end']) : date('Y-m-d');

        $start_date_sql = $start_date_raw . ' 00:00:00';
        $end_date_sql   = $end_date_raw . ' 23:59:00';

        $all_rows = [];

        ///////////////////////////////////
        //  0️⃣ Report Metadata
        ///////////////////////////////////
        $all_rows[] = ["AGILE STORE LOCATOR ANALYTICS EXPORT"];
        $all_rows[] = ['Generated At:', current_time('Y-m-d H:i:s')];
        $all_rows[] = ['Date Range:', $start_date_raw . ' to ' . $end_date_raw];
        $all_rows[] = ['Site URL:', site_url()];
        $all_rows[] = [''];
        $all_rows[] = ['-----------------------------------------------------'];
        $all_rows[] = [''];

      

        ///////////////////////////////////
        //  1️⃣ Searches Data
        ///////////////////////////////////
        $searches_query = $wpdb->prepare(
            'SELECT search_str AS location, ip_address, created_on
       FROM `' . ASL_PREFIX . 'stores_view`
       WHERE is_search = 1
       AND created_on BETWEEN %s AND %s',
            $start_date_sql,
            $end_date_sql
        );

        $searches = $wpdb->get_results($searches_query, ARRAY_A);

        $all_rows[] = ['*** Searches Data ***'];
        $all_rows[] = ['location', 'ip_address', 'created_on'];

        if ($searches) {
            foreach ($searches as $row) {
                $all_rows[] = [$row['location'], $row['ip_address'], $row['created_on']];
            }
        } else {
            $all_rows[] = ['(No search data available)', '', ''];
        }

        $all_rows[] = [''];
        $all_rows[] = ['-----------------------------------------------------'];
        $all_rows[] = [''];

        ///////////////////////////////////
        //  2️⃣ Top Stores Views
        ///////////////////////////////////
        $stores_query = $wpdb->prepare(
            'SELECT COUNT(*) AS Views, sv.`store_id`, s.title, s.city, s.state, s.country
       FROM `' . ASL_PREFIX . 'stores_view` AS sv
       LEFT JOIN `' . ASL_PREFIX . 'stores` AS s ON sv.`store_id` = s.`id`
       WHERE sv.store_id IS NOT NULL
       AND sv.created_on BETWEEN %s AND %s
       GROUP BY sv.store_id
       ORDER BY Views DESC',
            $start_date_sql,
            $end_date_sql
        );

        $top_stores = $wpdb->get_results($stores_query, ARRAY_A);

        $all_rows[] = ['*** Top Store Views ***'];
        $all_rows[] = ['Views', 'Store ID', 'Store Name', 'City', 'State', '% of Total'];

        $total_views = 0;
        if ($top_stores) {
            foreach ($top_stores as $row) {
                $total_views += intval($row['Views']);
            }

            foreach ($top_stores as $row) {
                $percent    = $total_views ? round(($row['Views'] / $total_views) * 100, 2) . '%' : '0%';
                $all_rows[] = [
                    $row['Views'],
                    $row['store_id'],
                    $row['title'],
                    $row['city'],
                    $row['state'],
                    $percent
                ];
            }
        } else {
            $all_rows[] = ['(No store data available)', '', '', '', '', '', ''];
        }

        $all_rows[] = [''];
        $all_rows[] = ['-----------------------------------------------------'];
        $all_rows[] = [''];

        ///////////////////////////////////
        //  3️⃣ Aggregate Totals Section
        ///////////////////////////////////
        $total_searches_count = $searches ? count($searches) : 0;

        $all_rows[] = ['*** Aggregate Totals ***'];
        $all_rows[] = ['Metric', 'Value'];
        $all_rows[] = ['Total Searches', $total_searches_count];
        $all_rows[] = ['Total Store Views', $total_views];
        $all_rows[] = [''];

    
        ///////////////////////////////////
        //  4️⃣ Generate and Send CSV
        ///////////////////////////////////
        $csv = new \AgileStoreLocator\Admin\CSV\Reader();
        $csv->setRows($all_rows);
        $csv->write(\AgileStoreLocator\Admin\CSV\Reader::DOWNLOAD, 'asl-analytics-export.csv');
        die;
    }

    /**
     * [date_diff Return the difference between dates]
     * @param  [type] $start [description]
     * @param  [type] $end   [description]
     * @return [type]        [description]
     */
    private function date_diff($start, $end)
    {
        $datediff   = strtotime($end) - strtotime($start);
        return round($datediff / (60 * 60 * 24));
    }
}
