<?php

namespace AgileStoreLocator\Admin;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use AgileStoreLocator\Admin\Base;

/**
 * Attribute Manager for Brand & Special or any other dropdown
 *
 * @link       https://agilestorelocator.com
 * @since      4.7.32
 *
 * @package    AgileStoreLocator
 * @subpackage AgileStoreLocator/Admin/Attribute
 */

class Attribute extends Base
{
    private $attr_tables;

    /**
     * [__construct description]
     */
    public function __construct()
    {
        $this->attr_tables = \AgileStoreLocator\Model\Attribute::get_controls_keys();

        parent::__construct();
    }

    /**
     * [delete_attribute Delete Attribute]
     * @return [type] [description]
     */
    public function delete_attribute()
    {
        global $wpdb;

        $response          = new \stdclass();
        $response->success = false;

        $table  = isset($_REQUEST['name']) ? sanitize_text_field($_REQUEST['name']) : null;
        $title  = isset($_REQUEST['title']) ? sanitize_text_field($_REQUEST['title']) : null;
        $value  = isset($_REQUEST['value']) ? sanitize_text_field($_REQUEST['value']) : null;

        $multiple = isset($_REQUEST['multiple']) ? $_REQUEST['multiple'] : null;
        $delete_sql;
        $cResults;

        //  To filter the table name
        $table = (in_array($table, $this->attr_tables)) ? $table : $this->attr_tables[0];

        if ($multiple) {
            //  Clean it
            $item_ids      = implode(',', array_map('intval', $_POST['item_ids']));

            $delete_sql    = 'DELETE FROM ' . ASL_PREFIX . $table . ' WHERE id IN (' . $item_ids . ')';
            $cResults      = $wpdb->get_results('SELECT * FROM ' . ASL_PREFIX . $table . ' WHERE id IN (' . $item_ids . ')');
        } else {
            $category_id   = intval($_REQUEST['category_id']);

            $delete_sql    = 'DELETE FROM ' . ASL_PREFIX . $table . ' WHERE id = ' . $category_id;
            $cResults      = $wpdb->get_results('SELECT * FROM ' . ASL_PREFIX . $table . ' WHERE id = ' . $category_id);
        }

        if (count($cResults) != 0) {
            if ($wpdb->query($delete_sql)) {
                $response->success = true;
            } else {
                $response->error = esc_attr__('Error occurred while deleting record', 'asl_locator');
                $response->msg   = $wpdb->show_errors();
            }
        } else {
            $response->error = esc_attr__('Error occurred while deleting record', 'asl_locator');
        }

        if ($response->success) {
            $response->msg = $title . ' ' . esc_attr__('deleted successfully', 'asl_locator');
        }

        return $this->send_response($response);
    }

    /**
     * [add_attribute description]
     */
    public function add_attribute()
    {
        global $wpdb;

        $response          = new \stdclass();
        $response->success = false;

        $table  = isset($_REQUEST['name']) ? sanitize_text_field($_REQUEST['name']) : null;
        $title  = isset($_REQUEST['title']) ? sanitize_text_field($_REQUEST['title']) : null;
        $value  = isset($_REQUEST['value']) ? sanitize_text_field($_REQUEST['value']) : null;
        $ordr   = isset($_REQUEST['ordr']) && is_numeric($_REQUEST['ordr']) ? $_REQUEST['ordr'] : 0;

        //  Filter the Table Name
        $table = (in_array($table, $this->attr_tables)) ? $table : $this->attr_tables[0];

        $value = stripslashes($value);

        if ($value && $wpdb->insert(ASL_PREFIX . $table, ['name' => $this->clean_input($value), 'ordr' => $ordr, 'lang' => $this->lang])) {
            $response->msg     = $title . esc_attr__(' added successfully', 'asl_locator');
            $response->success = true;
        } else {
            $response->msg = esc_attr__('Error occurred while saving record', 'asl_locator');
        }

        return $this->send_response($response);
    }

    /**
     * [update_attribute description]
     * @return [type] [description]
     */
    public function update_attribute()
    {
        global $wpdb;

        $response          = new \stdclass();
        $response->success = false;

        $table  = isset($_REQUEST['name']) ? sanitize_text_field($_REQUEST['name']) : null;
        $title  = isset($_REQUEST['title']) ? sanitize_text_field($_REQUEST['title']) : null;
        $value  = isset($_REQUEST['value']) ? sanitize_text_field($_REQUEST['value']) : null;
        $at_id  = isset($_REQUEST['id']) ? sanitize_text_field($_REQUEST['id']) : null;
        $ordr   = isset($_REQUEST['ordr']) && is_numeric($_REQUEST['ordr']) ? sanitize_text_field($_REQUEST['ordr']) : 0;

        //  Filter the Table Name
        $table = (in_array($table, $this->attr_tables)) ? $table : $this->attr_tables[0];

        $value = stripslashes($value);

        if ($at_id && $value && $wpdb->update(ASL_PREFIX . $table, ['name' => $this->clean_input($value), 'ordr' => $ordr], ['id' => $at_id])) {
            $response->msg     = $title . ' ' . esc_attr__('Updated Successfully', 'asl_locator');
            $response->success = true;
        } else {
            $response->msg = esc_attr__('Error occurred while saving record', 'asl_locator');
        }

        return $this->send_response($response);
    }

    /**
     * [get_attributes Get the Attribute]
     * @return [type] [description]
     */
    public function get_attributes()
    {
        global $wpdb;

        // Pagination and table request
        $start  = isset($_REQUEST['iDisplayStart']) ? intval($_REQUEST['iDisplayStart']) : 0;
        $length = isset($_REQUEST['iDisplayLength']) && $_REQUEST['iDisplayLength'] != '-1'
                  ? intval($_REQUEST['iDisplayLength']) : 10;
        $sEcho  = isset($_REQUEST['sEcho']) ? intval($_REQUEST['sEcho']) : 1;
        $table  = isset($_REQUEST['type']) ? sanitize_text_field($_REQUEST['type']) : null;

        // Validate requested table against allowed attribute tables
        if (!$table || !in_array($table, $this->attr_tables, true)) {
            $table = $this->attr_tables[0]; // fallback to default safe table
        }

        // Define and whitelist columns
        $acolumns        = ['id', 'id', 'name', 'ordr', 'created_on'];
        $allowed_columns = ['id', 'name', 'ordr', 'created_on'];

        $clause     = [];
        $sql_params = [];

        // Filtering with validation and SQL injection protection
        if (isset($_REQUEST['filter']) && is_array($_REQUEST['filter'])) {
            foreach ($_REQUEST['filter'] as $key => $value) {
                if (!$key || !$value || $key === 'undefined' || $value === 'undefined') {
                    continue;
                }

                $key   = sanitize_text_field($key);
                $value = sanitize_text_field($value);

                if (in_array($key, $allowed_columns, true)) {
                    $clause[]     = "`$key` LIKE %s";
                    $sql_params[] = '%' . $wpdb->esc_like($value) . '%';
                }
            }
        }

        // Always filter by language
        $clause[]     = '`lang` = %s';
        $sql_params[] = $this->lang;

        $sWhere = $clause ? 'WHERE ' . implode(' AND ', $clause) : '';
        $sLimit = "LIMIT $start, $length";

        // Sorting with safety checks
        $sOrder = '';
        if (isset($_REQUEST['iSortCol_0']) && isset($_REQUEST['iSortingCols'])) {
            for ($i = 0; $i < intval($_REQUEST['iSortingCols']); $i++) {
                $col_index = intval($_REQUEST['iSortCol_' . $i]);
                $sort_dir  = (isset($_REQUEST['sSortDir_' . $i]) && strtolower($_REQUEST['sSortDir_' . $i]) === 'asc') ? 'ASC' : 'DESC';

                if (isset($acolumns[$col_index]) && in_array($acolumns[$col_index], $allowed_columns, true)) {
                    $sOrder = "ORDER BY `{$acolumns[$col_index]}` $sort_dir";
                    break;
                }
            }
        }

        // Final table name
        $db_table = ASL_PREFIX . $table;
        $fields   = implode(',', $acolumns);

        // Data query
        $sql         = "SELECT $fields FROM $db_table";
        $data_query  = "$sql $sWhere $sOrder $sLimit";
        $data_output = $wpdb->get_results($wpdb->prepare($data_query, ...$sql_params));

        // Trigger plugin activator if a table column is missing
        if (!$data_output && $wpdb->last_error) {
            \AgileStoreLocator\Activator::activate();
        }

        // Count query
        $sqlCount       = "SELECT COUNT(*) as count FROM $db_table";
        $count_query    = "$sqlCount $sWhere";
        $r              = $wpdb->get_results($wpdb->prepare($count_query, ...$sql_params));
        $iFilteredTotal = isset($r[0]->count) ? intval($r[0]->count) : 0;

        // Output format
        $output = [
            'sEcho'                => $sEcho,
            'iTotalRecords'        => $iFilteredTotal,
            'iTotalDisplayRecords' => $iFilteredTotal,
            'aaData'               => []
        ];

        // Row formatting
        foreach ($data_output as $row) {
            $row->action = '<div class="edit-options">
            <a data-ordr="' . esc_attr($row->ordr) . '" data-value="' . esc_attr($row->name) . '" data-id="' . esc_attr($row->id) . '" title="Edit" class="edit_attr"><svg width="14" height="14"><use xlink:href="#i-edit"></use></svg></a>
            <a title="Delete" data-id="' . esc_attr($row->id) . '" class="delete_attr g-trash"><svg width="14" height="14"><use xlink:href="#i-trash"></use></svg></a>
        </div>';

            $row->check = '<div class="custom-control custom-checkbox">
            <input type="checkbox" data-id="' . esc_attr($row->id) . '" class="custom-control-input" id="asl-chk-' . esc_attr($row->id) . '">
            <label class="custom-control-label" for="asl-chk-' . esc_attr($row->id) . '"></label>
        </div>';

            $output['aaData'][] = $row;
        }

        return $this->send_response($output);
    }
}
