<?php
namespace BooklyPackages\Backend\Components\Dialogs\Package;

use Bookly\Lib as BooklyLib;
use Bookly\Lib\Entities\Customer;
use Bookly\Lib\Entities\Payment;
use Bookly\Backend\Components\Dialogs\Queue\NotificationList;
use BooklyPackages\Lib;
use BooklyPro\Lib\Config;
use BooklyPro\Lib\Utils;

class Ajax extends BooklyLib\Base\Ajax
{
    /**
     * @inheritDoc
     */
    protected static function permissions()
    {
        return array( '_default' => 'supervisor' );
    }

    /**
     * Get package data when editing an package.
     */
    public static function getDataForPackage()
    {
        $response = array( 'success' => false, 'data' => array() );

        $package = new Lib\Entities\Package();
        if ( $package->load( self::parameter( 'id' ) ) ) {
            $response['success'] = true;
            $response['data']['staff_id'] = (int) $package->getStaffId();
            $response['data']['service_id'] = (int) $package->getServiceId();
            $response['data']['location_id'] = (int) $package->getLocationId();
            $response['data']['internal_note'] = $package->getInternalNote();
            $response['data']['customer_id'] = (int) $package->getCustomerId();
            $response['data']['payment_title'] = '';
            if ( $package->getPaymentId() ) {
                $payment = Payment::find( $package->getPaymentId() );
                $response['data']['payment_id'] = (int) $package->getPaymentId();
                $response['data']['payment_type'] = $payment->getTotal() === $payment->getPaid() ? 'full' : 'partial';
                $response['data']['payment_title'] = BooklyLib\Entities\Payment::paymentInfo(
                    $payment->getPaid() + $payment->getChildPaid(),
                    $payment->getTotal(),
                    $payment->getType(),
                    $payment->getStatus()
                );
            }

            if ( Customer::query( 'c' )->count() >= Customer::REMOTE_LIMIT ) {
                $customer = Customer::find( $package->getCustomerId() );
                $name = $customer->getFullName();
                if ( $customer->getEmail() != '' || $customer->getPhone() != '' ) {
                    $name .= ' (' . trim( $customer->getEmail() . ', ' . $customer->getPhone(), ', ' ) . ')';
                }

                $response['data']['customer_data'][] = array(
                    'id' => (int) $customer->getId(),
                    'name' => $name,
                );
            }
        }
        wp_send_json( $response );
    }

    /**
     * Save package form (for both create and edit).
     */
    public static function savePackageForm()
    {
        $response = array( 'success' => false );

        $package_id = (int) self::parameter( 'id', 0 );
        $staff_id = self::parameter( 'staff_id' ) ?: null;
        $service_id = (int) self::parameter( 'service_id' );
        $location_id = self::parameter( 'location_id' ) ?: null;
        $customer_id = (int) self::parameter( 'customer_id' );
        $internal_note = self::parameter( 'internal_note' );
        $notification = self::parameter( 'notification' );

        $staff_service = new BooklyLib\Entities\StaffService();
        $staff_service->loadBy( array(
            'staff_id' => $staff_id,
            'service_id' => $service_id,
            'location_id' => BooklyLib\Proxy\Locations::prepareStaffLocationId( $location_id, $staff_id ) ?: null,
        ) );

        if ( ! $service_id ) {
            $response['errors']['service_required'] = true;
        }
        if ( ! $customer_id ) {
            $response['errors']['customer_required'] = true;
        }

        $queue = new NotificationList();

        // If no errors then try to save the package.
        if ( ! isset ( $response['errors'] ) ) {
            $price = self::parameter( 'payment_price' );
            if ( self::parameter( 'payment_action' ) === 'create' ) {
                $payment = new Payment();
                $payment
                    ->setType( Payment::TYPE_LOCAL )
                    ->setStatus( Payment::STATUS_PENDING )
                    ->setTax( self::parameter( 'payment_tax' ) ?: 0 )
                    ->setTotal( get_option( 'bookly_taxes_in_price' ) === 'excluded' ? $price + $payment->getTax() : $price )
                    ->setPaid( 0 )
                    ->save();
            } else {
                $payment = null;
            }

            $package = new Lib\Entities\Package();
            $package->load( $package_id );
            $package
                ->setLocationId( $location_id )
                ->setStaffId( $staff_id )
                ->setServiceId( $service_id )
                ->setCustomerId( $customer_id )
                ->setInternalNote( $internal_note );

            if ( $payment && $payment->getId() ) {
                $package->setPaymentId( $payment->getId() );
            } elseif ( self::parameter( 'payment_action' ) === 'attach' ) {
                $package->setPaymentId( self::parameter( 'payment_id' ) );
            }
            $package->save();

            if ( $payment ) {
                $pkg_details = new Lib\DataHolders\Details\Package();
                $pkg_details->setPackage( $package, $price );
                $details = $payment->getDetailsData();
                $details->addDetails( $pkg_details )
                    ->setCustomer( Customer::find( $customer_id ) );

                if ( Config::needCreateWCOrder( $payment->getTotal() ) ) {
                    $wc_order = Utils\Common::createWCOrder( $service_id, $payment->getTotal(), $payment->getTax(), $details->getCustomer()->getWpUserId() );
                    $details->setData( array( 'gateway_ref_id' => $wc_order->get_id() ) );
                    $payment
                        ->setType( Payment::TYPE_WOOCOMMERCE )
                        ->setStatus( Payment::STATUS_COMPLETED )
                        ->setPaid( $payment->getTotal() );
                }

                $payment->save();
            }

            $response['success'] = true;
            $response['package_id'] = $package->getId();
            if ( $notification ) {
                Lib\Notifications\Package\Sender::send( $package, 'created', '', $queue );
            }
        }

        update_user_meta( get_current_user_id(), 'bookly_packages_form_send_notifications', $notification );

        $list = $queue->getList();
        if ( $list ) {
            $db_queue = new BooklyLib\Entities\NotificationQueue();
            $db_queue
                ->setData( json_encode( array( 'all' => $list ) ) )
                ->save();

            $response['queue'] = array( 'token' => $db_queue->getToken(), 'all' => $queue->getInfo() );
        }

        wp_send_json( $response );
    }
}