<?php
/**
 * Contacter
 * Voice feedback form for your website for saving and transcribing user voice messages to text.
 * Exclusively on https://1.envato.market/contacter
 *
 * @encoding        UTF-8
 * @version         1.7.8
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Contacter;

use Merkulove\Contacter\Unity\Plugin;
use Merkulove\Contacter\Unity\Settings;
use Merkulove\Contacter\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class used to add contacter_form post type.
 *
 * @since 1.0.0
 *
 **/
final class CFOptionsMetaBox {

	/**
	 * The one true CFOptionsMetaBox.
	 *
	 * @var CFOptionsMetaBox
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

	}

	/**
	 * Render "Options" metabox with all fields.
	 *
	 * @param $contacter_form - Post Object.
	 *
	 * @since 1.0.0
	 **/
	public function render_metabox( $contacter_form ) {

		/** Render Nonce field to validate on save. */
		$this->render_nonce();

		?>
		<div class="mdp-options-box">
			<table class="form-table">
				<tbody>
				<?php

				/** Render Record Duration */
				$this->record_duration( $contacter_form );

                /** Render "Start recording" Button. */
                $this->start_recording( $contacter_form );

				/** Render "Skip recording" Button. */
				$this->skip_recording( $contacter_form );

                /** Render Before text field. */
                $this->before_text( $contacter_form );

				/** Render After text field. */
				$this->after_text( $contacter_form );

				/** Render Speak Now text field. */
				$this->speak_now_text( $contacter_form );

				/** Render Send recording text field. */
				$this->send_text( $contacter_form );

				/** Render Thank you text field. */
				$this->thanks_text( $contacter_form );

				/** Render Additional Fields. */
				$this->additional_fields( $contacter_form );

				/** Render Attachments properties */
				$this->attachment_properties( $contacter_form );

				/** Record Form Suffix */
				$this->form_suffix( $contacter_form );

				/** Render Form Notification. */
				$this->form_notification( $contacter_form );

				?>
				</tbody>
			</table>
		</div>
		<?php
	}

	/**
	 * Save "Options" metabox with all fields.
	 *
	 * @param $post_id - Post Object.
	 * @since 1.0.0
	 **/
	public function save_metabox( $post_id ) {

		/** Options fields keys. */
		$k = [
			'mdp_before_txt', // Before Text.
			'mdp_align', // Align.
			'mdp_btn_margin', // Margin.
			'mdp_btn_padding', // Padding.
            'mdp_btn_radius', // Radius.
			'mdp_btn_icon', // Icon.
			'mdp_btn_icon_position', // Icon Position.
            'mdp_btn_caption', // Caption
            'mdp_btn_size', // Size
            'mdp_btn_color', // Text/Icon color.
            'mdp_btn_color_hover', // Text/Icon Hover color.
            'mdp_btn_bg_color', // Background color.
            'mdp_btn_bg_color_hover', // Background Hover color.
			'mdp_btn_hover_animation', // Hover Animations.
            'mdp_skip_button',
			'mdp_btn_margin_skip', // Margin.
			'mdp_btn_padding_skip', // Padding.
			'mdp_btn_radius_skip', // Radius.
			'mdp_btn_icon_skip', // Icon.
			'mdp_btn_icon_position_skip', // Icon Position.
			'mdp_btn_caption_skip', // Caption
			'mdp_btn_size_skip', // Size
			'mdp_btn_color_skip', // Text/Icon color.
			'mdp_btn_color_hover_skip', // Text/Icon Hover color.
			'mdp_btn_bg_color_skip', // Background color.
			'mdp_btn_bg_color_hover_skip', // Background Hover color.
            'mdp_after_txt', // After Text.
			'mdp_speak_now_txt', // Speak Now message.
            'mdp_send_txt', // Send Recording message.
            'mdp_thanks_txt', // Thank You message.
            'mdp_additional_fields', // Additional Fields.
			'mdp_additional_fields_fb', // Form Builder fields.
			'mdp_additional_fields_res',
            'mdp_form_record_duration',
			'mdp_form_file_original',
            'mdp_form_file_size',
            'mdp_form_mime',
            'mdp_field_record_suffix',
			'mdp_field_notification'
        ];

		/** Save each field. */
		foreach ( $k as $field ) {
			$value = ( isset( $_POST[$field] ) ? wp_kses_post( $_POST[$field] ) : '' );

			if ( in_array( $field, ['mdp_before_txt', 'mdp_after_txt', 'mdp_speak_now_txt', 'mdp_send_txt', 'mdp_thanks_txt'] ) AND $value === '' ) {
				$value = ' ';
			}

			update_post_meta( $post_id, $field, $value );
        }

    }

	/**
	 * Render Before text field.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function before_text( $contacter_form ) {

		/** Get Before Text field value from meta if it's already been entered. */
		$before_txt = get_post_meta( $contacter_form->ID, 'mdp_before_txt', true );

		/** Default value. */
		if ( empty( $before_txt ) ) {
			$before_txt = Settings::get_instance()->options['msg_before_txt'];
		}

		/** Empty field. */
		if ( ' ' === $before_txt ) {
			$before_txt = '';
		}

		?>
		<tr>
			<th scope="row">
				<label for="mdpbeforetxt"><?php esc_html_e( 'Before Text', 'contacter' ); ?>:</label>
			</th>
			<td>
				<?php wp_editor( $before_txt, 'mdpbeforetxt', ['textarea_rows' => 5, 'textarea_name' => 'mdp_before_txt'] ); ?>
				<p class="description"><?php esc_html_e( 'Enter text before "Start recording" button or leave blank to do not use the field.', 'contacter' ); ?></p>
			</td>
		</tr>
		<?php
	}

    /**
     * Render Record Duration button fieldset.
     *
     * @param $contacter_form - Current Contacter Form Object.
     *
     * @since 1.3.1
     **/
    private function record_duration( $contacter_form ) {
        ?>
        <tr>
            <th scope="row">
                <label for="mdp_form_record_duration"><?php esc_html_e( 'Max duration', 'contacter' ); ?>:</label>
            </th>
            <td>
                <?php esc_html_e( 'Max records duration for the current form', 'contacter' ); ?>
                <?php

                /** Record Duration */
                $this->form_record_duration( $contacter_form );

                ?>
            </td>
        </tr>
        <?php
    }

    /**
     * Render Attachments properties fieldset.
     *
     * @param $contacter_form - Current Contacter Form Object.
     *
     * @since 1.3.1
     **/
    private function attachment_properties( $contacter_form ) {
        ?>
        <tr>
            <th scope="row">
                <label for="mdp_form_attachment_properties"><?php esc_html_e( 'Attachment properties', 'contacter' ); ?>:</label>
            </th>
            <td>
                <?php

                // File original name
                $this->form_file_original( $contacter_form );

                // Meme types
                $this->form_mime( $contacter_form );

                // File size limit
                $this->form_file_size( $contacter_form );

                ?>
            </td>
        </tr>
        <?php
    }

	/**
     * Render Form suffix property
	 * @param $contacter_form
	 */
	private function form_suffix( $contacter_form ) {
		?>
        <tr>
            <th scope="row">
                <label for="mdp_field_record_suffix"><?php esc_html_e( 'Record suffix', 'contacter' ); ?>: </label>
            </th>
            <td>
				<?php

				// Record Form Suffix
				$this->record_suffix( $contacter_form );

				?>
            </td>
        </tr>
		<?php
	}

	/**
	 * Notification settings row
	 * @param $contacter_form
	 *
	 * @return void
	 */
	private function form_notification( $contacter_form ) {

		?>
		<tr>
			<th scope="row">
				<label for="mdp_field_notification"><?php esc_html_e( 'Form notification', 'contacter' ); ?>: </label>
			</th>
			<td>
				<?php $this->form_notification_field( $contacter_form ); ?>
			</td>
		</tr>
		<?php

	}

	/**
     * Record form suffix
	 * @param $contacter_form
	 */
    private function record_suffix( $contacter_form ) {

        /** Get size value from meta if it's already been entered. */
        $value = get_post_meta( $contacter_form->ID, 'mdp_field_record_suffix', true );

        ?>
        <div class="mdp-control-field">
            <?php
            UI::get_instance()->render_input(
                $value,
                esc_html__( 'Record Name Suffix', 'contacter' ),
                esc_html__( 'Enter the name of the field, the content of which will be added as a suffix to the name of the record', 'contacter' ),
                [
                    'name' => 'mdp_field_record_suffix',
                    'id' => 'mdp_field_record_suffix',
                ]
            );
            ?>
        </div>
        <?php

    }

	/**
	 * Render form notification field
	 * @param $contacter_form
	 */
	private function form_notification_field( $contacter_form ) {

		/** Get size value from meta if it's already been entered. */
		$value = get_post_meta( $contacter_form->ID, 'mdp_field_notification', true );

		?><div class="mdp-control-field"><?php
			UI::get_instance()->render_select(
				$this->notifictations_list(),
				$value ?? 'all',
				esc_html__( 'Notification', 'contacter' ),
				esc_html__( 'Select the notification that will be triggered after the form is submitted', 'contacter' ),
				[
					'name' => 'mdp_field_notification',
					'id' => 'mdp_field_notification',
					'class' => 'mdp-chosen',
					'data-placeholder' => esc_html__( 'Select notification', 'contacter' ),
					'data-allow-clear' => 'true',
				]
			);
		?></div><?php

	}

	/**
	 * Get notifications list
	 * @return array
	 */
	private function notifictations_list(): array {

		$notifications = get_posts( [
			'post_type' => 'notification',
			'posts_per_page' => -1,
		] );

		$notifications_list = [];

		$notifications_list['all'] = esc_html__( 'All', 'contacter' );
		foreach ( $notifications as $notification ) {
			$notifications_list[ 'form-' . $notification->ID ] = $notification->post_title;
		}
		$notifications_list['none'] = esc_html__( 'No one', 'contacter' );

		return $notifications_list;

	}

	/**
	 * Render Start Recording button fieldset.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function start_recording( $contacter_form ) {
		?>
        <tr>
            <th scope="row">
                <label for="mdpaftertxt"><?php esc_html_e( 'Start recording button', 'contacter' ); ?>:</label>
            </th>
            <td>
               <fieldset class="mdp-contacter-start-btn-box">
                    <?php

                    /** Render Align. */
                    $this->align( $contacter_form );

                    /** Border Radius Padding. */
                    $this->btn_radius( $contacter_form );

                    /** Render Margin. */
                    $this->btn_margin( $contacter_form );

                    /** Render Padding. */
                    $this->btn_padding( $contacter_form );

                    /** Button Caption. */
                    $this->btn_caption( $contacter_form );

                    /** Button Icon. */
                    $this->btn_icon( $contacter_form );

                    /** Button Icon Position. */
                    $this->btn_icon_position( $contacter_form );

                    /** Icon/Font size. */
                    $this->btn_size( $contacter_form );

                    /** Button Text Color. */
                    $this->btn_color( $contacter_form );

                    /** Button Background Color. */
                    $this->btn_bg_color( $contacter_form );

                    /** Button Text Hover Color. */
                    $this->btn_color_hover( $contacter_form );

                    /** Button Background Color. */
                    $this->btn_bg_color_hover( $contacter_form );

                    /** Button Hover Animations. */
                    $this->btn_hover_animation( $contacter_form );

                    ?>
               </fieldset>
               <p class="description"><?php esc_html_e( 'Customize the Look & Feel of the "Start recording" button.', 'contacter' ); ?></p>
            </td>
        </tr>
		<?php
    }

	/**
	 * Render skip Recording button fieldset.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function skip_recording( $contacter_form )
    {

	    $mdp_skip_button = get_post_meta( $contacter_form->ID, 'mdp_skip_button', true );
		?>
        <tr>
            <th scope="row">
                <label for="mdpaftertxt"><?php esc_html_e( 'Skip recording button', 'contacter' ); ?>:</label>
            </th>
            <td>
                <div class="mdp-skip-button-switcher">
		            <?php
		            UI::get_instance()->render_switcher(
			            $mdp_skip_button,
			            esc_html__('Display the Skip button', 'contacter' ),
			            '',
			            [
				            'name' => 'mdp_skip_button',
				            'id' => 'mdp_skip_button'
			            ]
		            );
		            ?>
                    <p class="switch-description"><?php esc_html_e( 'Add a button to skip the recording stage. Skip button will show only if form have at least one additional field.', 'contacter' ); ?></p>
                </div>
                <fieldset class="mdp-contacter-skip-btn-box">
					<?php

					/** Border Radius Padding. */
					$this->btn_radius( $contacter_form, '_skip' );

					/** Render Margin. */
					$this->btn_margin( $contacter_form, '_skip' );

					/** Render Padding. */
					$this->btn_padding( $contacter_form, '_skip' );

					/** Button Caption. */
					$this->btn_caption( $contacter_form, '_skip' );

					/** Button Icon. */
					$this->btn_icon( $contacter_form, '_skip' );

					/** Button Icon Position. */
					$this->btn_icon_position( $contacter_form, '_skip' );

					/** Icon/Font size. */
					$this->btn_size( $contacter_form, '_skip' );

					/** Button Text Color. */
					$this->btn_color( $contacter_form, '_skip' );

					/** Button Background Color. */
					$this->btn_bg_color( $contacter_form, '_skip' );

					/** Button Text Hover Color. */
					$this->btn_color_hover( $contacter_form, '_skip' );

					/** Button Background Color. */
					$this->btn_bg_color_hover( $contacter_form, '_skip' );

					?>
                </fieldset>

            </td>
        </tr>
		<?php
	}

	/**
	 * Render Margin slider for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_margin( $contacter_form, $suffix = '' ) {

		/** Get Margin value from meta if it's already been entered. */
		$mdp_btn_margin = get_post_meta( $contacter_form->ID, 'mdp_btn_margin' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_margin === '' ) {
			$mdp_btn_margin = '10';
		}

        ?>
        <div class="mdp-control-field mdp-control-field-2">
            <?php
            /** Margin slider. */
            UI::get_instance()->render_slider(
	            $mdp_btn_margin,
	            0,
	            100,
	            1,
	            esc_html__( 'Button Margin', 'contacter' ),
	            esc_html__( 'Button Margin', 'contacter' ). ': ' .
	            '<strong>' . $mdp_btn_margin . '</strong>' . ' ' .
	            esc_html__( 'pixel(s)', 'contacter' ),
	            [
		            'name' => 'mdp_btn_margin' . $suffix,
		            'id' => 'mdp_btn_margin' . $suffix,
	            ]
            );
            ?>
        </div>
        <?php
    }

	/**
	 * Render Icon/Caption size for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_size( $contacter_form, $suffix = '' ) {

		/** Get size value from meta if it's already been entered. */
		$mdp_btn_size = get_post_meta( $contacter_form->ID, 'mdp_btn_size' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_size === '' ) {
			$mdp_btn_size = '18';
		}

		?>
        <div class="mdp-control-field mdp-control-field-2">
			<?php
			/** Icon/Caption size slider. */
			UI::get_instance()->render_slider(
				$mdp_btn_size,
				10,
				100,
				1,
				esc_html__( 'Size', 'contacter' ),
				esc_html__( 'Icon/Caption size', 'contacter' ) . ': ' .
				'<strong>' . $mdp_btn_size . '</strong> ' .
				esc_html__( 'pixel(s)', 'contacter' ),
				[
					'name' => 'mdp_btn_size' . $suffix,
					'id' => 'mdp_btn_size' . $suffix,
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Padding slider for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_padding( $contacter_form, $suffix = '' ) {

		/** Get Padding value from meta if it's already been entered. */
		$mdp_btn_padding = get_post_meta( $contacter_form->ID, 'mdp_btn_padding' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_padding === '' ) {
			$mdp_btn_padding = '20';
		}

		?>
        <div class="mdp-control-field mdp-control-field-2">
			<?php
			/** Padding slider. */
			UI::get_instance()->render_slider(
				$mdp_btn_padding,
				0,
				100,
				1,
				esc_html__( 'Button Padding', 'contacter' ),
				esc_html__( 'Button Padding', 'contacter' ) . ': ' .
				'<strong>' . $mdp_btn_padding . '</strong> ' .
				esc_html__( 'pixel(s)', 'contacter' ),
				[
					'name' => 'mdp_btn_padding' . $suffix,
					'id' => 'mdp_btn_padding' . $suffix,
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Border Radius slider for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_radius( $contacter_form, $suffix = '' ) {

		/** Get Radius value from meta if it's already been entered. */
		$mdp_btn_radius = get_post_meta( $contacter_form->ID, 'mdp_btn_radius' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_radius === '' ) {
			$mdp_btn_radius = '50';
		}

		?>
        <div class="mdp-control-field">
			<?php
			/** Radius slider. */
			UI::get_instance()->render_slider(
				$mdp_btn_radius,
				0,
				100,
				1,
				esc_html__( 'Button Radius', 'contacter' ),
				esc_html__( 'Button Radius', 'contacter' ) . ': ' .
				'<strong>' . $mdp_btn_radius . '</strong> ' .
				esc_html__( 'pixel(s)', 'contacter' ),
				[
					'name' => 'mdp_btn_radius' . $suffix,
					'id' => 'mdp_btn_radius' . $suffix,
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Button Icon for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_icon( $contacter_form, $suffix = '' ) {

		/** Get icon value from meta if it's already been entered. */
		$mdp_btn_icon = get_post_meta( $contacter_form->ID, 'mdp_btn_icon' . $suffix, true );

		/** Default value. */
		if ( '' === $mdp_btn_icon ) {
			$mdp_btn_icon = $suffix === '' ? '_contacter/waves.svg' : 'font-awesome/arrow-right.svg';
		}

		/** We use this to detect empty icon and first time loading. */
		if ( ' ' === $mdp_btn_icon ) {
			$mdp_btn_icon = '';
		}

		?>
        <div class="mdp-control-field">
			<?php
			/** Button icon icon. */
			UI::get_instance()->render_icon(
				$mdp_btn_icon,
				'',
				esc_html__( 'Select icon for button', 'contacter' ),
				[
					'name' => 'mdp_btn_icon' . $suffix,
					'id' => 'mdp_btn_icon' . $suffix
				],
				[
					'_contacter.json',
					'font-awesome.json',
					'material.json',
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Align for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function align( $contacter_form, $suffix = '' ) {

		/** Get align value from meta if it's already been entered. */
		$mdp_align = get_post_meta( $contacter_form->ID, 'mdp_align' . $suffix, true );

		/** Default value. */
		if ( $mdp_align === '' ) {
			$mdp_align = 'left';
		}

		/** Align options. */
		$options = [
			'none'      => esc_html__( 'None', 'contacter' ),
			'left'      => esc_html__( 'Left', 'contacter' ),
			'center'    => esc_html__( 'Center', 'contacter' ),
			'right'     => esc_html__( 'Right', 'contacter' ),
		];

		?><div class="mdp-control-field"><?php

		/** Render Align dropdown. */
		UI::get_instance()->render_select(
			$options,
			$mdp_align, // Selected option.
			esc_html__( 'Align', 'contacter' ),
			esc_html__( 'Choose how to align the button and other form elements.', 'contacter' ),
			[
				'name' => 'mdp_align' . $suffix,
				'id' => 'mdp_align' . $suffix
			]
		);

		?></div><?php

	}

	/**
	 * Render Button Icon Position for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_icon_position( $contacter_form, $suffix = '' ) {

		/** Get icon position value from meta if it's already been entered. */
		$mdp_btn_icon_position = get_post_meta( $contacter_form->ID, 'mdp_btn_icon_position' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_icon_position === '' ) {
			$mdp_btn_icon_position = 'before';
		}

		/** Icon Position options. */
		$options = [
			'none'   => esc_html__( 'Hide', 'contacter' ),
			'before' => esc_html__( 'Before', 'contacter' ),
			'after'  => esc_html__( 'After', 'contacter' ),
			'above'  => esc_html__( 'Above', 'contacter' ),
			'bellow' => esc_html__( 'Bellow', 'contacter' ),
		];

		?><div class="mdp-control-field mdp-control-field-2"><?php

		/** Render Icon Position dropdown. */
		UI::get_instance()->render_select(
			$options,
			$mdp_btn_icon_position, // Selected option.
			esc_html__( 'Icon Position', 'contacter' ),
			esc_html__( 'Position of the icon inside the button', 'contacter' ),
			[
				'name' => 'mdp_btn_icon_position' . $suffix,
				'id' => 'mdp_btn_icon_position' . $suffix
			]
		);

        ?></div><?php

    }

	/**
	 * Render Hover Animations for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_hover_animation( $contacter_form, $suffix = '' ) {

		/** Get hover animation value from meta if it's already been entered. */
		$mdp_btn_hover_animation = get_post_meta( $contacter_form->ID, 'mdp_btn_hover_animation' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_hover_animation === '' ) {
			$mdp_btn_hover_animation = 'none';
		}

		/** Hover Animations options. */
		$options = [
			'none'                  => esc_html__( 'None', 'contacter' ),
			'fade'                  => esc_html__( 'Fade', 'contacter' ),
			'bounce'                => esc_html__( 'Bounce', 'contacter' ),
			'flip-x'                => esc_html__( 'Flip X', 'contacter' ),
			'flip-y'                => esc_html__( 'Flip Y', 'contacter' ),
			'scale'                 => esc_html__( 'Scale', 'contacter' ),
			'wobble'                => esc_html__( 'Wobble', 'contacter' ),
			'rotate'                => esc_html__( 'Rotate', 'contacter' )
		];

		?><div class="mdp-control-field"><?php

		/** Render Hover Animations dropdown. */
		UI::get_instance()->render_select(
			$options,
			$mdp_btn_hover_animation, // Selected option.
			esc_html__( 'Hover animation', 'contacter' ),
			esc_html__( 'Button hover animation', 'contacter' ),
			[
				'name' => 'mdp_btn_hover_animation' . $suffix,
				'id' => 'mdp_btn_hover_animation' . $suffix
			]
		);

		?></div><?php

	}

	/**
	 * Render Caption filed for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_caption( $contacter_form, $suffix = '' ) {

		/** Get Caption value from meta if it's already been entered. */
		$mdp_btn_caption = get_post_meta( $contacter_form->ID, 'mdp_btn_caption' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_caption === '' ) {
			$mdp_btn_caption = $suffix === '' ? esc_html__( 'Record', 'contacter' ) : esc_html__( 'Skip', 'contacter' );
		}

		?>
        <div class="mdp-control-field">
			<?php
			/** Caption input. */
			UI::get_instance()->render_input(
				$mdp_btn_caption,
				esc_html__( 'Caption', 'contacter' ),
				esc_html__( 'Start record button caption', 'contacter' ),
				[
					'name' => 'mdp_btn_caption' . $suffix,
					'id' => 'mdp_btn_caption' . $suffix
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Text/Icon color for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_color( $contacter_form, $suffix = '' ) {

		/** Get Caption value from meta if it's already been entered. */
		$mdp_btn_color = get_post_meta( $contacter_form->ID, 'mdp_btn_color' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_color === '' ) {
			$mdp_btn_color = '#fff';
		}

		?>
        <div class="mdp-control-field mdp-control-field-2">
			<?php
			/** Text/Icon Color colorpicker. */
			UI::get_instance()->render_colorpicker(
				$mdp_btn_color,
				esc_html__( 'Text Color', 'contacter' ),
				esc_html__( 'Select icon and text color', 'contacter' ),
				[
					'name' => 'mdp_btn_color' . $suffix,
					'id' => 'mdp_btn_color' . $suffix,
					'readonly' => 'readonly'
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Text/Icon Hover color for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_color_hover( $contacter_form, $suffix = '' ) {

		/** Get Caption value from meta if it's already been entered. */
		$mdp_btn_color_hover = get_post_meta( $contacter_form->ID, 'mdp_btn_color_hover' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_color_hover === '' ) {
			$mdp_btn_color_hover = '#0274e6';
		}

		?>
        <div class="mdp-control-field mdp-control-field-2">
			<?php
			/** Text/Icon Color colorpicker. */
			UI::get_instance()->render_colorpicker(
				$mdp_btn_color_hover,
				esc_html__( 'Text Hover Color', 'contacter' ),
				esc_html__( 'Select icon and text hover color', 'contacter' ),
				[
					'name' => 'mdp_btn_color_hover' . $suffix,
					'id' => 'mdp_btn_color_hover' . $suffix,
					'readonly' => 'readonly'
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Background color for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_bg_color( $contacter_form, $suffix = '' ) {

		/** Get Caption value from meta if it's already been entered. */
		$mdp_btn_bg_color = get_post_meta( $contacter_form->ID, 'mdp_btn_bg_color' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_bg_color === '' ) {
			$mdp_btn_bg_color = '#0274e6';
		}

		?>
        <div class="mdp-control-field mdp-control-field-2">
			<?php
			/** Text/Icon Color colorpicker. */
			UI::get_instance()->render_colorpicker(
				$mdp_btn_bg_color,
				esc_html__( 'Background Color', 'contacter' ),
				esc_html__( 'Select button background color', 'contacter' ),
				[
					'name' => 'mdp_btn_bg_color' . $suffix,
					'id' => 'mdp_btn_bg_color' . $suffix,
					'readonly' => 'readonly'
				]
			);
			?>
        </div>
		<?php
	}

	/**
	 * Render Background Hover color for Start button.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function btn_bg_color_hover( $contacter_form, $suffix = '' ) {

		/** Get Caption value from meta if it's already been entered. */
		$mdp_btn_bg_color_hover = get_post_meta( $contacter_form->ID, 'mdp_btn_bg_color_hover' . $suffix, true );

		/** Default value. */
		if ( $mdp_btn_bg_color_hover === '' ) {
			$mdp_btn_bg_color_hover = '#fff';
		}

		?>
        <div class="mdp-control-field mdp-control-field-2">
			<?php
			/** Text/Icon Color colorpicker. */
			UI::get_instance()->render_colorpicker(
				$mdp_btn_bg_color_hover,
				esc_html__( 'Background Hover Color', 'contacter' ),
				esc_html__( 'Select button background hover color', 'contacter' ),
				[
					'name' => 'mdp_btn_bg_color_hover' . $suffix,
					'id' => 'mdp_btn_bg_color_hover' . $suffix,
					'readonly' => 'readonly'
				]
			);
			?>
        </div>
		<?php
	}

    /**
     * Render Max Record Duration
     *
     * @param $contacter_form
     *
     * @since 1.3.1
     */
    private function form_record_duration ( $contacter_form ) {

        /** Get size value from meta if it's already been entered. */
        $mdp_form_record_duration = get_post_meta( $contacter_form->ID, 'mdp_form_record_duration', true );

        /** @var array $options - Plugin options */
        $options = Settings::get_instance()->options;

        /** Default value. */
        if ( $mdp_form_record_duration === '' ) {
            $mdp_form_record_duration = $options['max_duration'];
        }

        ?>
        <div class="mdp-control-field">
            <?php
            /** Record duration */
            UI::get_instance()->render_slider(
                $mdp_form_record_duration,
                0,
                300,
                1,
                esc_html__( 'Duration', 'contacter' ),
                esc_html__( 'Max recording duration: ', 'contacter' ) .
                '<strong>' . $mdp_form_record_duration . '</strong>' .
                esc_html__( ' seconds. Set 0 to unlimited record time.', 'contacter' ),
                [
                    'name' => 'mdp_form_record_duration',
                    'id' => 'mdp_form_record_duration',
                ]
            );
            ?>
        </div>
        <?php

    }

    /**
     * Attachments file size limit limit
     * @param $contacter_form
     */
    private function form_file_size( $contacter_form ) {

        /** Get size value from meta if it's already been entered. */
        $mdp_form_file_size = get_post_meta( $contacter_form->ID, 'mdp_form_file_size', true );

        /** Default value. */
        if ( $mdp_form_file_size === '' ) {
            $mdp_form_file_size = 256;
        }

        ?>
        <div class="mdp-control-field">
            <?php
            /** Record duration */
            UI::get_instance()->render_slider(
                $mdp_form_file_size,
                256,
                10240,
                256,
                esc_html__( 'File Size', 'contacter' ),
                esc_html__( 'Max attachments file size', 'contacter' ) . ': ' .
                '<strong>' . $mdp_form_file_size . '</strong> kB',
                [
                    'name' => 'mdp_form_file_size',
                    'id' => 'mdp_form_file_size',
                ]
            );
            ?>
        </div>
        <?php

    }

	/**
     * Attachments original name
	 * @param $contacter_form
	 *
	 * @return void
	 */
	private function form_file_original( $contacter_form ) {

		/** Get size value from meta if it's already been entered. */
		$mdp_form_file_original = get_post_meta( $contacter_form->ID, 'mdp_form_file_original', true );

		/** Default value. */
		if ( $mdp_form_file_original === '' ) {
			$mdp_form_file_original = 'on';
		}

		?>
        <div class="mdp-control-field mdp-form-file-original">
			<?php
            UI::get_instance()->render_switcher(
	            $mdp_form_file_original,
	            esc_html__('Use original file name', 'contacter' ),
	            '',
	            [
		            'name' => 'mdp_form_file_original',
		            'id' => 'mdp_form_file_original'
	            ]
            );
			?>
        </div>
		<?php

	}

    /**
     * Attachments MIME type
     * @param $contacter_form
     */
    private function form_mime( $contacter_form ) {

        /** Get size value from meta if it's already been entered. */
        $mdp_form_mime = get_post_meta( $contacter_form->ID, 'mdp_form_mime', true );

        /** Default value. */
        if ( $mdp_form_mime === '' ) {
            $mdp_form_mime = '*';
        }

        ?>
        <div class="mdp-control-field">
            <?php
            UI::get_instance()->render_input(
                $mdp_form_mime,
                esc_html__( 'Mime types', 'contacter' ),
                esc_html__( 'List the attachments ', 'contacter' ) .
                '<a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/MIME_types" target="_blank">' .
                esc_html__( 'MIME Types', 'contacter' ) .
                '</a>' .
                esc_html__( ' separated by commas. Or put * to accept any files.', 'contacter' ),
                [
                    'name' => 'mdp_form_mime',
                    'id' => 'mdp_form_mime',
                ]
            );
            ?>
        </div>
        <?php

    }

	/**
	 * Render After text field.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function after_text( $contacter_form ) {

		/** Get After Text field value from meta if it's already been entered. */
		$after_txt = get_post_meta( $contacter_form->ID, 'mdp_after_txt', true );

		/** Default value. */
		if ( empty( $after_txt ) ) {
			$after_txt = Settings::get_instance()->options['msg_after_txt'];
		}

		/** Empty field. */
		if ( ' ' === $after_txt ) {
			$after_txt = '';
		}
		?>
        <tr>
            <th scope="row">
                <label for="mdpaftertxt"><?php esc_html_e( 'After Text', 'contacter' ); ?>:</label>
            </th>
            <td>
				<?php wp_editor( $after_txt, 'mdpaftertxt', ['textarea_rows' => 5, 'textarea_name' => 'mdp_after_txt'] ); ?>
                <p class="description"><?php esc_html_e( 'Enter text after "Start recording" button or leave blank to do not use the field.', 'contacter' ); ?></p>
            </td>
        </tr>
		<?php
	}

	/**
	 * Render Thank You message field.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function thanks_text( $contacter_form ) {

		/** Get Thank you Text field value from meta if it's already been entered. */
		$thanks_txt = get_post_meta( $contacter_form->ID, 'mdp_thanks_txt', true );

		/** Default value. */
		if ( empty( $thanks_txt ) ) {
			$thanks_txt = Settings::get_instance()->options['msg_thank_you'];
		}
		?>
        <tr>
            <th scope="row">
                <label><?php esc_html_e( 'Thank You Text', 'contacter' ); ?>:</label>
            </th>
            <td>
				<?php wp_editor( $thanks_txt, 'mdpthankstxt', ['textarea_rows' => 5, 'textarea_name' => 'mdp_thanks_txt'] ); ?>
            </td>
        </tr>
		<?php
	}

	/**
	 * Render Send recording field.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function send_text( $contacter_form ) {

		/** Get Send Text field value from meta if it's already been entered. */
		$send_txt = get_post_meta( $contacter_form->ID, 'mdp_send_txt', true );

		/** Default value. */
		if ( empty( $send_txt ) ) {
			$send_txt = Settings::get_instance()->options['msg_send'];
		}
		?>
        <tr>
            <th scope="row">
                <label><?php esc_html_e( 'Send Recording Text', 'contacter' ); ?>:</label>
            </th>
            <td>
				<?php wp_editor( $send_txt, 'mdpsendtxt', ['textarea_rows' => 5, 'textarea_name' => 'mdp_send_txt'] ); ?>
            </td>
        </tr>
		<?php
	}

	/**
	 * Render Speak Now field.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function speak_now_text( $contacter_form ) {

		/** Get Speak Now Text field value from meta if it's already been entered. */
		$speak_now_txt = get_post_meta( $contacter_form->ID, 'mdp_speak_now_txt', true );

		/** Default value. */
		if ( empty( $speak_now_txt ) ) {
			$speak_now_txt = Settings::get_instance()->options['msg_speak_now'];
		}
		?>
        <tr>
            <th scope="row">
                <label for="mdpaftertxt"><?php esc_html_e( 'Speak Now Text', 'contacter' ); ?>:</label>
            </th>
            <td>
				<?php wp_editor( $speak_now_txt, 'mdpspeaknowtxt', ['textarea_rows' => 5, 'textarea_name' => 'mdp_speak_now_txt'] ); ?>
                <p class="description"><?php esc_html_e( 'You can use special placeholders: {timer}, {max-duration}, {countdown}.', 'contacter' ); ?></p>
            </td>
        </tr>
		<?php
	}

	/**
	 * Render Additional fields switcher.
	 *
	 * @param $contacter_form - Current Contacter Form Object.
	 *
	 * @since 1.0.0
	 **/
	private function additional_fields( $contacter_form ) {

		/** Get Additional fields switcher value from meta if it's already been entered. */
		$mdp_additional_fields = get_post_meta( $contacter_form->ID, 'mdp_additional_fields', true );
		$mdp_additional_fields_fb = get_post_meta( $contacter_form->ID, 'mdp_additional_fields_fb', true );
		$mdp_additional_fields_res = get_post_meta( $contacter_form->ID, 'mdp_additional_fields_res', true );

		/** Default value. Additional Fields switcher. */
		if ( '' === $mdp_additional_fields ) {
			$mdp_additional_fields = 'off';
		}

		/** Default value. Form in JSON. */
		if ( '' === $mdp_additional_fields_fb ) {
			$mdp_additional_fields_fb = '[{"type":"text","label":"First Name","placeholder":"Enter your first name","className":"mdp-form-control mdp-form-control-name","name":"mdp-contacter-first-name","subtype":"text"},{"type":"text","subtype":"email","required":true,"label":"E-mail","placeholder":"Enter your e-mail","className":"mdp-form-control mdp-form-control-email","name":"mdp-contacter-e-mail"}]';
		}

		/** Default value. Form in HTML. */
		if ( '' === $mdp_additional_fields_res ) {
			$mdp_additional_fields_res = '<div class="rendered-form"><div class="fb-text form-group field-first-name"><label for="first-name" class="fb-text-label">First Name</label></div><div class="fb-text form-group field-e-mail"><label for="e-mail" class="fb-text-label">E-mail<span class="fb-required">*</span></label></div></div>';
		}

		?>
        <tr>
            <th scope="row">
                <label for="mdpaftertxt"><?php esc_html_e( 'Additional fields', 'contacter' ); ?>:</label>
            </th>
            <td>
                <div class="mdp-additional-fields-switcher">
                <?php
                /** Render Additional fields switcher. */
                UI::get_instance()->render_switcher(
	                $mdp_additional_fields,
	                esc_html__('Additional Fields', 'contacter' ),
	                '',
	                [
		                'name' => 'mdp_additional_fields',
		                'id' => 'mdp_additional_fields'
	                ]
                );
                ?>
                <p class="switch-description"><?php esc_html_e( 'Show to user a small form after recording a voice message.', 'contacter' ); ?></p>
                </div>

                <div class="mdp-form-builder-box">

                    <!--suppress HtmlFormInputWithoutLabel -->
                    <input name="mdp_additional_fields_fb"
                           type="hidden"
                           id="mdp_additional_fields_fb"
	                       value="<?php esc_attr_e( $mdp_additional_fields_fb ); ?>"
                    >
                    <!--suppress HtmlFormInputWithoutLabel -->
	                <input name="mdp_additional_fields_res"
	                       type="hidden"
	                       id="mdp_additional_fields_res"
	                       value="<?php esc_attr_e( $mdp_additional_fields_res ); ?>"
	                >
                    <div id="mdp-form-builder-editor"></div>

                </div>

            </td>
        </tr>
		<?php
	}

	/**
	 * Render Nonce field to validate form request came from current site.
	 *
	 * @since 1.0.0
	 **/
	private function render_nonce() {

		wp_nonce_field( Plugin::get_basename(), 'options_metabox_fields_nonce' );

	}

	/**
	 * Main CFOptionsMetaBox Instance.
	 *
	 * Insures that only one instance of CFOptionsMetaBox exists in memory at any one time.
	 *
	 * @static
	 * @return CFOptionsMetaBox
	 * @since 1.0.0
	 **/
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof CFOptionsMetaBox ) ) {
			self::$instance = new CFOptionsMetaBox;
		}

		return self::$instance;
	}

} // End Class CFOptionsMetaBox.
