<?php
/**
 * Contacter
 * Voice feedback form for your website for saving and transcribing user voice messages to text.
 * Exclusively on https://1.envato.market/contacter
 *
 * @encoding        UTF-8
 * @version         1.7.8
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Contacter;

use Merkulove\Contacter\Unity\Plugin;
use Merkulove\Contacter\Unity\Settings;
use Merkulove\Contacter\Unity\TabGeneral;
use WP_Query;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Settings class used to modify default plugin settings.
 *
 * @since 1.0.0
 *
 **/
final class Config {

	/**
	 * The one true Settings.
	 *
     * @since 1.0.0
     * @access private
	 * @var Config
	 **/
	private static $instance;

	public static function language_options() {

		/** Prepare languages list. */
		return [
			'af-ZA'=> esc_html( 'Afrikaans (Suid-Afrika)' ),
			'am-ET'=> esc_html( 'አማርኛ (ኢትዮጵያ)' ),
			'hy-AM'=> esc_html( 'Հայ (Հայաստան)' ),
			'az-AZ'=> esc_html( 'Azərbaycan (Azərbaycan)' ),
			'id-ID'=> esc_html( 'Bahasa Indonesia (Indonesia)' ),
			'ms-MY'=> esc_html( 'Bahasa Melayu (Malaysia)' ),
			'bn-BD'=> esc_html( 'বাংলা (বাংলাদেশ)' ),
			'bn-IN'=> esc_html( 'বাংলা (ভারত)' ),
			'ca-ES'=> esc_html( 'Català (Espanya)' ),
			'cs-CZ'=> esc_html( 'Čeština (Česká republika)' ),
			'da-DK'=> esc_html( 'Dansk (Danmark)' ),
			'de-DE'=> esc_html( 'Deutsch (Deutschland)' ),
			'en-AU'=> esc_html( 'English (Australia)' ),
			'en-CA'=> esc_html( 'English (Canada)' ),
			'en-GH'=> esc_html( 'English (Ghana)' ),
			'en-GB'=> esc_html( 'English (Great Britain)' ),
			'en-IN'=> esc_html( 'English (India)' ),
			'en-IE'=> esc_html( 'English (Ireland)' ),
			'en-KE'=> esc_html( 'English (Kenya)' ),
			'en-NZ'=> esc_html( 'English (New Zealand)' ),
			'en-NG'=> esc_html( 'English (Nigeria)' ),
			'en-PH'=> esc_html( 'English (Philippines)' ),
			'en-SG'=> esc_html( 'English (Singapore)' ),
			'en-ZA'=> esc_html( 'English (South Africa)' ),
			'en-TZ'=> esc_html( 'English (Tanzania)' ),
			'en-US'=> esc_html( 'English (United States)' ),
			'es-AR'=> esc_html( 'Español (Argentina)' ),
			'es-BO'=> esc_html( 'Español (Bolivia)' ),
			'es-CL'=> esc_html( 'Español (Chile)' ),
			'es-CO'=> esc_html( 'Español (Colombia)' ),
			'es-CR'=> esc_html( 'Español (Costa Rica)' ),
			'es-EC'=> esc_html( 'Español (Ecuador)' ),
			'es-SV'=> esc_html( 'Español (El Salvador)' ),
			'es-ES'=> esc_html( 'Español (España)' ),
			'es-US'=> esc_html( 'Español (Estados Unidos)' ),
			'es-GT'=> esc_html( 'Español (Guatemala)' ),
			'es-HN'=> esc_html( 'Español (Honduras)' ),
			'es-MX'=> esc_html( 'Español (México)' ),
			'es-NI'=> esc_html( 'Español (Nicaragua)' ),
			'es-PA'=> esc_html( 'Español (Panamá)' ),
			'es-PY'=> esc_html( 'Español (Paraguay)' ),
			'es-PE'=> esc_html( 'Español (Perú)' ),
			'es-PR'=> esc_html( 'Español (Puerto Rico)' ),
			'es-DO'=> esc_html( 'Español (República Dominicana)' ),
			'es-UY'=> esc_html( 'Español (Uruguay)' ),
			'es-VE'=> esc_html( 'Español (Venezuela)' ),
			'eu-ES'=> esc_html( 'Euskara (Espainia)' ),
			'fil-PH'=> esc_html( 'Filipino (Pilipinas)' ),
			'fr-CA'=> esc_html( 'Français (Canada)' ),
			'fr-FR'=> esc_html( 'Français (France)' ),
			'gl-ES'=> esc_html( 'Galego (España)' ),
			'ka-GE'=> esc_html( 'ქართული (საქართველო)' ),
			'gu-IN'=> esc_html( 'ગુજરાતી (ભારત)' ),
			'hr-HR'=> esc_html( 'Hrvatski (Hrvatska)' ),
			'zu-ZA'=> esc_html( 'IsiZulu (Ningizimu Afrika)' ),
			'is-IS'=> esc_html( 'Íslenska (Ísland)' ),
			'it-IT'=> esc_html( 'Italiano (Italia)' ),
			'jv-ID'=> esc_html( 'Jawa (Indonesia)' ),
			'kn-IN'=> esc_html( 'ಕನ್ನಡ (ಭಾರತ)' ),
			'km-KH'=> esc_html( 'ភាសាខ្មែរ (កម្ពុជា)' ),
			'lo-LA'=> esc_html( 'ລາວ (ລາວ)' ),
			'lv-LV'=> esc_html( 'Latviešu (latviešu)' ),
			'lt-LT'=> esc_html( 'Lietuvių (Lietuva)' ),
			'hu-HU'=> esc_html( 'Magyar (Magyarország)' ),
			'ml-IN'=> esc_html( 'മലയാളം (ഇന്ത്യ)' ),
			'mr-IN'=> esc_html( 'मराठी (भारत)' ),
			'nl-NL'=> esc_html( 'Nederlands (Nederland)' ),
			'ne-NP'=> esc_html( 'नेपाली (नेपाल)' ),
			'nb-NO'=> esc_html( 'Norsk bokmål (Norge)' ),
			'pl-PL'=> esc_html( 'Polski (Polska)' ),
			'pt-BR'=> esc_html( 'Português (Brasil)' ),
			'pt-PT'=> esc_html( 'Português (Portugal)' ),
			'ro-RO'=> esc_html( 'Română (România)' ),
			'si-LK'=> esc_html( 'සිංහල (ශ්රී ලංකාව)' ),
			'sk-SK'=> esc_html( 'Slovenčina (Slovensko)' ),
			'sl-SI'=> esc_html( 'Slovenščina (Slovenija)' ),
			'su-ID'=> esc_html( 'Urang (Indonesia)' ),
			'sw-TZ'=> esc_html( 'Swahili (Tanzania)' ),
			'sw-KE'=> esc_html( 'Swahili (Kenya)' ),
			'fi-FI'=> esc_html( 'Suomi (Suomi)' ),
			'sv-SE'=> esc_html( 'Svenska (Sverige)' ),
			'ta-IN'=> esc_html( 'தமிழ் (இந்தியா)' ),
			'ta-SG'=> esc_html( 'தமிழ் (சிங்கப்பூர்)' ),
			'ta-LK'=> esc_html( 'தமிழ் (இலங்கை)' ),
			'ta-MY'=> esc_html( 'தமிழ் (மலேசியா)' ),
			'te-IN'=> esc_html( 'తెలుగు (భారతదేశం)' ),
			'vi-VN'=> esc_html( 'Tiếng Việt (Việt Nam)' ),
			'tr-TR'=> esc_html( 'Türkçe (Türkiye)' ),
			'ur-PK'=> esc_html( 'اردو (پاکستان)' ),
			'ur-IN'=> esc_html( 'اردو (بھارت)' ),
			'el-GR'=> esc_html( 'Ελληνικά (Ελλάδα)' ),
			'bg-BG'=> esc_html( 'Български (България)' ),
			'ru-RU'=> esc_html( 'Русский (Россия)' ),
			'sr-RS'=> esc_html( 'Српски (Србија)' ),
			'uk-UA'=> esc_html( 'Українська (Україна)' ),
			'he-IL'=> esc_html( 'עברית (ישראל)' ),
			'ar-IL'=> esc_html( 'العربية (إسرائيل)' ),
			'ar-JO'=> esc_html( 'العربية (الأردن)' ),
			'ar-AE'=> esc_html( 'العربية (الإمارات)' ),
			'ar-BH'=> esc_html( 'العربية (البحرين)' ),
			'ar-DZ'=> esc_html( 'العربية (الجزائر)' ),
			'ar-SA'=> esc_html( 'العربية (السعودية)' ),
			'ar-IQ'=> esc_html( 'العربية (العراق)' ),
			'ar-KW'=> esc_html( 'العربية (الكويت)' ),
			'ar-MA'=> esc_html( 'العربية (المغرب)' ),
			'ar-TN'=> esc_html( 'العربية (تونس)' ),
			'ar-OM'=> esc_html( 'العربية (عُمان)' ),
			'ar-PS'=> esc_html( 'العربية (فلسطين)' ),
			'ar-QA'=> esc_html( 'العربية (قطر)' ),
			'ar-LB'=> esc_html( 'العربية (لبنان)' ),
			'ar-EG'=> esc_html( 'العربية (مصر)' ),
			'fa-IR'=> esc_html( 'فارسی (ایران)' ),
			'hi-IN'=> esc_html( 'हिन्दी (भारत)' ),
			'th-TH'=> esc_html( 'ไทย (ประเทศไทย)' ),
			'ko-KR'=> esc_html( '한국어 (대한민국)' ),
			'zh-TW'=> esc_html( '國語 (台灣)' ),
			'yue-Hant-HK'=> esc_html( '廣東話 (香港)' ),
			'ja-JP'=> esc_html( '日本語（日本）' ),
			'zh-HK'=> esc_html( '普通話 (香港)' ),
			'zh'=> esc_html( '普通话 (中国大陆)' )
		];

	}


	/**
     * Prepare plugin settings by modifying the default one.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function prepare_settings() {

        /** Get default plugin settings. */
        $tabs = Plugin::get_tabs();

        /** Shorthand access to plugin settings. */
        $options = Settings::get_instance()->options;

	    /** General tab */

	    $key = 'max_duration';
	    $default = Settings::get_instance()->options[ $key ] ?? 60;
	    $tabs['general']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Max Duration', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Max recording duration', 'contacter' ) . ': '.
		                           '<strong>' . $default . '</strong>' .
		                           esc_html__( ' seconds. Set 0 to unlimited record time.', 'contacter' ),
		    'min'               => 0,
		    'max'               => 300,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $key = 'sample_rate';
	    $default = Settings::get_instance()->options[ $key ] ?? 44100;
	    $tabs['general']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Sample rate', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Recording sample rate', 'contacter' )  . ': ' .
		                           '<strong>' . $default . '</strong>' .
		                           esc_html__( ' Hz. Decreasing this setting will make it impossible to recognize text from audio.', 'contacter' ),
		    'min'               => 4000,
		    'max'               => 48000,
		    'step'              => 4000,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $tabs['general']['fields']['accent_color'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Accent Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Accent color', 'contacter' ),
		    'description'       => esc_html__( 'Select accent color', 'contacter' ),
		    'default'           => '#0274e6',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $tabs['general']['fields']['media_library'] = [
		    'type'              => 'switcher',
		    'label'             => esc_html__( 'Media Library', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Visible in Media Library', 'contacter' ),
		    'description'       => esc_html__( 'Make the audio visible and available in the Media Library', 'contacter' ),
		    'default'           => 'off',
	    ];

	    $tabs['general']['fields']['show_download_link'] = [
		    'type'              => 'switcher',
		    'label'             => esc_html__( 'Download Link', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Show Download Link', 'contacter' ),
		    'description'       => esc_html__( 'Show download link to audio on frontend', 'contacter' ),
		    'default'           => 'off',
	    ];

	    $tabs['general']['fields']['download_link_text'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Download Link Text', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Enter Text value', 'contacter' ),
		    'description'       => esc_html__( 'Text for download link', 'contacter' ),
		    'default'           => esc_html__( 'Download record', 'contacter' ),
		    'attr'              => [
			    'maxlength' => '4500'
		    ]
	    ];

	    $tabs['general']['fields']['file_name'] = [
		    'type'              => 'switcher',
		    'label'             => esc_html__( 'Custom record name', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Custom file name', 'contacter' ),
		    'description'       => esc_html__( 'Use custom file name pattern for the new records', 'contacter' ),
		    'default'           => 'off',
	    ];

	    $tabs['general']['fields']['file_name_pattern'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'File name pattern', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'File name', 'contacter' ),
		    'description'       => '<code>[form-id]</code> – ' . esc_html__( 'ID of the Contacter form', 'contacter' ) . '<br>' .
		    '<code>[date]</code> – ' . esc_html__( 'Date of the record in format YYYY-MM-DD', 'contacter' ) . '<br>' .
		    '<code>[time]</code> – '. esc_html__( 'Time of the record in format hh:mm:ss', 'contacter' ),
		    'default'           => 'contacter-[form-id]-[date]-[time]',
		    'attr'              => [
			    'maxlength' => '4500'
		    ]
	    ];

	    $tabs['general']['fields']['show_record_id'] = [
		    'type'              => 'switcher',
		    'label'             => esc_html__( 'Display record ID', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Record ID', 'contacter' ),
		    'description'       => esc_html__( 'Display ID of the record on front-end and back-end', 'contacter' ),
		    'default'           => 'off',
	    ];

	    if ( defined('PHP_MAJOR_VERSION') && PHP_MAJOR_VERSION >= 7 ) {
		    $tabs['general']['fields']['speech_recognition'] = [
			    'type'              => 'switcher',
			    'label'             => esc_html__( 'Speech Recognition', 'contacter' ),
			    'placeholder'       => esc_html__( 'Enable Speech Recognition', 'contacter' ),
			    'description'       => esc_html__( 'Convert audio to text by applying powerful neural network models.', 'contacter' ),
			    'default'           => 'off',
		    ];
	    } else {
		    ?>
		    <div class="mdc-text-field-helper-text mdc-text-field-helper-text--persistent"
		         xmlns="http://www.w3.org/1999/html">
			    <?php esc_html_e( 'This functionality requires PHP 7 or higher.', 'contacter' ); ?>
		    </div>
		    <?php
	    }

	    $tabs['general']['fields']['dnd-api-key'] = [
		    'type'              => 'custom',
		    'render'            => [ self::get_instance(), 'dnd_api_key' ],
		    'label'             => esc_html__( 'API Key File', 'contacter' ) . ':',
		    'default'           => '',
	    ];

	    $tabs['general']['fields']['language'] = [
		    'type'              => 'select',
		    'label'             => esc_html__( 'Language', 'contacter' ),
		    'placeholder'       => esc_html__( 'Language', 'contacter' ),
		    'description'       => esc_html__( 'Select Language to convert audio to text.', 'contacter' ),
		    'default'           => 'en-US',
		    'options'           => self::language_options()
	    ];

	    $tabs['general']['fields']['punctuation'] = [
		    'type'              => 'switcher',
		    'label'             => esc_html__( 'Punctuation', 'contacter' ),
		    'placeholder'       => esc_html__( 'Punctuation', 'contacter' ),
		    'description'       => esc_html__( 'Accurately punctuates transcriptions (e.g., commas, question marks) with machine learning', 'contacter' ),
		    'default'           => 'on',
	    ];

	    /** Float button */

	    $offset = 1;
	    $tabs = array_slice( $tabs, 0, $offset, true ) +
	            [ 'floatbutton' => [
		            'enabled'       => true,
		            'class'         => TabGeneral::class, // Handler
		            'label'         => esc_html__( 'Floating Button', 'contacter' ),
		            'title'         => esc_html__( 'Floating Button', 'contacter' ),
		            'show_title'    => true,
		            'icon'          => 'edit_attributes', // Icon for tab
		            'fields'        => []
	            ] ] +
	            array_slice( $tabs, $offset, NULL, true );

	    $tabs['floatbutton']['fields']['show_fbutton'] = [
		    'type'              => 'switcher',
		    'label'             => esc_html__( 'Show Floating Button', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Floating button for all pages', 'contacter' ),
		    'description'       => esc_html__( 'Show or hide floating button for all pages. Alternatively button cat be added by shortcodes', 'contacter' ),
		    'default'           => 'on',
	    ];

	    /** Prepare Contacter forms options. */
        $forms = $this->get_forms();

        $forms_array = $this->get_forms();
	    reset($forms_array );
	    $first_form_id = key( $forms_array );

	    if ( count( $forms ) ) {

            $tabs['floatbutton']['fields']['fbutton_c_form'] = [
                'type'              => 'select',
                'label'             => esc_html__( 'Form to show', 'contacter' ) . ': ',
                'placeholder'       => esc_html__( 'Contacter Form', 'contacter' ),
                'description'       => esc_html__( 'Select Contacter Form to show by Floating Button click', 'contacter' ),
                'default'           => $first_form_id,
                'options'           => $forms
            ];

	    }

	    $tabs['floatbutton']['fields']['fbutton_position'] = [
		    'type'              => 'select',
		    'label'             => esc_html__( 'Position', 'contacter' ),
		    'placeholder'       => esc_html__( 'Button Position', 'contacter' ),
		    'description'       => esc_html__( 'Select a place on the page to display Floating Button', 'contacter' ),
		    'default'           => 'bottom-right',
		    'options'           => [
			    'top-left'      => esc_html__( 'Top Left', 'contacter' ),
			    'top-right'     => esc_html__( 'Top Right', 'contacter' ),
			    'left-center'   => esc_html__( 'Left Center', 'contacter' ),
			    'right-center'  => esc_html__( 'Right Center', 'contacter' ),
			    'bottom-left'   => esc_html__( 'Bottom Left', 'contacter' ),
			    'bottom-center' => esc_html__( 'Bottom Center', 'contacter' ),
			    'bottom-right'  => esc_html__( 'Bottom Right', 'contacter' ),
		    ]
	    ];

	    $tabs['floatbutton']['fields']['fbutton_caption'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Caption', 'contacter' ),
		    'placeholder'       => esc_html__( 'Button Caption', 'contacter' ),
		    'description'       => esc_html__( 'Start recording button caption', 'contacter' ),
		    'default'           => esc_html__( 'Button Caption', 'contacter' ),
		    'attr'              => [
			    'class'     => 'mdp-test-class',
			    'maxlength' => '4500'
		    ]
	    ];

        $tabs['floatbutton']['fields']['fbutton_icon'] = [
            'type'              => 'icon',
            'label'             => esc_html__( 'Icon', 'contacter' ),
            'placeholder'       => '',
            'description'       => esc_html__( 'Select icon for button', 'contacter' ),
            'default'           => '_contacter/contacter.svg',
            'meta'              => [
                '_contacter.json',
                'font-awesome.json',
                'material.json'
            ]
        ];

	    $tabs['floatbutton']['fields']['fbutton_icon_position'] = [
		    'type'              => 'select',
		    'label'             => esc_html__( 'Icon Position', 'contacter' ),
		    'placeholder'       => esc_html__( 'Icon Position', 'contacter' ),
		    'default'           => 'before',
		    'options'           => [
			    'none'   => esc_html__( 'Hide', 'contacter' ),
			    'before' => esc_html__( 'Before', 'contacter' ),
			    'after'  => esc_html__( 'After', 'contacter' ),
			    'above' => esc_html__( 'Above', 'contacter' ),
			    'bellow'   => esc_html__( 'Bellow', 'contacter' ),
		    ]
	    ];

	    $key = 'fbutton_size';
	    $default = Settings::get_instance()->options[ $key ] ?? 24;
	    $tabs['floatbutton']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Icon/Caption size', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Icon/Caption size', 'contacter' ) . ': <strong>' . $default . '</strong> ' . esc_html__( 'pixel(s)', 'contacter' ),
		    'min'               => 10,
		    'max'               => 100,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $key = 'fbutton_margin';
	    $default = Settings::get_instance()->options[ $key ] ?? 10;
	    $tabs['floatbutton']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Button Margin', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Button Margin', 'contacter' ) . ': <strong>' . $default . '</strong> ' . esc_html__( 'pixel(s)', 'contacter' ),
		    'min'               => 0,
		    'max'               => 100,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $key = 'fbutton_padding';
	    $default = Settings::get_instance()->options[ $key ] ?? 20;
	    $tabs['floatbutton']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Button Padding', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Button Padding', 'contacter' ) . ': <strong>' . $default . '</strong> ' . esc_html__( 'pixel(s)', 'contacter' ),
		    'min'               => 0,
		    'max'               => 100,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $key = 'fbutton_border_radius';
	    $default = Settings::get_instance()->options[ $key ] ?? 50;
	    $tabs['floatbutton']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Border Radius', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Border Radius', 'contacter' ) . ': <strong>' . $default . '</strong> ' . esc_html__( 'pixel(s)', 'contacter' ),
		    'min'               => 0,
		    'max'               => 100,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $tabs['floatbutton']['fields']['fbutton_color'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Icon/Caption Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Text Color', 'contacter' ),
		    'description'       => esc_html__( 'Select icon and text color', 'contacter' ),
		    'default'           => '#ffffff',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $tabs['floatbutton']['fields']['fbutton_color_hover'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Icon/Caption Hover Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Text Hover Color', 'contacter' ),
		    'description'       => esc_html__( 'Select icon and text hover color', 'contacter' ),
		    'default'           => '#0274e6',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $tabs['floatbutton']['fields']['fbutton_bgcolor'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Background Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Background Color', 'contacter' ),
		    'description'       => esc_html__( 'Select button background color', 'contacter' ),
		    'default'           => '#0274e6',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $tabs['floatbutton']['fields']['fbutton_bgcolor_hover'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Background Hover Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Background Hover Color', 'contacter' ),
		    'description'       => esc_html__( 'Select button hover background color', 'contacter' ),
		    'default'           => '#ffffff',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $key = 'fbutton_entrance_timeout';
	    $default = Settings::get_instance()->options[ $key ] ?? 0;
	    $tabs['floatbutton']['fields'][ $key ] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Entrance Timeout', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Entrance Timeout', 'contacter' )  . ': ' .
		                           '<strong>' . $default . '</strong>' .
		                           esc_html__( ' seconds', 'contacter' ),
		    'min'               => 0,
		    'max'               => 100,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $tabs['floatbutton']['fields']['fbutton_animation'] = [
		    'type'              => 'select',
		    'label'             => esc_html__( 'Entrance animation', 'contacter' ),
		    'placeholder'       => esc_html__( 'Entrance animation', 'contacter' ),
		    'description'       => esc_html__( 'Button Entrance animation', 'contacter' ),
		    'default'           => 'fade',
		    'options'           => [
			    'none'                  => esc_html__( 'None', 'contacter' ),
			    'bounce'                => esc_html__( 'Bounce', 'contacter' ),
			    'fade'                  => esc_html__( 'Fade', 'contacter' ),
			    'flip-x'                => esc_html__( 'Flip X', 'contacter' ),
			    'flip-y'                => esc_html__( 'Flip Y', 'contacter' ),
			    'scale'                 => esc_html__( 'Scale', 'contacter' ),
			    'wobble'                => esc_html__( 'Wobble', 'contacter' ),
			    'rotate'                => esc_html__( 'Rotate', 'contacter' )
		    ]
	    ];

	    $tabs['floatbutton']['fields']['fbutton_hover_animation'] = [
		    'type'              => 'select',
		    'label'             => esc_html__( 'Hover animation', 'contacter' ),
		    'placeholder'       => esc_html__( 'Hover animation', 'contacter' ),
		    'description'       => esc_html__( 'Button hover animation', 'contacter' ),
		    'default'           => 'none',
		    'options'           => [
			    'none'                  => esc_html__( 'None', 'contacter' ),
			    'bounce'                => esc_html__( 'Bounce', 'contacter' ),
			    'fade'                  => esc_html__( 'Fade', 'contacter' ),
			    'flip-x'                => esc_html__( 'Flip X', 'contacter' ),
			    'flip-y'                => esc_html__( 'Flip Y', 'contacter' ),
			    'scale'                 => esc_html__( 'Scale', 'contacter' ),
			    'wobble'                => esc_html__( 'Wobble', 'contacter' ),
			    'rotate'                => esc_html__( 'Rotate', 'contacter' )
		    ]
	    ];

	    /** Modal popup */

	    $offset = 2;
	    $tabs = array_slice( $tabs, 0, $offset, true ) +
	            [ 'modal' => [
		            'enabled'       => true,
		            'class'         => TabGeneral::class, // Handler
		            'label'         => esc_html__( 'Modal Popup', 'contacter'  ),
		            'title'         => esc_html__( 'Modal Popup', 'contacter' ),
		            'show_title'    => true,
		            'icon'          => 'web_asset', // Icon for tab
		            'fields'        => []
	            ] ] +
	            array_slice( $tabs, $offset, NULL, true );

	    $key = 'countdown_duration';
	    $default = Settings::get_instance()->options[$key] ?? 0;
	    $tabs['modal']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Countdown duration', 'contacter' )  . ':',
		    'description'       => esc_html__( 'Pre-record countdown', 'contacter' )  . ': ' .
		                           '<strong>' . $default . '</strong>' .
		                           esc_html__( ' seconds. Set 0 to disable countdown.', 'contacter' ),
		    'min'               => 0,
		    'max'               => 10,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $tabs['modal']['fields']['modal_overlay_color'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Overlay Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Overlay Color', 'contacter' ),
		    'description'       => esc_html__( 'Select modal overlay background-color', 'contacter' ),
		    'default'           => '#0253ee',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $tabs['modal']['fields']['modal_bg_color'] = [
		    'type'              => 'colorpicker',
		    'label'             => esc_html__( 'Background Color', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Background Color', 'contacter' ),
		    'description'       => esc_html__( 'Select modal background-color', 'contacter' ),
		    'default'           => '#ffffff',
		    'attr'              => [
			    'readonly'      => 'readonly',
		    ]
	    ];

	    $key = 'modal_radius';
	    $default = Settings::get_instance()->options[$key] ?? 10;
	    $tabs['modal']['fields'][$key] = [
		    'type'              => 'slider',
		    'label'             => esc_html__( 'Border Radius', 'contacter' ) . ':',
		    'description'       => esc_html__( 'Border Radius', 'contacter' )  . ': ' .
		                           '<strong>' . $default . '</strong> ' .
		                           esc_html__( 'pixel(s)', 'contacter' ),
		    'min'               => 0,
		    'max'               => 100,
		    'step'              => 1,
		    'default'           => $default,
		    'discrete'          => true,
	    ];

	    $tabs['modal']['fields']['modal_animation'] = [
		    'type'              => 'select',
		    'label'             => esc_html__( 'Animation', 'contacter' ),
		    'placeholder'       => esc_html__( 'Animation', 'contacter' ),
		    'description'       => esc_html__( 'Modal entrance animation', 'contacter' ),
		    'default'           => 'fade',
		    'options'           => [
			    'none'                  => esc_html__( 'None', 'contacter' ),
			    'bounce'                => esc_html__( 'Bounce', 'contacter' ),
			    'fade'                  => esc_html__( 'Fade', 'contacter' ),
			    'flip-x'                => esc_html__( 'Flip X', 'contacter' ),
			    'flip-y'                => esc_html__( 'Flip Y', 'contacter' ),
			    'scale'                 => esc_html__( 'Scale', 'contacter' ),
			    'slide-tr'              => esc_html__( 'Slide to right', 'contacter' ),
			    'slide-tl'              => esc_html__( 'Slide to left', 'contacter' ),
			    'slide-tt'              => esc_html__( 'Slide to top', 'contacter' ),
			    'slide-tb'              => esc_html__( 'Slide to bottom', 'contacter' ),
			    'rotate'                => esc_html__( 'Rotate', 'contacter' ),
			    'wobble'                => esc_html__( 'Wobble', 'contacter' )
		    ]
	    ];

	    /** Messages popup */

	    $offset = 3;
	    $tabs = array_slice( $tabs, 0, $offset, true ) +
	            [ 'messages' => [
		            'enabled'       => true,
		            'class'         => TabGeneral::class, // Handler
		            'label'         => esc_html__( 'Messages', 'contacter'  ),
		            'title'         => esc_html__( 'Messages', 'contacter' ),
                    'description'   => esc_html__( 'These are default messages that will be used to create new forms. Changing the settings on this page will not change the settings for existing forms.', 'contacter' ),
		            'show_title'    => true,
		            'icon'          => 'spellcheck', // Icon for tab
		            'fields'        => []
	            ] ] +
	            array_slice( $tabs, $offset, NULL, true );

	    $tabs['messages']['fields']['msg_before_txt'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Before Text', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'We would love to hear from you!', 'contacter' ) . '</h4><p>' . esc_html__( 'Please record your message.', 'contacter' ) . '</p>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_after_txt'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'After Text', 'contacter' ) . ':',
		    'default'           => '<p>' . esc_html__( 'Record, Listen, Send', 'contacter' ) . '</p>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_speak_now'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Speak Now', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Speak now', 'contacter' ) . '</h4><div>{countdown}</div>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_allow_access'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Allow Access', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Allow access to your microphone', 'contacter' ) . '</h4><p>' . esc_html__( 'Click "Allow" in the permission dialog. It usually appears under the address bar in the upper left side of the window. We respect your privacy.', 'contacter' ) . '</p>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_mic_access_err'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Access error', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Microphone access error', 'contacter' ) . '</h4><p>' . esc_html__( 'It seems your microphone is disabled in the browser settings. Please go to your browser settings and enable access to your microphone.', 'contacter' ) . '</p>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_reset_recording'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Reset recording', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Reset recording', 'contacter' ) . '</h4><p>' . esc_html__( 'Are you sure you want to start a new recording? Your current recording will be deleted.', 'contacter' ) . '</p>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_send'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Send recording', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Send your recording', 'contacter' ) . '</h4>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_sending_error'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( 'Sending error', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Oops, something went wrong', 'contacter' ) . '</h4><p>' . esc_html__( 'Error occurred during uploading your audio. Please click the Retry button to try again.', 'contacter' ) . '</p>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['msg_thank_you'] = [
		    'type'              => 'editor',
		    'label'             => esc_html__( '"Thank you" message', 'contacter' ) . ':',
		    'default'           => '<h4>' . esc_html__( 'Thank you', 'contacter' ) . '</h4>',
		    'attr'              => [
			    'textarea_rows' => '3',
		    ]
	    ];

	    $tabs['messages']['fields']['btn_stop'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Stop button text', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Stop', 'contacter' ),
		    'description'       => esc_html__( 'Text for Stop button.', 'contacter' ),
		    'default'           => esc_html__( 'Stop', 'contacter' ),
	    ];

	    $tabs['messages']['fields']['btn_pause'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Pause button text', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Pause', 'contacter' ),
		    'description'       => esc_html__( 'Text for Pause button.', 'contacter' ),
		    'default'           => esc_html__( 'Pause', 'contacter' ),
	    ];

	    $tabs['messages']['fields']['btn_send'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Send button text', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Send', 'contacter' ),
		    'description'       => esc_html__( 'Text for Send button.', 'contacter' ),
		    'default'           => esc_html__( 'Send', 'contacter' ),
	    ];

	    $tabs['messages']['fields']['btn_reset'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Reset button text', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Reset', 'contacter' ),
		    'description'       => esc_html__( 'Text for Reset button.', 'contacter' ),
		    'default'           => esc_html__( 'Reset', 'contacter' ),
	    ];


	    $tabs['messages']['fields']['btn_resume'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Resume button text', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Resume', 'contacter' ),
		    'description'       => esc_html__( 'Text for Resume button.', 'contacter' ),
		    'default'           => esc_html__( 'Resume', 'contacter' ),
	    ];

	    $tabs['messages']['fields']['btn_start_new'] = [
		    'type'              => 'text',
		    'label'             => esc_html__( 'Start a new Message button', 'contacter' ) . ':',
		    'placeholder'       => esc_html__( 'Start a New Message', 'contacter' ),
		    'description'       => esc_html__( 'Text for Start a New Message button.', 'contacter' ),
		    'default'           => esc_html__( 'Start a New Message', 'contacter' ),
	    ];

	    if ( isset( $options[ 'show_record_id' ] ) && $options[ 'show_record_id' ] === 'on' ) {

		    $tabs['messages']['fields']['msg_message_id'] = [
			    'type'              => 'text',
			    'label'             => esc_html__( 'Message ID', 'contacter' ) . ':',
			    'placeholder'       => esc_html__( 'Message ID', 'contacter' ) . ':',
			    'description'       => esc_html__( 'Text displayed in front of the message ID on the "Thank you" screen', 'contacter' ),
			    'default'           => esc_html__( 'Message ID', 'contacter' ),
		    ];

	    }

        /** Set updated tabs. */
        Plugin::set_tabs( $tabs );

        /** Refresh settings. */
        Settings::get_instance()->get_options();

    }

	/**
     * Get forms id and post title
	 * @return array
	 */
    public function get_forms() {

	    /** Prepare Contacter forms options. */
	    $wp_query = new WP_Query;
	    $query    = [
		    'post_type'   => ContacterForm::POST_TYPE,
		    'post_status' => [ 'publish' ]
	    ];
	    $c_forms  = $wp_query->query( $query );

	    $forms = [];
	    foreach ( $c_forms as $c_form ) {
		    $forms[ 'ID-' . $c_form->ID ] = $c_form->post_title;
	    }

	    return $forms;

    }

	/**
	 * Render Drag & Drop API Key field.
	 *
	 * @since 1.0.5
	 * @access public
	 **/
	public static function dnd_api_key() {

		$key_exist = false;
		if ( Settings::get_instance()->options['dnd-api-key'] ) { $key_exist = true; }

		?>
        <div class="mdp-dnd">
            <!--suppress HtmlFormInputWithoutLabel -->
            <div class="mdc-text-field mdc-input-width mdc-text-field--outlined mdc-hidden">
                <!--suppress HtmlFormInputWithoutLabel -->
                <input  type="text"
                        class="mdc-text-field__input"
                        name="mdp_contacter_general_settings[dnd-api-key]"
                        id="mdp-contacter-general-settings-dnd-api-key"
                        value="<?php esc_attr_e( Settings::get_instance()->options['dnd-api-key'] ); ?>"
                >
                <div class="mdc-notched-outline mdc-notched-outline--upgraded mdc-notched-outline--notched">
                    <div class="mdc-notched-outline__leading"></div>
                    <div class="mdc-notched-outline__notch">
                        <label for="mdp-contacter-general-settings-dnd-api-key" class="mdc-floating-label mdc-floating-label--float-above"><?php esc_html_e( 'API Key', 'contacter' ); ?></label>
                    </div>
                    <div class="mdc-notched-outline__trailing"></div>
                </div>
            </div>
            <div id="mdp-api-key-drop-zone" class="<?php if ( $key_exist ) : ?>mdp-key-uploaded<?php endif; ?>">
				<?php if ( $key_exist ) : ?>
                    <span class="material-icons">check_circle_outline</span><?php esc_html_e( 'API Key file exist', 'contacter' ); ?>
                    <span class="mdp-drop-zone-hover"><?php esc_html_e( 'Drop Key file here or click to upload', 'contacter' ); ?></span>
				<?php else : ?>
                    <span class="material-icons">cloud</span><?php esc_html_e( 'Drop Key file here or click to upload.', 'contacter' ); ?>
				<?php endif; ?>
            </div>
			<?php if ( $key_exist ) : ?>
                <div class="mdp-messages mdc-text-field-helper-line mdc-text-field-helper-text mdc-text-field-helper-text--persistent">
					<?php esc_html_e( 'Drag and drop or click on the form to replace API key. |', 'contacter' ); ?>
                    <a href="#" class="mdp-contacter-reset-key-btn"><?php esc_html_e( 'Reset API Key', 'contacter' ); ?></a>
                </div>
			<?php else : ?>
                <div class="mdp-messages mdc-text-field-helper-line mdc-text-field-helper-text mdc-text-field-helper-text--persistent">
					<?php esc_html_e( 'Drag and drop or click on the form to add API key.', 'contacter' ); ?>
                </div>
			<?php endif; ?>
            <input id="mdp-dnd-file-input" type="file" name="name" class="mdc-hidden" />
        </div>
		<?php

	}

	/**
	 * Main Settings Instance.
	 * Insures that only one instance of Settings exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return Config
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
