<?php
/**
 * Contacter
 * Voice feedback form for your website for saving and transcribing user voice messages to text.
 * Exclusively on https://1.envato.market/contacter
 *
 * @encoding        UTF-8
 * @version         1.7.8
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Contacter;

use Exception;
use Merkulove\Contacter\Unity\Plugin;
use Merkulove\Contacter\Unity\Settings;
use WP_Post;
use WP_Query;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class used to add contacter_form post type.
 *
 * @since 1.0.0
 *
 **/
final class ContacterForm {

	/**
	 * contacter_form Post Type name.
	 *
	 * @const POST_TYPE
	 * @since 1.0.0
	 **/
	const POST_TYPE = 'contacter_form';

	/**
	 * The one true ContacterForm.
	 *
	 * @var ContacterForm
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Fire meta box setup on the post editor screen. */
		add_action( 'load-post.php', [ $this, 'meta_boxes_setup' ] );
		add_action( 'load-post-new.php', [ $this, 'meta_boxes_setup' ] );

		/** Add Shortcode and Messages columns to Contacter Form list. */
		add_filter( 'manage_' . self::POST_TYPE . '_posts_columns', [$this, 'columns_head'], 10 );
		add_action( 'manage_' . self::POST_TYPE . '_posts_custom_column', [$this, 'columns_body'], 10, 2 );

		/** Add mark for form that is used for the float button. */
		add_filter( 'display_post_states', [$this, 'fbutton_form_mark'], 10, 2 );

		/** You cannot delete the form used for the Floating Button. */
		add_action( 'wp_trash_post', [$this, 'before_delete_post'] );
		add_action( 'before_delete_post', [$this, 'before_delete_post'] );

		/** Add message "You can't delete the form used for the Floating Button." */
		add_action( 'admin_notices', [$this, 'delete_form_warning'] );

	}

	/**
	 * Add message "You can't delete the form used for the Floating Button."
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 **/
	public function delete_form_warning() {

	    if ( isset( $_GET['mdp-delete-form-warning'] ) AND $_GET['mdp-delete-form-warning'] === '1' ) {
            ?>
            <div id=mdp-delete-form-warning" class="notice notice-warning is-dismissible">
                <p><?php esc_html_e( 'You can\'t delete the form used for the Floating Button.', 'contacter' ); ?></p>
            </div>
            <?php
        }

	}

	/**
	 * You cannot delete the form used for the Floating Button.
	 *
	 * @param $post_id - The post id that is being deleted.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function before_delete_post( $post_id ) {

		/** Process only contacter_form posts. */
		if ( ContacterForm::POST_TYPE !== get_post_type( $post_id ) ) { return; }

        $form_id = str_replace( 'ID-', '', Settings::get_instance()->options['fbutton_c_form'] );

		if ( $post_id == $form_id ) {

			wp_redirect( admin_url( '/edit.php?post_type=contacter_form&mdp-delete-form-warning=1' ) );
			exit();
        }

	}

	/**
	 * Add mark for form that is used for the float button.
	 *
	 * @param $post_states
	 * @param $post
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return array
	 **/
	public function fbutton_form_mark( $post_states, $post ) {

		if ( ContacterForm::POST_TYPE !== $post->post_type ) { return $post_states; }

		$form_id = str_replace( 'ID-', '', Settings::get_instance()->options['fbutton_c_form'] );

		/** Mark form that is used for floating button. */
		if ( $post->ID == $form_id ) {
			$post_states[] = esc_html__( 'Floating Button', 'contacter' );
		}

		return $post_states;

	}

	/**
	 * Add content for Shortcode and Messages columns.
	 *
	 * @param $column_name
	 * @param $form_id
	 *
	 * @throws Exception
	 * @since  1.0.0
	 * @access public
	 **/
	public function columns_body( $column_name, $form_id ) {

	    /** Shortcode column. */
		if ( 'mdp_contacter_form_shortcode' === $column_name ) {
			?>
            <code>[contacter id="<?php esc_attr_e( $form_id ); ?>"]</code>
            <a
                class="mdp-contacter-form-shortcode"
                data-clipboard-text='[contacter id="<?php esc_attr_e( $form_id ); ?>"]'
                href="#"
                title="<?php esc_html_e( 'Copy to Clipboard', 'contacter' ); ?>"
                data-copy-text='<svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="clone" class="svg-inline--fa fa-clone fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 0H144c-26.51 0-48 21.49-48 48v48H48c-26.51 0-48 21.49-48 48v320c0 26.51 21.49 48 48 48h320c26.51 0 48-21.49 48-48v-48h48c26.51 0 48-21.49 48-48V48c0-26.51-21.49-48-48-48zM362 464H54a6 6 0 0 1-6-6V150a6 6 0 0 1 6-6h42v224c0 26.51 21.49 48 48 48h224v42a6 6 0 0 1-6 6zm96-96H150a6 6 0 0 1-6-6V54a6 6 0 0 1 6-6h308a6 6 0 0 1 6 6v308a6 6 0 0 1-6 6z"></path></svg>'
                data-copied-text='<svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="check-square" class="svg-inline--fa fa-check-square fa-w-14" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M400 32H48C21.49 32 0 53.49 0 80v352c0 26.51 21.49 48 48 48h352c26.51 0 48-21.49 48-48V80c0-26.51-21.49-48-48-48zm0 400H48V80h352v352zm-35.864-241.724L191.547 361.48c-4.705 4.667-12.303 4.637-16.97-.068l-90.781-91.516c-4.667-4.705-4.637-12.303.069-16.971l22.719-22.536c4.705-4.667 12.303-4.637 16.97.069l59.792 60.277 141.352-140.216c4.705-4.667 12.303-4.637 16.97.068l22.536 22.718c4.667 4.706 4.637 12.304-.068 16.971z"></path></svg>'>
                <svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="clone" class="svg-inline--fa fa-clone fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 0H144c-26.51 0-48 21.49-48 48v48H48c-26.51 0-48 21.49-48 48v320c0 26.51 21.49 48 48 48h320c26.51 0 48-21.49 48-48v-48h48c26.51 0 48-21.49 48-48V48c0-26.51-21.49-48-48-48zM362 464H54a6 6 0 0 1-6-6V150a6 6 0 0 1 6-6h42v224c0 26.51 21.49 48 48 48h224v42a6 6 0 0 1-6 6zm96-96H150a6 6 0 0 1-6-6V54a6 6 0 0 1 6-6h308a6 6 0 0 1 6 6v308a6 6 0 0 1-6 6z"></path></svg>
            </a>
            <br>
            <br>
            <code>[contacter-click id="<?php esc_attr_e( $form_id ); ?>"]Your Content[/contacter-click]</code>
            <a
                    class="mdp-contacter-form-shortcode"
                    data-clipboard-text='[contacter-click id="<?php esc_attr_e( $form_id ); ?>"]Your Content[/contacter-click]'
                    href="#"
                    title="<?php esc_html_e( 'Copy to Clipboard', 'contacter' ); ?>"
                    data-copy-text='<svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="clone" class="svg-inline--fa fa-clone fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 0H144c-26.51 0-48 21.49-48 48v48H48c-26.51 0-48 21.49-48 48v320c0 26.51 21.49 48 48 48h320c26.51 0 48-21.49 48-48v-48h48c26.51 0 48-21.49 48-48V48c0-26.51-21.49-48-48-48zM362 464H54a6 6 0 0 1-6-6V150a6 6 0 0 1 6-6h42v224c0 26.51 21.49 48 48 48h224v42a6 6 0 0 1-6 6zm96-96H150a6 6 0 0 1-6-6V54a6 6 0 0 1 6-6h308a6 6 0 0 1 6 6v308a6 6 0 0 1-6 6z"></path></svg>'
                    data-copied-text='<svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="check-square" class="svg-inline--fa fa-check-square fa-w-14" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M400 32H48C21.49 32 0 53.49 0 80v352c0 26.51 21.49 48 48 48h352c26.51 0 48-21.49 48-48V80c0-26.51-21.49-48-48-48zm0 400H48V80h352v352zm-35.864-241.724L191.547 361.48c-4.705 4.667-12.303 4.637-16.97-.068l-90.781-91.516c-4.667-4.705-4.637-12.303.069-16.971l22.719-22.536c4.705-4.667 12.303-4.637 16.97.069l59.792 60.277 141.352-140.216c4.705-4.667 12.303-4.637 16.97.068l22.536 22.718c4.667 4.706 4.637 12.304-.068 16.971z"></path></svg>'>
                <svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="clone" class="svg-inline--fa fa-clone fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 0H144c-26.51 0-48 21.49-48 48v48H48c-26.51 0-48 21.49-48 48v320c0 26.51 21.49 48 48 48h320c26.51 0 48-21.49 48-48v-48h48c26.51 0 48-21.49 48-48V48c0-26.51-21.49-48-48-48zM362 464H54a6 6 0 0 1-6-6V150a6 6 0 0 1 6-6h42v224c0 26.51 21.49 48 48 48h224v42a6 6 0 0 1-6 6zm96-96H150a6 6 0 0 1-6-6V54a6 6 0 0 1 6-6h308a6 6 0 0 1 6 6v308a6 6 0 0 1-6 6z"></path></svg>
            </a>
            <?php
		}

		/** Messages column. */
		if ( 'mdp_contacter_form_messages' === $column_name ) :
            ?>
            <div class="mdp-contacter-form-messages-count">
                <?php $this->new_messages_link( $form_id ); ?>&nbsp;<?php $this->total_messages_link( $form_id ); ?>
            </div>
            <?php
		endif;

	}

	private function new_messages_link( $form_id, $show_zeros = false ) {

		/** Select only 'pending' records. */
		$query = new WP_Query( [
			'post_type' => ContacterRecord::POST_TYPE,
			'post_status' => ['pending'],
			'meta_key' => 'mdp_cform_id',
			'meta_value' => $form_id
		] );

		if ( $query->found_posts > 0 ) {
			?><a class="mdp-contacter-form-messages-count-new" title="<?php esc_html_e( 'New', 'contacter' ); ?>" href="<?php echo esc_url( admin_url( 'edit.php?s&post_status=pending&post_type=contacter_record&action=-1&m=0&mdp-contacter-filter-by-form=' . $form_id . '&filter_action=Filter&paged=1&action2=-1' ) ); ?>"><?php esc_html_e( $query->found_posts ); ?></a><?php
		} else {
		    if ( $show_zeros ) {
			    ?><span class="mdp-contacter-form-messages-count-new" title="<?php esc_html_e( 'New', 'contacter' ); ?>"><?php esc_html_e( $query->found_posts ); ?></span><?php
            }
        }

    }

	private function total_messages_link( $form_id, $show_zeros = false ) {

		$query = new WP_Query( [
			'post_type' => ContacterRecord::POST_TYPE,
			'post_status' => ['publish', 'pending', 'draft'],
			'meta_key' => 'mdp_cform_id',
			'meta_value' => $form_id
		] );

		if ( $query->found_posts > 0 ) {
			?><a class="mdp-contacter-form-messages-count-total" title="<?php esc_html_e( 'Total', 'contacter' ); ?>" href="<?php echo esc_url( admin_url( 'edit.php?s&post_status=all&post_type=contacter_record&action=-1&m=0&mdp-contacter-filter-by-form=' . $form_id . '&filter_action=Filter&paged=1&action2=-1' ) ); ?>"><?php esc_html_e( $query->found_posts ); ?></a><?php
        } else {
		    if ( $show_zeros ) {
			    ?><span class="mdp-contacter-form-messages-count-total" title="<?php esc_html_e( 'Total', 'contacter' ); ?>"><?php esc_html_e( $query->found_posts ); ?></span><?php
            }
        }

    }

	/**
	 * Add title for Shortcode and Messages columns.
	 *
	 * @param array $columns
	 *
	 * @return array
	 * @since 1.0.0
	 * @access public
	 **/
	public function columns_head( $columns ) {

		/** Add new column key to the existing columns. */
		$columns['mdp_contacter_form_shortcode'] = esc_html__( 'Shortcode', 'contacter' );
		$columns['mdp_contacter_form_messages'] = esc_html__( 'Messages', 'contacter' );

		/** Define a new order. (•̀o•́)ง */
		$new_order = ['cb', 'title', 'mdp_contacter_form_shortcode', 'mdp_contacter_form_messages', 'date'];

		/** Order columns like set in $new_order. */
		$new = [];
		foreach ( $new_order as $col_name ) {
			$new[ $col_name ] = $columns[ $col_name ];
		}

		/** Return a new column array to WordPress. */
		return $new;
	}

	/**
	 * Meta box setup function.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function meta_boxes_setup() {

		/** Add meta boxes on the 'add_meta_boxes' hook. */
		add_action( 'add_meta_boxes', [ $this, 'add_meta_boxes' ] );

		/** Save meta box values on the 'save_post' hook. */
		add_action( 'save_post', [ $this, 'save_meta_boxes' ], 1, 2 );

	}

	/**
	 * Create Options, Shortcode and Messages meta boxes to be displayed on Contacter Form editor screen.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function add_meta_boxes() {

		/** Options metabox. */
		add_meta_box( 'mdp-options-mbox', esc_html__( ' Form Options', 'contacter' ), [$this, 'options_metabox'], self::POST_TYPE, 'normal', 'default' );

		/** Shortcode metabox. */
		add_meta_box( 'mdp-shortcode-mbox', esc_html__( 'Shortcode', 'contacter' ), [$this, 'shortcode_metabox' ], self::POST_TYPE, 'side', 'default' );

		/** Messages metabox. */
		add_meta_box( 'mdp-messages-mbox', esc_html__( 'Voice Messages', 'contacter' ), [$this, 'messages_metabox'], self::POST_TYPE, 'side', 'default' );

	}

	/**
	 * Display Messages meta box.
	 *
	 * @param WP_Post $contacter_form
	 *
	 * @throws Exception
	 * @since  1.0.0
	 * @access public
	 **/
	public function messages_metabox( $contacter_form ) {
		?>
        <div class="mdp-contacter-form-messages-count">
	        <?php esc_html_e( 'Messages: ', 'contacter' ); ?>
            <?php $this->new_messages_link( $contacter_form->ID, true ); ?>&nbsp;<?php $this->total_messages_link( $contacter_form->ID, true ); ?>
        </div>
        <?php
	}


	/**
	 * Display Shortcode meta box.
	 *
	 * @param WP_Post $contacter_form
	 **/
	public function shortcode_metabox( $contacter_form ) {

        ?>
        <p>
            <code>[contacter id="<?php echo esc_html( $contacter_form->ID ); ?>"]</code>
            <a
                    class="mdp-contacter-form-shortcode"
                    data-clipboard-text='[contacter id="<?php esc_attr_e( $contacter_form->ID ); ?>"]'
                    href="#"
                    title="<?php esc_html_e( 'Copy to Clipboard', 'contacter' ); ?>"
                    data-copy-text='<svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="clone" class="svg-inline--fa fa-clone fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 0H144c-26.51 0-48 21.49-48 48v48H48c-26.51 0-48 21.49-48 48v320c0 26.51 21.49 48 48 48h320c26.51 0 48-21.49 48-48v-48h48c26.51 0 48-21.49 48-48V48c0-26.51-21.49-48-48-48zM362 464H54a6 6 0 0 1-6-6V150a6 6 0 0 1 6-6h42v224c0 26.51 21.49 48 48 48h224v42a6 6 0 0 1-6 6zm96-96H150a6 6 0 0 1-6-6V54a6 6 0 0 1 6-6h308a6 6 0 0 1 6 6v308a6 6 0 0 1-6 6z"></path></svg>'
                    data-copied-text='<svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="check-square" class="svg-inline--fa fa-check-square fa-w-14" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M400 32H48C21.49 32 0 53.49 0 80v352c0 26.51 21.49 48 48 48h352c26.51 0 48-21.49 48-48V80c0-26.51-21.49-48-48-48zm0 400H48V80h352v352zm-35.864-241.724L191.547 361.48c-4.705 4.667-12.303 4.637-16.97-.068l-90.781-91.516c-4.667-4.705-4.637-12.303.069-16.971l22.719-22.536c4.705-4.667 12.303-4.637 16.97.069l59.792 60.277 141.352-140.216c4.705-4.667 12.303-4.637 16.97.068l22.536 22.718c4.667 4.706 4.637 12.304-.068 16.971z"></path></svg>'>
                <svg aria-hidden="true" focusable="false" data-prefix="far" data-icon="clone" class="svg-inline--fa fa-clone fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 0H144c-26.51 0-48 21.49-48 48v48H48c-26.51 0-48 21.49-48 48v320c0 26.51 21.49 48 48 48h320c26.51 0 48-21.49 48-48v-48h48c26.51 0 48-21.49 48-48V48c0-26.51-21.49-48-48-48zM362 464H54a6 6 0 0 1-6-6V150a6 6 0 0 1 6-6h42v224c0 26.51 21.49 48 48 48h224v42a6 6 0 0 1-6 6zm96-96H150a6 6 0 0 1-6-6V54a6 6 0 0 1 6-6h308a6 6 0 0 1 6 6v308a6 6 0 0 1-6 6z"></path></svg>
            </a>
        </p><?php
	}

	/**
	 * Display Options meta box.
	 *
	 * @param WP_Post $contacter_form
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 **/
	public function options_metabox( $contacter_form ) {

		/** Render "Options" metabox with all fields. */
		CFOptionsMetaBox::get_instance()->render_metabox( $contacter_form );

	}

	/**
	 * Save meta box values.
	 *
	 * @param $post_id
	 * @param $post
	 *
	 * @return void
	 * @since 1.0.0
	 * @access public
	 **/
	public function save_meta_boxes( $post_id, $post ) {

		/** Work only with contacter_form post type. */
	    if ( 'contacter_form' !== $post->post_type ) { return; }

		/** Verify the nonce before proceeding. */
		if (
             ! isset( $_POST['options_metabox_fields_nonce'] ) ||
             ! wp_verify_nonce( $_POST['options_metabox_fields_nonce'], Plugin::get_basename() )
        ) {
			return;
		}

		/** Get the post type object. */
		$post_type = get_post_type_object( $post->post_type );

		/** Check if the current user has permission to edit the post. */
		if ( ! current_user_can( $post_type->cap->edit_post, $post_id ) ) {
			return;
		}

		/** Save "Options" metabox with all fields. */
		CFOptionsMetaBox::get_instance()->save_metabox( $post_id );

	}

	/**
	 * Register post type.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function register_post_type() {

		register_post_type( self::POST_TYPE, [
			'public'              => false,
			'labels'              => [
				'name'                  => esc_html__( 'Contacter Forms', 'contacter' ),
				'singular_name'         => esc_html__( 'Contacter Form', 'contacter' ),
				'add_new'               => esc_html__( 'Add New', 'contacter' ),
				'add_new_item'          => esc_html__( 'Add New', 'contacter' ),
				'new_item'              => esc_html__( 'New Contacter Form', 'contacter' ),
				'edit_item'             => esc_html__( 'Edit Contacter Form', 'contacter' ),
				'view_item'             => esc_html__( 'View Contacter Form', 'contacter' ),
				'view_items'            => esc_html__( 'View Contacter Forms', 'contacter' ),
				'search_items'          => esc_html__( 'Search Contacter Forms', 'contacter' ),
				'not_found'             => esc_html__( 'No Contacter Forms found', 'contacter' ),
				'not_found_in_trash'    => esc_html__( 'No Contacter Forms found in Trash', 'contacter' ),
				'all_items'             => esc_html__( 'Forms', 'contacter' ),
				'archives'              => esc_html__( 'Contacter Forms Archives', 'contacter' ),
				'attributes'            => esc_html__( 'Contacter Form Attributes', 'contacter' ),
				'insert_into_item'      => esc_html__( 'Insert to Contacter Form', 'contacter' ),
				'uploaded_to_this_item' => esc_html__( 'Uploaded to this Contacter Form', 'contacter' ),
				'menu_name'             => esc_html__( 'Contacter', 'contacter' )
			],
			'exclude_from_search' => true,
			'publicly_queryable'  => false,
			'menu_position'       => false,
			'show_in_rest'        => false,
			'show_in_menu'        => 'edit.php?post_type=' . ContacterRecord::POST_TYPE,
			'rest_base'           => self::POST_TYPE,
			'supports'            => [ 'title' ],
			'show_ui'               => true,
		] );

	}

	/**
	 * Create default contacter form.
	 *
	 * @static
	 * @since 1.0.0
	 **/
	public function create_default_form() {

		/** Check do we have some forms. */
		$query = new WP_Query( [
			'post_type'     => self::POST_TYPE,
			'post_status'   => 'publish',
		] );

		/** Yes. We have forms. */
		if ( $query->found_posts > 0 ) { return true; }

		/**
		 * Oops, it looks like there are no forms at all.
         * So create new one.
		 **/
		$form_id = wp_insert_post( [
			'post_type'     => self::POST_TYPE,
			'post_title'    => esc_html__( 'Default Form', 'contacter' ),
			'post_status'   => 'publish',
		] );

		/** Fill meta fields. */
		if ( ! $form_id ) { return false; }

		/** Add default form meta */
		$this->default_form_meta( $form_id );

        return true;

	}

	/**
     * Add default form meta
	 * @param $form_id
	 *
	 * @return void
	 */
    private function default_form_meta( $form_id ) {

	    /** Align. */
	    add_post_meta( $form_id, 'mdp_align', 'center' );
	    add_post_meta( $form_id, 'mdp_align_skip', 'center' );

	    /** Margin. */
	    add_post_meta( $form_id, 'mdp_btn_margin', '10' );
	    add_post_meta( $form_id, 'mdp_btn_margin_skip', '10' );

	    /** Padding. */
	    add_post_meta( $form_id, 'mdp_btn_padding', '20' );
	    add_post_meta( $form_id, 'mdp_btn_padding_skip', '20' );

	    /** Radius. */
	    add_post_meta( $form_id, 'mdp_btn_radius', '50' );
	    add_post_meta( $form_id, 'mdp_btn_radius_skip', '50' );

	    /** Icon. */
	    add_post_meta( $form_id, 'mdp_btn_icon', 'material/record-voice-microphone-button.svg' );
	    add_post_meta( $form_id, 'mdp_btn_icon_skip', 'material/play-next-button.svg' );

	    /** Position. */
	    add_post_meta( $form_id, 'mdp_btn_icon_position', 'before' );
	    add_post_meta( $form_id, 'mdp_btn_icon_position_skip', 'before' );

	    /** Button Caption. */
	    add_post_meta( $form_id, 'mdp_btn_caption', esc_html__( 'Record', 'contacter' ) );
	    add_post_meta( $form_id, 'mdp_btn_caption_skip', esc_html__( 'Skip', 'contacter' ) );

	    /** Button size. */
	    add_post_meta( $form_id, 'mdp_btn_size', '18' );
	    add_post_meta( $form_id, 'mdp_btn_size_skip', '18' );

	    /** Button color. */
	    add_post_meta( $form_id, 'mdp_btn_color', '#fff' );
	    add_post_meta( $form_id, 'mdp_btn_color_skip', '#fff' );

	    /** BG color. */
	    add_post_meta( $form_id, 'mdp_btn_bg_color', '#0274e6' );
	    add_post_meta( $form_id, 'mdp_btn_bg_color_skip', '#0274e6' );

	    /** Hover animation. */
	    add_post_meta( $form_id, 'mdp_btn_hover_animation', 'none' );

	    /** Before Text field value. */
	    add_post_meta( $form_id, 'mdp_before_txt', '<h4>' . esc_html__( 'We would love to hear from you!', 'contacter' ) . '</h4><p>' . esc_html__( 'Please record your message.', 'contacter' ) . '</p>' );

	    /** After Text field value. */
	    add_post_meta( $form_id, 'mdp_after_txt', '<p>' . esc_html__( 'Record, Listen, Send', 'contacter' ) . '</p>' );

	    /** Speak now Text field value. */
	    add_post_meta( $form_id, 'mdp_speak_now_txt', '<h4>' . esc_html__( 'Speak now', 'contacter' ) . '</h4><div>{countdown}</div>' );

	    /** Send Text field value. */
	    add_post_meta( $form_id, 'mdp_send_txt', '<h4>' . esc_html__( 'Send your recording', 'contacter' ) . '</h4>' );

	    /** Thanks Text field value. */
	    add_post_meta( $form_id, 'mdp_thanks_txt', '<h4>' . esc_html__( 'Thank you', 'contacter' ) . '</h4>' );

	    /** Additional fields switcher. */
	    add_post_meta( $form_id, 'mdp_additional_fields', 'off' );

    }

	/**
	 * Main ContacterForm Instance.
	 *
	 * Insures that only one instance of ContacterForm exists in memory at any one time.
	 *
	 * @static
	 * @return ContacterForm
	 * @since 1.0.0
	 **/
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof ContacterForm ) ) {
			self::$instance = new ContacterForm;
		}

		return self::$instance;
	}

} // End Class ContacterForm.
