<?php
/**
 * Contacter
 * Voice feedback form for your website for saving and transcribing user voice messages to text.
 * Exclusively on https://1.envato.market/contacter
 *
 * @encoding        UTF-8
 * @version         1.7.8
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Contacter;

use Merkulove\Contacter\Unity\Plugin;
use Merkulove\Contacter\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class used to implement shortcodes.
 *
 * @since 1.0.0
 *
 **/
final class Shortcodes {

	/**
	 * The one true Shortcodes.
	 *
	 * @var Shortcodes
	 * @since 1.0.0
	 **/
	private static $instance;

    /**
     * Plugin settings
     * @var array
     * @since 1.3.1
     */
	private static $options;

    /**
     * Max form record duration
     * @var string
     * @since 1.3.1
     */
    private static $max_duration;

	/**
	 * Sets up a new Shortcodes instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Initializes plugin shortcodes. */
		add_action( 'init', [$this, 'shortcodes_init'] );

		/** @var array options Plugin settings */
        self::$options = Settings::get_instance()->options;

	}

	/**
	 * Initializes shortcodes.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 **/
	public function shortcodes_init() {

		/** Add plugin shortcode [contacter id=""]. Works everywhere on site. */
		add_shortcode( 'contacter', [ $this, 'contacter_shortcode' ] );

		/** Add plugin shortcode [contacter-click id=""]Your Content[/contacter-click]. Works everywhere on site. */
		add_shortcode( 'contacter-click', [ $this, 'contacter_click_shortcode' ] );


	}

	/**
	 * Add plugin shortcode [contacter-click id=""]Your Content[/contacter-click].
	 *
	 * @param array $atts - Shortcodes attributes.
	 *
	 * @param       $content
	 *
	 * @since  1.0.5
	 * @access public
	 * @return string
	 **/
	public function contacter_click_shortcode( $atts = [], $content = '' ) {

		/** Filter shortcode attributes. */
		$atts = shortcode_atts( [
			'id' => '',
			'title' => ''
		], $atts );

		/** Nothing to show without any parameters. */
		if ( '' === $atts['id'] && '' === $atts['title'] ) { return ''; }

		/** We can't show anything without content. */
		if ( ! $content ) { return ''; }

		/** Unique id for current shortcode. */
		$id = $this->get_shortcode_id( $atts );

		/** Get Contacter form data. */
		$c_form = $this->get_cform( $atts );

		/** Contacter form not found. */
		if ( ! $c_form ) { return ''; }

		/** Contacter form not published. */
		if ( 'publish' !== $c_form->post_status) { return ''; }

		/** Enqueue styles and scripts only if shortcode used on this page. */
		$this->enqueue( $id, $c_form );

		/** Get align. */
		$mdp_align = get_post_meta( $c_form->ID, 'mdp_align', true );

		/** @var string $max_duration */
        $max_form_record_duration = get_post_meta( $c_form->ID, 'mdp_form_record_duration', true );
        self::$max_duration = empty( $max_form_record_duration ) ? self::$options[ 'max_duration' ] : $max_form_record_duration;

		ob_start();

		?>

        <!-- Start Contacter WordPress Plugin -->
        <div id="<?php esc_attr_e( $id ); ?>"
             class="mdp-contacter-form-box mdp-step-start mdp-align-<?php esc_attr_e( $mdp_align ); ?>"
             cform-name="<?php esc_attr_e( $c_form->post_title ); ?>"
             cform-id="<?php esc_attr_e( $c_form->ID ); ?>"
             max-duration="<?php esc_attr_e( self::$max_duration ); ?>"
             style="display: none;"
        >

	        <?php $this->render_custom_content_start_box( $c_form, $content ); // Start Recording Step with custom content. ?>

			<?php $this->render_allow_access_box(); // Allow access to microphone step. ?>

			<?php $this->render_mic_access_err_box( $c_form ); // Microphone access error step. ?>

            <?php $this->render_countdown_box(); ?>

			<?php $this->render_recording_box( $c_form ); // Speak Now Step. ?>

			<?php $this->render_reset_box( $c_form ); // Reset Step. ?>

			<?php $this->render_error_box(); // Error Step. ?>

			<?php $this->render_send_box( $c_form ); // Send Step. ?>

			<?php $this->render_thanks_box( $c_form ); // Thank You Step. ?>

        </div>
        <!-- End Contacter WordPress Plugin -->
		<?php

		return ob_get_clean();

	    return $content;
    }

	/**
	 * Add Contacter by shortcode [contacter].
	 *
	 * @param array $atts - Shortcodes attributes.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string
	 **/
	public function contacter_shortcode( $atts = [] ) {

		/** Filter shortcode attributes. */
		$atts = shortcode_atts( [
			'id' => '',
			'title' => ''
		], $atts );

		/** Nothing to show without any parameters. */
		if ( '' === $atts['id'] && '' === $atts['title'] ) { return ''; }

		/** Unique id for current shortcode. */
		$id = $this->get_shortcode_id( $atts );

		/** Get Contacter form data. */
		$c_form = $this->get_cform( $atts );

		/** Contacter form not found. */
		if ( ! $c_form ) { return ''; }

		/** Contacter form not published. */
        if ( 'publish' !== $c_form->post_status) { return ''; }

		/** Enqueue styles and scripts only if shortcode used on this page. */
		$this->enqueue( $id, $c_form );

		/** Get align. */
		$mdp_align = get_post_meta( $c_form->ID, 'mdp_align', true );

        /** @var string $max_duration */
        $max_form_record_duration = get_post_meta( $c_form->ID, 'mdp_form_record_duration', true );
        self::$max_duration = empty( $max_form_record_duration ) ? self::$options[ 'max_duration' ] : $max_form_record_duration;

		ob_start();

		?>

		<!-- Start Contacter WordPress Plugin -->
		<div id="<?php esc_attr_e( $id ); ?>"
             class="mdp-contacter-form-box mdp-step-start mdp-align-<?php esc_attr_e( $mdp_align ); ?>"
             cform-name="<?php esc_attr_e( $c_form->post_title ); ?>"
             cform-id="<?php esc_attr_e( $c_form->ID ); ?>"
             max-duration="<?php esc_attr_e( self::$max_duration ); ?>"
		     style="display: none;"
        >

            <?php $this->render_start_box( $c_form ); // Start Recording Step. ?>

			<?php $this->render_allow_access_box(); // Allow access to microphone step. ?>

			<?php $this->render_mic_access_err_box( $c_form ); // Microphone access error step. ?>

            <?php $this->render_countdown_box(); ?>

			<?php $this->render_recording_box( $c_form ); // Speak Now Step. ?>

			<?php $this->render_reset_box( $c_form ); // Reset Step. ?>

			<?php $this->render_error_box(); // Error Step. ?>

			<?php $this->render_send_box( $c_form ); // Send Step. ?>

			<?php $this->render_thanks_box( $c_form ); // Thank You Step. ?>

		</div>
		<!-- End Contacter WordPress Plugin -->
		<?php

		return ob_get_clean();

	}

	/**
	 * Render Thank You Step.
	 *
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_thanks_box( $c_form ) {

        $options = Settings::get_instance()->options;

		$thanks_txt = get_post_meta( $c_form->ID, 'mdp_thanks_txt', true );

		?>
		<div class="mdp-contacter-thanks-box">
			<?php echo wp_kses_post( $thanks_txt ); ?>
            <?php if ( $options[ 'show_record_id' ] === 'on' ) :?>
            <div class="mdp-contacter-thanks-box--post-id">
                <p><?php echo esc_html( $options[ 'msg_message_id' ] ) ?? esc_html__( 'Message ID', 'contacter' ) . ': ';?><b></b></p>
            </div>
            <?php endif; ?>
			<?php if ( $options['btn_start_new'] !== '' ) {
				$button_txt = wp_kses_post( $options['btn_start_new'] ); ?>
			<button class="mdp-restart" aria-label="<?php esc_html_e( $button_txt, 'contacter' ); ?>">
				<span><?php esc_html_e( $button_txt, 'contacter' ); ?></span>
			</button>
			<?php } ?>
        </div>
		<?php
	}

	/**
	 * Render Send Step.
	 *
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_send_box( $c_form ) {

		$options = Settings::get_instance()->options;

	    /** Get Additional fields. */
        $send_txt = get_post_meta( $c_form->ID, 'mdp_send_txt', true );
		$additional_fields = get_post_meta( $c_form->ID, 'mdp_additional_fields', true );
		$fields_res = get_post_meta( $c_form->ID, 'mdp_additional_fields_res', true );

		/** Get buttons text */
        $btn_send = wp_kses_post( $options['btn_send'] );
        $btn_reset = wp_kses_post( $options['btn_reset'] );

		?>
		<div class="mdp-contacter-send-box">
            <?php if ( 'on' === $additional_fields ) : ?><form novalidate=""><?php endif; ?>

                <?php echo wp_kses_post( $send_txt ); ?>
                <div class="mdp-contacter-player-box"></div>

                <?php if ( 'on' === $options['show_download_link'] ) : ?>
                    <p class="mdp-contacter-download-link-box">
                        <a href="#"><?php esc_html_e( $options['download_link_text'] ); ?></a>
                    </p>
                <?php endif; ?>

		        <?php if ( 'on' === $additional_fields ) : ?>
                    <div class="mdp-contacter-additional-fields"
                         additional-fields="<?php esc_attr_e( $fields_res ); ?>"
                         user-login="<?php esc_attr_e( wp_get_current_user()->user_login ); ?>"
                         user-ip="<?php esc_attr_e( $this->get_real_ip() ); ?>"
                    ></div>
                <?php endif; ?>

                <div class="mdp-send-btns mdp-hover-<?php esc_attr_e( get_post_meta( $c_form->ID, 'mdp_btn_hover_animation', true ) ) ?>">
                    <button class="mdp-send-rec-btn" aria-label="<?php esc_html_e( $btn_send, 'contacter' ); ?>">
                        <span><?php esc_html_e( $btn_send, 'contacter' ); ?></span>
                    </button>
                    <button class="mdp-reset-rec-btn" aria-label="<?php esc_html_e( $btn_reset, 'contacter' ); ?>">
                        <span><?php esc_html_e( $btn_reset, 'contacter' ); ?></span>
                    </button>
                </div>

			<?php if ( 'on' === $additional_fields ) : ?></form><?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Return real user IP.
	 *
	 * @since  1.0.5
	 * @access public
	 * @return string
	 **/
	private function get_real_ip() {

		/** Check ip from share internet. */
		if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {

			$ip = $_SERVER['HTTP_CLIENT_IP'];

        /** To check ip is pass from proxy. */
		} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {

			$ip = $_SERVER['HTTP_X_FORWARDED_FOR'];

		} else {

			$ip = $_SERVER['REMOTE_ADDR'];

		}

		return $ip;

	}

	/**
	 * Render Error Step.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_error_box() {
		?>
		<div class="mdp-contacter-error-box">
			<?php echo wp_kses_post( Settings::get_instance()->options['msg_sending_error'] ); ?>
		</div>
		<?php
	}

	/**
	 * Render Reset Step.
	 *
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_reset_box( $c_form ) {

        /** Get buttons text */
        $btn_reset = wp_kses_post( Settings::get_instance()->options['btn_reset'] );
        $btn_resume = wp_kses_post( Settings::get_instance()->options['btn_resume'] );

		?>
		<div class="mdp-contacter-reset-box mdp-hover-<?php esc_attr_e( get_post_meta( $c_form->ID, 'mdp_btn_hover_animation', true ) ) ?>">
			<?php echo wp_kses_post( Settings::get_instance()->options['msg_reset_recording'] ); ?>
			<div class="mdp-speak-now-btns">
				<button class="mdp-reset-rec-yes" aria-label="<?php esc_html_e( $btn_reset, 'contacter' ); ?>">
                    <span><?php esc_html_e( $btn_reset, 'contacter' ); ?></span>
                </button>
				<button class="mdp-reset-rec-no" aria-label="<?php esc_html_e( $btn_resume, 'contacter' ); ?>">
                    <span><?php esc_html_e( $btn_resume, 'contacter' ); ?></span>
                </button>
			</div>
		</div>
		<?php
	}

	/**
     * Render countdown box
	 */
	private function render_countdown_box()
    {
        ?>
        <div class="mdp-contacter-countdown-box">
            <span class="mdp-contacter-countdown-box--counter"><?php esc_attr_e( Settings::get_instance()->options[ 'countdown_duration' ] ); ?></span>
        </div>
        <?php
    }

	/**
	 * Render Speak Now Step.
	 *
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_recording_box( $c_form ) {

        /** Get buttons text */
	    $btn_stop = wp_kses_post( Settings::get_instance()->options['btn_stop'] );
	    $btn_pause = wp_kses_post( Settings::get_instance()->options['btn_pause'] );

		?>
		<div class="mdp-contacter-recording-box">
			<?php
			/** Get Speak Now content. */
			$mdp_speak_now_msg = get_post_meta( $c_form->ID, 'mdp_speak_now_txt', true );

			/** Replace {timer} {max-duration} {countdown} placeholders. */
			$mdp_speak_now_msg = $this->replace_placeholders( $mdp_speak_now_msg );

			/** Show Speak Now content. */
			echo wp_kses_post( $mdp_speak_now_msg );
			?>
            <div class="mdp-contacter-recording-animation">
                <canvas width="384" height="60">
                    <div><?php esc_html_e( 'Canvas not available.', 'contacter' ); ?></div>
                </canvas>
            </div>
			<div class="mdp-speak-now-btns mdp-hover-<?php esc_attr_e( get_post_meta( $c_form->ID, 'mdp_btn_hover_animation', true ) ) ?>">
				<button class="mdp-stop-rec-btn" aria-label="<?php esc_html_e( $btn_stop, 'contacter' );  ?>">
                    <span><?php esc_html_e( $btn_stop, 'contacter' );  ?></span>
                </button>
				<button class="mdp-reset-rec-btn" aria-label="<?php esc_html_e( $btn_pause, 'contacter' );  ?>">
                    <span><?php esc_html_e( $btn_pause, 'contacter' );  ?></span>
                </button>
			</div>
		</div>
		<?php
	}

	/**
	 * Render Microphone access error Step.
	 *
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_mic_access_err_box( $c_form ) {
		?>
        <div class="mdp-contacter-mic-access-err-box">
			<?php echo wp_kses_post( Settings::get_instance()->options['msg_mic_access_err'] ); ?>
            <div class="mdp-speak-now-btns mdp-hover-<?php esc_attr_e( get_post_meta( $c_form->ID, 'mdp_btn_hover_animation', true ) ) ?>">
                <button class="mdp-mic-access-err-reload-btn" aria-label="<?php esc_html_e( 'Try again', 'contacter' ); ?>">
                    <span><?php esc_html_e( 'Try again', 'contacter' ); ?></span>
                </button>
            </div>
        </div>
		<?php
	}

	/**
	 * Render Allow access to microphone Step.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_allow_access_box() {
        ?>
        <div class="mdp-contacter-allow-access-box">
            <?php echo wp_kses_post( Settings::get_instance()->options['msg_allow_access'] ); ?>
        </div>
        <?php
    }

    private function render_custom_content_start_box( $c_form, $content ) {

	    /** Get Start Box Settings. */
	    $mdp_before_txt = get_post_meta( $c_form->ID, 'mdp_before_txt', true );
	    $mdp_after_txt = get_post_meta( $c_form->ID, 'mdp_after_txt', true );

	    ?>
        <div class="mdp-contacter-start-box">
		    <?php if ( $mdp_before_txt ) : ?>
                <div class="mdp-contacter-before-txt"><?php echo wp_kses_post( $mdp_before_txt ); ?></div>
		    <?php endif; ?>

            <div class="mdp-contacter-start-btn-box">
                <div class="mdp-contacter-start-btn mdp-contacter-custom">
                    <?php echo wp_kses_post( $content ); ?>
                </div>
            </div>

		    <?php if ( $mdp_after_txt ) : ?>
                <div class="mdp-contacter-after-txt"><?php echo wp_kses_post( $mdp_after_txt ); ?></div>
		    <?php endif; ?>
        </div>
	    <?php

    }

	/**
	 * Render Start Recording Step.
	 *
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function render_start_box( $c_form ) {

		/** Get Start Box Settings. */
		$mdp_before_txt = get_post_meta( $c_form->ID, 'mdp_before_txt', true );
		$mdp_after_txt = get_post_meta( $c_form->ID, 'mdp_after_txt', true );
		$mdp_btn_caption = wp_kses_post( get_post_meta( $c_form->ID, 'mdp_btn_caption', true ) );
		$mdp_btn_icon = get_post_meta( $c_form->ID, 'mdp_btn_icon', true );
		$mdp_btn_icon_position = get_post_meta( $c_form->ID, 'mdp_btn_icon_position', true );
		$mdp_btn_hover_animation = get_post_meta( $c_form->ID, 'mdp_btn_hover_animation', true );

		$mdp_additional_fields = get_post_meta( $c_form->ID, 'mdp_additional_fields', true ) === 'on';
		$mdp_skip_button = get_post_meta( $c_form->ID, 'mdp_skip_button', true ) === 'on';
		$mdp_btn_caption_skip = wp_kses_post( get_post_meta( $c_form->ID, 'mdp_btn_caption_skip', true ) );
		$mdp_btn_icon_skip = get_post_meta( $c_form->ID, 'mdp_btn_icon_skip', true );
		$mdp_btn_icon_position_skip = get_post_meta( $c_form->ID, 'mdp_btn_icon_position_skip', true );
		$mdp_btn_hover_animation_skip = get_post_meta( $c_form->ID, 'mdp_btn_hover_animation_skip', true );

		?>
		<div class="mdp-contacter-start-box">
			<?php if ( $mdp_before_txt ) : ?>
				<div class="mdp-contacter-before-txt"><?php echo wp_kses_post( $mdp_before_txt ); ?></div>
			<?php endif; ?>

			<div class="mdp-contacter-start-btn-box mdp-hover-<?php esc_attr_e( $mdp_btn_hover_animation ); ?>">
				<button class="mdp-contacter-start-btn mdp-icon-position-<?php esc_attr_e( $mdp_btn_icon_position ); ?>" aria-label="<?php esc_html_e( $mdp_btn_caption, 'contacter' ); ?>">

					<?php if ( $mdp_btn_icon_position !== 'none' ) : ?>
						<span class="mdp-contacter-start-btn-icon"><?php Caster::get_instance()->inline_svg_e( $mdp_btn_icon ); ?></span>
					<?php endif; ?>

					<?php if ( $mdp_btn_caption ) : ?>
						<span class="mdp-contacter-start-btn-caption"><?php esc_html_e( $mdp_btn_caption, 'contacter' ); ?></span>
					<?php endif; ?>

				</button>

				<?php if ( $mdp_skip_button && $mdp_additional_fields ) : ?>
                    <div class="mdp-contacter-skip-btn-box mdp-hover-<?php esc_attr_e( $mdp_btn_hover_animation_skip ); ?>">
                        <button class="mdp-contacter-skip-btn mdp-icon-position-<?php esc_attr_e( $mdp_btn_icon_position_skip ); ?>" aria-label="<?php esc_html_e( $mdp_btn_caption_skip, 'contacter' ); ?>">

							<?php if ( $mdp_btn_icon_position_skip !== 'none' ) : ?>
                                <span class="mdp-contacter-skip-btn-icon"><?php Caster::get_instance()->inline_svg_e( $mdp_btn_icon_skip ); ?></span>
							<?php endif; ?>

                            <span class="mdp-contacter-skip-btn-caption"><?php esc_html_e( $mdp_btn_caption_skip, 'contacter' ); ?></span>

                        </button>
                    </div>
				<?php endif; ?>

			</div>



			<?php if ( $mdp_after_txt ) : ?>
				<div class="mdp-contacter-after-txt"><?php echo wp_kses_post( $mdp_after_txt ); ?></div>
			<?php endif; ?>
		</div>
		<?php

	}

	/**
	 * Enqueue styles and scripts only if shortcode used on this page.
	 *
	 * @param string $id - current shortcode id.
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 **/
	private function enqueue( $id, $c_form ) {

		/**
		 * Enqueue styles only if shortcode used on this page.
		 * Enqueue here because we need to get inline css styles for current shortcode.
		 */
		wp_enqueue_style(
			'mdp-contacter',
			Plugin::get_url() . 'css/contacter' . Plugin::get_suffix() . '.css',
			array(),
			Plugin::get_version()
		);
		wp_enqueue_style(
			'mdp-contacter-shortcode',
			Plugin::get_url() . 'css/contacter' . Plugin::get_suffix() . '.css',
			array(),
			Plugin::get_version()
		);

		/** Get inline CSS for current shortcode. */
		$css = $this->get_shortcode_inline_css( $id, $c_form );

		/** Add inline styles.. */
		wp_add_inline_style( 'mdp-contacter-shortcode', $css );

		/** Enqueue JavaScript only if shortcode used on this page. */
		wp_enqueue_script( 'mdp-contacter-recorder' );
		wp_enqueue_script( 'mdp-contacter' );

	}

	/**
	 * Return inline css styles for current shortcode.
	 *
	 * @param string $id - current shortcode id.
	 * @param object $c_form - current Contacter Form.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string - inline css styles.
	 */
	private function get_shortcode_inline_css( $id, $c_form ) {

		/** Get Contacter Form params. */
		$btn_caption = get_post_meta( $c_form->ID, 'mdp_btn_caption', true );
		$btn_margin = get_post_meta( $c_form->ID, 'mdp_btn_margin', true );
		$btn_padding = get_post_meta( $c_form->ID, 'mdp_btn_padding', true );
		$btn_radius = get_post_meta( $c_form->ID, 'mdp_btn_radius', true );
		$btn_color = get_post_meta( $c_form->ID, 'mdp_btn_color', true );
		$btn_color_hover = get_post_meta( $c_form->ID, 'mdp_btn_color_hover', true );
		$btn_bg_color = get_post_meta( $c_form->ID, 'mdp_btn_bg_color', true );
		$btn_bg_color_hover = get_post_meta( $c_form->ID, 'mdp_btn_bg_color_hover', true );
		$btn_size = get_post_meta( $c_form->ID, 'mdp_btn_size', true );

		/** Get Contacter Form params. */
		$btn_caption_skip = get_post_meta( $c_form->ID, 'mdp_btn_caption_skip', true );
		$btn_margin_skip = get_post_meta( $c_form->ID, 'mdp_btn_margin_skip', true );
		$btn_padding_skip = get_post_meta( $c_form->ID, 'mdp_btn_padding_skip', true );
		$btn_radius_skip = get_post_meta( $c_form->ID, 'mdp_btn_radius_skip', true );
		$btn_color_skip = get_post_meta( $c_form->ID, 'mdp_btn_color_skip', true );
		$btn_color_hover_skip = get_post_meta( $c_form->ID, 'mdp_btn_color_hover_skip', true );
		$btn_bg_color_skip = get_post_meta( $c_form->ID, 'mdp_btn_bg_color_skip', true );
		$btn_bg_color_hover_skip = get_post_meta( $c_form->ID, 'mdp_btn_bg_color_hover_skip', true );
		$btn_size_skip = get_post_meta( $c_form->ID, 'mdp_btn_size_skip', true );

		/** If button have caption make it rectangle. */
		if ( $btn_caption ) {
			$btn_padding = "calc({$btn_padding}px / 2) {$btn_padding}px";
		/** Else make it square. */
		} else {
			$btn_padding = "{$btn_padding}px";
		}

		/** If button have caption make it rectangle. */
		if ( $btn_caption_skip ) {
			$btn_padding_skip = "calc({$btn_padding_skip}px / 2) {$btn_padding_skip}px";
			/** Else make it square. */
		} else {
			$btn_padding_skip = "{$btn_padding_skip}px";
		}

        $css = Assets::get_instance()->get_contacter_inline_css();

		// language=CSS
        // @noinspection CssUnusedSymbol
		$css .= "
		    #{$id} {
            --contacter-btn-color: {$btn_bg_color};		    
            --contacter-btn-color--hover: {$btn_bg_color_hover};		    
		    }
		    #{$id} .mdp-contacter-start-btn:not(.mdp-contacter-custom),
		    #{$id} .mdp-speak-now-btns button,
		    #{$id} .mdp-contacter-thanks-box button,
		    #{$id} .mdp-send-btns button {
		        margin: {$btn_margin}px;
		        padding: {$btn_padding};
                border-radius: {$btn_radius}px;
                color: {$btn_color};
                background: {$btn_bg_color}; 
		    }
		    
		    #{$id} .mdp-contacter-start-btn:not(.mdp-contacter-custom):hover,
		    #{$id} .mdp-speak-now-btns button:hover,
		    #{$id} .mdp-contacter-thanks-box button:hover,
		    #{$id} .mdp-send-btns button:hover {
                fill: {$btn_color_hover};
                color: {$btn_color_hover};
                background: {$btn_bg_color_hover}; 
		    }
		    
		    #{$id} .mdp-contacter-start-btn:not(.mdp-contacter-custom):hover svg {
		        fill: {$btn_color_hover};
                color: {$btn_color_hover};
		    }
		    
		    #{$id} .mdp-contacter-start-btn.mdp-contacter-custom { cursor: pointer; }
		    
		    #{$id} .mdp-contacter-start-btn-icon svg {
		        width: {$btn_size}px;
                height: {$btn_size}px;
                fill: {$btn_color};
		    }
		    
		    #{$id} .mdp-contacter-start-btn .mdp-contacter-start-btn-caption {
		        font-size: {$btn_size}px;
		        line-height: 1.3;
		    }

		";

		$css .= "
		    #{$id} .mdp-contacter-skip-btn:not(.mdp-contacter-custom) {
		        margin: {$btn_margin_skip}px;
		        padding: {$btn_padding_skip};
                border-radius: {$btn_radius_skip}px;
                color: {$btn_color_skip};
                background: {$btn_bg_color_skip}; 
		    }
		    
		    #{$id} .mdp-contacter-skip-btn:not(.mdp-contacter-custom):hover {
                fill: {$btn_color_hover_skip};
                color: {$btn_color_hover_skip};
                background: {$btn_bg_color_hover_skip}; 
		    }
		    
		     #{$id} .mdp-contacter-skip-btn:not(.mdp-contacter-custom):hover svg {
		        fill: {$btn_color_hover_skip};
                color: {$btn_color_hover_skip};
		     }
		    
		    #{$id} .mdp-contacter-skip-btn.mdp-contacter-custom { cursor: pointer; }
		    
		    #{$id} .mdp-contacter-skip-btn-icon svg {
		        width: {$btn_size_skip}px;
                height: {$btn_size_skip}px;
                fill: {$btn_color_skip};
		    }
		    
		    #{$id} .mdp-contacter-skip-btn .mdp-contacter-skip-btn-caption {
		        font-size: {$btn_size_skip}px;
		        line-height: 1.3;
		    }

		";

		return $css;
	}

	/**
	 * Return Contacter Form object.
	 *
	 * @param array $atts - Shortcodes attributes.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return object - Contacter Form
	 **/
	private function get_cform( $atts ) {

		/** Get Contacter form data. */
		if ( ! empty( $atts['id'] ) ) {
			$c_form = get_post( $atts['id'] );
		} else {
			$c_form = get_page_by_title( $atts['title'], 'OBJECT', ContacterForm::POST_TYPE );
		}

		return $c_form;
	}

	/**
	 * Return unique id for current shortcode.
	 *
	 * @param array $atts - Shortcodes attributes.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string
	 **/
	private function get_shortcode_id( $atts ) {

		/** $call_count will be initialized on the first time call. */
		static $call_count = 0;

		/** call_count will be incremented each time the method gets called. */
		$call_count ++;

		return 'mdp-contacter-' . md5( json_encode( $atts ) ) . '-' . $call_count;

	}

	/**
	 * Replace {timer} {max-duration} {countdown} placeholders.
	 *
	 * @param string $message - Content for Speak Now step.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string
	 **/
	private function replace_placeholders( $message ) {

		/** Replace {timer}. */
		$timer   = '<p class="mdp-contacter-recording-timer">00:00</p>';
		$message = str_replace( '{timer}', $timer, $message );

		/** Replace {countdown}. */
		$countdown   = '<p class="mdp-contacter-recording-countdown">00:00</p>';
		$message = str_replace( '{countdown}', $countdown, $message );

		/** Replace {max-duration}. */
        if ( '0' === self::$max_duration ) { self::$max_duration = '∞'; }
		$max_duration_msg =
			'<p class="mdp-contacter-recording-max-duration">' .
                esc_html__( 'Max duration', 'contacter' ) .
				' <strong>' . self::$max_duration . '</strong> '.
				esc_html__( 'seconds', 'contacter' ) .
            '</p>';
		$message          = str_replace( '{max-duration}', $max_duration_msg, $message );

		return $message;
	}

    /**
     * Returns translated HTML
     *
     * @param $html
     */
	public function translate_html( $html ) {

        $string = strip_tags( $html );
        $tags = explode( $string, $html );
        $tags_count = preg_match( '/<\//', $html );

        // No html tags
        if ( $tags_count === 0 ) {

            esc_html_e( $string, 'contacter' );

        // One Closed tag
        } else if ( $tags_count === 1 ) {

            if ( count( $tags ) > 1 ) {

                echo wp_kses_post( $tags[ 0 ] );
                esc_html_e( $string, 'contacter' );
                echo wp_kses_post( $tags[ count( $tags ) - 1 ] );

            } else {

                echo wp_kses_post( $html );

            }


        }

    }

	/**
	 * Main Shortcodes Instance.
	 *
	 * Insures that only one instance of Shortcodes exists in memory at any one time.
	 *
	 * @static
	 * @return Shortcodes
	 * @since 1.0.0
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

} // End Class Shortcodes.
