<?php 
/**
 * Functions to allow styling of the templating system
 *
 * General core functions available on both the front-end and admin.
 *
 * @package /inc
 */

 if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Adds custom classes to the array of body classes.
 */
function pls_body_woocommerce_classes( $classes ) {
	
	if(  pls_is_catalog() ) { //Catalog Loop Class
		$classes[] = 'pls-catalog-page';
		
		if(  pls_get_option( 'shop-page-wide-layout', 0 ) ){
			$classes[] = 'pls-shop-page-wide-layout';
		}
		
		if(  pls_get_option( 'shop-page-off-canvas-sidebar', 0 ) ){
			$classes[] = 'pls-off-canvas-sidebar';
		}
		
		if(  pls_get_option( 'ajax-filter', 0 ) ){
			$classes[] = 'pls-catalog-ajax-filter';
		}		
	}
	
	if( is_product() ) { //Single Product Class
		
		$product_content_background = pls_get_post_meta( 'product_content_background' );
		if( ! $product_content_background || 'default' == $product_content_background ) {
			$product_content_background = pls_get_option( 'product-content-background', 0 );
		}
		
		$has_product_content_background = false;
		
		if( $product_content_background && $product_content_background != 'none' ){
			$has_product_content_background = true;
		}
	
		//$classes[]		= pls_get_product_gallery_style();
		$classes[]		= ( $has_product_content_background ) ? 'pls-product-content-background' : '';
		
		if( pls_get_option( 'single-product-quick-buy', 0 ) ) {
			$classes[] = 'pls-single-product-quick-buy';
		}
	}
	
	if( is_cart() && pls_get_option( 'cart-auto-update', 1 ) ) {
		$classes[] = 'has-auto-update-cart';
	}
	
	$classes = apply_filters( 'pls_body_woocommerce_classes', $classes );
	
	return $classes;
}

/**
 * Product loop row classes
 */
if ( ! function_exists( 'pls_product_row_classes' ) ):
	function pls_product_row_classes(){
		
		$product_style 		= pls_get_loop_prop( 'product-style' );
		$products_columns 	= pls_get_loop_prop( 'products-columns' );
		
		if( $product_style == 'products-widget' ){
			$classes[]	= 'product_list_widget';
			return implode( ' ', $classes );
		}
		
		if( $product_style == 'products-card' ){
			$classes[]	= 'product_list_widget row';
			$classes[] 	= 'grid-col-xl-'.pls_get_products_columns();			
			$classes[] 	= 'grid-col-lg-'.pls_get_loop_prop( 'products-columns-small-device' );
			$classes[] 	= 'grid-col-md-'.pls_get_loop_prop( 'products-columns-tablet' );
			$classes[] 	= 'grid-col-'.pls_get_loop_prop( 'products-columns-mobile' );
			return implode( ' ', $classes );
		}
		
		$classes[]	 		= 'products-wrap';
		$classes[] 			= $product_style;
				
		if( 'horizontal' == pls_get_loop_prop( 'product_view_mode' ) ) {
			$classes[] 		= 'pls-product-horizontal';
		}
		if( pls_get_loop_prop( 'name' ) == 'pls-slider' ){
			$classes[] 		= 'grid-view';
			$classes[] 		= 'swiper-wrapper';
			$classes[] 		= 'slider-col-xl-'.pls_get_loop_prop( 'slides_to_show' );
			$classes[] 		= 'slider-col-lg-'.pls_get_loop_prop( 'slides_to_show_small_device' );
			$classes[] 		= 'slider-col-md-'.pls_get_loop_prop( 'slides_to_show_tablet' );
			$classes[] 		= 'slider-col-'.pls_get_loop_prop( 'slides_to_show_mobile' );
		}else{			
			$classes[] 		= 'row';	
			if( 'grid-list' == pls_get_loop_prop( 'products_view' ) || 'list-view' == pls_get_loop_prop( 'products_view' ) ){
				$classes[] 		= 'list-view';
			}else{
				$classes[] 		= 'grid-view';
			}	
			$classes[] 	= 'grid-col-xl-'.pls_get_products_columns();			
			$classes[] 	= 'grid-col-lg-'.pls_get_loop_prop( 'products-columns-small-device' );
			$classes[] 	= 'grid-col-md-'.pls_get_loop_prop( 'products-columns-tablet' );
			$classes[] 	= 'grid-col-'.pls_get_loop_prop( 'products-columns-mobile' );
		}
		if ( pls_get_option( 'product-quick-shop', 0 ) && in_array( $product_style, [ 'product-style-1', 'product-style-2', 'product-style-5' ] ) ) { 
			$classes[] 	= 'has-quick-shop';
		}
		$classes = apply_filters( 'pls_product_row_classes', $classes );
		
		return implode( ' ', $classes );
	}
endif;

/**
 * Product loop classes
 */
if( ! function_exists( 'pls_product_loop_classes' ) ):
	function pls_product_loop_classes() {
		$classes = array();		
		if( pls_get_loop_prop( 'name' ) == 'pls-slider' ){
			$classes[] = 'swiper-slide';
		}
		return apply_filters( 'pls_product_loop_classes', $classes );
	}
endif;

/**
 * Single product classes
 */
function pls_woocommerce_product_class (){
	$classes	= [];
	$gallery_style				= pls_get_product_gallery_style();
	$product_content_layout 	= pls_get_product_content_layout();
	$product_content_fullwidth 	= pls_get_post_meta( 'product_content_fullwidth' );
	
	if( ! $product_content_fullwidth || 'default' == $product_content_fullwidth ) {
		$product_content_fullwidth = pls_get_option( 'product-content-fullwidth', 0 );
	}elseif( $product_content_fullwidth == 'enable' ) {
		$product_content_fullwidth = true;
	}elseif( $product_content_fullwidth == 'disable' ) {
			$product_content_fullwidth = false;
	}
	
	if ( 'product-gallery-horizontal' == $gallery_style ){
		$product_content_layout = 'style-2';
	}
	
	$classes[]		= 'pls-single-product';
	$classes[]		= 'pls-product-content-'. $product_content_layout;
	$classes[]		= ( $product_content_fullwidth ) ? 'pls-product-content-fullwidth' : '';
	$classes[]		= $gallery_style;
	
	return $classes;
}
//add_filter ( 'post_class', 'pls_woocommerce_product_class' );

function pls_get_product_gallery_style() {
	
	$gallery_layout = pls_get_post_meta( 'product_gallery_style' );
	
	if( ! $gallery_layout ) {
		$gallery_layout = pls_get_loop_prop( 'product-gallery-style' );	
	}	
	return $gallery_layout;
}

/**
 * Adds extra post classes for products.
 *
 * @return array
 */
function pls_get_product_content_layout() {
	
	$gallery_layout 		= pls_get_product_gallery_style();
	$product_content_layout = 'style-1';
	
	if ( 'product-gallery-horizontal' == $gallery_layout || 'product-gallery-horizontal-2' == $gallery_layout ){
			$product_content_layout = 'style-2';
	}
		
	return $product_content_layout;
}

/**
 * Mini Cart Slide
 */
if( ! function_exists( 'pls_wc_minicart_slide' ) ) :
	function pls_wc_minicart_slide(){
	
		if ( ! pls_get_option( 'header-minicart-popup', 1 ) || is_cart() || is_checkout() ){ 
			return; 
		} ?>
		
		<div class="pls-minicart-slide">
			
			<?php wc_get_template( 'cart/minicart-cross-sells.php' );	?>
			
			<div class="pls-minicart-items">
				<div class="pls-minicart-header">
					<h3 class="minicart-title"><?php echo apply_filters( 'pls_minicart_header_text', esc_html__( 'Shopping Cart', 'modave' ) );?></h3>
					<a href="#" class="pls-close-btn"><?php esc_html_e( 'Close', 'modave' ); ?></a>
				</div>
				<div class="woocommerce widget_shopping_cart">				
					<div class="widget_shopping_cart_content">					
						<?php woocommerce_mini_cart();?>
					</div>
				</div>
				<?php pls_wc_mincart_blocks(); ?>
			</div>
		</div>
		<?php
	}
endif;

/**
 * Mini Cart Cross Sells
 */
if( ! function_exists( 'pls_minicart_cross_sells' ) ) :
	function pls_minicart_cross_sells(){
		
		$products = array_filter( array_map( 'wc_get_product', WC()->cart->get_cross_sells() ), 'wc_products_array_filter_visible' );		
		return $products;
	}
endif;

/**
 * Canvas Sidebar
 */
if( ! function_exists( 'pls_wc_canvas_sidebar' ) ) :
	function pls_wc_canvas_sidebar() {
		
		if( 'full-width' == pls_get_layout() || ! pls_get_option( 'sidebar-canvas-mobile', 0 ) ) {
			return;
		}
		if( pls_get_option( 'mobile-bottom-navbar', 0 ) ){
			$mobile_elemets = (array)pls_get_option( 'mobile-navbar-elements',  array(
				'enabled'  => array(
					'shop'  		=> esc_html__( 'Shop', 'modave' ),
					'category'  	=> esc_html__( 'Category', 'modave' ),
					'sidebar'  		=> esc_html__( 'Sidebar/Filters', 'modave' ),
					'wishlist' 		=> esc_html__( 'Wishlist', 'modave' ),
					'cart'     		=> esc_html__( 'Cart', 'modave' ),							
				) ) );
			
			if( ! isset( $mobile_elemets['enabled'] ) ){
				$mobile_elemets['enabled'] =  array(
					'shop'  		=> esc_html__( 'Shop', 'modave' ),
					'category'  	=> esc_html__( 'Category', 'modave' ),
					'sidebar'  		=> esc_html__( 'Sidebar/Filters', 'modave' ),
					'wishlist' 		=> esc_html__( 'Wishlist', 'modave' ),
					'cart'     		=> esc_html__( 'Cart', 'modave' ),							
				);
			}
			if( array_key_exists( 'sidebar', $mobile_elemets['enabled'] ) ){
				return;
			}
		} ?>
		
		<div class="pls-canvas-sidebar">
			<div class="canvas-sidebar-icon"><?php esc_html_e( 'Sidebar', 'modave' )?></div>
		</div>
		<?php
	}
endif;

/**
 * Quick View Popup
 */
if( ! function_exists( 'pls_wc_quickview_popup' ) ) :
	function pls_wc_quickview_popup() { 
		if( ! pls_get_option( 'product-quickview-button', 1 ) ){
			return;
		} ?>
		<div class="pls-quick-view-slide">
			<a href="#" class="pls-close-btn"><?php esc_html_e( 'Close', 'modave' ); ?></a>
			<div class="pls-quick-view-content woocommerce"></div>
		</div>
		<?php
	}
endif;	
/**
 * User Login Signup Popup
 */
if( ! function_exists( 'pls_wc_login_signup_popup' ) ) :
	function pls_wc_login_signup_popup() {
		
		if( ! pls_get_option( 'show-login-register-popup', 1 ) ){
			return;
		}
		if ( ! shortcode_exists( 'woocommerce_my_account' ) || is_user_logged_in() ) {
			return;
		}
		if( is_account_page() || is_checkout() || is_page('vendor-registration') ){
			return;
		} 
		$account_link 	= get_permalink( get_option( 'woocommerce_myaccount_page_id' ) );
		?>	
		<div id="pls-wc-login-register-popup" class="pls-wc-login-register-popup woocommerce mfp-hide">
			<div class="pls-wc-login-form">
				<h2><?php esc_html_e( 'Login', 'modave' ); ?></h2>
			
				<form class="woocommerce-form woocommerce-form-login login" method="post" action="<?php echo esc_url( $account_link );?>">

					<?php do_action( 'woocommerce_login_form_start' ); ?>

				<p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">					
					<input type="text" class="woocommerce-Input woocommerce-Input--text input-text" name="username" id="username" placeholder="<?php esc_attr_e( 'Username or email address *', 'modave' ); ?>" autocomplete="username" value="<?php echo ( ! empty( $_POST['username'] ) ) ? esc_attr( wp_unslash( $_POST['username'] ) ) : ''; ?>" /><?php // @codingStandardsIgnoreLine ?>
				</p>
				<p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">					
					<input class="woocommerce-Input woocommerce-Input--text input-text" type="password" name="password" id="password" placeholder="<?php esc_attr_e( 'Password *', 'modave' ); ?>" autocomplete="current-password" />
				</p>

				<?php do_action( 'woocommerce_login_form' ); ?>

				<p class="form-row woocommerce-rememberme-lost_password">
						<label class="woocommerce-form__label woocommerce-form__label-for-checkbox woocommerce-form-login__rememberme">
						<input class="woocommerce-form__input woocommerce-form__input-checkbox" name="rememberme" type="checkbox" id="rememberme" value="forever" /> <span><?php esc_html_e( 'Remember me', 'modave' ); ?></span>
					</label>
					<a class="woocommerce-LostPassword" href="<?php echo esc_url( wp_lostpassword_url() ); ?>"><?php esc_html_e( 'Forgot your password?', 'modave' ); ?></a>
				</p>
				
				<p class="woocommerce-login-button">
					<?php wp_nonce_field( 'woocommerce-login', 'woocommerce-login-nonce' ); ?>
					<button type="submit" class="woocommerce-button button woocommerce-form-login__submit<?php echo esc_attr( wc_wp_theme_get_element_class_name( 'button' ) ? ' ' . wc_wp_theme_get_element_class_name( 'button' ) : '' ); ?>" name="login" value="<?php esc_attr_e( 'Log in', 'modave' ); ?>"><?php esc_html_e( 'Log in', 'modave' ); ?>
					</button>
					<?php				
					if ( 'yes' === get_option( 'woocommerce_enable_myaccount_registration' ) ) : ?>
						<a class="pls-new-register" href="<?php echo esc_url( $account_link ); ?>#register"><?php  echo esc_html__(' New to Register?','modave');?></a>
					<?php endif; ?>					
				</p>
				
				<?php do_action( 'woocommerce_login_form_end' ); ?>
				
				</form>				
			</div>
		</div>
		<?php
	}
endif;

/** 	
 * Ensure cart contents update when products are added to the cart via AJAX
 */
if( ! function_exists( 'pls_mini_cart_data' ) ) :
	function pls_mini_cart_data( $array ) {
		$count  		= WC()->cart->get_cart_contents_count();
		$count_hidden 	= ( $count < 1 ) ? ' pls-hidden' : '';
		$cart_count 	= '<span class="pls-header-cart-count'.$count_hidden.'">'. $count .'</span>';
		$cart_total 	= '<span class="pls-header-cart-total">'. $count .'</span>';
		$cart_item_text = '<span class="pls-header-cart-item-text">'. $count .' '._n( 'item', 'items', $count, 'modave' ).'</span>';
	
		$array['span.pls-header-cart-count'] 		= $cart_count;
		$array['span.pls-header-cart-total'] 		= $cart_total;
		$array['span.pls-header-cart-item-text'] 	= $cart_item_text;
		
		ob_start();
		wc_get_template( 'cart/minicart-cross-sells.php' );
		$minicart_cross_sells_products = ob_get_clean();
		$array['.pls-minicart-cross-sells'] = $minicart_cross_sells_products;
		
		ob_start();
		wc_get_template( 'cart/mini-cart-totals.php' );
		$mini_cart_total = ob_get_clean();
		$array['.pls-minicart-cart-totals'] = $mini_cart_total;
		
		ob_start();
		wc_get_template( 'cart/minicart-block-shipping.php' );
		$shipping_total = ob_get_clean();
		$array['.pls-minicart-shipping'] = $shipping_total;
		
		return $array;
	}	
	add_filter( 'woocommerce_add_to_cart_fragments', 'pls_mini_cart_data', 30 );
endif;

if( ! function_exists( 'pls_woosw_fragments' ) ) :
	function pls_woosw_fragments( $fragments ) {
		
		$wishlist_count	= WPcleverWoosw::get_count();; 
		$count_hidden 	= ( $wishlist_count < 1 ) ? ' pls-hidden' : ''; 		

		$fragments['.pls-header-wishlist .pls-header-wishlist-count']        = '<span class="pls-header-wishlist-count' . $count_hidden . '" data-count="' . $wishlist_count . '">' . $wishlist_count . '</span>';
		
		$fragments['.item-wishlist .pls-header-wishlist-count'] = '<span class="pls-header-wishlist-count">' . $wishlist_count . '</span>';
		
		return $fragments;
	}
	add_filter( 'woosw_fragments', 'pls_woosw_fragments' );
endif;

if( ! function_exists( 'pls_empty_mini_cart_button' ) ) :
	/** 	
	 * Empty Mini Cart Shop Button
	 */
	function pls_empty_mini_cart_button(){?>
	<p class="woocommerce-empty-mini-cart__buttons">
		<a class="button" href="<?php echo esc_url( apply_filters( 'woocommerce_return_to_shop_redirect', wc_get_page_permalink( 'shop' ) ) ); ?>"><?php echo apply_filters( 'pls_woocommerce_empty_mini_cart_button_text', esc_html__( 'Start Shopping', 'modave' ) );?></a>
	</p>
	<?php }
endif;

if ( ! function_exists( 'pls_shop_page_categories' ) ) :
	/**
	 * Shop Page Categories.
	 */
	function pls_shop_page_categories() {
			
		global $wp_query;
		$args 		= [ 'taxonomy' => 'product_cat', 'hide_empty ' => pls_get_option( 'shop-page-hide-empty-category', 0 ), 'parent' => 0, 'number' => pls_get_option( 'shop-page-categories-limit', 6 ), 'orderby' => 'ID', 'order' => 'DESC' ];
		
		$current_cat_id = isset( $wp_query->queried_object->term_id ) ? $wp_query->queried_object->term_id : 0 ;
		
		if( pls_get_option( 'shop-page-current-child-cat', 0 ) ){
			$args['parent'] = $current_cat_id;
		}		
		if( ! empty( pls_get_option( 'shop-page-selected-categories', [] ) ) ){
			unset( $args['parent'] );
			$args['include'] = pls_get_option( 'shop-page-selected-categories', [] );
		}
		if( ! empty( pls_get_option( 'shop-page-exculde-categories', [] ) ) ){
			$args['exclude'] = pls_get_option( 'shop-page-exculde-categories', [] );
		}
		
		$categories_style	= pls_get_option( 'shop-page-categories-style', 'rounded' );
		$columns			= pls_get_option( 'shop-page-categories-columns', 6 );
		
		$cat_args 		= apply_filters( 'pls_shop_page_category_args', $args );
		$categories 	= get_categories( $cat_args );
		if( empty( $categories ) ){
			return;
		}
		$current_active = false;
		$results 		= [];			
		foreach( $categories as $cat ){
			$cat_link = get_term_link( $cat );
			if( $current_cat_id == $cat->term_id ){
				$current_active = true;
			}
			$results[$cat->term_id] = [
				'name' 		=> $cat->name,
				'term_id' 	=> $cat->term_id,
				'slug' 		=> $cat->slug,
				'link' 		=> $cat_link,
				'current_active' => $current_active,				
			];
			$current_active = false;
		}
		$categories 	= apply_filters( 'pls_shop_page_categories' , $results ) ;		
		$slider_wrap	= $swiper_slide = $cat_class = '';
		$cat_class	   .=	'swiper-wrapper slider-col-lg-'.$columns.' slider-col-md-4 slider-col-2'; 
		$cat_slider_data	= array(
			'slider_loop'			=> false,
			'slider_navigation'		=> 'yes',
			'slider_dots'			=> false,
			'slides_to_show' 		=> $columns,
			'slides_to_show_tablet'	=> 4,
			'slides_to_show_mobile' => 2,
		);
		$slider_data 	= shortcode_atts( pls_slider_options(), $cat_slider_data );
		$slider_wrap	= 'pls-slider swiper';
		$cat_class		.= ' pls-category-'.$categories_style;
		$swiper_slide 	.= ' swiper-slide';			
		?>
		<div class="pls-shop-categories <?php echo esc_attr( $slider_wrap ); ?>">
			<div class="products pls-products-categories <?php echo esc_attr( $cat_class );?>" data-slider_options="<?php echo esc_attr( pls_slider_attributes($slider_data) ); ?>">
				<?php foreach ( $categories as $key => $data ) {
					$class = 'product-category cat-item-'.$key;
					if( $data['current_active'] ){
						$class .= ' has-current';
					}
					$class 			.= $swiper_slide;					
					$thumbnail_id 	= get_term_meta( $key, 'thumbnail_id', true );
					$thumb_url 		= wp_get_attachment_image_url( $thumbnail_id, 'woocommerce_thumbnail' ); ?>
					
					<div class="<?php echo esc_attr( $class ); ?>">
						<div class="pls-category-inner">
							<?php if( $thumb_url ){ ?>
								<div class="pls-category-image">
									<img src="<?php echo esc_url( $thumb_url );?>" alt="<?php echo esc_attr( $data['name'] ); ?>" />
								</div>
							<?php } ?>
							<div class="pls-category-thumb-content">
								<a href="<?php echo esc_url( $data['link'] ); ?>">
									<div class="pls-category-title">
										<?php echo esc_html( $data['name'] ); ?>
									</div>
								</a>
							</div>
						</div>
					</div>
				<?php } ?>
			</div>
		</div>
	<?php }	
endif;

/**
 * Shop Loop Header
 */
if( ! function_exists( 'pls_wc_shop_loop_header' ) ) :
	function pls_wc_shop_loop_header(){ 
	
		if( pls_get_loop_prop( 'is_shortcode' ) ){
			return;
		}
		
		wc_get_template_part( 'content', 'product_header' );
		
	}
endif;

if ( ! function_exists( 'pls_product_filter_top' ) ) :
	/**
	 * Shop top filter button
	 */
	function pls_product_filter_top() {
		
		if( pls_get_loop_prop( 'is_shortcode' ) ){
			return;
		}
		
		if( ! pls_get_option( 'shop-top-filter', 0 ) ) {
			return;
		}
		
		$filter_text =  apply_filters( 'pls_filter_button_text', esc_html__( 'Filters', 'modave' ) ); ?>
		
		<span class="pls-product-filter-btn"><?php echo esc_html( $filter_text );?></span>
	
	<?php }
endif;

if ( ! function_exists( 'pls_wc_loop_product_view' ) ) :
	/**
	 * Products view grid/list style on shop page
	 */
	function pls_wc_loop_product_view() {
		
		if( ! pls_get_loop_prop( 'products-view-icon' ) || ! wc_get_loop_prop( 'is_paginated' ) || ! woocommerce_products_will_display() ){ 
			return; 
		}
		
		$product_view = pls_get_loop_prop( 'products_view' );
		$produc_view_icons = pls_get_option( 'products-view-icon', [ 'grid-list', 'grid-two-col','grid-three-col','grid-four-col' ] );
		if( empty( $produc_view_icons ) ) {
			return;
		}
		$products_columns 	= pls_woocommerce_product_columns_text();
		if( 'list-view' == $product_view || 'grid-list' == $product_view ){
			$products_columns = 'grid-list';
		} ?>
		<div class="pls-products-view">
			<?php 
			foreach($produc_view_icons as $view_style ){ ?>
				<a class="pls-tooltip <?php echo esc_attr( $view_style )?> <?php echo esc_attr( $products_columns == $view_style ) ? 'active' : ''; ?>" data-shopview="<?php echo esc_attr( $view_style )?>" href="<?php echo esc_url( add_query_arg( array( 'view' => $view_style ) ) );?>"><?php echo esc_html(pls_woocommerce_product_grid_list_text( $view_style ) );?></a>
			<?php } ?>
		</div><?php 
	}
endif;

if ( ! function_exists( 'pls_woocommerce_product_columns_text' ) ) :
	
	function pls_woocommerce_product_columns_text() {
		$products_columns = pls_get_products_columns();
		$list = [ 
			'2' => 'grid-two-col',
			'3' => 'grid-three-col',
			'4' => 'grid-four-col',
			'5' => 'grid-five-col',
		];
		return isset( $list[$products_columns] ) ? $list[$products_columns] : '';
	}
endif;

if ( ! function_exists( 'pls_woocommerce_product_grid_list_text' ) ) :
	function pls_woocommerce_product_grid_list_text( $list_type ) {
		$list = [ 
			'grid-two-col' 		=> esc_html__( '2 Columns', 'modave' ),
			'grid-three-col' 	=> esc_html__( '3 Columns', 'modave' ),
			'grid-four-col'		=> esc_html__( '4 Columns', 'modave' ),
			'grid-five-col'		=> esc_html__( '5 Columns', 'modave' ),
			'grid-list' 		=> esc_html__( 'List', 'modave' ),
		];
		return isset( $list[$list_type] ) ? $list[$list_type] : '';
	}
endif;

if ( ! function_exists( 'pls_wc_loop_show_product_per_page' ) ) :
	/**
	 * Show number of products per page on product loop
	 */
	function pls_wc_loop_show_product_per_page() {
			
		if( ! pls_get_option( 'products-per-page-dropdown', 0 ) || ! wc_get_loop_prop( 'is_paginated' ) || ! woocommerce_products_will_display() ) {
			return;
		}
		
		$show_numbers = pls_get_shop_viewnumbers();
		$loop_shop_per_page = pls_loop_shop_per_page();
		
		if( !empty( $show_numbers ) ) { ?>
			<div class="pls-product-show">
				<form class="show-products-number" method="get">
					<span><?php esc_html_e( 'Show:', 'modave' ); ?></span>
					<select class="show-number per_page" name="per_page">
						<?php foreach( $show_numbers as $show_per_page ) { 	?> 
							<option value="<?php echo esc_attr($show_per_page); ?>" <?php selected( $show_per_page, $loop_shop_per_page );?>><?php echo absint($show_per_page);?></option>
						<?php } ?>
					</select>
					<?php
					foreach( $_GET as $name => $value ) {
						if ( 'per_page' != $name ) {
							printf( '<input type="hidden" name="%s" value="%s">', esc_attr( $name ), esc_attr( $value ) );
						}
					} ?>
				</form>
			</div>
		<?php }
	}
endif;

if ( ! function_exists( 'pls_wc_shop_top_filter_widgets' ) ) :
	/**
	 * Display Filter Widgets on Shop Page
	 */
	function pls_wc_shop_top_filter_widgets() {
		
		if( pls_get_loop_prop( 'is_shortcode' ) || ! pls_get_option( 'shop-top-filter', 0 ) ){
			return;
		} ?>
		
		<div class="pls-shop-top-filter-widgets" style="display:none">
			<div class="pls-filter-inner row">
				<?php 
				if ( is_active_sidebar( 'shop-filters-sidebar' ) ) {
					dynamic_sidebar( 'shop-filters-sidebar' );
				}else{	?>
					<div class="col">
						<?php esc_html_e( 'No, Any filters available.', 'modave' ); ?>
					</div>
				<?php
				} ?>
			</div>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_active_filter_widgets' ) ) :
	/**
	 * Display Active Filter on shop header
	 */
	function pls_wc_active_filter_widgets() { 
	
		if( pls_get_loop_prop( 'is_shortcode' ) || ! class_exists( 'Pls_Widget_Layered_Nav_Filters' ) ){
			return;
		} ?>
		<div class="pls-active-filters">
			<?php the_widget( 'Pls_Widget_Layered_Nav_Filters', array( 'title' => '' ), array() ); ?>
		</div>
	<?php
	}
endif;

if ( ! function_exists( 'pls_wc_clear_filters_btn' ) ) :
	/**
	 * Display Cleat Filter button on shop header
	 */
	function pls_wc_clear_filters_btn() { 
			global $wp;  
			$url 	= home_url( add_query_arg( array( $_GET ), $wp->request ) );
			$filters = array( 'filter_', 'rating_filter', 'min_price', 'max_price', 'product_visibility', 'stock','onsale' );
			$need_clear = false;
				
			foreach ( $filters as $filter ) {
				if ( strpos( $url, $filter ) ) {
					$need_clear = true;
				}
			}				
				
			if ( $need_clear ) {
				$reset_url = strtok( $url, '?' );
				if ( isset( $_GET['post_type'] ) ) {
					$reset_url = add_query_arg( 'post_type', wc_clean( wp_unslash( $_GET['post_type'] ) ), $reset_url );
				} ?>
				<div class="pls-clear-filters-wrapp">
					<a class="pls-clear-filters" href="<?php echo esc_url( $reset_url ); ?>"><?php echo esc_html__( 'Remove All', 'modave' ); ?></a>
				</div>
				<?php
			}
		}
endif;

if ( ! function_exists( 'pls_wc_off_canvas_sidebar' ) ) :
	/**
	 * Product Off Canvas Sidebar Button
	 */
	function pls_wc_off_canvas_sidebar() {
			
		if( pls_get_option( 'shop-top-filter', 0 ) ) {
			return;
		}
		if ( pls_is_vendor_page() ){
			return;
		}
		$filter_text =  pls_get_option( 'off-canvas-button-text', esc_html__( 'Filters', 'modave' ) ); ?>
		
		<span class="pls-product-off-canvas-btn"><?php echo esc_html( $filter_text ); ?></span>
	
	<?php }
endif;

if ( ! function_exists( 'pls_wc_onsale_product' ) ) :
	/**
	 * Product Onsale 
	 */
	function pls_wc_onsale_product() {
			
		if( ! pls_get_option( 'product-onsale', 0 ) ) {
			return;
		}
		
		if (  isset( $_GET['onsale'] ) &&  $_GET['onsale'] == 'onsale' ) {
            $onsale_selected = ' chosen';
            $salelink = remove_query_arg('onsale');
        } else {
            $onsale_selected = '';
            $salelink = add_query_arg('onsale','onsale');
        }
		?>
		<div class="widget widget_layered_nav pls-product-onsale">
			<ul>
				<li class="wc-layered-nav-term<?php echo esc_attr($onsale_selected);?>">
					<a href="<?php echo esc_url($salelink); ?>"> <?php esc_html_e( 'Show only products on sale', 'modave' )?> </a>
				</a>
			</ul>
		</div>
	<?php }
endif;

if ( ! function_exists( 'pls_product_status_onsale_query' ) ) :
	/**
	 * Product Onsale Query
	 */
	function pls_product_status_onsale_query( $query ) {
		/* Onsale Product*/
        if ( isset( $_GET['onsale'] ) &&  $_GET['onsale'] == 'onsale'  ) {
            $query->set ( 'post__in', wc_get_product_ids_on_sale() );
        }
	}
endif;

add_action( 'woocommerce_product_query', 'pls_product_status_onsale_query');


if( ! function_exists( 'pls_loop_shop_per_page' ) ) :
	/**
	 * Set per page product loop product page
	 */
	function pls_loop_shop_per_page(){
		
		$shop_loop_per_page = pls_get_loop_prop( 'products-per-page' );
		if ( isset( $_GET[ 'per_page' ] ) ) {
			return sanitize_text_field( $_GET[ 'per_page' ] );
		}
		
		return $shop_loop_per_page;
	}
	add_filter( 'loop_shop_per_page', 'pls_loop_shop_per_page', 20 );
endif;

if ( ! function_exists( 'pls_woocommerce_loop_product_wrapper' ) ) :
	/**
	 * Product loop wrapper start
	 */
	function pls_woocommerce_loop_product_wrapper() { ?>
		<div class="pls-product-inner">
	<?php }
endif;

if ( ! function_exists( 'pls_woocommerce_product_wrapper_end' ) ) :
	/**
	 * Product loop wrapper end
	 */
	function pls_woocommerce_product_wrapper_end() { ?>
		</div>
	<?php }
endif;

if ( ! function_exists( 'pls_woocommerce_before_shop_loop_item_title' ) ) :
	/**
	 * Product loop image
	 */
	function pls_woocommerce_before_shop_loop_item_title() { ?>
		<div class="pls-product-image">
			<?php
			/**
			 * Hook: pls_woocommerce_before_shop_loop_item_title.
			 *
			 * @hooked pls_wc_template_loop_product_thumbnail - 10
			 */
			 do_action( 'pls_woocommerce_before_shop_loop_item_title' );?>
		 </div>
		 <?php 
	}
endif;

if ( ! function_exists( 'woocommerce_template_loop_category_title' ) ) {

	/**
	 * Show the subcategory title in the product loop.
	 *
	 * @param object $category Category object.
	 */
	function woocommerce_template_loop_category_title( $category ) {
		?>
		<div class="pls-category-title">
			<?php echo esc_html( $category->name ); ?>
		</div>
		<?php
		if ( $category->count > 0 ) { ?>
			<div class="pls-product-count-wrap">
				<?php 
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters( 'woocommerce_subcategory_count_html', ' <mark class="count">(' . esc_html( $category->count ) . ')</mark>', $category );
				?>
				<span class="pls-category-arrow"></span>
			</div>
			<?php 
		}
	}
}

if ( ! function_exists( 'pls_subcategory_count_html' ) ) :
	/**
	 * Categories loop products count
	 */
	function pls_subcategory_count_html( $html, $category ) { 
			 return sprintf(
				'<span class="product-count">%1$s</span>',
				sprintf( _n( '%s Item', '%s Items', $category->count, 'modave' ), $category->count )
			);
	}
	add_filter('woocommerce_subcategory_count_html', 'pls_subcategory_count_html', 10, 2);
endif;

if ( ! function_exists( 'pls_wc_template_loop_product_thumbnail' ) ) :
	/**
	 * Get the product thumbnail, slider for the loop.
	 */
	function pls_wc_template_loop_product_thumbnail() {
				
		global $product;

		$image_size 	= apply_filters( 'single_product_archive_thumbnail_size', 'woocommerce_thumbnail' );		
		$attachment_ids = $product->get_gallery_image_ids();
		$hover_image 	= '';
		$mobile_hover_image	= true;
			
		if( wp_is_mobile() && ! pls_get_option( 'mobile-product-hover-image', 1 ) ) {
			$mobile_hover_image	= false;
		}
		
		if ( $mobile_hover_image && ! empty( $attachment_ids[0] ) ) {
			$hover_image = pls_get_image_html( $attachment_ids[0] , $image_size, 'hover-image' );
		}
		
		$link = apply_filters( 'woocommerce_loop_product_link', get_the_permalink(), $product );
		
		$target = '_self';
		if( pls_get_option( 'open-product-page-new-tab', 0 ) ){
			$target = '_blank';
		}
		
		$html = '<a href="'. esc_url( $link ) .'" class="woocommerce-LoopProduct-link" target="'.$target.'" aria-label="' . esc_attr( get_the_title() )  . '">';		
			$html .=  $product ? pls_get_post_thumbnail( $image_size, 'front-image' ) : '';			
			if( '' != $hover_image && pls_get_option( 'product-hover-image', 1 ) ):
				$html .= $hover_image;
			endif;			
		$html .= '</a>';
		
		echo apply_filters( 'pls_wc_template_loop_product_thumbnail', $html );
	}
endif;

if ( ! function_exists( 'pls_woocommerce_shop_loop_item_title' ) ) :
	/**
	 * Product loop title hooke
	 */
	function pls_woocommerce_shop_loop_item_title() { 
		/**
		 * Hook: pls_woocommerce_shop_loop_item_title.
		 *
		 * @hooked pls_woocommerce_loop_product_info_wrapper - 5
		 * @hooked pls_product_loop_category - 15
		 * @hooked woocommerce_template_loop_product_title - 20
		 * @hooked woocommerce_template_single_excerpt - 20
		 * @hooked pls_woocommerce_product_wrapper_end - 50
		 */
		 do_action( 'pls_woocommerce_shop_loop_item_title' );
	}
endif;

if ( ! function_exists( 'pls_woocommerce_loop_product_info_wrapper' ) ) :
	/**
	 * Product loop info wrapper start
	 */
	function pls_woocommerce_loop_product_info_wrapper() { ?>
		<div class="pls-product-info">
	<?php }
endif;

if( ! function_exists( 'pls_wc_product_loop_categories' ) ) :
	
	function pls_wc_product_loop_categories() { 

		global $product;
		
		if( ! pls_get_option( 'product-category', 0 ) ) { return; } ?>
		
		<div class="pls-product-cats">
			<?php echo wc_get_product_category_list( $product->get_id(), ', ' );?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'woocommerce_template_loop_product_title' ) ) :
	/**
	 * Show the product title in the product loop. By default this is an H3.
	 */
	function woocommerce_template_loop_product_title() {
		
		if( ! pls_get_option( 'product-title', 1 ) ) { return; }
		
		global $product;

		$link = apply_filters( 'woocommerce_loop_product_link', get_the_permalink(), $product );
		
		$target = '_self';
		if( pls_get_option( 'open-product-page-new-tab', 0 ) ){
			$target = '_blank';
		}
		echo '<h3 class="product-title"><a href="' . esc_url( $link ) . '" target="'.$target.'">' . get_the_title() . '</a></h3>';
	}
endif;

if ( ! function_exists( 'pls_product_labels' ) ) :
	/**
	 * Product labels
	 */
	function pls_product_labels( $sale_label = '' ) {
		global $product;
		$output 				= array();
		$sale_percentage_label 	= ( $sale_label == 'percentage' ) ? $sale_label : pls_get_option( 'sale-product-label-text-options', 'percentage' );
		
		if ( pls_get_option( 'product-new-label', 0 ) ) {
			
			$postdate 		= get_the_time( 'Y-m-d' );
			$postdatestamp 	= strtotime( $postdate );
			$newness 		= pls_get_option( 'product-newness-days', 90 ); 
			$new_label_text	= pls_get_option( 'new-product-label-text', 'New' );

			if ( ( time() - ( 60 * 60 * 24 * $newness ) ) < $postdatestamp ) {
				$output['new'] = '<span class="new">' . $new_label_text . '</span>';
			}					
		}
		
		if( $product->is_on_sale() && pls_get_option( 'sale-product-label', 1 ) ) {		
			$percentage = '';
			if( $product->get_type() == 'variable' && $sale_percentage_label =='percentage' ){				
				$available_variations = $product->get_variation_prices();
				$max_value = 0;
				foreach( $available_variations['regular_price'] as $key => $regular_price ) {					
					$sale_price = $available_variations['sale_price'][$key];					
					if ( $sale_price < $regular_price ) {
						$percentage = round( ( ( $regular_price - $sale_price ) / $regular_price ) * 100 );
						if ( $percentage > $max_value ) {
							$max_value = $percentage;
						}
					}
				}
				$percentage = $max_value;
				
			} elseif ( ( $product->get_type() == 'simple' || $product->get_type() == 'external' ) && $sale_percentage_label =='percentage' ) {				
				$percentage = round( ( ( $product->get_regular_price() - $product->get_sale_price() ) / $product->get_regular_price() ) * 100 );
			}
			if ( $percentage ) {	
				$sale_label_text = pls_get_option( 'sale-product-label-percentage-text', '' );
				$sale_minus = ( empty( $sale_label_text ) ) ? '-' : '';
				$output['sale'] = '<span class="on-sale">'.$sale_minus.''. $percentage . '% ' . $sale_label_text . '</span>';
			}else{				
				if($product->is_on_sale() && $sale_percentage_label == 'percentage' ){
					/* Fixed issue for you may also like variable products*/
					$percentage = 0;
					if($product->get_regular_price() && $product->get_sale_price()){
						$percentage = round( ( ( $product->get_regular_price() - $product->get_sale_price() ) / $product->get_regular_price() ) * 100 );
					}
					if( $percentage > 0 ){
						$sale_label_text = pls_get_option( 'sale-product-label-percentage-text', '' );
						$sale_minus = ( empty( $sale_label_text ) ) ? '-' : '';
						$output['sale'] = '<span class="on-sale">'.$sale_minus.''. $percentage . '% ' . $sale_label_text . '</span>';
					}
				} else {
					$sale_label_text = pls_get_option( 'sale-product-label-text', 'Sale' );
					$output['sale'] = '<span class="on-sale">' . $sale_label_text . '</span>';
				}
				
			}
		}		

		if ( $product->is_featured() && pls_get_option( 'featured-product-label', 1 ) ) {
			$featured_label_text = pls_get_option( 'featured-product-label-text', 'Hot' );
			$output['featured'] = '<span class="featured">' . $featured_label_text . '</span>';
		}	
		
		if ( pls_get_option( 'pre-order-product-label', 1 ) && $product->is_on_backorder() ) {
			$preorder_label_text	= pls_get_option( 'pre-order-product-label-text', esc_html__( 'Pre Order', 'modave' ) );				
			$output['pre_order']	= '<span class="pre-order">' . esc_html( $preorder_label_text ) . '</span>';
		}
		
		if( !$product->is_in_stock() && pls_get_option( 'outofstock-product-label', 1 ) ){
			$out_stock_label_text = pls_get_option( 'outofstock-product-label-text', 'Out Of Stock' );
			$output['out_of_stock'] = '<span class="out-of-stock">' . $out_stock_label_text . '</span>';
		}
		if ( ! is_user_logged_in() && pls_get_option( 'login-to-see-price', 0 ) ) {
			if( isset( $output['sale'] ) ){
				unset( $output['sale'] );
			}
		}		
		return apply_filters( 'pls_product_labels', $output );
	}
endif;

if ( ! function_exists( 'pls_wc_product_labels' ) ) :
	/**
	 * Product labels
	 */
	function pls_wc_product_labels() {
		
		if( ! pls_get_option( 'product-labels', 1 ) ){ return; }
		
		$output_labels = pls_product_labels();
		$html='';
		$current_filter = current_filter();
		
		if( isset( $output_labels['out_of_stock'] ) ){			
			unset( $output_labels['out_of_stock'] );			
		}		
		if ( ! empty( $output_labels ) ) {
			$html = '<div class="pls-product-labels">' . implode( '', $output_labels ) . '</div>';
		}
		echo apply_filters( 'pls_wc_product_labels', $html, $output_labels );
	}
endif;

if ( ! function_exists( 'pls_wc_product_widget_sale_label' ) ) :
	/**
	 * Sale label on product widget image
	 */
	function pls_wc_product_widget_sale_label() {
		global $product;
		if( ! $product->is_on_sale() || ( ! pls_get_option( 'sale-product-label', 1 )  ) ) { return; }; ?>
		
		<div class="pls-product-discount-label">
			<?php
			$sale = pls_product_labels();		
			echo isset ( $sale['sale'] ) ? $sale['sale'] : '';
			?>
		</div> <?php
	}
endif;	

if ( ! function_exists( 'pls_wc_single_product_sale_label' ) ) :
	/**
	 * Add Single Product Sale Label After Price
	 */
	function pls_wc_single_product_sale_label() {
		global $product;
		if( ! $product->is_on_sale() || ( pls_get_option( 'sale-product-label', 1 ) && ! pls_get_option( 'product-price-discount-label', 0 ) ) ) { return; } ?>
		
		<div class="pls-product-discount-label">
			<?php
			$sale = pls_product_labels( 'percentage' );		
			echo isset ( $sale['sale'] ) ? $sale['sale'] : '';
			?>
		</div> <?php
	}
endif;

if ( ! function_exists( 'pls_is_schedule_product' ) ) :
	function pls_is_schedule_product(){
		global $product;
		$has_schedule = false;
		$sale_time = '';
		
		if ( ! $product->is_in_stock() ) : 
			return; 
		endif;
		
		if ( $product->is_on_sale() ) : 
			$sale_time = get_post_meta( $product->get_id(), '_sale_price_dates_to', true );
		endif;
			
		/* variable product */
		if( $product->has_child() && $product->is_on_sale() ){
			$vsale_end = array();
			
			$pvariables = $product->get_children();
			foreach($pvariables as $pvariable){
				$vsale_end[] = (int)get_post_meta( $pvariable, '_sale_price_dates_to', true );
			}			
			/* get the latest time */
			$sale_time = max( $vsale_end );				
		}
		
		if( $product->is_on_sale() && $sale_time ) {
			$has_schedule =  true;
		}
		return $has_schedule;
	}
endif;

if ( ! function_exists( 'pls_wc_product_sale_marquee' ) ) :
	/**
	 * Product Sale Marquee
	 */
	function pls_wc_product_sale_marquee() {
		global $product;
		
		if( ! $product->is_on_sale() || ( ! pls_get_option( 'sale-product-label', 1 ) || ! pls_get_option( 'product-sale-marquee', 0 ) ) ) { return; }; 
		
		if( pls_get_loop_prop( 'product-countdown' ) ){
			if( pls_is_schedule_product() ){
				return;
			}
		}
		$sale 				= pls_product_labels( 'percentage' );				
		$sale_per 			= isset ( $sale['sale'] ) ? $sale['sale'] : '';		
		$sale_per 			= preg_replace('/(?<!\w)-/', '', $sale_per);		
		$marquee_text 		= pls_get_option( 'product-sale-marquee-text', 'Hot Sale {per} off.' );				
		$marquee_text  		= str_replace( '{per}', $sale_per, $marquee_text );		
		$sale_marque_string = '<div class="pls-marquee-item">'. $marquee_text .'</div><div class="pls-marquee-separator"></div>';
		$marquee_array 		= array( $sale_marque_string, $sale_marque_string, $sale_marque_string, $sale_marque_string, $sale_marque_string );		
		$sale_marque_output = implode( '', $marquee_array ); ?>		
		<div class="pls-product-sale-marquee">
			<div class="pls-marquee">
				<div class="pls-marquee-items">
					<?php echo wp_kses_post( $sale_marque_output ); ?>
				</div>
			</div>
			<div class="pls-marquee">
				<div class="pls-marquee-items">
					<?php echo wp_kses_post( $sale_marque_output ); ?>
				</div>
			</div>
		</div> 
		<?php
	}
endif;

if ( ! function_exists( 'pls_get_product_size_terms' ) ) :
	/**
	 * Product Size Terms
	 */
	function pls_get_product_size_terms() { 
		global $product;
		$attributes			= $product->get_attributes();
		$product_size_term 	= '';
		
		foreach ( $attributes as $attribute_name => $options ) {			
			if ($attribute_name === 'pa_size') {
				$product_size_term = $attribute_name;
				break;
			}
		}
		
		if( empty( $product_size_term )){ 
			return;
		}
				
		$terms 		= wc_get_product_terms( $product->get_id(), $product_size_term, array( 'fields' => 'all' ) );			
		
		if( is_wp_error( $terms ) || empty( $terms ) ){
			return;
		}
		
		$term_names = [];		
		foreach ($terms as $index => $term_data) {
			if ($index >= 4) {
				break;
			}
			$term_names[] = '<span>' . $term_data->name . '</span>';
		}
		
		return implode( ' ', $term_names );
	}
endif;

if ( ! function_exists( 'pls_wc_product_variation_label' ) ) :
	/**
	 * Product loop variation label
	 */
	function pls_wc_product_variation_label() { 
		global $product;
			
		if( ! pls_get_option( 'product-variation-size', 0 ) || $product->get_type() != 'variable' ){
			return;
		}
		
		$term_list = pls_get_product_size_terms();
		if( empty( $term_list ) ){
			return;
		} ?>
		<div class="pls-product-variation-label">
			<?php echo wp_kses_post( $term_list ); ?>
		</div> 
	<?php }	
endif;	

if ( ! function_exists( 'pls_wc_product_loop_cart_button' ) ) :
	/**
	 * Product loop cart button
	 */
	function pls_wc_product_loop_cart_button() {
		
		if( ! pls_get_option( 'product-buttons', 1 ) || ! pls_get_option( 'product-cart-button', 1 ) ||  pls_get_option( 'catalog-mode', 0 ) || ( ! is_user_logged_in() && pls_get_option( 'login-to-see-price', 0 ) ) ) {
			return; 
		} ?>
		
		<div class="pls-cart-button">
			<?php
			/**
			 * Hook: pls_wc_product_loop_cart_button.
			 *
			 * @hooked woocommerce_template_loop_add_to_cart - 10
			 */
			 do_action( 'pls_wc_product_loop_cart_button' );?>
		 </div>
		<?php 
	}
endif;

if ( ! function_exists( 'pls_woocommerce_product_loop_wishlist_button' ) ) :
	/**
	 * Product loop wishlist button
	 */
	function pls_woocommerce_product_loop_wishlist_button() {
		
		if( ! pls_get_option( 'product-buttons', 1 ) || ! pls_get_option( 'product-wishlist-button', 1 ) ){
			return;
		} 
		if ( ! class_exists( 'WPCleverWoosw' ) ) {
			return; 
		}
		global $product;
		$product_id = $product->get_id(); ?>
		
		<div class="pls-whishlist-btn">
			<?php echo do_shortcode( '[woosw id="' . $product_id . '" type="link"]' ); ?>
		</div>		
	<?php }
endif;

if ( ! function_exists( 'pls_woocommerce_product_loop_compare_button' ) ) :
	/**
	 * Product loop compare button
	 */
	function pls_woocommerce_product_loop_compare_button() {
		
		if( ! pls_get_option( 'product-buttons', 1 ) || ! pls_get_option( 'product-compare-button', 1 ) ) {
			return;
		}
		if ( ! class_exists( 'WPCleverWoosc' ) ) {
			return; 
		}
		global $product;
		$product_id = $product->get_id(); ?>
		
		<div class="pls-compare-btn">
			<?php echo do_shortcode( '[woosc id="' . $product_id . '" type="link"]' ); ?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_compare_add_product_url' ) ) :
	function pls_compare_add_product_url( $product_id ) {

		$action_add = 'yith-woocompare-add-product';

		$url_args = array(
			'action' => $action_add,
			'id'     => $product_id,
		);

		return apply_filters( 'yith_woocompare_add_product_url',
			esc_url_raw( add_query_arg( $url_args ) ),
			$action_add );
	}
endif;

if ( ! function_exists( 'pls_wc_product_loop_quick_view_button' ) ) :
	/**
	 * Product loop quick view button
	 */
	function pls_wc_product_loop_quick_view_button( $class = '', $product_id = null ) {
		
		$product_id = $product_id ? $product_id : get_the_ID();
		if( ! pls_get_option( 'product-buttons', 1 ) || ! pls_get_option( 'product-quickview-button', 1 ) ) {
			return; 
		} ?>
		
		<div class="pls-quickview-button">
			<a class="pls-quickview-btn <?php echo esc_attr($class); ?>" href="<?php echo esc_url( get_the_permalink() );?>" data-id="<?php echo esc_attr( $product_id );?>"><?php esc_html_e( 'Quick View', 'modave' )?></a>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_quantity_button_plus' ) ) :
	/**
	 * Quantity Button Plus
	 */
	function pls_wc_quantity_button_plus() { ?>
		<label class="plus"></label>
	<?php }
endif;

if ( ! function_exists( 'pls_wc_quantity_button_minus' ) ) :
	/**
	 * Quantity Button Minus
	 */
	function pls_wc_quantity_button_minus() { ?>
		<label class="minus"></label>
		
	<?php }
endif;

if ( ! function_exists( 'pls_wc_stock_progress_bar' ) ) :
	/**
	 * Product Stock Available & Sold Progress Bar
	 */
	function pls_wc_stock_progress_bar() {
		if( ! pls_get_loop_prop( 'products-stock-progressbar' ) ){
			return;
		}
		global $product;
		$product_error 		= false;
		$productId 			= get_the_ID();	
		$stock_available 	= false;	
		$stock_sold 		= ( $total_sales = get_post_meta( $productId, 'total_sales', true ) ) ? round( $total_sales ) : 0;
		$stock_available 	= ( $stock = get_post_meta($productId, '_stock', true ) ) ? round( $stock ) : 0;
		$percentage 		= $stock_available > 0 ? round( $stock_sold / ( $stock_available + $stock_sold ) * 100 ) : 0;
		if( $stock_available ) : ?>
			<div class="pls-product-stock-progressbar">
				<div class="pls-progress">
					<span class="pls-progress-bar active" style="<?php echo esc_attr( 'width:' . $percentage . '%' ); ?>"><?php echo esc_html( $percentage ).'%'; ?></span>
				</div>
				<div class="pls-product-stock-label">
					<span class="pls-stock-available"><?php esc_html_e( 'Available:', 'modave' );?> <span><?php echo esc_html($stock_available ); ?></span></span>
					<span class="pls-stock-sold"><?php esc_html_e( 'Sold:', 'modave' );?> <span><?php echo esc_html( $stock_sold ); ?></span></span>
				</div>
			</div>
		<?php endif;
	}
endif;

if ( ! function_exists( 'pls_woocommerce_after_shop_loop_item' ) ):
	/**
	 * Product after shop loop wrapper end
	 */
	function pls_woocommerce_after_shop_loop_item() {
		/**
		 * Hook: pls_woocommerce_after_shop_loop_item.
		 *
		 * @hooked pls_woocommerce_product_wrapper_end - 10
		 * @hooked pls_woocommerce_product_wrapper_end - 20
		 * @hooked pls_woocommerce_product_wrapper_end - 30
		 */
		 do_action( 'pls_woocommerce_after_shop_loop_item' );
	}
endif;
	
if ( ! function_exists( 'pls_single_product_photoswipe_btn' ) ) :
	/**
	 * Single product photoswipe button
	 */
	function pls_single_product_photoswipe_btn(){
		
		if( ! pls_get_option( 'product-gallery-lightbox', 1 ) ) {
			return; 
		} ?>
		
		<div class="product-photoswipe-btn">
			<a href="#" class="pls-product-image-full"><?php esc_html_e('Lightbox', 'modave'); ?></a>
		</div>		
	<?php
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_video_btn' ) ) :
	/**
	 * Single product video button
	 */
	function pls_wc_single_product_video_btn(){
		
		if( ! pls_get_option( 'product-video', 1 ) ) {
			return;
		}
		$prefix 	= PLS_PREFIX;
		$video_url 	= get_post_meta(get_the_ID(),  $prefix .'product_video', true );
		if( ! empty( $video_url ) ){ ?>
			<div class="product-video-btn">
				<a href="<?php echo esc_url( $video_url ); ?>" class="pls-video-popup"><?php esc_html_e( 'Watch Video', 'modave' ); ?></a>
			</div>
			
		<?php }
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_degree360_btn' ) ) :
	/**
	 * Single product 360 degree View button
	 */
	function pls_wc_single_product_degree360_btn(){
		
		if( ! pls_get_option( 'product-360-degree', 1 ) ) {
			return;
		}
		global $post;
		if ( ! $post ) {
			return;
		}
		$prefix 			= PLS_PREFIX;
		$gallery_images 	= get_post_meta($post->ID,  $prefix .'product_360_degree_images' );
		
		if( ! empty( $gallery_images ) ){ ?>
			<div class="product-360-degree-btn">
				<a href="#pls-360-degree-wrapper" ><?php esc_html_e('360 Degree', 'modave'); ?></a>
			</div>			
		<?php }
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_360_degree_content' ) ) :
	/**
	 * Single Product 360 Degree Content
	 */
	function pls_wc_single_product_360_degree_content(){
		
		if( ! pls_get_option( 'product-360-degree', 1 ) ) {
			return;
		}
		global $post;
		if ( ! $post ) {
			return;
		}
		$prefix 	= PLS_PREFIX;
		$gallery_images 	= get_post_meta( $post->ID,  $prefix .'product_360_degree_images' );
		if( empty( $gallery_images ) ){
			return;
		}
		$image_array = array();
		foreach ( $gallery_images as $attachment_id ) {
			$image_src = wp_get_attachment_image_url( $attachment_id, 'woocommerce_single' );
			if( $image_src ){
				$image_array[] = "'" . $image_src . "'";
			}		
		}
		$frames_count  = count( $image_array );
		$images_js_string = implode( ',', $image_array );	?>
		<div id="pls-360-degree-wrapper" class="pls-360-degree-wrapper mfp-hide">
			<ol class="pls-360-degree-images"></ol>	
			<div class="spinner">
				<span>0%</span>
			</div>
		</div>
		<?php
		wp_enqueue_script( 'threesixty' );
		wp_add_inline_script('threesixty',
			'jQuery(document).ready(function( $ ) {
				$(".pls-360-degree-wrapper").ThreeSixty({
					totalFrames: ' . esc_js( $frames_count ) . ',
					endFrame: ' . esc_js( $frames_count ) . ',
					currentFrame: 1,
					imgList: ".pls-360-degree-images",
					progress: ".spinner",
					imgArray: ' . '[' . $images_js_string . ']' . ', 
					width: 300,
					height: 300,
					responsive: true,
					navigation: true,
					position: "bottom-center",
				});
			});',
			'after'
		);
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_top' ) ):
	/**
	 * Before Single Product
	 */
	function pls_wc_single_product_top() { 
		global $product;
		if ( ! $product || ! is_singular( 'product' ) ) {
			return;
		}
		
		wc_get_template_part( 'content','single-product-top' );
	}
endif;

if( ! function_exists( 'pls_wc_single_product_navigation' ) ) :
	/**
	 * Single Product Navigation
	 */
	function pls_wc_single_product_navigation(){
		
		if( ! pls_get_option( 'single-product-navigation', 1 ) ) { return; }
	
		$next = get_next_post();
	    $prev = get_previous_post();

	    $next = ( ! empty( $next ) ) ? wc_get_product( $next->ID ) : false;
	    $prev = ( ! empty( $prev ) ) ? wc_get_product( $prev->ID ) : false; ?>
		
		<div class="pls-product-navigation">
			<?php if ( ! empty( $prev ) ): ?>
				<div class="pls-product-nav-btn pls-product-nav-prev">
					<a href="<?php echo esc_url( $prev->get_permalink() ); ?>">
						<?php esc_html_e( 'Previous Product', 'modave' ); ?>
					</a>
				</div>
			<?php endif ?>
			<div class="pls-product-nav-btn pls-product-nav-archive">				
				<a href="<?php echo esc_url( wc_get_page_permalink( 'shop' ) ); ?>">
					<?php esc_html_e( 'Archive Product', 'modave' ); ?>
				</a>
			</div>
			<?php if ( ! empty( $next ) ): ?>
				<div class="pls-product-nav-btn pls-product-nav-next">				
					<a href="<?php echo esc_url( $next->get_permalink() ); ?>">
						<?php esc_html_e( 'Next Product', 'modave' ); ?>
					</a>
				</div>
			<?php endif ?>
		</div>
	<?php }
endif;

if ( ! function_exists( 'pls_wc_single_product_category' ) ):
	/**
	 * Single Product Category
	 */
	function pls_wc_single_product_category() {
		global $product; ?>
		<div class="pls-product-cat">
			<?php echo wc_get_product_category_list( $product->get_id(), ', ', '<span class="posted_in">', '</span>' );?>
		</div>
	<?php }	
endif;

if ( ! function_exists( 'pls_wc_single_product_rating_info' ) ):
	/**
	 * Single Product Rating Info
	 */
	function pls_wc_single_product_rating_info() { ?>	
		<div class="pls-product-rating-info"><?php
			/**
			 * Hook: pls_wc_single_product_rating_info.
			 *
			 * @hooked woocommerce_template_single_rating - 5
			 * @hooked pls_wc_single_product_sold_count - 10
			 */
			 do_action( 'pls_wc_single_product_rating_info' );
		?></div>
	<?php
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_price_info' ) ):
	/**
	 * Single Product Price Info
	 */
	function pls_wc_single_product_price_info() { ?>
		<div class="pls-product-price-info"><?php 
			/**
			 * Hook: pls_wc_single_product_price_info.
			 *
			 * @hooked woocommerce_template_single_price - 5
			 * @hooked pls_wc_single_product_sale_label - 10
			 */
			 do_action( 'pls_wc_single_product_price_info' );
		?></div>
	<?php
	}
endif;

if ( ! function_exists( 'pls_wc_product_sale_countdown' ) ):
	/**
	 * Sale Product Countdown
	 */
	function pls_wc_product_sale_countdown() {
		
		$current_filter = current_filter();
		if( ( ! pls_get_loop_prop( 'product-countdown' )  && ! in_array( $current_filter, ['woocommerce_single_product_summary','woocommerce_single_product_summary_first'] ) ) || ( is_product() && ( $current_filter == 'woocommerce_single_product_summary' || $current_filter == 'woocommerce_single_product_summary_first' ) && ! pls_get_option( 'single-product-countdown', 1 ) ) ) {
			return; 
		}
		
		global $product;
		$html = $sale_time 	= $offer_text = $offer_html = '';
		$timezone 			= wc_timezone_string();
		
		if ( ! $product->is_in_stock() ) : 
			return; 
		endif;

		if ( $product->is_on_sale() ) : 
			$sale_time = get_post_meta( $product->get_id(), '_sale_price_dates_to', true );
		endif;
		
		/* variable product */
		if( $product->has_child() && $product->is_on_sale() ){
			$vsale_end = array();
			
			$pvariables = $product->get_children();
			foreach($pvariables as $pvariable){
				$vsale_end[] = (int)get_post_meta( $pvariable, '_sale_price_dates_to', true );
			}			
			/* get the latest time */
			$sale_time = max( $vsale_end );				
		}
		
		if( $product->is_on_sale() && $sale_time ) {
			wp_enqueue_script( 'countdown' );
			$sale_time = $sale_time;
			$sale_time = date( 'Y-m-d H:i:s', $sale_time ); 
			if ( is_product() && ( $current_filter == 'woocommerce_single_product_summary' || $current_filter == 'woocommerce_single_product_summary_first' ) && pls_get_option( 'single-product-countdown', 1 ) ) { ?>
				<div class="pls-product-countdown-timer">
					<span class="pls-product-countdown-title"><?php echo apply_filters( 'pls_product_countdown_title', wp_kses_post( 'Hurry Up!<br/> Offer ends in:' ), 'modave' ); ?></span>
					<div class="pls-product-countdown" data-end-date="<?php echo esc_attr( $sale_time );?>" data-timezone="<?php echo esc_attr( $timezone );?>" data-countdown-style="countdown-box"></div>
				</div>
			<?php } elseif ( 'horizontal' == pls_get_loop_prop( 'product_view_mode' ) ) { ?>
				<div class="pls-product-countdown-timer">
					<div class="pls-product-countdown" data-end-date="<?php echo esc_attr( $sale_time );?>" data-timezone="<?php echo esc_attr( $timezone );?>" data-countdown-style="countdown-box"></div>
				</div>
			<?php } else { ?>
				<div class="pls-product-countdown-timer">
					<div class="pls-product-countdown" data-end-date="<?php echo esc_attr( $sale_time );?>" data-timezone="<?php echo esc_attr( $timezone );?>" data-countdown-style="countdown-text"></div>
				</div>
			<?php }
		}
		
		echo apply_filters( 'pls_wc_product_sale_countdown', $html, $sale_time, $timezone );
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_stock_progress_bar' ) ) :
	/**
	 * Single Product Stock Available & Sold Progress Bar
	 */
	function pls_wc_single_product_stock_progress_bar() {
		
		if( ! pls_get_option( 'single-product-stock-progress-bar', 1 ) ) {
			return; 
		} 
		
		global $product;
		$product_error 		= false;
		$productId 			= get_the_ID();	
		$stock_available 	= false;	
		$stock_sold 		= ( $total_sales = get_post_meta( $productId, 'total_sales', true ) ) ? round( $total_sales ) : 0;
		$stock_available 	= ( $stock = get_post_meta($productId, '_stock', true ) ) ? round( $stock ) : 0;
		$percentage 		= $stock_available > 0 ? round( $stock_sold / ( $stock_available + $stock_sold ) * 100 ) : 0;
		if( $stock_available ) : ?>
			<div class="pls-product-stock-progressbar">
				<div class="pls-ps-progressbar-title">
					<?php echo apply_filters( 'pls_product_stock_progressbar_title', esc_html__( 'Sold It:', 'modave' ) ); ?>
				</div>
				<div class="pls-ps-progressbar-content">
					<div class="pls-progress">
						<span class="pls-progress-bar active" style="<?php echo esc_attr( 'width:' . $percentage . '%' ); ?>"></span>
					</div>
					<div class="pls-product-stock-label">
						<span class="pls-stock-sold"><?php echo number_format_i18n( $percentage ); ?><?php esc_html_e( '% Sold', 'modave' );?></span> - 
						<span class="pls-stock-available">
							<?php printf( // WPCS: XSS OK.
								esc_html( _nx( 'Only %1$02s Item left in stock!', 'Only %1$02s Item(s) left in stock!', $stock_available, 'stock available', 'modave' ) ),
								number_format_i18n( $stock_available ),
							); ?>
						</span>
					</div>
				</div>
			</div>
		<?php endif;
	}
endif;

if( ! function_exists( 'pls_single_product_summary_separator' ) ) :
	/**
	 * Product Summary Separator
	 */
	function pls_single_product_summary_separator(){		
		
		?>
		<div class="pls-product-summary-separator"></div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_single_product_variation_availability' ) ):
	/**
	 * Single Product Variation Availability
	 */
	function pls_single_product_variation_availability() {
		global $product; 
		
		if( $product && $product->get_type() != 'variable' || ! pls_enable_variation_price_change() ){
			return;
		} ?>
		<div class="pls-product-availability"></div>
	<?php }
endif;

if ( ! function_exists( 'pls_get_products_availability' ) ) :
	/* Change In Stock Text */
	function pls_get_products_availability( $availability, $_product ) {
	  	
		if ( ! $_product->is_in_stock() ) {
			$availability['availability']	= pls_get_option( 'single-product-availability-outstock-msg', 'Sold Out' );
			$availability['class']			= 'out-of-stock';
		} elseif ( $_product->managing_stock() && $_product->is_on_backorder( 1 ) ) {
			$availability['availability']	= $_product->backorders_require_notification() ? esc_html__( 'Pre Order', 'modave' ) : '';
			$availability['class']			= 'onbackorder';
		} elseif ( ! $_product->managing_stock() && $_product->is_on_backorder( 1 ) ) {
			$availability['availability'] 	= esc_html__( 'Pre Order', 'modave' );
			$availability['class'] 			= 'onbackorder';
		} elseif ( $_product->managing_stock() ) {
			$stock_amount 	= $_product->get_stock_quantity();
			$stockQty		= pls_get_option( 'single-product-availability-lowstock-qty', 5 );
			if( $stock_amount <= $stockQty ){
				$stock_string 					= pls_get_option( 'single-product-availability-hurry-left-msg', 'Hurry, Only {qty} left.' );
				$stock_outputstring  			= str_replace('{qty}',$stock_amount,$stock_string); 
				$availability['availability'] 	= $stock_outputstring;
				$availability['class'] 			= 'min-stock';
				
			}else{
				$stock_string 					= pls_get_option( 'single-product-availability-instock-msg', 'In Stock' );
				$stock_outputstring  			= str_replace('{qty}',$stock_amount,$stock_string); 
				$availability['availability'] 	= $stock_outputstring;
				$availability['class'] 			= 'in-stock';
			}			
		} else {
			$stock_string 						= pls_get_option( 'single-product-availability-instock-msg', 'In Stock' );
			$stock_outputstring  				= str_replace('{qty}','',$stock_string); 
			$availability['availability'] 		= $stock_outputstring;
			$availability['class']				= 'in-stock';
		}
		return $availability;
	}
endif;
add_filter( 'woocommerce_get_availability', 'pls_get_products_availability', 1, 2 );

if( ! function_exists( 'pls_wc_single_product_availability' ) ) :
	/**
	 * Single Product Stock Availability
	 */
	function pls_wc_single_product_availability(){
		
		if( ! pls_get_option( 'single-product-availability', 0 ) ) {
			return;
		}
		global $product; ?>
		<span class="pls-product-stock"><?php
			echo wc_get_stock_html( $product ); // WPCS: XSS ok.
		?></span>
		<?php
	}
endif;

if( ! function_exists( 'pls_preorder_button_text' ) ) :
	function pls_preorder_button_text( $button_text, $product ) {
		if ( $product->is_on_backorder( ) ) {
			$button_text = esc_html__( 'Pre Order', 'modave' );
		}
		return $button_text;
	}
	add_filter( 'woocommerce_product_single_add_to_cart_text', 'pls_preorder_button_text', 10, 2 );
	add_filter( 'woocommerce_product_add_to_cart_text', 'pls_preorder_button_text', 10, 2 );
endif;

if( ! function_exists( 'pls_wc_single_product_brands' ) ) :
	/**
	 * Single Product Brands
	 */
	function pls_wc_single_product_brands(){		
		
		if( ! pls_get_option( 'single-product-brands', 0 ) ) {
			return;
		}
		
		$brands = get_the_terms( get_the_ID(), 'product_brand' );	
		if( is_wp_error( $brands ) || empty ( $brands ) ){
			return;
		}
		
		$brands_list = array();	
		
		foreach( $brands as $brand ): 
			$brand_link 	= get_term_link( $brand, 'product_brand' ); 
			$brand_class 	= 'pls-brand-title'; 
			$brands_list[] 	= '<a class="' . esc_attr( $brand_class ) . '" href="' . esc_url( get_term_link( $brand ) ) . '" title="' . esc_attr( $brand->name ) . '">' . esc_html( $brand->name ) . '</a>';
		endforeach;	?>
		<span class="pls-product-brands">
			<?php
				echo _n( 'Brand: ', 'Brands: ', count( $brands_list ), 'modave' );
				if( !empty( $brands_list ) ){
					echo implode(',', $brands_list);
				} ?>
		</span>
		<?php
	}
endif;

if( ! function_exists( 'pls_woocommerce_grouped_product_list_image' ) ) :
	/**
	 * Group Product added image in grouped product list
	 */
	function pls_woocommerce_grouped_product_list_image( $product ){
		$image = $product->get_image('thumbnail', array('class' => 'pls-grouped-product-img') ); 
		$thumbnail = '<div class="product-thumbnail">'.$image.'</div>';
		echo '<td class="woocommerce-grouped-product-list-item__thumbnail">'.$thumbnail.'</td>';
	}
	add_action( 'woocommerce_grouped_product_list_before_label', 'pls_woocommerce_grouped_product_list_image' );
endif;

if( ! function_exists( 'pls_wc_add_quick_buy_pid' ) ) :
	/* Quick buy button*/
	function pls_wc_add_quick_buy_pid() {
		
		if( ! pls_get_option( 'single-product-quick-buy', 1 ) ) {
			return;
		}
		
		global $product;
		if ( $product != null ) {
			echo '<input type="hidden" class="pls_quick_buy_product_' . esc_attr( $product->get_id() ). '" value="' . esc_attr( $product->get_id() ) . '"  />';
		}
	}
endif;

if( ! function_exists( 'pls_wc_add_quick_buy_button' ) ) :
	function pls_wc_add_quick_buy_button(){
		
		if( ! pls_get_option( 'single-product-quick-buy', 1 ) ) { 
			return;
		}
		
		global $product;
		$html = '';

		if ( $product == null ) {
			return;
		}
		if ( $product->get_type() == 'external' ) {
			return;
		}
		$pid 			= $product->get_id();
		$type 			= $product->get_type();
		$label 			= pls_get_option( 'product-quickbuy-button-text', 'Buy It Now' );
		$quick_buy_btn_style 	= 'button';
		$class 			= '';
		$defined_class 	= 'pls_quick_buy_' . $type . ' pls_quick_buy_' . $pid;
		$defined_attrs 	= 'name="pls_quick_buy_button"  data-product-type="' . esc_attr( $type ) . '" data-pls-product-id="' . esc_attr($pid ) . '"';
		echo '<div class="pls-quick-buy">';

		if ( $quick_buy_btn_style == 'button' ) {
			echo '<button  class="pls_quick_buy_button '.esc_attr( $defined_class ).'" value="' . esc_attr($label) . '" type="button" ' . $defined_attrs . '>' . esc_attr($label) . '</button>';
		}
		echo  '</div>';
	}
endif;

if( ! function_exists('pls_wc_quick_buy_redirect') ) :
	/**
	 * Function to redirect user after qucik buy button is submitted
	 */
	function pls_wc_quick_buy_redirect( $url ) {
		if ( isset( $_REQUEST['pls_quick_buy'] ) && $_REQUEST['pls_quick_buy'] == true ) {
			$redirect = 'checkout';
			if ( $redirect == 'cart' ) {
				return wc_get_cart_url();
			} elseif ( $redirect == 'checkout' ) {
				return wc_get_checkout_url();
			}
		}
		return $url;
	}
endif;

if( ! function_exists( 'pls_attribute_size_chart' ) ) :
	/**
	 * Single Product Size Chart
	 */
	function pls_attribute_size_chart( $attribute_name = '' ){
		
		if( wp_doing_ajax() ){
			return;
		}
		
		$size_chart_attribute = pls_get_option('size-chart-attribute','');
		if( empty( $size_chart_attribute ) ||  $size_chart_attribute != $attribute_name ){
			return;
		}
		if( ! pls_get_option( 'single-product-size-chart', 0 ) ) {
			return;
		}
		$prefix 	= PLS_PREFIX;
		$chart_id 	= get_post_meta(get_the_ID(),  $prefix.'size_guide', true );
		if( empty( trim($chart_id) ) ) {
			return;
		} ?>		
		<div class="product-sizechart">
			<a href="#" data-id="<?php echo esc_attr($chart_id);?>" class="pls-ajax-size-chart"><?php echo apply_filters( 'pls_single_product_sizechart_label', esc_html__('Size Guide', 'modave') );?></a>
		</div>
		<?php
	}
endif;

if( ! function_exists( 'pls_wc_single_product_size_chart' ) ) :
	/**
	 * Single Product Size Chart
	 */
	function pls_wc_single_product_size_chart(){
		
		if( ! pls_get_option( 'single-product-size-chart', 0 ) ) {
			return;
		}
		
		global $product;
		$size_chart_attribute = pls_get_option('size-chart-attribute','');
		if( !empty( $size_chart_attribute ) &&  $product->is_type( 'variable' ) ){
			return;
		}
		$prefix 	= PLS_PREFIX;
		$chart_id 	= get_post_meta(get_the_ID(),  $prefix.'size_guide', true );
		if( empty( trim($chart_id) ) ) {
			return;
		} ?>		
		<div class="product-sizechart">
			<a href="#" data-id="<?php echo esc_attr($chart_id);?>" class="pls-ajax-size-chart"><?php echo apply_filters( 'pls_single_product_sizechart_label', esc_html__('Size Guide', 'modave') );?></a>
		</div>
		<?php 
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_wishlist_button' ) ) :
	/**
	 * Product single wishlist button
	 */
	function pls_wc_single_product_wishlist_button() {
		
		if ( ! class_exists( 'WPCleverWoosw' ) ) {
			return; 
		}
		
		global $product;
		$product_id = $product->get_id(); ?>
		
		<div class="pls-whishlist-btn">
			<?php echo do_shortcode( '[woosw id="' . $product_id . '" type="link"]' ); ?>
		</div>		
	<?php }
endif;

if ( ! function_exists( 'pls_wc_single_product_compare_button' ) ) :
	/**
	 * Product single compare button
	 */
	function pls_wc_single_product_compare_button() {
		if ( ! class_exists( 'WPCleverWoosc' ) ) {
			return; 
		}
		global $product;
		$product_id = $product->get_id(); ?>
		
		<div class="pls-compare-btn">
			<?php echo do_shortcode( '[woosc id="' . $product_id . '" type="link"]' ); ?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_share' ) ) :
	/**
	 * Single Product Share
	 */
	function pls_wc_single_product_share() {
		
		if( ! pls_get_option( 'single-product-share', 1 ) ) { return; } ?>
		
		<?php if ( function_exists( 'pls_social_share' ) ) { 
			global $product; ?>
			<div class="pls-product-share">
				<span class="share-label"> <?php esc_html_e( 'Share', 'modave' ); ?> </span>
			</div>
			<div id="pls-product-share-popup" class="pls-product-share-popup mfp-hide">
				<h5 class="pls-share-popup-title"><?php esc_html_e( 'Share', 'modave' )?></h5>
				<?php pls_social_share(
					array(
						'type' 		=> 'share', 
						'style' 	=> 'icons-fill-colour',
						'size' 		=> 'icons-size-small',
						'el_class' 	=> 'pls-arrow'
					)
				); ?>
				<div class="pls-copy-link-wrap">
					<h6 class="pls-copy-link-title"><?php esc_html_e( 'Copy URL', 'modave' )?></h6>
					<form class="pls-product-share-form">
						<input id="pls-product-share-url" type="text" value="<?php echo esc_url( $product->get_permalink() ); ?>" readonly>
						<button class="button pls-copy-btn" data-copy="<?php esc_attr_e( 'Copy', 'modave' )?>" data-copied="<?php esc_attr_e( 'Copied', 'modave' )?>"> 
							<?php esc_html_e( 'Copy', 'modave' )?>
						</button>
					</form>
				</div>
			</div>
		<?php 
		}
	}
endif;

if( ! function_exists('pls_wc_single_product_view_full_details') ) :
	/**
	 * Single Product View Full Details
	 */
	function pls_wc_single_product_view_full_details() {
		global $product;
		?>
		<a class="pls-product-view-full-details" href="<?php echo esc_url( $product->get_permalink() ); ?>"><?php esc_html_e( 'View Full Details','modave' ); ?></a>
		<?php 
	}
endif;
	
if( ! function_exists('pls_wc_single_product_delivery_return') ) :
	/**
	 * Single Product Delivery Return
	 */
	function pls_wc_single_product_delivery_return() {
		
		if( ! pls_get_option( 'product-delivery-return', 0 ) ){
			return;
		}
		
		$class = '';
		$block_id = pls_get_option( 'delivery-return-terms', 0 ); 
		if( $block_id ){
			$class = ' pls-ajax-block';
		} ?>
		<div class="pls-deliver-return<?php echo esc_attr($class);?>" data-id="<?php echo esc_attr($block_id);?>">
			<?php echo esc_html( pls_get_option( 'delivery-return-label', 'Delivery & Return' ) ); ?>
		</div>
		<?php
	}
endif;

if( ! function_exists( 'pls_wc_single_product_ask_question' ) ) :
	/**
	 * Single Product Ask a Quesion
	 */
	function pls_wc_single_product_ask_question() {
		
		if( ! pls_get_option( 'product-ask-quetion', 0 ) ){
			return;
		}
		
		$class = '';
		$form_id = pls_get_option( 'ask-question-form', 0 );
		if( $form_id ){
			$class = ' pls-ask-questions-ajax';
		} ?>					
		<div class="pls-ask-questions<?php echo esc_attr( $class );?>" data-id="<?php echo esc_attr( $form_id );?>">
			<?php echo esc_html( pls_get_option( 'ask-quetion-label', 'Ask a Question' ) ); ?>
		</div>
		<?php
	}
endif;

if( ! function_exists('pls_wc_aks_question_popup_form') ) :
	/**
	 * Ask Questions Popup Form
	 */
	function pls_wc_aks_question_popup_form(){
		if( ! pls_get_option( 'product-ask-quetion', 0 ) ){
			return;
		}
		if ( function_exists('is_product') && !is_product()){
			return;
		}
		global $product;
		$product_title 	= $product->get_name(); 
		$form_id 		= pls_get_option( 'ask-question-form', 0 );
		?>
		<div id="pls-ask-questions-popup" class="pls-ask-questions-popup mfp-hide">
			<h3 class="ask-questions-form-tile"> 
				<?php echo esc_html( pls_get_option( 'ask-quetion-form-title', 'Ask a Question' ) ); ?>
			</h3>
			<?php 
			$form_shortcode = apply_filters('pls_ask_questions_form','[contact-form-7 id="'.$form_id.'" product-title="'.$product_title.'"]');
			echo do_shortcode( $form_shortcode ); 			
			?>
		</div>
		<?php
	}
endif;

if( ! function_exists('pls_shortcode_atts_wpcf7_filter') ) :
	/**
	 * Custom attribute add to form
	 */
	function pls_shortcode_atts_wpcf7_filter( $out, $pairs, $atts ) {
		$my_attr = 'product-title';
		if ( isset( $atts[$my_attr] ) ) {
			$out[$my_attr] = $atts[$my_attr];
		}
		return $out;
	}
	add_filter( 'shortcode_atts_wpcf7', 'pls_shortcode_atts_wpcf7_filter', 10, 3 );
endif;

if( ! function_exists('pls_wc_single_product_estimated_delivery') ) :
	/**
	 * Single Product Estimated Delivery Time
	 */
	function pls_wc_single_product_estimated_delivery() {
		
		if( ! pls_get_option( 'product-estimated-delivery', 0 ) ) { 
			return;			
		}
		
		global $product;
		
		if( !$product->is_in_stock() ){
			return;
		}
		
		if( $product->is_downloadable() ){
			return;
		}
		?>
		<div class="pls-estimated-delivery">
			<div class="pls-delivery-text">
				<?php echo wp_kses_post( pls_get_option( 'estimated-delivery-text', 'Estimated Delivery:&nbsp;&nbsp;<span>12-26 days</span> (International), <span>3-6 days</span> (United States)' ) ); ?>
			</div>
		</div>
		<?php
	}
endif;

if( ! function_exists('pls_wc_single_product_return_refund') ) :
	/**
	 * Single Product Return and Refund
	 */
	function pls_wc_single_product_return_refund() {
		
		if( ! pls_get_option( 'product-return-refund', 0 ) ) { 
			return;			
		} ?>
		<div class="pls-product-return">
			<div class="pls-product-return-text">
				<?php echo wp_kses_post( pls_get_option( 'product-return-refund-text', 'Return within <span>45 days</span> of purchase. Duties &amp; taxes are non-refundable.' ) ); ?>
			</div>
		</div>
		<?php
	}
endif;

if( ! function_exists('pls_wc_single_product_store_location') ) :
	/**
	 * Single Product Store Location
	 */
	function pls_wc_single_product_store_location() {
		
		if( ! pls_get_option( 'product-store-location', 0 ) ){
			return;
		}
		
		$class = '';
		$block_id = pls_get_option( 'store-locatation-block', 0 ); 
		if( ! empty( trim( $block_id ) ) ){
			$class = ' pls-ajax-block';
		} ?>
		<div class="pls-product-store-location<?php echo esc_attr($class);?>" data-id="<?php echo esc_attr($block_id); ?>">
			<div class="pls-product-store-location-text">
				<?php echo esc_html( pls_get_option( 'store-location-label', 'View Store Information' ) ); ?>
			</div>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_sold_count' ) ) :
	/**
	 * Single Product Sold Count
	 */
	function pls_wc_single_product_sold_count() {
		
		global $product;
		
		if( ! pls_get_option( 'product-sold-count', 0 ) || ! $product || ! $product->is_in_stock() ) { 
			return;
		}		
		
		$has_enable 			= false;		
		$selected_products 		= pls_get_option( 'product-sold-specific-products', array() );
		$selected_categories 	= pls_get_option( 'product-sold-specific-categories', array() );
		$product_id 			= $product->get_ID();
		
		if( empty( $selected_products ) && empty( $selected_categories ) ) {
			$has_enable = true;
		}	
		
		if( ! empty( $selected_products ) ) {
			if( in_array( $product_id, $selected_products ) ) {
				$has_enable = true;
			}
		}
		
		if( ! empty( $selected_categories ) ) {			
			$terms = get_the_terms( $product->get_ID(), 'product_cat' );
			if( ! is_wp_error( $terms ) && $terms ) {
				foreach( $terms as $term ) {
					if( in_array( $term->term_id, $selected_categories ) ) {
						$has_enable = true;
						break;
					}
				}
			}
		}
		
		$has_enable = apply_filters( 'pls_enabled_sold_count', $has_enable );
		
		if( ! $has_enable ){
			return;
		}		
		
		$key_name 			= 'pls_sold_count_' . $product_id;
		$cache_time 		= (int) pls_get_option( 'product-sold-count-cache', 3600 );
        $sold_count_html 	= get_transient( $key_name );
        //$sold_count_html 	= false;
		
		if ( ! $sold_count_html ) {
			$number			= pls_get_option( 'product-sold-count-number', array( 1 => 1, 2 => 20 ) );
			$min 			= isset( $number[1] ) ? (int) $number[1] : 1;
			$max 			= isset( $number[2] ) ? (int) $number[2] : 20;		
			$sold_count 	= rand( $min, $max );
			
			$time			= pls_get_option( 'product-sold-count-time', array( 1 => 1, 2 => 5 ) );
			$tmin 			= isset($time[1]) ? (int) $time[1] : 1;
			$tmax			= isset($time[2]) ? (int) $time[2] : 5;			
			$sold_time 		= rand( $tmin, $tmax );
			$time_text 		= $sold_time > 1 ? esc_html__( 'hours', 'modave' ) : esc_html__( 'hour', 'modave' );
			$message		= sprintf(
                esc_html__( '%s sold in last %s %s', 'modave' ),
                $sold_count,
                $sold_time,
				$time_text
            );
			$sold_count_html = '<div class="pls-product-sold-count">'. esc_html( $message ) .'</div>';
			set_transient( $key_name, $sold_count_html, $cache_time );
		}
		echo apply_filters( 'pls_product_sold_count_html', $sold_count_html );
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_visitor_count' ) ) :
	/**
	 * Single Product Visitor Count
	 */
	function pls_wc_single_product_visitor_count() {
		
		if( ! pls_get_option( 'single-product-visitor-count', 0 ) ) { 
			return;
		}
		
		$number			= pls_get_option( 'random-visitor-number', array( 1 => 20, 2 => 50, ) );
		$min 			= $number['1'];
		$max 			= $number['2'];
		$delay 			=  pls_get_option( 'visitor-count-delay-time', '5' );
		$visitor_count 	= rand( $min, $max );
		$enable_enterval = '';
		if( $delay  > 0 ){
			$enable_enterval 	= ' pls-visitor-change';
		}		
		$visitor_count_btml = '<span class="product-visitor-count">'.$visitor_count.'</span>';
		$count_message		= pls_get_option( 'visitor-count-text', '{visitor_count} People viewing this product right now!' );
		$count_message		= str_replace( '{visitor_count}', $visitor_count_btml, $count_message );
		
		
		$visitor_count_html = '<div class="pls-visitor-count'.$enable_enterval.'" data-min="'.$min.'" data-max="'.$max.'" data-delay="'.$delay.'">'. $count_message .'</div>';
		
		echo apply_filters('pls_product_visitor_count', $visitor_count_html );
	}
endif;

if ( ! function_exists( 'pls_wc_single_product_trust_badge' ) ) :
	/**
	 * Single Product Trust Badge
	 */
	function pls_wc_single_product_trust_badge() {
		
		if( ! pls_get_option( 'single-product-trust-badge', 0 ) ) { 
			return;
		}
		
		$trust_badge_url = pls_get_option( 'trust-badge-image', array( 'url' => PLS_IMAGES.'trust_badge.png' ) );
		
		if( empty( $trust_badge_url ) ) { 
			return;
		}
		
		ob_start(); ?>
		<div class="pls-product-trust-badge">
			<label><?php echo esc_html( pls_get_option( 'trust-badge-label', 'Guaranteed Safe Checkout:' ) ); ?></label>
			<img src="<?php echo esc_url( $trust_badge_url['url'] ); ?>" alt="<?php esc_attr_e( 'Trues Badge', 'modave'); ?>"/>
		</div>
		<?php 
		$badge_html = ob_get_clean();			
		echo apply_filters( 'pls_product_trust_badge', $badge_html );
	}
endif;

if ( ! function_exists( 'pls_single_product_histogram_reviews' ) ):
	/**
	 * Single Product Histogram Reviews
	 */
	function pls_single_product_histogram_reviews() {
		
		global $product;
		$average 		= $product->get_average_rating();
		$rating_count 	= $product->get_rating_count();
		$rating_counts 	= $product->get_rating_counts();
		$review_count 	= $product->get_review_count(); ?>
		
		<div class="pls-product-rating-histogram">
			<div class="pls-product-rating-avg-wrapper">
				<div class="pls-product-rating-avg"><?php echo round( $average, 1 );?></div>
				<?php echo wc_get_rating_html( $average, $rating_count ); // WPCS: XSS ok. ?>
				<div class="pls-product-rating-count">
					<span>(<?php echo sprintf( _n( '%s Rating', '%s Ratings', $review_count, 'modave' ), number_format_i18n( $review_count ) );?>)</span>
				</div>
			</div>
			<div class="pls-rating-histogram-wrapper">
				<div class="pls-rating-histogram">
					<?php for ( $r=5; $r > 0; $r-- ){ 
						$rating_class	= ( $r >= 3 ) ? 'good' : ( ( $r == 2 ) ? 'poor' : 'bad' );
						$rating_percentage = 0;
						if ( isset( $rating_counts[$r] ) && $review_count > 0 ) {
							$rating_percentage = ( round( $rating_counts[$r] / $review_count, 2 ) * 100 );
						} ?>																	
						<div class="pls-rating-bar">									
							<div class="pls-rating-star"><?php echo number_format_i18n( $r ); ?></div>
							<div class="pls-progress">
								<div class="pls-progress-bar <?php echo esc_attr( $rating_class ); ?>" style="width:<?php echo esc_attr( $rating_percentage ); ?>%"></div>
							</div>
							<?php if ( isset( $rating_counts[$r] ) ) : ?>
								<div class="pls-rating-count"><?php echo number_format_i18n( $rating_percentage ); ?>%</div>
							<?php else : ?>
								<div class="pls-rating-count zero"><?php echo number_format_i18n( 0 )?></div>
							<?php endif; ?>
						</div>
					<?php }?>							
				</div>
			</div>
		</div>
		<?php 
	}
endif;

if ( ! function_exists( 'pls_output_recently_viewed_products' ) ) :
	/**
	 * Single Product Share
	 */
	function pls_output_recently_viewed_products() {
		
		$recently_viewed_products = pls_get_recently_viewed_products();	
		
		if( ! empty( $recently_viewed_products ) ){
			$args['recently_viewed_products'] = $recently_viewed_products;
			// Set global loop values.
			wc_set_loop_prop( 'name', 'recently-viewed' );
			wc_get_template( 'single-product/recently-viewed.php', $args );
		}
	}
endif;

if( ! function_exists('pls_reduce_woocommerce_min_strength_requirement') ) :
	/** 
	 *Reduce the strength requirement on the woocommerce password.
	 * 
	 * Strength Settings
	 * 3 = Strong (default)
	 * 2 = Medium
	 * 1 = Weak
	 * 0 = Very Weak / Anything
	 */
	function pls_reduce_woocommerce_min_strength_requirement( $strength ) {
		if( pls_get_option( 'manage-password-strength', 0 ) )
			return pls_get_option( 'user-password-strength', 3 );
		else
			return 3;		 
	}
	add_filter( 'woocommerce_min_password_strength', 'pls_reduce_woocommerce_min_strength_requirement' );
endif;

/**
 * My Account Page
 */
if ( ! function_exists( 'pls_before_account_navigation' ) ) :
	/**
	 * Add wrap and user info to the account navigation
	 */
	function pls_before_account_navigation() {

		// Name to display
		$current_user = wp_get_current_user();

		if ( $current_user->display_name ) {
			$name = $current_user->display_name;
		} else {
			$name = esc_html__( 'Welcome!', 'modave' );
		}
		$name = apply_filters( 'pls_user_profile_name_text', $name );

		echo '<div class="MyAccount-navigation-wrapper">';
			echo '<div class="pls-user-profile">';
				echo '<div class="user-avatar">'. get_avatar( $current_user->user_email, 128 ) .'</div>';
				echo '<div class="user-info">';
					echo '<h5 class="display-name">'. esc_attr( $name ) .'</h5>';
				echo '</div>';
			echo '</div>';
	}
endif;

if ( ! function_exists( 'pls_after_account_navigation' ) ) :
	/**
	 * Add wrap to the account navigation.
	 */
	function pls_after_account_navigation() {
		echo '</div>';
	}
endif;

if ( ! function_exists( 'pls_wc_before_account_orders' ) ) :
	/**
	 *  My Orders Page Title
	 */
	function pls_wc_before_account_orders( $has_orders) {
		?>
		<div class="section-title">
			<h2><?php esc_html_e( 'My Orders', 'modave' ); ?></h2>
			<p><?php esc_html_e( 'Your recent orders are displayed in the table below.', 'modave' ); ?></p>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_before_account_downloads' ) ) :
	/**
	 *  My Downloads Page Title
	 */
	function pls_wc_before_account_downloads( $has_orders) {
		?>
		<div class="section-title">
			<h2><?php esc_html_e( 'My Downloads', 'modave' ); ?></h2>
			<p><?php esc_html_e( 'Your digital downloads are displayed in the table below.', 'modave' ); ?></p>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_my_account_my_address_description' ) ):
	/**
	 *  My Address Page Title
	 */
	function pls_wc_my_account_my_address_description( $address_desc ) {
		
		$address_title = '<div class="section-title">';
		$address_title .= '<h2>'.esc_html__('Address','modave').'</h2>';
		$address_title .= '<p>' . $address_desc . '</p>';
		$address_title .= '</div>';
		return $address_title;
	}
endif;


if ( ! function_exists( 'pls_wc_myaccount_edit_account_heading' ) ) :
	/**
	 * Edit Account Heading
	 */
	function pls_wc_myaccount_edit_account_heading() {
		?>
		<div class="section-title">
			<h2><?php esc_html_e( 'My account', 'modave' ) ?></h2>
			<p><?php esc_html_e( 'Edit your account details or change your password', 'modave' ); ?></p>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_free_shipping_bar' ) ) :
	/**
	 * Free Shipping Progress Bar
	 */
	function pls_wc_free_shipping_bar() {
		
		if( ! pls_get_option( 'free-shipping-bar', 0 ) ) {
			return;
		}
		
		if( empty( pls_get_option( 'free-shipping-amount', '200' ) ) ) {
			return;
		}
		
		$subtotal 		= WC()->cart->subtotal;
		$minimum_amount = pls_get_option( 'free-shipping-amount', 200 );
		if( $subtotal < $minimum_amount ){
			$remaining 	= $minimum_amount - $subtotal;
			$percentage = round( ( $subtotal / $minimum_amount ) * 100 ) ;
			$missing_amount = wc_price( $remaining );
			$free_shipping_text = pls_get_option( 'free-shipping-msg', 'Buy {missing_amount} more get <span>Free Shipping</span>' );
			$free_shipping_text = str_replace( '{missing_amount}', $missing_amount, $free_shipping_text );
			$class = 'active';
			
		}else{
			$free_shipping_text = pls_get_option( "free-shipping-complete-msg", "Congratulations! You've got free shipping!" );
			$percentage 		= 100;
			$class 				= 'completed';
		} ?>
		<div class="pls-freeshipping-bar <?php echo esc_attr($class); ?>">
			<div class="freeshipping-bar-msg"><?php echo wp_kses_post( $free_shipping_text ); ?></div>
			<div class="freeshipping-bar">				
				<span class="pls-progress-bar active" style="width:<?php echo esc_attr( $percentage ); ?>%"></span>
			</div>
		</div>
	<?php }
endif;

if ( ! function_exists( 'pls_wc_mincart_totals' ) ) :
	/** 
	 * Mini Cart Totals
	 */
	function pls_wc_mincart_totals() {		
		wc_get_template( 'cart/mini-cart-totals.php' );
	}
endif; 
 

if ( ! function_exists( 'pls_wc_mincart_footer_items_enale' ) ) :
	
	function pls_wc_mincart_footer_items_enale(){
		
		if( pls_get_option( 'mini-cart-note', 0 ) || pls_get_option( 'mini-cart-shipping', 0 ) || pls_get_option( 'mini-cart-coupon', 0 )){
			return true;
		}
		return false;
	}

endif;

if ( ! function_exists( 'pls_wc_mincart_footer_items' ) ) :
	/** 
	 * Mini Cart Items
	 */
	function pls_wc_mincart_footer_items() {
		
		if( !pls_wc_mincart_footer_items_enale() ){
			return;
		} ?>
		<div class="pls-minicart-action-btns">			
			 <?php 
			 if (  pls_get_option( 'mini-cart-note', 0 ) && apply_filters( 'woocommerce_enable_order_notes_field', 'yes' === get_option( 'woocommerce_enable_order_comments', 'yes' ) ) ) { ?>
                <a href="#" class="pls-minicart-action-btn pls-minicart-note-btn" data-action_name="note">
					<span><?php esc_html_e( 'Note', 'modave' ); ?></span>
                </a>
			 <?php } ?>
			 <?php if ( pls_get_option( 'mini-cart-shipping', 0 ) &&  'yes' === get_option( 'woocommerce_enable_shipping_calc' ) && WC()->cart->needs_shipping() && WC()->cart->show_shipping() ) { ?>
                <a href="#" class="pls-minicart-action-btn pls-minicart-shipping-btn" data-action_name="shipping">
					<span><?php esc_html_e( 'Shipping', 'modave' ); ?></span>
                </a>
			 <?php } ?>
			  <?php if (  pls_get_option( 'mini-cart-coupon', 0 ) && wc_coupons_enabled() ) { ?>
                <a href="#" class="pls-minicart-action-btn pls-minicart-coupon-btn" data-action_name="coupon">
					<span><?php esc_html_e( 'Coupon', 'modave' ); ?></span>
                </a>
			 <?php } ?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'pls_wc_mincart_blocks' ) ) :
	/** 
	 * Mini Cart Blocks
	 */
	function pls_wc_mincart_blocks() {	
		if (  pls_get_option( 'mini-cart-note', 0 ) ) {
			wc_get_template( 'cart/minicart-block-note.php' );
		}
		if ( pls_get_option( 'mini-cart-shipping', 0 ) ) { 
			wc_get_template( 'cart/minicart-block-shipping.php' );
		}
		if (  pls_get_option( 'mini-cart-coupon', 0 ) ) { 
			wc_get_template( 'cart/minicart-block-coupan.php' );
		}
	}
endif;

if ( ! function_exists( 'pls_wc_cart_page_wrapper' ) ) :
	/**
	 * Cart Page Wrapper Start
	 */
	function pls_wc_cart_page_wrapper() { ?>
		<div class="woocommerce-cart-wrapper">
	<?php }
endif;

if ( ! function_exists( 'pls_wc_cart_page_wrapper_end' ) ) :
	/**
	 * Cart Page Wrapper End
	 */
	function pls_wc_cart_page_wrapper_end() { ?>
		</div>
	<?php }
endif;

if ( ! function_exists( 'pls_wc_sticky_add_to_cart_button' ) ) :
	/**
	 * Single Product Sticky Add To Cart Button
	 */
	function pls_wc_sticky_add_to_cart_button(){
		
		global $product;		
		$stick_add_to_cart = pls_get_option( 'sticky-add-to-cart-button', 0 );
		
		if ( !$product || ! $stick_add_to_cart || ! is_singular( 'product' ) || $product->is_type('grouped') || $product->is_type('woosb') || !$product->is_in_stock() ) {
			return;
		} ?>
		
		<div class="pls-sticky-add-to-cart">
			<div class="container">
				<div class="row">
					<div class="col pls-sticky-add-to-cart-left">
						<div class="pls-sticky-product-image">
							<?php echo woocommerce_get_product_thumbnail( 'woocommerce_gallery_thumbnail'); ?>
						</div>
						<div class="pls-sticky-product-info">
							<div class="pls-sticky-product-title"><?php the_title(); ?></div>
							<?php if( wc_review_ratings_enabled() ) {
								echo wc_get_rating_html( $product->get_average_rating() );
							} ?>
							<span class="price"><?php echo wp_kses_post( $product->get_price_html() ); ?></span>
						</div>
					</div>
					<div class="col-auto pls-sticky-add-to-cart-right">						
						<a href="<?php echo esc_url( '#' ); ?>" class="button <?php echo esc_attr( $product->get_type() );?>" rel="nofollow">
							<?php echo esc_attr( $product->add_to_cart_text() ); ?>
						</a>
					</div>
				</div>
			</div>
		</div>
	<?php }
endif;

function pls_single_products_tabs(){
	global $product;
	if ( ! $product || ! is_singular( 'product' ) ) {
			return;
	}
	$upsleess_ids  				= $product->get_upsell_ids();
	$posts_per_page 			= 10;
	$related_products 			= array_filter( array_map( 'wc_get_product', wc_get_related_products( $product->get_id(), $posts_per_page, $upsleess_ids ) ), 'wc_products_array_filter_visible' );
	$recently_viewed_products	= pls_get_recently_viewed_products();	
	$tabs 						= [];
	$enable_upsell				= pls_get_option( 'upsells-products', 1 );
	$enable_related				= pls_get_option( 'related-products', 1 );
	$enable_recently_view		= pls_get_option( 'recently-viewed-products', 0 );
	if( ! empty( $upsleess_ids ) && $enable_upsell ){
		$tabs['upsells-tab'] = apply_filters( 'pls_product_upsells_tab_title', esc_html__( 'Upselles Products', 'modave' ) );
	}
	if( ! empty( $related_products ) && $enable_related  ){
		$tabs['related-tab'] = apply_filters( 'pls_product_related_tab_title', esc_html__( 'Related Products', 'modave' ) );
	}			
	if( ! empty( $recently_viewed_products ) && $enable_recently_view  ){
		$tabs['recently-tab'] = apply_filters( 'pls_product_recently_tab_title', esc_html__( 'Recently Viewed', 'modave' ) );
	}
	
	if( empty( $tabs ) ){
		return;
	} ?>
	<div class="pls-single-products-tabs pls-tabs">
		<ul class="nav nav-tabs">
			<?php
			$i 			= 0;
			$active_tab = 1;
			foreach( $tabs as $key => $value ){ 
				$i++;
				$tab_nav_class = array( 'nav-link' );
				if ( $i == $active_tab ){
					$tab_nav_class[] = 'active';
					$tab_nav_class[] = 'loaded';
				}
				
				$expanded 	= ( $i == $active_tab ) ? 'true' : 'false';	
				$tab_nav_class = array( 'nav-link' );
				if ( $i == $active_tab ){
					$tab_nav_class[] = 'active';
					$tab_nav_class[] = 'loaded';
					
				}
				$nav_class = implode( ' ', array_filter( $tab_nav_class ) );
				$tab_id = 'pls-tab-'.$i; ?>
				<li class="nav-item" data-tab-index="<?php echo esc_attr( $i ); ?>">
					<a id="nav-<?php echo esc_attr( $tab_id ); ?>" 
					class="<?php echo esc_attr( $nav_class ); ?>" 
					href="#<?php echo esc_attr( $tab_id ); ?>" 
					data-href="<?php echo esc_attr( $tab_id ); ?>" 
					data-toggle="tab" role="tab" aria-controls="<?php echo esc_attr( $tab_id ); ?>" 
					aria-selected="<?php echo esc_attr( $expanded ); ?>">
					<?php echo esc_html( $value ); ?></a>
				</li>
			<?php } ?>
		</ul>		
		<div class="tab-content">
			<?php 
			$i 			= 0;
			$active_tab = 1;
			foreach( $tabs as $key => $value ){
				$i++; 
				$tab_content_class = ($i == $active_tab ) ? 'tab-pane fade show active' : 'tab-pane fade'; 
				$tab_id = 'pls-tab-'.$i; ?>			
				<div class="<?php echo esc_attr( $tab_content_class ); ?>"
					 id="<?php echo esc_attr( $tab_id ); ?>"
					role="panel" aria-labelledby="nav-<?php echo esc_attr($tab_id );?>">
					<?php 
					if( $key == 'upsells-tab' ){
						woocommerce_upsell_display();
					}
					if( $key == 'related-tab' ){
						woocommerce_related_products( [ 'posts_per_page' => 10 ] );
					}
					if( $key == 'recently-tab' ){
						pls_output_recently_viewed_products();
					} ?>
				</div>
			<?php } ?>
		</div>
	</div>
	<?php
}