<?php
/**
 * WP_User_Channel_Admin class
 *
 * Handles the admin settings for the WP User Channel plugin.
 *
 * @link              https://1.envato.market/mgXE4y
 * @since             1.0.0
 * @package           WP_User_Channel
 */

// Prevent direct access.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class WP_User_Channel_Admin {

	/**
	 * Saves the settings for the WP User Channel.
	 *
	 * This function checks user permissions, verifies the nonce, and updates user capabilities based on the selected roles.
	 */
	public static function save_settings() {
		if ( ! current_user_can( 'administrator' ) || ! isset( $_POST['update_wp_user_channel'] ) ) {
			return;
		}

		if ( wp_verify_nonce( $_POST['update_wp_user_channel'], 'update_wp_user_channel_security' ) ) {
			$data = wp_parse_args( $_POST['wp_user_channel'], array(
				'default_roles' => array(),
			) );

			// Update the option with the new settings.
			update_option( 'wp_user_channel', wp_unslash( $data ) );

			add_action( 'admin_notices', function(){
				?>
					<div class="notice notice-success is-dismissible">
					<p><?php esc_html_e( 'Settings saved.', 'wp-user-channel' ); ?></p>
					</div>				
				<?php
			} );
		}
	}

	/**
	 * Retrieves the current settings for the WP User Channel.
	 *
	 * @return array The settings for the WP User Channel.
	 */
	public static function get_settings() {
		return wp_parse_args( get_option( 'wp_user_channel' ), array(
			'max_channels' 	=>	5,
			'default_roles' => 	array()
		) );
	}

	/**
	 * Adds the settings page under the Settings menu.
	 */
	public static function setting_page() {
		add_submenu_page(
			'options-general.php',
			esc_html__( 'WP User Channel', 'wp-user-channel' ),
			esc_html__( 'WP User Channel', 'wp-user-channel' ),
			'administrator',
			'wp-user-channel',
			array( __CLASS__, 'load_setting_page' )
		);
	}

	/**
	 * Loads the settings page template.
	 */
	public static function load_setting_page() {
		load_template( WP_USER_CHANNEL_PATH . 'admin/partials/settings.php' );
	}

    /**
     *
     * Filter user table
     * 
     * @param  array $columns
     * @return array new $columns
     * 
     */
	public static function table_user_column( $columns ){
        return array_merge( $columns, array(
            'manage_channels'   =>  esc_html__( 'Create Channels', 'streamtube-core' )
        ) );
	}

    /**
     *
     * Filter user table
     * 
     * @param string $output
     * @param string $column_name
     * @param innt $user_id
     *
     * 
     */
    public static function table_user_column_content( $output, $column_name, $user_id ){
    	switch ( $column_name ) {
    		case 'manage_channels':

    		$is_allowed = user_can( $user_id, 'add_channels' );

    			$output = sprintf(
    				'<button type="button" data-nonce="%1$s" data-user-id="%2$s" class="button-assign-channel-cap button button-%3$s button-small btn btn-%3$s btn-sm">%4$s</button>',
    				esc_attr( wp_create_nonce( 'assign_channel_cap_' . $user_id ) ),
    				esc_attr( $user_id ),
    				$is_allowed ? 'primary' : 'secondary',
    				$is_allowed ? esc_html__( 'Allowed', 'wp-user-channel' ) : esc_html__( 'N/A', 'wp-user-channel' )
    			);
    			
    		break;
    	}

    	return $output;
    }

    public static function assign_channel_cap( $user_id = 0 ){
    	$user = get_userdata( $user_id );

    	if( $user instanceof WP_User ){
    		if( $user->has_cap( 'add_channels' ) ){
    			$user->remove_cap( 'add_channels' );	
    		}else{
    			$user->add_cap( 'add_channels' );
    		}
    	}

    	return self::table_user_column_content( null, 'manage_channels', $user_id );
    }

    public static function assign_role_cap( $role = '' ){

    	$role = get_role( $role );

    	if( $role ){
    		if( $role->has_cap( 'add_channels' ) ){
    			return $role->remove_cap( 'add_channels' );
    		}else{
    			return $role->add_cap( 'add_channels' );
    		}
    	}
    	return false;
    }

    public static function ajax_assign_channel_cap(){

    	$user_id = isset( $_POST['user_id'] ) ? absint( $_POST['user_id'] ) : 0;

    	if( ! current_user_can( 'administrator' ) || ! $user_id ){
    		exit;
    	}

    	check_ajax_referer( 'assign_channel_cap_' . $user_id, 'nonce' );

    	$response = self::assign_channel_cap( $user_id );

    	if( is_wp_error( $response ) ){
    		wp_send_json_error( $response );
    	}

    	wp_send_json_success( $response );
    }

    public static function ajax_assign_role_cap(){

    	$role = isset( $_POST['role'] ) ? wp_unslash( $_POST['role'] ) : '';

    	if( ! current_user_can( 'administrator' ) || ! $role ){
    		exit;
    	}

    	check_ajax_referer( 'assign_role_cap_' . $role, 'nonce' );

    	$response = self::assign_role_cap( $role );

    	if( is_wp_error( $response ) ){
    		wp_send_json_error( $response );
    	}

    	wp_send_json_success( $response );
    }        

    /**
     * Load admin scripts
     */
    public static function enqueue_scripts(){
    	wp_enqueue_script(
    		WP_USER_CHANNEL, 
    		WP_USER_CHANNEL_URL . 'admin/assets/scripts.js',
    		array('jquery'), 
    		filemtime( WP_USER_CHANNEL_PATH . 'admin/assets/scripts.js' ), 
    		true
    	);

    	wp_enqueue_style( 
    		WP_USER_CHANNEL, 
    		WP_USER_CHANNEL_URL . 'admin/assets/style.css', 
    		array(), 
    		filemtime( WP_USER_CHANNEL_PATH . 'admin/assets/style.css' )
    	);
    }

    public static function action_links( $actions ){
		$links = array();

		$links[] = sprintf(
			'<a href="%s">%s</a>',
			esc_url( admin_url( 'options-general.php?page=wp-user-channel' ) ),
			esc_html__( 'Settings', 'wp-user-channel' )
		);

		return array_merge( $actions, $links );	    	
    }
}
