<?php
/**
 * Google OAuth
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

class StreamTube_Core_Google_OAuth{

    const OAUTH_ID      = 'google';

    /**
     * Holds the Google client ID
     * @var string
     */
    public $client_id;
    
    /**
     * Holds the Google client secret key
     * @var string
     */
    public $client_secret;
    
    /**
     * Holds the Redirect Uri
     * @var string
     */
    public $redirect_uri;

    /**
     *
     * Holds scopes
     * 
     * @var array
     */
    protected $scopes   = array();

    /**
     *
     * Holds the callback state
     * 
     * @var array
     */
    protected $state    = array();

    /**
     *
     * Holds the service
     * 
     * @var string
     */
    protected $service  = 'google';

    /**
     *
     * Define the OAuth token url
     * 
     */
    const URL_OAUTH_TOKEN = 'https://accounts.google.com/o/oauth2/token';

    function __construct( $args = array() ){

        $args = wp_parse_args( $args, array(
            'client_id'     =>  '',
            'client_secret' =>  '',
            'redirect_uri'  =>  add_query_arg( array(
                'oauth' =>  'google'
            ), home_url('/') ),
            'scopes'        =>  array(),
            'state'         =>  array(),
            'service'       =>  'google'// e.g youtube, google (for one-click login)
        ) );

        $this->client_id        = $args['client_id'];

        $this->client_secret    = $args['client_secret'];

        $this->redirect_uri     = $args['redirect_uri'];

        if( $args['scopes'] ){
            $this->scopes = $args['scopes'];
        }

        $this->state    = $args['state'];

        if( $args['service'] ){
            $this->service = $args['service'];
        }
    }

    /**
     *
     * Set scopes
     * 
     * @param array $scopes
     */
    public function set_scopes( $scopes = array() ){
        $this->scopes = $scopes;
    }

    /**
     *
     * Set service
     * 
     * @param string $service
     */
    public function set_service( $service = '' ){
        if( $service ){
            $this->service = $service;
        }
    }

    /**
     *
     * Call API
     * 
     * @param  string $url
     * @param  array  $args
     * @return array|WP_Errro
     */
    public function _call_api( $url, $args = array() ){
        $response = wp_remote_request( $url, $args );

        if( is_wp_error( $response ) ){
            return $response;
        }

        return json_decode( wp_remote_retrieve_body( $response ), true );
    }       

    public function _can_auth(){
        if( $this->client_id && $this->client_secret && $this->redirect_uri ){
            return true;
        }
        return false;
    }

    public function _is_valid_token( $token = array() ){
        if( is_array( $token ) && array_key_exists( 'access_token', $token ) ){
            return true;
        }
        return false;
    }    

    /**
     *
     * Get the OAuth URL
     * 
     * @return string
     */
    public function _create_oauth_url( $state = array() ){

        if( $state ){
            $this->state = $state;
        }

        if( is_array( $this->state ) ){
            $this->state = base64_encode( json_encode( $this->state ) );
        }

        $url    =   add_query_arg( array(
            'client_id'         =>  $this->client_id,
            'redirect_uri'      =>  $this->redirect_uri,
            'scope'             =>  implode( "+", $this->scopes ),
            'response_type'     =>  'code',
            'approval_prompt'   =>  'force',
            'access_type'       =>  'offline',
            'state'             =>  $this->state
        ), 'https://accounts.google.com/o/oauth2/auth' );
        
        return $url;
    }

    /**
     *
     * Check token expiration
     * 
     * @param  integer $user_id
     */
    public function _is_user_token_expired( $user_id = 0 ){
        return get_transient( 'google_token_' . $this->service . '_' . $user_id ) ? false : true;
    }

    /**
     *
     * Set token transient based on `expires_in` value
     * 
     * @param integer $user_id
     * @param array   $token
     * 
     */
    public function _set_user_token_transient( $user_id = 0, $token = array() ){
        return set_transient( 'google_token_' . $this->service . '_' . $user_id, $token, absint( $token['expires_in'] ) );
    }

    /**
     *
     * Update token into user metadata
     * 
     * @param  integer $user_id
     * @param array $token;
     * 
     */
    public function _update_user_token( $user_id = 0, $token = array() ){
        if( ! $token ){
            return false;
        }

        return update_user_meta( $user_id, 'google_token_' . $this->service, $token );
    }

    /**
     *
     * Get user token
     * Refresh token and reupdate token if expired
     * 
     * @param  integer $user_id
     * @return array|WP_Error|false
     */
    public function _get_user_token( $user_id = 0 ){

        $token = get_user_meta( $user_id, 'google_token_' . $this->service, true );

        if( ! $token || ! $this->_is_valid_token( $token ) ){
            return false;
        }

        $refresh_token = $token['refresh_token'];

        if( $this->_is_user_token_expired( $user_id ) ){
            // Expired
            $token = $this->_refresh_token( $token );

            if( is_wp_error( $token ) ){
                return $token;
            }

            if( ! array_key_exists( 'refresh_token' , $token ) ){
                $token = array_merge( $token, compact( 'refresh_token' ) );
            }

            if( $this->_is_valid_token( $token ) ){
                $this->_update_user_token( $user_id, $token );
                $this->_set_user_token_transient( $user_id, $token );
            }   
        }

        return $token;
    }

    /**
     *
     * Delete user and transient token
     * 
     * @param  integer $user_id
     * 
     */
    public function _revoke_user_token( $user_id = 0 ){
        delete_user_meta( $user_id, 'google_token_' . $this->service );
        delete_transient( 'google_token_' . $this->service . '_' . $user_id );
    }

    /**
     *
     * Call API to get access token
     * 
     * @param  string $code
     * @return array|WP_Error
     */
    public function _request_token( $code = '' ) {
        return $this->_call_api( self::URL_OAUTH_TOKEN, array(
            'method'    =>  'POST',
            'body'      =>  array(
                'code'              =>  $code,
                'client_id'         =>  $this->client_id,
                'client_secret'     =>  $this->client_secret,
                'redirect_uri'      =>  $this->redirect_uri,
                'grant_type'        =>  'authorization_code'        
            )
        ) );
    }

    /**
     *
     * Refresh token
     * 
     * @param  string|array $token
     * @return array|WP_Error
     */
    public function _refresh_token( $token = '' ){

        $refresh_token = '';

        if( is_array( $token ) && array_key_exists( 'refresh_token', $token ) ){
            $refresh_token = $token['refresh_token'];
        }

        if( is_string( $token ) ){
            $refresh_token = $token;
        }

        return $this->_call_api( self::URL_OAUTH_TOKEN, array(
            'method'    =>  'POST',
            'body'      =>  array(
                'client_id'         =>  $this->client_id,
                'client_secret'     =>  $this->client_secret,
                'refresh_token'     =>  $refresh_token,
                'grant_type'        =>  'refresh_token'    
            )
        ) );
    }

    /**
     * Create revoke token to delete all transient and user token
     */
    public function _revoke_token_url(){

        return add_query_arg(
            array(
                'oauth'     =>  self::OAUTH_ID
            ),
            wp_nonce_url( $_SERVER['REQUEST_URI'], 'revoke_token_' . get_current_user_id(), 'revoke_token' )
        );
    }
}