<?php

/**
 * Define the Role functionality
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

class StreamTube_Core_Role{

    /**
     *
     * Check if given user can promote user
     * 
     * @param  integer $user_id
     *
     * @return boolean
     */
    public static function can_promote_users( $user_id = 0 ){
        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        return user_can( $user_id, 'promote_users' );
    }

    /**
     *
     * Check if capability can be deleted
     *
     * @return false if it's admin capability
     * 
     */
    public static function can_delete_capability( $capability = '' ){

        if( ! self::can_promote_users() || self::is_admin_capability( $capability ) ){
            return false;
        }

        return true;
    }

    /**
     * Check if a capability can be ungranted for a given role.
     *
     * @param string $role       The role to check.
     * @param string $capability The capability to check.
     *
     * @return bool True if the capability can be ungranted, false otherwise.
     */
    public static function can_ungrant_capability( $role = '', $capability = '' ) {

        // Check if the current user has the ability to promote users.
        if ( ! self::can_promote_users() ) {
            return false;
        }

        // Prevent ungranting admin capabilities from admin roles.
        if ( self::is_admin_role( $role ) && self::is_admin_capability( $capability ) ) {
            return false;
        }

        // If no restrictions apply, allow ungranting the capability.
        return true;
    }

    /**
     *
     * Check if given role is admin role
     * 
     */
    public static function is_admin_role( $role = '' ){

        $role_id = '';

        if( is_object( $role ) ){
            $role_id = $role->name;
        }else{
            $role_id = $role;
        }

        return in_array( 
            $role_id , 
            apply_filters( 
                'streamtube/core/admin_roles', 
                array( 'super-admin', 'administrator' )
            ) 
        ) ? true : false;
    }

    /**
     *
     * Get default admin capabilities
     * 
     * @return array
     */
    public static function get_admin_capabilities(){
        return array(
            'create_sites',
            'delete_sites',
            'manage_network',
            'manage_sites',
            'manage_network_users',
            'manage_network_plugins',
            'manage_network_themes',
            'manage_network_options',
            'upgrade_network',
            'setup_network',            
            'activate_plugins',
            'edit_plugins',
            'update_plugins',
            'delete_plugins',
            'install_plugins',
            'resume_plugins',
            'edit_themes',
            'update_themes',
            'install_themes',
            'delete_themes',
            'resume_themes',
            'edit_users',
            'create_users',
            'delete_users',
            'edit_dashboard',
            'edit_theme_options',
            'export',
            'import',
            'list_users',
            'manage_categories',
            'manage_links',
            'manage_options',
            'moderate_comments',
            'promote_users',
            'read',
            'remove_users',
            'switch_themes',
            'upload_files',
            'customize',
            'delete_site'
        );
    }

    /**
     *
     * Check if given capability is the default cap of admin
     * 
     */
    public static function is_admin_capability( $capability = '' ){
        return in_array( $capability , self::get_admin_capabilities() );
    }

    /**
     *
     * Get all available capabilities
     * 
     * @return array
     */
    public static function get_all_capabilities( $role = '' ){
        $capabilities   = array();

        foreach ( get_editable_roles() as $role_id => $value ) {

            if( $role && $role === $role_id ){
                $capabilities = array_merge( $capabilities, array_keys( $value['capabilities'] ) );
            }
            else{
                $capabilities = array_merge( $capabilities, array_keys( $value['capabilities'] ) );
            }
            
        }

        $capabilities = array_unique( array_filter( array_values( $capabilities ) ) );

        return $capabilities;
    }    

    /**
     *
     * Delete given capability
     *
     * @param string $capability
     *
     * @return WP_Error|true
     * 
     */
    public function delete_capability( $capability = '' ){
        
        if( ! $capability ){
            return new WP_Error(
                'empty_capability',
                esc_html__( 'Capability is empty.', 'streamtube-core' )
            );            
        }

        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to add role.', 'streamtube-core' )
            );          
        }

        global $wp_roles;

        foreach( array_keys( $wp_roles->roles ) as $role ) {
            $wp_roles->remove_cap( $role, $capability );

            /**
             *
             * Fires after capability removed
             *
             * @param WP_Role $role
             * @param string $capability
             * 
             */
            do_action( 'streamtube/core/removed_role_capability', $role, $capability );             
        }

        return true;
    }    

    /**
     *
     * Add role
     * 
     * @param string $role
     * @param array  $capabilities
     */
    public function add_role( $role = '', $capabilities = array() ){

        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to add role.', 'streamtube-core' )
            );          
        }        

        if( ! $capabilities || ( is_array( $capabilities ) && ! in_array( 'read' , $capabilities ) ) ){
            $capabilities = array( 'read' );
        }

        if( ! $role ){
            return new WP_Error(
                'empty_role',
                esc_html__( 'Role is empty.', 'streamtube-core' ),
            );      
        }

        if( get_role( $role ) ){
            return new WP_Error(
                'empty_exists',
                esc_html__( 'Role already exists.', 'streamtube-core' ),
            );             
        }    

        $results = add_role( sanitize_title( $role ), $role, $capabilities );

        /**
         *
         * Fires after capability removed
         *
         * @param WP_Role $_role
         * @param string $capability
         * 
         */
        do_action( 'streamtube/core/added_role', $results, $role, $capabilities );            

        return $results;
    }

    /**
     *
     * Delete role
     * 
     * @param  string $role
     *
     * @return WP_Error|remove_role()
     * 
     */
    public function delete_role( $role = '' ){

        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to delete this role.', 'streamtube-core' )
            );          
        }

        if( self::is_admin_role( $role ) ){
            return new WP_Error(
                'cannot_delete_admin',
                esc_html__( 'You cannot delete Administrator and Editor roles.', 'streamtube-core' )
            );             
        }

        return remove_role( $role );
    }      

    /**
     *
     * Add capability to given role
     * 
     * @param  string $role
     * @param  string $capability
     */
    public function add_role_capability( $role = '', $capability = '' ){

        $capability = trim( sanitize_title( $capability ) );

        if( ! $role || ! is_string( $role ) || ! $capability || ! is_string( $capability ) ){
            return new WP_Error(
                'invalid_requested',
                esc_html__( 'Invalid Requested', 'streamtube-core' )
            );
        }        

        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to add capability.', 'streamtube-core' )
            );          
        }

        $role = get_role( $role );

        if( ! $role ){
            return new WP_Error(
                'role_not_exist',
                esc_html__( 'Role does not exist.', 'streamtube-core' )
            );
        }

        if( $role->has_cap( $capability ) ){
            return new WP_Error(
                'capability_already_granted',
                esc_html__( 'The capability has already been granted.', 'streamtube-core' )
            );             
        }

        get_role( 'administrator' )->add_cap( $capability );

        $role->add_cap( $capability );

        /**
         *
         * Fires after capability added
         *
         * @param WP_Role $role
         * @param string $capability
         * 
         */
        do_action( 'streamtube/core/added_role_capability', $role, $capability );

        return compact( 'role', 'capability' );
    }

    public function remove_role_capability( $role = '', $capability = '' ){

        if( ! $capability || ! is_string( $capability ) ){
           return new WP_Error(
                'invalid_requested',
                esc_html__( 'Invalid Requested', 'streamtube-core' )
            );
        }

        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to add capability.', 'streamtube-core' )
            );          
        }        

        if( $role ){
            return get_role( $role )->remove_cap( $capability );

            /**
             *
             * Fires after capability removed
             *
             * @param WP_Role $role
             * @param string $capability
             * 
             */
            do_action( 'streamtube/core/removed_role_capability', $role, $capability );
        }

        return $this->delete_capability( $capability );
    }

    /**
     *
     * Remove or add capability for given role
     *
     * @param string $role
     * @param string $capability
     *
     * @return true|WP_Error
     * 
     */
    public function update_role_capability( $role = '', $capability = '' ){

        $role       = trim( $role );
        $capability = trim( $capability );

        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to do this action.', 'streamtube-core' )
            );          
        }

        $_role = get_role( $role );

        if( ! $_role ){
            return new WP_Error(
                'role_not_found',
                esc_html__( 'Role was not found', 'streamtube-core' )
            );
        }

        /**
         *
         * Fires after capability removed
         *
         * @param WP_Role $_role
         * @param string $capability
         * 
         */
        do_action( 'streamtube/core/before_grant_role_capability', $_role, $capability );        

        if( $_role->has_cap( $capability ) ){

            if( self::is_admin_role( $_role->name ) && self::is_admin_capability( $capability ) ){
                return new WP_Error(
                    'cannot_remove_admin_capability',
                    esc_html__( 'Cannot remove admin capability', 'streamtube-core' )
                );                
            }

            $result = $_role->remove_cap( $capability );         

            /**
             *
             * Fires after capability removed
             *
             * @param WP_Role $_role
             * @param string $capability
             * 
             */
            do_action( 'streamtube/core/removed_role_capability', $_role, $capability );

            return array(
                'action'        =>  'removed',
                'capability'    =>  $capability
            );
        }else{
            $result = $_role->add_cap( $capability );

            /**
             *
             * Fires after capability added
             *
             * @param WP_Role $_role
             * @param string $capability
             * 
             */
            do_action( 'streamtube/core/added_role_capability', $_role, $capability );

            return array(
                'action'        =>  'added',
                'capability'    =>  $capability
            );
        }
    }     

    /**
     *
     * Bulk update role capabilities
     * 
     */
    public function update_role_capabilities( $role, $capabilities = array() ){
        if( ! self::can_promote_users() ){
            return new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to do this action.', 'streamtube-core' )
            );          
        }

        $role = get_role( $role );

        if( ! $role ){
            return new WP_Error(
                'role_not_exists',
                esc_html__( 'Role does not exist.', 'streamtube-core' ),
            );             
        }

        $old_caps = $role->capabilities;

        foreach ( $old_caps as $capability => $v ) {

            // Do not delete capability if it's admin and a default.
            if( self::is_admin_role( $role->name ) && self::is_admin_capability( $capability ) ){
                continue;
            }

            $role->remove_cap( $capability );

            /**
             *
             * Fires after capability removed
             *
             * @param WP_Role $role
             * @param string $capability
             * 
             */
            do_action( 'streamtube/core/removed_role_capability', $role, $capability );            
        }

        $capabilities = array_unique( array_values( $capabilities ) );

        for ( $i = 0;  $i < count( $capabilities );  $i++) { 
            $role->add_cap( $capabilities[$i] );

            /**
             *
             * Fires after capability added
             *
             * @param WP_Role $_role
             * @param string $capability
             * 
             */
            do_action( 'streamtube/core/added_role_capability', $role, $capabilities[$i] );            
        }

        $response = compact( 'old_caps', 'capabilities' );

        /**
         *
         * Fires after capability removed
         *
         * @param WP_Role $_role
         * @param array $response
         * 
         */
        do_action( 'streamtube/core/updated_role_capabilities', $role, $response );          

        return $response;
    }     

    /**
     *
     * Add capability to given role
     * 
     * @param  string $role
     * @param  string $capability
     */
    public function ajax_add_role_capability(){

        $http_data = wp_parse_args( $_REQUEST, array(
            '_wpnonce'      =>  '',
            'data'          =>  '',
            'capability'    =>  ''
        ) );

        extract( $http_data );

        check_ajax_referer( 'add-capability-' . $data );

        $response = $this->add_role_capability( $data, $capability );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        wp_send_json_success( array_merge( $response, array(
            'output'    =>  streamtube_core_the_capability_list_item( $response['capability'], $response['role']->name )
        ) ) );
    }     

    /**
     *
     * Do AJAX granting/ungraning capability
     * 
     */
    public function ajax_update_role_capability(){

        $http_data = wp_parse_args( $_REQUEST, array(
            'capability'    =>  '',
            'role'          =>  ''
        ) );

        extract( $http_data );

        check_ajax_referer( 'edit-cap-' . $capability );

        $response = $this->update_role_capability( $role, $capability );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        wp_send_json_success( 
            array(
                'data'      =>  $response,
                'message'   =>  sprintf(
                    esc_html__( 'The %s capability has been granted for %s role.', 'streamtube-core' ),
                    $capability,
                    $role
                )
            )
        );
    } 

    /**
     *
     * Do AJAX adding role and updating capabilities
     * 
     */
    public function ajax_add_role(){

        check_ajax_referer( 'add-role-' . get_current_user_id() );

        $http_data = wp_parse_args( $_POST, array(
            'role'          =>  '',
            'capabilities'  =>  array(),
            'edit_role'     =>  ''
        ) );

        extract( $http_data );

        if( ! empty( $edit_role ) ){
            $response = $this->update_role_capabilities( $edit_role, $capabilities );
        }else{
            $response = $this->add_role( $role, $capabilities );
        }    

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        if( ! $response ){
            wp_send_json_error( new WP_Error(
                'undefined_error',
                esc_html__( 'Undefined error: It seems the role already exists or cannot be added.', 'streamtube-core' )
            ) );
        }   

        wp_send_json_success( array(
            'message'   =>  $edit_role ? esc_html__( 'Role updated.', 'streamtube-core' ) : esc_html__( 'Role added.', 'streamtube-core' ),
            'data'      =>  $response,
            'output'    =>  ! $edit_role ? streamtube_core_the_role_list_item( $response->name ) : ''
        ) );
    }

    /**
     *
     * Do AJAX deleting role
     * 
     */
    public function ajax_delete_role(){

        $http_data = wp_parse_args( $_REQUEST, array(
            '_wpnonce'  =>  '',
            'data'      =>  ''
        ) );

        extract( $http_data );

        check_ajax_referer( 'delete-role-' . $data );

        $response = $this->delete_role( $data );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        wp_send_json_success( esc_html__( 'Role deleted.', 'streamtube-core' ) );
    }   

    /**
     *
     * Do AJAX deleting capability
     * 
     */
    public function ajax_delete_capability(){

        $http_data = wp_parse_args( $_REQUEST, array(
            '_wpnonce'  =>  '',
            'data'      =>  ''
        ) );

        extract( $http_data );

        check_ajax_referer( 'delete-capability-' . $data );

        $response = $this->delete_capability( $data );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        wp_send_json_success( array(
            'message'   =>  esc_html__( 'Capability deleted.', 'streamtube-core' )
        ) );
    }

    public function ajax_search_capabilities(){
        check_ajax_referer( '_wpnonce' );

        $http_data = wp_parse_args( $_POST, array(
            'capability'    =>  ''
        ) );

        extract( $http_data );

        $capability = trim( $capability );
        $caps = array();

        if( empty( $capability ) ){
            wp_send_json_error( new WP_Error(
                'empty_capability',
                esc_html__( 'Capability is empty.', 'streamtube-core' )
            ) );
        }

        $capabilities = self::get_all_capabilities();

        for ($i=0; $i < count( $capabilities ); $i++) { 
            if( strpos( $capabilities[$i] , $capability ) !== false && ! self::is_admin_capability( $capabilities[$i] ) ){
                $caps[] = $capabilities[$i];
            }
        }

        wp_send_json_success( $caps );
    }
}