<?php
/**
 * Handles file upload logic for Streamtube Core
 * 
 * @package Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Streamtube_Core_Upload
 * Manages file upload permissions, validations, and actual upload process.
 *
 * @since 1.0.0
 */
class Streamtube_Core_Upload {

	public $user_id = 0;
	public $file_size = 0;
	public $file_name = '';
	public $tmp_name = '';
	public $file_type = '';
	public $file_error = false;
	public $file_audio = true;
	public $chunks = 0;
	public $wp_errors = null;

	/**
	 * Constructor to initialize the upload process.
	 * 
	 * @param array $args Arguments for initialization.
	 * @since 1.0.0
	 */
	public function __construct( $args = array() ) {
		$args = wp_parse_args( $args, array(
			'user_id'  => get_current_user_id(),
			'size'     => 0,
			'name'     => '',
			'tmp_name' => '',
			'type'     => '',
			'audio'    => true,
			'chunks'   => 0,
			'error'    => false
		) );

		$this->tmp_name   = $args['tmp_name'];
		$this->user_id    = $args['user_id'];
		$this->file_size  = $args['tmp_name'] ? filesize( $args['tmp_name'] ) : $args['size'];
		$this->file_name  = $args['name'];
		$this->file_type  = $args['type'];
		$this->file_error = $args['error'];
		$this->file_audio = $args['audio'];
		$this->chunks     = $args['chunks'];

		$this->wp_errors = new WP_Error();

		$this->check_file();
	}

	/**
	 *
	 * Get allowed extensions including audio and video
	 * 
	 * @return array
	 */
	public function get_allowed_extensions() {
		$extensions = wp_get_video_extensions();

		if ( $this->file_audio ) {
			$extensions = array_merge( $extensions, wp_get_audio_extensions() );
		}
		return array_unique( $extensions );
	}

	/**
	 *
	 * Get file mimetype by reading file content
	 * 
	 */
	public function get_file_mime_type() {

		if ( ! $this->tmp_name || ! $this->file_name ) {
			return '';
		}

		if ( $this->tmp_name ) {
			// Read first 8 bytes of the file
			$file_data = file_get_contents( $this->tmp_name, false, null, 0, 8 );

			// Get file mime type using finfo
			$finfo     = finfo_open( FILEINFO_MIME_TYPE );
			$mime_type = finfo_file( $finfo, $this->tmp_name );
			finfo_close( $finfo );

			return $mime_type;
		}

		if ( $this->file_name ) {
			$check = wp_check_filetype( $this->file_name );
			return $check['type'];
		}
	}

	/**
	 *
	 * Get WP_Error
	 * 
	 * @return WP_Error
	 */
	public function get_errors() {
		return $this->wp_errors;
	}

	/**
	 * Check if the file can be uploaded based on permissions and validations.
	 *
	 * @return WP_Error|true Returns WP_Error object if validation fails, or true if allowed.
	 * @since 1.0.0
	 */
	public function check_file() {

		// Check for file errors (empty or missing file)
		if ( is_int( $this->file_error ) && $this->file_error !== 0 || ! $this->file_size || $this->file_size === 0 || ( ! $this->file_type && ! $this->file_name ) ) {
			$this->wp_errors->add(
				'file_error',
				esc_html__( 'File was not found or is empty.', 'streamtube-core' )
			);
		}

		$max_upload_size = wp_max_upload_size();

		if ( $max_upload_size < 0 ) {
			$this->wp_errors->add(
				'upload_disabled',
				esc_html__( 'Video uploads are currently disabled.', 'streamtube-core' )
			);
		}

		if ( $max_upload_size > 0 && $this->file_size > $max_upload_size ) {
			$this->wp_errors->add(
				'exceeds_max_upload_size',
				sprintf(
					esc_html__( 'The %s file size %s exceeds the maximum allowed size: %s', 'streamtube-core' ),
					$this->file_name,
					size_format( $this->file_size ),
					size_format( $max_upload_size )
				)
			);
		}

		if ( $this->file_name ) {

			// Validate file type
			$file_type = wp_check_filetype( $this->file_name );

			if ( ! $file_type['ext'] || ! in_array( strtolower( $file_type['ext'] ), $this->get_allowed_extensions() ) ) {
				$this->wp_errors->add(
					'file_type_not_allowed',
					esc_html__( 'File type is not allowed.', 'streamtube-core' )
				);
			}
		}

		$mime_type = $this->file_type ? explode( '/', $this->file_type ) : explode( '/', $this->get_file_mime_type() );

		if ( ! is_array( $mime_type ) || ! in_array( $mime_type[0], array( 'audio', 'video' ) ) ) {
			$this->wp_errors->add(
				'mime_type_not_allowed',
				esc_html__( 'Mime type is not allowed.', 'streamtube-core' )
			);
		}

		if ( $this->chunks ) {
			if ( ! class_exists( 'BigFileUploads' ) || ! method_exists( 'BigFileUploads', 'ajax_chunk_receiver' ) ) {
				$this->wp_errors->add(
					'BigFileUploads_not_found',
					esc_html__( 'BigFileUploads plugin was not found.', 'streamtube-core' )
				);
			}
		}

		// Check if the user has permission to upload
		if ( ! Streamtube_Core_Permission::can_upload( $this->user_id ) ) {
			$this->wp_errors->add(
				'no_permission',
				esc_html__( 'You do not have permission to upload videos.', 'streamtube-core' )
			);
		}

		// Check if the user is verified
		if ( get_option( 'upload_files_verified_user' ) && ! Streamtube_Core_Permission::is_verified( $this->user_id ) ) {
			$this->wp_errors->add(
				'not_verified',
				esc_html__( 'Sorry, you have not been verified yet.', 'streamtube-core' )
			);
		}

		// Apply filter for custom errors
		$this->wp_errors = apply_filters(
			'streamtube/core/upload/video/errors',
			$this->wp_errors,
			array(
				'user_id' => $this->user_id,
				'size'    => $this->file_size,
				'name'    => $this->file_name,
				'error'   => $this->file_error
			)
		);

		// Return errors if validation fails
		if ( $this->wp_errors->get_error_code() ) {
			return $this->wp_errors;
		}

		return true;
	}

	/**
	 * Handle the file upload process.
	 *
	 * @param string $file_id_or_url File ID or URL for upload.
	 * @param int $post_parent_id Post ID to associate with the uploaded file.
	 * @return int|WP_Error Attachment ID or WP_Error on failure.
	 * @since 1.0.0
	 */
	public function do_upload( $file_id = '', $post_parent_id = 0 ) {

		$attachment_id = media_handle_upload( $file_id, $post_parent_id );

		if ( is_wp_error( $attachment_id ) ) {
			return $attachment_id;
		}

		if ( is_int( $attachment_id ) ) {
			$attach_data = wp_generate_attachment_metadata( $attachment_id, get_attached_file( $attachment_id ) );
			wp_update_attachment_metadata( $attachment_id, $attach_data );
		}

		return $attachment_id;
	}
}
