<?php
/**
 * Define the dashboard functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * Define the profile functionality
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Streamtube_Core_User_Dashboard extends Streamtube_Core_User {

	const ENDPOINT = 'dashboard';

	protected $post;

	protected $comment;

	public function __construct() {
		$this->post = new Streamtube_Core_Post();

		$this->comment = new Streamtube_Core_Comment();
	}

	/**
	 *
	 * Check if current page is dashboard
	 * 
	 * @return boolean
	 */
	public static function is_dashboard() {
		global $wp_query;

		if ( is_author() && array_key_exists( self::ENDPOINT, $wp_query->query_vars ) ) {
			return true;
		}

		return false;
	}

	/**
	 *
	 * Add dashboard endpoint
	 * 
	 */
	public function add_endpoints() {
		add_rewrite_endpoint( self::ENDPOINT, EP_AUTHORS );
	}

	/**
	 * 
	 * @param  integer $user_id
	 * @param  string  $value
	 * @return string
	 *
	 * @since  1.0.0
	 * 
	 */
	public function get_endpoint( $user_id = 0, $endpoint = '' ) {

		if ( ! $user_id ) {
			return;
		}

		$url = get_author_posts_url( $user_id );

		if ( ! get_option( 'permalink_structure' ) ) {
			return add_query_arg( array(
				self::ENDPOINT => $endpoint
			), $url );
		}

		return trailingslashit( $url ) . self::ENDPOINT . '/' . $endpoint;
	}

	/**
	 *
	 * Filter menu, fix missing params
	 * 
	 * @param  array &$items
	 * 
	 */
	protected function filter_menu( &$items ) {
		foreach ( $items as $menu_id => $menu ) {
			// Ensure 'parent' is set, defaulting to 'dashboard' if not provided
			$menu = wp_parse_args( $menu, array(
				'title'     => '',
				'alt_title' => '',
				'parent'    => self::ENDPOINT,
				'icon'      => '',
				'cap'       => 'read',
				'url'       => '',
				'callback'  => '',
				'priority'  => 10
			) );

			// Process submenu items if they exist
			if ( array_key_exists( 'submenu', $menu ) && is_array( $menu['submenu'] ) ) {
				if ( $menu['url'] !== '#' ) {
					$_menu = array(
						'title'    => $menu['alt_title'] ? $menu['alt_title'] : $menu['title'],
						'cap'      => $menu['cap'],
						'callback' => $menu['callback'],
						'url'      => $menu['url'],
						'icon'     => $menu['icon'],
						'priority' => 0
					);

					/**
					 *
					 * Filter the default submenu
					 *
					 * @param array $_menu
					 * @param string $menu_id
					 * 
					 */
					$menu['submenu'][ $menu_id ] = apply_filters(
						'streamtube/core/dashboard/menu/default_submenu',
						$_menu,
						$menu_id
					);
				}

				foreach ( $menu['submenu'] as $submenu_id => $submenu ) {

					// Update each submenu item with the appropriate parent path
					$menu['submenu'][ $submenu_id ] = array_merge( $submenu, array(
						'parent' => ! empty( $menu['parent'] ) ? $menu['parent'] . '/' . $menu_id : ''
					) );

					// Update the main $items array with the modified submenu
					$items[ $menu_id ]['submenu'] = $menu['submenu'];
				}
			}

			$items[ $menu_id ] = $menu;
		}
	}

	/**
	 * Sorts menu items by priority.
	 *
	 * @param array &$items Reference to the menu items array.
	 */
	protected function uasort( &$items ) {
		uasort( $items, function ($item1, $item2) {
			return $item1['priority'] <=> $item2['priority'];
		} );
	}

	/**
	 *
	 * Menu items
	 * 
	 * @return array
	 */
	public function get_menu_items() {

		$items = array();

		$items['dashboard'] = array(
			'title'    => esc_html__( 'Dashboard', 'streamtube-core' ),
			'icon'     => 'icon-gauge',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/dashboard.php' );
			},
			'cap'      => 'read',
			'priority' => 0
		);

		$items['comments'] = array(
			'title'    => esc_html__( 'Comments', 'streamtube-core' ),
			'badge'    => $this->comment->get_pending_comments_badge(),
			'desc'     => esc_html__( 'All comments', 'streamtube-core' ),
			'icon'     => 'icon-comment',
			'callback' => function () {
				streamtube_core_load_template( 'comment/table-comments.php' );
			},
			'parent'   => self::ENDPOINT,
			'cap'      => 'publish_posts',
			'priority' => 20
		);

		$items['advertising'] = array(
			'title'    => esc_html__( 'Advertising', 'streamtube-core' ),
			'desc'     => esc_html__( 'Advertising', 'streamtube-core' ),
			'icon'     => 'icon-flash',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/advertising.php' );
			},
			'parent'   => self::ENDPOINT,
			'cap'      => 'manage_vast_tag',
			'priority' => 50
		);

		$items['site-settings'] = array(
			'title'    => esc_html__( 'Settings', 'streamtube-core' ),
			'icon'     => 'dashicons dashicons-admin-generic',
			'parent'   => self::ENDPOINT,
			'callback' => function () {
				streamtube_core_load_template( 'site-settings/capabilities.php' );
			},
			'url'      => '#',
			'cap'      => 'manage_options',
			'submenu'  => array(
				'xcapabilities'   => array(
					'title'    => esc_html__( 'Capabilities', 'streamtube-core' ),
					'icon'     => 'icon-wrench',
					'callback' => function () {
						streamtube_core_load_template( 'site-settings/capabilities.php' );
					},
					'cap'      => function () {
						return StreamTube_Core_Role::can_promote_users();
					},
					'priority' => 10
				),
				'user-roles'      => array(
					'title'    => esc_html__( 'Roles', 'streamtube-core' ),
					'icon'     => 'icon-wrench',
					'callback' => function () {
						streamtube_core_load_template( 'site-settings/roles.php' );
					},
					'cap'      => function () {
						return StreamTube_Core_Role::can_promote_users();
					},
					'priority' => 20
				),
				'max-upload-size' => array(
					'title'    => esc_html__( 'Max Upload Sizes', 'streamtube-core' ),
					'icon'     => 'icon-wrench',
					'callback' => function () {
						streamtube_core_load_template( 'site-settings/max-upload-size.php' );
					},
					'cap'      => 'manage_options',
					'priority' => 30
				),
				'storage'         => array(
					'title'    => esc_html__( 'Storages', 'streamtube-core' ),
					'icon'     => 'icon-wrench',
					'callback' => function () {
						streamtube_core_load_template( 'site-settings/storage.php' );
					},
					'cap'      => 'manage_options',
					'priority' => 40
				)
			),
			'priority' => 99998
		);

		$items['users'] = array(
			'title'    => esc_html__( 'Users', 'streamtube-core' ),
			'icon'     => 'icon-users',
			'callback' => function () {
				streamtube_core_load_template( 'user/user-table.php' );
			},
			'cap'      => 'list_users',
			'priority' => 99990,
			'parent'   => self::ENDPOINT,
		);

		$items['backend'] = array(
			'title'    => esc_html__( 'Backend', 'streamtube-core' ),
			'icon'     => 'icon-wordpress',
			'callback' => function () {},
			'cap'      => 'activate_plugins',
			'url'      => admin_url(),
			'priority' => 99999
		);

		/**
		 *
		 * @since 1.0.8
		 * 
		 */
		$items = apply_filters( 'streamtube/core/user/dashboard/menu/items', $items );

		$this->filter_menu( $items );

		$this->uasort( $items );

		return $items;
	}

	/**
	 *
	 * Personal settings menu
	 * 
	 */
	public function register_profile_menu( $menu_items ) {

		$menu_items = wp_parse_args( $menu_items, array(
			'account' => array()
		) );

		$submenu = array();

		$submenu['personal'] = array(
			'title'    => esc_html__( 'General', 'streamtube-core' ),
			'icon'     => 'icon-edit',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/account/personal.php' );
			},
			'priority' => 10
		);

		$submenu['social-profiles'] = array(
			'title'    => esc_html__( 'Social Profiles', 'streamtube-core' ),
			'icon'     => 'icon-share',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/account/social-profiles.php' );
			},
			'priority' => 20
		);

		// Check if bp is active and avatar enabled
		if ( ! function_exists( 'bp_disable_avatar_uploads' ) || bp_disable_avatar_uploads() ) {
			$submenu['change-profile-avatar'] = array(
				'title'    => esc_html__( 'Avatar', 'streamtube-core' ),
				'icon'     => 'icon-user-circle',
				'callback' => function () {
					streamtube_core_load_template( 'user/dashboard/account/avatar.php' );
				},
				'priority' => 30
			);
		}

		if ( ! function_exists( 'bp_disable_cover_image_uploads' ) || bp_disable_cover_image_uploads() ) {
			$submenu['change-profile-photo'] = array(
				'title'    => esc_html__( 'Profile Photo', 'streamtube-core' ),
				'icon'     => 'icon-picture',
				'callback' => function () {
					streamtube_core_load_template( 'user/dashboard/account/photo.php' );
				},
				'priority' => 40
			);
		}

		$submenu['storage'] = array(
			'title'    => esc_html__( 'Storage', 'streamtube-core' ),
			'icon'     => 'icon-upload-cloud',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/account/storage.php' );
			},
			'priority' => 100
		);

		$submenu['application-password'] = array(
			'title'    => esc_html__( 'App Password', 'streamtube-core' ),
			'icon'     => 'icon-lock',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/account/application-password.php' );
			},
			'priority' => 100
		);

		if ( ! $menu_items['account'] ) {
			$menu_items['account'] = array(
				'title'    => esc_html__( 'Account', 'streamtube-core' ),
				'icon'     => 'icon-user-circle-o',
				'url'      => '#',
				'callback' => function () {
					streamtube_core_load_template( 'user/dashboard/account/personal.php' );
				},
				'parent'   => 'dashboard',
				'submenu'  => $submenu,
				'cap'      => 'read',
				'priority' => 1000
			);
		} else {
			$menu_items['account'] = array_merge( $menu_items['account'], array(
				'priority' => 1000,
				'icon'     => 'icon-user-circle-o',
				'submenu'  => $submenu,
				'parent'   => 'dashboard'
			) );
		}

		return $menu_items;
	}

	/**
	 *
	 * Register post types and related taxonomies items
	 * 
	 */
	public function register_menu_post_types( $items ) {

		$post_types = streamtube_core_get_public_post_types();

		foreach ( $post_types as $post_type ) {

			$post_type_object = get_post_type_object( $post_type );
			$menu_name        = $post_type_object->labels->menu_name;

			$menu_icon = $post_type_object->menu_icon;

			if ( strpos( $menu_icon, 'dashicons' ) !== false ) {
				$menu_icon = sprintf( 'dashicons %s', $menu_icon );
			}

			switch ( $post_type_object->name ) {
				case 'youtube_importer':
					$menu_name = esc_html__( 'YouTube', 'streamtube-core' );
					break;
			}

			$items[ $post_type_object->name ] = array(
				'title'     => $menu_name,
				'alt_title' => sprintf(
					esc_html__( 'All %s', 'streamtube-core' ),
					$menu_name
				),
				'badge'     => $this->post->get_pending_posts_badge( $post_type_object->name ),
				'desc'      => sprintf(
					esc_html__( 'All %s', 'streamtube-core' ),
					$menu_name
				),
				'icon'      => $menu_icon ? $menu_icon : 'dashicons dashicons-admin-post',
				'parent'    => 'dashboard',
				'cap'       => function () use ($post_type_object) {
					return current_user_can( $post_type_object->cap->create_posts );
				},
				'callback'  => function () use ($post_type_object) {

					$GLOBALS['post_type'] = $post_type_object->name;

					streamtube_core_load_template( 'user/dashboard/post-type.php' );
				},
				'priority'  => $post_type_object->menu_position ? absint( $post_type_object->menu_position ) : 10
			);

			$items[ $post_type_object->name ]['submenu'][ $post_type_object->name ] = array(
				'title'    => $menu_name,
				'cap'      => $items[ $post_type_object->name ]['cap'],
				'priority' => 10
			);

			$taxonomies = get_object_taxonomies( $post_type_object->name, 'object' );

			foreach ( $taxonomies as $tax => $object ) {

				if (
					! in_array( $tax, array( 'post_format' ) ) &&
					$object->show_ui_frontend
				) {
					$menu_name = $object->labels->menu_name;

					switch ( $tax ) {
						case 'categories':
							$menu_name = esc_html__( 'Categories', 'streamtube-core' );
							break;
						case 'video_tag':
							$menu_name = esc_html__( 'Tags', 'streamtube-core' );
							break;
						case 'video_collection':
							$menu_name = esc_html__( 'Collections', 'streamtube-core' );
							break;
					}

					/**
					 *
					 * Filter taxonomy name
					 * 
					 * @param string name
					 * @param object Taxonomy object
					 */
					$menu_name = apply_filters( 'streamtube/core/dashboard/menu/taxonomy_name', $menu_name, $object );

					$items[ $post_type_object->name ]['submenu'][ $tax ] = array(
						'title'    => $menu_name,
						'icon'     => 'icon-minus',
						'callback' => function () use ($post_type_object, $tax) {
							$GLOBALS['post_type'] = $post_type_object->name;
							$GLOBALS['taxonomy'] = $tax;
							streamtube_core_load_template( 'user/dashboard/taxonomy.php' );
						},
						'cap'      => function () use ($object) {
							return current_user_can( $object->cap->edit_terms );
						},
						'priority' => 15
					);
				}
			}
		}

		return $items;
	}

	/**
	 *
	 * WP Menu callback content template
	 * 
	 */
	public function _wp_menu_callback( $item ) {
		return streamtube_core_load_template( 'user/dashboard/wp-menu-content.php', true, $item );
	}

	/**
	 *
	 * Since 4.0, Dashboard supports menu items retrieved from wp_get_nav_menu_items().
	 * 
	 */
	public function register_wp_menu_items( $menu_items ) {

		$locations = get_nav_menu_locations();

		if ( ! isset( $locations['dashboard'] ) || ! $locations['dashboard'] ) {
			return $menu_items;
		}

		$wp_nav_items = wp_get_nav_menu_items( $locations['dashboard'] );

		if ( $wp_nav_items ) {

			$wp_nav_items = new StreamTube_Core_WP_Menu( $wp_nav_items, array( $this, '_wp_menu_callback' ) );

			foreach ( $wp_nav_items->get_menu() as $item ) {
				$submenu = $item->submenu ? $wp_nav_items->build_submenu( $item->submenu ) : false;

				$menu_items[ sanitize_title( $item->title ) ] = array(
					'title'         => $item->title,
					'url'           => $item->url ?? '',
					'callback'      => array( $this, '_wp_menu_callback' ),
					'callback_args' => $item,
					'priority'      => $wp_nav_items::get_menu_priority( $item ),
					'cap'           => $wp_nav_items::get_menu_cap( $item ),
					'icon'          => $item->wpmi->icon,
					'submenu'       => $submenu
				);
			}
		}

		return $menu_items;
	}

	/**
	 *
	 * Register `dashboard` menu location
	 * 
	 * @since 4.0
	 */
	public function register_dashboard_menu_location() {
		register_nav_menus(
			array(
				'dashboard' => esc_html__( 'Custom Dashboard Menu', 'streamtube-core' )
			)
		);
	}

	/**
	 *
	 * @since 1.0.8
	 * 
	 * @return [type] [description]
	 */
	private function pre_get_menu_items() {
		$menu_items = $this->get_menu_items();

		$enabled_pages = get_option( 'user_dashboard_pages' );

		if ( ! $enabled_pages || ! is_array( $enabled_pages ) ) {
			return $menu_items;
		}

		$enabled_pages['dashboard'] = '1';

		foreach ( $menu_items as $key => $value ) {
			if ( array_key_exists( $key, $enabled_pages ) && empty( $enabled_pages[ $key ] ) ) {
				unset( $menu_items[ $key ] );
			}
			if ( array_key_exists( "{$key}_icon", $enabled_pages ) && ! empty( $enabled_pages[ "{$key}_icon" ] ) ) {
				$menu_items[ $key ]['icon'] = $enabled_pages[ "{$key}_icon" ];
			}

			if ( array_key_exists( "{$key}_icon_color", $enabled_pages ) && ! empty( $enabled_pages[ "{$key}_icon_color" ] ) ) {
				$menu_items[ $key ]['icon_color'] = $enabled_pages[ "{$key}_icon_color" ];
			}

			if ( array_key_exists( "{$key}_priority", $enabled_pages ) && (int) $enabled_pages[ "{$key}_priority" ] ) {
				$menu_items[ $key ]['priority'] = $enabled_pages[ "{$key}_priority" ];
			}
		}

		uasort( $menu_items, function ($item1, $item2) {
			return $item1['priority'] <=> $item2['priority'];
		} );

		return $menu_items;
	}

	private function get_request_endpoint() {
		global $wp_query;

		$menu_items = $this->get_menu_items();

		$endpoint = $wp_query->query_vars[ self::ENDPOINT ];

		if ( empty( $endpoint ) || ( is_string( $endpoint ) && $endpoint == 1 ) ) {
			return array_keys( $menu_items )[0];
		}

		return explode( '/', $endpoint );
	}

	private function get_parent_menu_item() {
		$request = $this->get_request_endpoint();

		if ( is_string( $request ) ) {
			return $request;
		}

		return is_array( $request ) && count( $request ) === 2 ? $request[0] : 'dashboard';
	}

	private function get_current_menu_item( $parent = '' ) {

		$depth = 0;

		$current = '';

		$menu_all_items = $this->get_menu_items();

		$menu_items = $this->pre_get_menu_items();

		if ( $parent ) {
			$depth = 1;
			if (
				array_key_exists( $parent, $menu_items )
				&& is_array( $menu_items[ $parent ] )
				&& array_key_exists( 'submenu', $menu_items[ $parent ] ) ) {
				$menu_items = $menu_items[ $parent ]['submenu'];
			}
		}

		$request = $this->get_request_endpoint();

		if ( is_string( $request ) ) {
			return $request;
		}

		/**
		 *
		 * Filter current request
		 * Enables other plugin filter the current menu item
		 * 
		 * @param string $current
		 * @param array $request
		 *
		 * 
		 */
		$current = apply_filters( 'streamtube/core/dashboard/menu/current', $current, $request );

		if ( $current ) {
			return $current;
		}

		if ( is_array( $request ) ) {
			$current = isset( $request[ $depth ] ) ? $request[ $depth ] : $request[0];

			if ( count( $request ) == 1 && ! array_key_exists( $current, $menu_all_items ) ) {
				return array_keys( $menu_items )[0];
			}

			if (
				! isset( $menu_all_items[ $request[0] ] )
				|| ! array_key_exists( 'submenu', $menu_all_items[ $request[0] ] )
				|| ! isset( $menu_all_items[ $request[0] ] )
				|| ! isset( $menu_all_items[ $request[0] ]['submenu'] ) ) {
				return $request[0];
			}
		}

		if ( $parent && $current == $parent ) {
			$current = array_keys( $menu_items )[0];
		}

		if ( ! array_key_exists( $current, $menu_items ) ) {
			$current = array_keys( $menu_items )[0];
		}

		if ( ! get_option( 'permalink_structure' ) && $parent ) {
			foreach ( $menu_items as $key => $value ) {
				if ( isset( $_GET[ $key ] ) ) {
					$current = $key;
				}
			}
		}

		return $current;
	}

	public function the_menu( $args = array(), $parent = '' ) {

		$menu_items = $this->pre_get_menu_items();

		if ( $parent ) {
			$menu_items = $menu_items[ $parent ]['submenu'];
		}

		$menu = new Streamtube_Core_Menu( array_merge( $args, array(
			'menu_items' => $menu_items,
			'current'    => $this->get_current_menu_item( $parent ),
			'icon'       => true
		) ) );

		return $menu->the_menu();
	}

	/**
	 * Load the menu template for the dashboard
	 */
	public function the_menu_template() {
		ob_start();

		// Load the user dashboard menu template
		streamtube_core_load_template( 'user/dashboard/menu.php' );

		/**
		 * Allow other functions to filter the navigation output
		 * This can be useful in certain cases where additional modifications are needed.
		 */
		echo apply_filters( 'streamtube/core/dashboard/primary_nav', ob_get_clean() );
	}

	/**
	 *
	 * Load the main content template
	 * 
	 * @param  string $parent
	 */
	public function the_main_template( $parent = '' ) {

		$menu_items = $this->pre_get_menu_items();

		if ( ! $parent ) {
			$parent = $this->get_parent_menu_item();
		}

		$current = $this->get_current_menu_item( $parent );

		if (
			$parent
			&& array_key_exists( $parent, $menu_items )
			&& is_array( $menu_items[ $parent ] )
			&& array_key_exists( 'submenu', $menu_items[ $parent ] ) ) {
			$menu_items = $menu_items[ $parent ]['submenu'];
		}

		$menu_item = wp_parse_args( (array) $menu_items[ $current ], array(
			'cap'           => 'read',
			'callback'      => __return_true(),
			'callback_args' => array()
		) );

		$cap      = $menu_item['cap'];
		$callback = $menu_item['callback'];

		if (
			( is_string( $cap ) && current_user_can( $cap ) ) ||
			( is_callable( $cap ) && call_user_func( $cap ) === true )
		) {
			if ( is_callable( $callback ) ) {
				call_user_func( $callback, $menu_item['callback_args'] );
			}
		} else {
			printf(
				'<div class="alert alert-warning">%s</div>',
				esc_html__( 'You do not have permission to access this page.', 'streamtube-core' )
			);
		}
	}

	/**
	 *
	 * Load the index template including header, sidebar, main content, footer
	 * Everything of the Dashboard Index page
	 * 
	 */
	public function the_index_template() {
		global $wp_query;

		if (
			( $this->is_my_profile() && self::is_dashboard() ) ||
			(
				array_key_exists( 'dashboard', $wp_query->query_vars ) &&
				is_user_logged_in() &&
				wp_get_current_user()->user_nicename === $wp_query->query_vars['author_name']
			)
		) {

			add_filter( 'sidebar_float', function ($show) {
				return false;
			}, 10, 1 );

			define( 'STREAMTUBE_CORE_IS_DASHBOARD_INDEX', true );

			streamtube_core_load_template( 'user/dashboard/index.php' );
			exit;
		}
	}

	/**
	 *
	 * Convert widgets to metaboxes
	 *
	 * @since 1.0.0
	 * 
	 */
	public function add_widget_metaboxes() {
		$metabox = new StreamTube_Core_Widget_Meta_Box( array(
			'sidebar' => 'user-dashboard',
			'screen'  => 'user_dashboard',
			'context' => 'normal'
		) );

		return $metabox->add_meta_boxes();
	}

	/**
	 *
	 * Generate the warning notice for capabilities
	 * 
	 */
	public function the_permission_warning() {
		if ( current_user_can( 'administrator' ) && ! get_transient( 'dismiss_perm_warning_' . get_current_user_id() ) ) {
			$screens = array(
				'permissions',
				'permissions/capabilities',
				'permissions/roles',
				'users'
			);

			if ( in_array( get_query_var( self::ENDPOINT ), $screens ) ) {
				streamtube_core_load_template( 'form/permission-warning.php' );
			}
		}
	}

	/**
	 *
	 * Handles the dismissal of permission warning notices.
	 * 
	 */
	public function dismiss_permissions_warning() {
		check_ajax_referer( 'permissions_warning', 'permissions_warning' );
		wp_send_json_success( set_transient( 'dismiss_perm_warning_' . get_current_user_id(), 1 ) );
	}

	/**
	 *
	 * Auto redirect to user dashboard once logged in
	 * 
	 * @since  1.0.0
	 * 
	 */
	public function login_redirect( $redirect_to, $requested_redirect_to, $user ) {

		if ( $user instanceof WP_User ) {

			if ( isset( $user->roles ) && in_array( 'administrator', $user->roles ) ) {
				return $redirect_to;
			}

			if ( ! $redirect_to ) {
				$redirect_to = home_url( '/' );
			}

			if ( strpos( $redirect_to, '/wp-admin' ) !== false ) {
				$redirect_to = home_url( '/' );
			}
		}

		return $redirect_to;
	}

	/**
	 * Redirect to dashboard if redirect_to_dashboard found
	 */
	public function dashboard_redirect() {
		global $wp_query;

		$endpoint = self::ENDPOINT;

		if ( isset( $wp_query->query_vars['redirect_to_dashboard'] ) ) {

			if ( ! empty( $wp_query->query_vars['redirect_to_dashboard'] ) ) {
				$endpoint = $wp_query->query_vars['redirect_to_dashboard'];
			}

			if ( is_user_logged_in() ) {
				wp_redirect( $this->get_endpoint( get_current_user_id(), $endpoint ) );
				exit;
			} else {
				wp_redirect( wp_login_url( add_query_arg(
					array(
						'redirect_to_dashboard' => $endpoint
					),
					home_url( '/' )
				) ) );
			}
		}
	}

	/**
	 *
	 * Get upload types
	 * 
	 * @return array
	 *
	 * @since  1.0.0
	 * 
	 */
	public function get_upload_types( $location = 'header' ) {

		$types = array();

		if ( StreamTube_Core_Permission::can_upload() ) {
			$types['upload'] = array(
				'text'     => esc_html__( 'Upload Video', 'streamtube-core' ),
				'icon'     => 'icon-upload-cloud',
				'priority' => 10
			);
		}

		if ( StreamTube_Core_Permission::can_embed() ) {
			$types['embed'] = array(
				'text'     => esc_html__( 'Add From Source', 'streamtube-core' ),
				'icon'     => 'icon-code',
				'priority' => 20
			);
		}

		$types = apply_filters( 'streamtube_core_upload_types', $types, $location );

		uasort( $types, function ($item1, $item2) {
			return ( $item1['priority'] ?? 0 ) <=> ( $item2['priority'] ?? 100 );
		} );

		return $types;
	}

	/**
	 * Add js class body
	 */
	public function add_js_body_class( $classes ) {
		if ( ! in_array( 'js', $classes ) ) {
			$classes[] = 'js';
		}

		if ( in_array( 'no-js', $classes ) ) {
			unset( $classes[ array_search( 'no-js', $classes ) ] );
		}
		return array_unique( array_filter( $classes ) );
	}

	/**
	 *
	 * Load dashicons only for author single page
	 * 
	 */
	public function enqueue_scripts() {
		if ( is_author() ) {
			wp_enqueue_style( 'dashicons' );
		}
	}
}