<?php
/**
 * WP Users List Table
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class StreamTube_Core_Users_List_Table extends WP_Users_List_Table {

	function __construct( $args = array() ) {
		parent::__construct( $args );
	}

	/**
	 * Gets the current page number.
	 *
	 * @since 3.1.0
	 *
	 * @return int
	 */
	public function get_pagenum() {
		$pagenum = isset( $_REQUEST['cpaged'] ) ? absint( $_REQUEST['cpaged'] ) : 0;

		if ( isset( $this->_pagination_args['total_pages'] ) && $pagenum > $this->_pagination_args['total_pages'] ) {
			$pagenum = $this->_pagination_args['total_pages'];
		}

		return max( 1, $pagenum );
	}

	/**
	 * Displays the pagination.
	 *
	 * @since 3.1.0
	 *
	 * @param string $which The location of the pagination: Either 'top' or 'bottom'.
	 */
	protected function pagination( $which ) {
		if ( empty( $this->_pagination_args ) ) {
			return;
		}

		$total_items     = $this->_pagination_args['total_items'];
		$total_pages     = $this->_pagination_args['total_pages'];
		$infinite_scroll = false;
		if ( isset( $this->_pagination_args['infinite_scroll'] ) ) {
			$infinite_scroll = $this->_pagination_args['infinite_scroll'];
		}

		if ( 'top' === $which && $total_pages > 1 ) {
			$this->screen->render_screen_reader_content( 'heading_pagination' );
		}

		$output = '<span class="displaying-num">' . sprintf(
			/* translators: %s: Number of items. */
			_n( '%s item', '%s items', $total_items ),
			number_format_i18n( $total_items )
		) . '</span>';

		$current              = $this->get_pagenum();
		$removable_query_args = wp_removable_query_args();

		$current_url = set_url_scheme( 'http://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );

		$current_url = remove_query_arg( $removable_query_args, $current_url );

		$page_links = array();

		$total_pages_before = '<span class="paging-input">';
		$total_pages_after  = '</span></span>';

		$disable_first = false;
		$disable_last  = false;
		$disable_prev  = false;
		$disable_next  = false;

		if ( 1 === $current ) {
			$disable_first = true;
			$disable_prev  = true;
		}
		if ( $total_pages === $current ) {
			$disable_last = true;
			$disable_next = true;
		}

		if ( $disable_first ) {
			$page_links[] = '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&laquo;</span>';
		} else {
			$page_links[] = sprintf(
				"<a class='first-page button' href='%s'>" .
				"<span class='screen-reader-text'>%s</span>" .
				"<span aria-hidden='true'>%s</span>" .
				'</a>',
				esc_url( remove_query_arg( 'cpaged', $current_url ) ),
				/* translators: Hidden accessibility text. */
				__( 'First page' ),
				'&laquo;'
			);
		}

		if ( $disable_prev ) {
			$page_links[] = '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&lsaquo;</span>';
		} else {
			$page_links[] = sprintf(
				"<a class='prev-page button' href='%s'>" .
				"<span class='screen-reader-text'>%s</span>" .
				"<span aria-hidden='true'>%s</span>" .
				'</a>',
				esc_url( add_query_arg( 'cpaged', max( 1, $current - 1 ), $current_url ) ),
				/* translators: Hidden accessibility text. */
				__( 'Previous page' ),
				'&lsaquo;'
			);
		}

		if ( 'bottom' === $which ) {
			$html_current_page  = $current;
			$total_pages_before = sprintf(
				'<span class="screen-reader-text">%s</span>' .
				'<span id="table-paging" class="paging-input">' .
				'<span class="tablenav-paging-text">',
				/* translators: Hidden accessibility text. */
				__( 'Current Page' )
			);
		} else {
			$html_current_page = sprintf(
				'<label for="current-page-selector" class="screen-reader-text">%s</label>' .
				"<input class='current-page' id='current-page-selector' type='text'
                    name='paged' value='%s' size='%d' aria-describedby='table-paging' />" .
				"<span class='tablenav-paging-text'>",
				/* translators: Hidden accessibility text. */
				__( 'Current Page' ),
				$current,
				strlen( $total_pages )
			);
		}

		$html_total_pages = sprintf( "<span class='total-pages'>%s</span>", number_format_i18n( $total_pages ) );

		$page_links[] = $total_pages_before . sprintf(
			/* translators: 1: Current page, 2: Total pages. */
			_x( '%1$s of %2$s', 'paging' ),
			$html_current_page,
			$html_total_pages
		) . $total_pages_after;

		if ( $disable_next ) {
			$page_links[] = '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&rsaquo;</span>';
		} else {
			$page_links[] = sprintf(
				"<a class='next-page button' href='%s'>" .
				"<span class='screen-reader-text'>%s</span>" .
				"<span aria-hidden='true'>%s</span>" .
				'</a>',
				esc_url( add_query_arg( 'cpaged', min( $total_pages, $current + 1 ), $current_url ) ),
				/* translators: Hidden accessibility text. */
				__( 'Next page' ),
				'&rsaquo;'
			);
		}

		if ( $disable_last ) {
			$page_links[] = '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&raquo;</span>';
		} else {
			$page_links[] = sprintf(
				"<a class='last-page button' href='%s'>" .
				"<span class='screen-reader-text'>%s</span>" .
				"<span aria-hidden='true'>%s</span>" .
				'</a>',
				esc_url( add_query_arg( 'cpaged', $total_pages, $current_url ) ),
				/* translators: Hidden accessibility text. */
				__( 'Last page' ),
				'&raquo;'
			);
		}

		$pagination_links_class = 'pagination-links';
		if ( ! empty( $infinite_scroll ) ) {
			$pagination_links_class .= ' hide-if-js';
		}
		$output .= "\n<span class='$pagination_links_class'>" . implode( "\n", $page_links ) . '</span>';

		if ( $total_pages ) {
			$page_class = $total_pages < 2 ? ' one-page' : '';
		} else {
			$page_class = ' no-pages';
		}
		$this->_pagination = "<div class='tablenav-pages{$page_class}'>$output</div>";

		echo $this->_pagination;
	}

	/**
	 * Gets a list of sortable columns for the list table.
	 *
	 * @since 3.1.0
	 *
	 * @return array Array of sortable columns.
	 */
	protected function get_sortable_columns() {
		$columns = array(
			'username'    => array(
				'login',
				false,
				esc_html__( 'Username', 'streamtube-core' ),
				esc_html__( 'Table ordered by Username.', 'streamtube-core' ),
				'asc'
			),
			'email'       => array(
				'email',
				false,
				esc_html__( 'E-mail', 'streamtube-core' ),
				esc_html__( 'Table ordered by E-mail.', 'streamtube-core' )
			),
			'name'        => array(
				'display_name',
				false,
				esc_html__( 'Name', 'streamtube-core' ),
				esc_html__( 'Table ordered by Displayed Name.', 'streamtube-core' )
			),
			'posts'       => array(
				'post_count',
				false,
				esc_html__( 'Posts', 'streamtube-core' ),
				esc_html__( 'Table ordered by post count.', 'streamtube-core' )
			),
			'video_count' => array(
				'video_count',
				false,
				esc_html__( 'Videos', 'streamtube-core' ),
				esc_html__( 'Table ordered by video count.', 'streamtube-core' )
			)
		);

		return apply_filters( "users_list_table/{$this->screen->id}/sortable_columns", $columns );
	}

	/**
	 * Define which columns are hidden
	 *
	 * @return array
	 */
	public function get_hidden_columns() {
		$hidden_columns = (array) get_user_option( 'manage' . $this->screen->id . 'columnshidden' );

		return is_array( $hidden_columns ) ? $hidden_columns : array();
	}

	/**
	 * Get number of items to display on a single page.
	 *
	 * @param string $option Optional. Option name. Default ''.
	 * @param int    $default_value Optional. Number of items to return if no user preference is set. Default 20.
	 * @return int Number of items to display on a single page.
	 */
	protected function get_items_per_page( $option = '', $default_value = 20 ) {

		$per_page = (int) get_user_option( 'edit_' . $this->screen->id . '_per_page' );

		return $per_page !== 0 ? absint( $per_page ) : $default_value;
	}

	/**
	 * Returns an associative array listing all the views that can be used
	 * with this table.
	 *
	 * Provides a list of roles and user count for that role for easy
	 * filtering of the user table.
	 *
	 * @since 3.1.0
	 *
	 * @global string $role
	 *
	 * @return string[] An array of HTML links keyed by their view.
	 */
	protected function get_views() {
		global $role, $wp;

		$wp_roles = wp_roles();

		$count_users = ! wp_is_large_user_count();

		$url = home_url( $wp->request );

		$role_links  = array();
		$avail_roles = array();
		$all_text    = __( 'All' );

		if ( $count_users ) {
			if ( $this->is_site_users ) {
				switch_to_blog( $this->site_id );
				$users_of_blog = count_users( 'time', $this->site_id );
				restore_current_blog();
			} else {
				$users_of_blog = count_users();
			}

			$total_users = $users_of_blog['total_users'];
			$avail_roles =& $users_of_blog['avail_roles'];
			unset( $users_of_blog );

			$all_text = sprintf(
				/* translators: %s: Number of users. */
				_nx(
					'All <span class="count">(%s)</span>',
					'All <span class="count">(%s)</span>',
					$total_users,
					'users'
				),
				number_format_i18n( $total_users )
			);
		}

		$role_links['all'] = array(
			'url'     => $url,
			'label'   => $all_text,
			'current' => empty( $role ),
		);

		foreach ( $wp_roles->get_names() as $this_role => $name ) {
			if ( $count_users && ! isset( $avail_roles[ $this_role ] ) ) {
				continue;
			}

			$name = translate_user_role( $name );
			if ( $count_users ) {
				$name = sprintf(
					/* translators: 1: User role name, 2: Number of users. */
					__( '%1$s <span class="count">(%2$s)</span>' ),
					$name,
					number_format_i18n( $avail_roles[ $this_role ] )
				);
			}

			$role_links[ $this_role ] = array(
				'url'     => esc_url( add_query_arg( 'role', $this_role, $url ) ),
				'label'   => $name,
				'current' => $this_role === $role,
			);
		}

		if ( ! empty( $avail_roles['none'] ) ) {

			$name = __( 'No role' );
			$name = sprintf(
				/* translators: 1: User role name, 2: Number of users. */
				__( '%1$s <span class="count">(%2$s)</span>' ),
				$name,
				number_format_i18n( $avail_roles['none'] )
			);

			$role_links['none'] = array(
				'url'     => esc_url( add_query_arg( 'role', 'none', $url ) ),
				'label'   => $name,
				'current' => 'none' === $role,
			);
		}

		return $this->get_views_links( $role_links );
	}

	public function get_bulk_actions() {

		$action = array();

		$action = array(
			'delete'       => esc_html__( 'Delete', 'streamtube-core' ),
			'verify'       => esc_html__( 'Verify', 'streamtube-core' ),
			'unverify'     => esc_html__( 'Unverify', 'streamtube-core' ),
			'deactivate'   => esc_html__( 'Deactivate', 'streamtube-core' ),
			'undeactivate' => esc_html__( 'Undeactivate', 'streamtube-core' )
		);

		return apply_filters( 'manage_frontend_users_list_table_actions', $action );
	}

	public function process_bulk_action() {

		$action = $this->current_action();

		$users = isset( $_REQUEST['users'] ) ? wp_unslash( $_REQUEST['users'] ) : false;

		if ( ! $users ) {
			return;
		}

		$redirect_uri = esc_url( remove_query_arg( array_keys( $_REQUEST ), $_SERVER['REQUEST_URI'] ) );

		do_action( 'manage_frontend_users_list_table_do_' . $action, $users, $redirect_uri );

		do_action( 'manage_frontend_users_list_table_do_action', $action, $users, $redirect_uri );
	}

	/**
	 * Prepares the users list for display.
	 *
	 * @since 3.1.0
	 *
	 * @global string $role
	 * @global string $usersearch
	 */
	public function prepare_items() {
		global $role, $usersearch;

		$usersearch = isset( $_REQUEST['s'] ) ? wp_unslash( trim( $_REQUEST['s'] ) ) : '';

		$role = isset( $_REQUEST['role'] ) ? $_REQUEST['role'] : '';

		//$per_page       = ( $this->is_site_users ) ? 'site_users_network_per_page' : 'users_per_page';
		$users_per_page = $this->get_items_per_page();

		$paged = $this->get_pagenum();

		$this->process_bulk_action();

		if ( 'none' === $role ) {
			$args = array(
				'number'  => $users_per_page,
				'offset'  => ( $paged - 1 ) * $users_per_page,
				'include' => wp_get_users_with_no_role( $this->site_id ),
				'search'  => $usersearch,
				'fields'  => 'all_with_meta',
			);
		} else {
			$args = array(
				'number' => $users_per_page,
				'offset' => ( $paged - 1 ) * $users_per_page,
				'role'   => $role,
				'search' => $usersearch,
				'fields' => 'all_with_meta',
			);
		}

		if ( '' !== $args['search'] ) {
			$args['search'] = '*' . $args['search'] . '*';
		}

		if ( $this->is_site_users ) {
			$args['blog_id'] = $this->site_id;
		}

		if ( isset( $_REQUEST['orderby'] ) ) {
			$args['orderby'] = $_REQUEST['orderby'];
		}

		if ( isset( $_REQUEST['order'] ) ) {
			$args['order'] = $_REQUEST['order'];
		}

		/**
		 * Filters the query arguments used to retrieve users for the current users list table.
		 *
		 * @since 4.4.0
		 *
		 * @param array $args Arguments passed to WP_User_Query to retrieve items for the current
		 *                    users list table.
		 */
		$args = apply_filters( 'users_list_table_query_args', $args );

		// Query the user IDs for this page.
		$wp_user_search = new WP_User_Query( $args );

		$this->items = $wp_user_search->get_results();

		$this->set_pagination_args(
			array(
				'total_items' => $wp_user_search->get_total(),
				'per_page'    => $users_per_page,
			)
		);
	}

	/**
	 *
	 * Remove unnecessary action for dashboard list
	 * 
	 */
	public static function filter_user_row_actions( $actions = array(), $user_object = null ) {

		if ( ! is_admin() ) {

			$url = remove_query_arg( array_keys( $_REQUEST ), $_SERVER['REQUEST_URI'] );

			$actions = array(
				'view' => isset( $actions['view'] ) ? $actions['view'] : ''
			);

			if ( $user_object instanceof WP_User ) {
				if ( current_user_can( 'delete_user', $user_object->ID ) && $user_object->ID !== get_current_user_id() ) {
					$actions['delete'] = sprintf(
						'<a href="%s">%s</a>',
						esc_url(
							wp_nonce_url(
								add_query_arg( array(
									'action'  => 'delete',
									'user_id' => $user_object->ID
								), $url ),
								'delete-' . $user_object->ID
							)
						),
						esc_html__( 'Delete', 'streamtube-core' )
					);
				}

				if ( current_user_can( 'promote_users', $user_object->ID ) ) {
					$actions['roles'] = sprintf(
						'<a href="%s">%s</a>',
						esc_url(
							wp_nonce_url(
								add_query_arg( array(
									'action'  => 'roles',
									'user_id' => $user_object->ID
								), $url ),
								'roles-' . $user_object->ID
							)
						),
						esc_html__( 'Roles', 'streamtube-core' )
					);

					$actions['capabilities'] = sprintf(
						'<a href="%s">%s</a>',
						esc_url(
							wp_nonce_url(
								add_query_arg( array(
									'action'  => 'capabilities',
									'user_id' => $user_object->ID
								), $url ),
								'capabilities-' . $user_object->ID
							)
						),
						esc_html__( 'Capabilities', 'streamtube-core' )
					);
				}

				if ( current_user_can( 'edit_user', $user_object->ID ) ) {

					$actions['change-password'] = sprintf(
						'<a href="%s">%s</a>',
						esc_url(
							wp_nonce_url(
								add_query_arg( array(
									'action'  => 'change-password',
									'user_id' => $user_object->ID
								), $url ),
								'change-password-' . $user_object->ID
							)
						),
						esc_html__( 'Change Password', 'streamtube-core' )
					);

					if ( user_can( $user_object->ID, get_post_type_object( 'video' )->cap->edit_posts ) ) {
						$actions['storage-quota'] = sprintf(
							'<a href="%s">%s</a>',
							esc_url(
								wp_nonce_url(
									add_query_arg( array(
										'action'  => 'storage-quota',
										'user_id' => $user_object->ID
									), $url ),
									'storage-quota-' . $user_object->ID
								)
							),
							esc_html__( 'Storage Quota', 'streamtube-core' )
						);
					}
				}
			}

		}

		return $actions;
	}
}