(function($) {
    "use strict";

	$.isJson = function(str) {
		try {
		JSON.parse(str);
		} catch (e) {
			return false;
		}
		return true;
	};  

    $.fn.playlistBlock = function() {

        var block = $(this);

        if( block.length != 0 ){

        	var embedSize = block.find( '.ratio' ).height();

        	var videoList = block.find( '.post-grid' );

        	videoList.css( 'height', embedSize ).removeClass( 'd-none' );

        	var embedReloader = '<div class="embed-reloader bg-black">';
        		embedReloader += '<div class="position-absolute top-50 start-50 translate-middle">';
        			embedReloader += $.getSpinner();
        		embedReloader += '</div>';	
        	embedReloader += '</div>';

        	block.find( '.post-permalink' ).on( 'click', function(e){
        		e.preventDefault();

        		if( block.find( '.embed-reloader' ).length == 0 ){
        			block.find( '.embed-wrap .ratio' ).append( embedReloader );
        		}
        		else{
        			block.find( '.embed-reloader' ).show();
        		}

        		var postItem = $(this).closest( '.post-item' );
        		var postGrid = $(this).closest( '.post-grid' );

        		postGrid.find( '.post-item' ).removeClass( 'active' );
        		postItem.addClass( 'active' );

        		var embedUrl = postItem.find( 'article' ).attr( 'data-embed-url' ) + '?autoplay=1&logo=0';

        		block.find( 'iframe' ).attr( 'src', embedUrl ).next().fadeOut( 'slow' );
        	} );
        }
    }

	/**
	 * Generate a secure random password
	 * @param {number} length - Desired length of the password
	 * @returns {string} - Securely generated password
	 */
	$.generateSecurePassword = function( length = 12 ) {
	    const charset = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*()_+[]{}|;:,.<>?";
	    let password = "";
	    
	    // Use crypto for secure random generation
	    const crypto = window.crypto || window.msCrypto; // For compatibility with older browsers
	    const randomValues = new Uint32Array(length);
	    crypto.getRandomValues(randomValues);

	    for (let i = 0; i < length; i++) {
	        const randomIndex = randomValues[i] % charset.length;
	        password += charset[randomIndex];
	    }
	    
	    return password;
	} 

	/**
	 * Generate a unique ID for a file based on its properties.
	 *
	 * @param {File} file - The file object.
	 * @returns {string} - A unique identifier for the file.
	 */
	$.generateFileId = function(file) {
	    // Combine file name, size, and last modified date to generate a unique key
	    const uniqueKey = `${file.name}-${file.size}-${file.lastModified}`;

	    // Convert the string to a Base64-compatible format
	    const utf8Encoder = new TextEncoder();
	    const utf8Array = utf8Encoder.encode(uniqueKey);

	    // Convert to Base64 string
	    const base64String = window.btoa(String.fromCharCode(...utf8Array));

	    // Remove non-alphanumeric characters to keep it clean
	    return base64String.replace(/[^a-zA-Z0-9]/g, '');
	};

	/**
	 * Generate a spinner HTML element.
	 * 
	 * @param {boolean} $wrap - Whether to wrap the spinner in a container (default: true).
	 * @param {string} $text - The spinner text color (default: 'secondary').
	 * @param {string} $spinner - The spinner style ('border' or 'grow', default: 'border').
	 * @param {string} $message - An optional message to display with the spinner.
	 * @return {string} The generated spinner HTML.
	 * 
	 * @since 1.0.0
	 */
	$.getSpinner = function($wrap = true, $text = 'secondary', $spinner = 'border', $message = '') {
	    let output = '';

	    if( ! $spinner ){
	    	$spinner = 'border';
	    }

	    // Ensure default values
	    $text = $text || 'secondary';

	    if ($wrap) {
	        output += '<div class="spinner-wrap d-flex align-items-center justify-content-center p-3">';
	    }

	    output += `
	        <div class="spinner spinner-${$spinner} text-${$text}" role="status">
	            <span class="visually-hidden">Loading...</span>
	        </div>
	    `;

	    if ($message) {
	        output += $message;
	    }

	    if ($wrap) {
	        output += '</div>';
	    }

	    return output;
	};

	$.playSound = function( type = 'success', loop = false ){

		var audioUrl = '';

		if( type == 'success' ){
			audioUrl = streamtube.sound.success;
		}

		if( type == 'warning' ){
			audioUrl = streamtube.sound.warning;
		}		

		if( ! audioUrl ){
			return;
		}

        const audio 	= document.createElement('audio');
        audio.src 		= audioUrl;
        audio.loop 		= loop;
        audio.play();
	}

	$.getErrorMessages = function( $wpErrors ){

		if ( typeof $wpErrors === 'object') {
			return $wpErrors.map(item => `[${item.code}] ${item.message}`).join('<br/>');
		}
		return $wpErrors;
	}

	/**
	 * Show toast
	 * @param string message
	 * @param string type
	 * @since 1.0.0
	 */
	$.showToast = function( message, type = true, delay = 2000, custom_icon = '', uniqueid = '' ){

		var output = '';
		var icon = '';

		switch( type ) {
			case 'success':
				icon = 'icon-ok-circled2';
			break;

			case 'danger':
			case 'warning':
				icon = 'icon-cancel-circled';
			break;

			case 'info':
				icon = 'icon-info-circled';
			break;			

			default:
				icon = 'icon-help-circled';
			break;
		}

		if( custom_icon != "" ){
			icon = custom_icon;
		}

		if ( typeof message === 'object') {
			message = $.getErrorMessages( message );
		}

		output += '<div class="toast-wrap position-fixed bottom-0 start-50 translate-middle-x p-3">';
			output += '<div id="notify-toast" class="toast fade hide border-0 p-1 align-items-center text-white bg-'+ type +'" role="alert" aria-live="assertive" aria-atomic="true">';
				output += '<div class="d-flex">';
					output += '<div class="toast-body">';
						output += '<div class="d-flex align-items-center">';
							output += '<span class="h3 me-2 '+icon+'"></span>';
							output += '<p class="m-0">'+ message +'</p>';
						output += '</div>';
					output += '</div><!--.toast-body-->';

					output += '<button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>';
				output += '</div><!--.d-flex-->';
			output += '</div><!--.toast-->';
		output += '</div><!--.toast-wrap-->';

		if( $('#notify-toast').length != 0 ){
			$('#notify-toast').remove();
		}

		$( 'body' ).append( output );

		if( delay == 2000 ){
			delay = streamtube.toast.delay;
		}

        $( document.body ).trigger( 
            'streamtube_toast_show',
            [ message, type, delay, custom_icon, uniqueid ]
        );		

		$('#notify-toast').toast({
			delay: delay
		});

		$('#notify-toast').toast( 'show' );
	}

	$.getEditorContentCSS =  function(){
		var themeMode 	= $( 'html' ).attr( 'data-theme' );

		return themeMode == 'dark' ? streamtube.dark_editor_css : streamtube.light_editor_css;
	}

	/**
	 * WP Editor init
	 */
	$.editorInit = function( fieldId = '', content_css ){

		wp.editor.initialize( fieldId, {
			tinymce : {
				toolbar1 	: streamtube.editor_toolbar.toString(),
				content_css : $.getEditorContentCSS()
			}
		} );
	}

	/**
	 *
	 * Remove editor
	 * 
	 */
	$.editorRemove = function( fieldId = ''){
		wp.editor.remove( fieldId );
	}

	/**
	 * Get ajaxUrl
	 * 
	 * @return string
	 *
	 * @since 1.0.0
	 * 
	 */
	$.getAjaxRequestUrl = function(){
		return streamtube.ajaxUrl;
	}

	/**
	 * Get rest URL
	 * @return string
	 *
	 * @since 1.0.6
	 * 
	 */
	$.getRestUrl = function( $endpoint = '' ){
		return streamtube.rest_url + $endpoint;
	}

	/**
	 *
	 * The main ajaxFormRequest handler
	 * @since 1.0.0
	 * 
	 */
	$.ajaxFormRequest = function( event ) {
		event.preventDefault();

		var form 		= $(this);
		var formId 		= form.attr( 'id' );
		var method 		= form.attr( 'method' );

		if( ! method ){
			method = 'POST';
		}

		var formData 	= new FormData(form[0]);
		var button 		= $(document.activeElement);

		if(  button.attr( 'type' ) == 'submit' && button.val() == '' ){
			button = $(event.originalEvent.submitter);
		}

		var action 		= formData.get( 'action' );

		var requestUrl 	= formData.get( 'request_url' );

		if( requestUrl === '#' ){
			return false;
		}

		if( ! requestUrl ){
			// Should use in admin-ajax
			
			var formNonce = formData.get('_wpnonce');

			if( ! formNonce ){
				formData.append( '_wpnonce', streamtube._wpnonce );
			}
			
			requestUrl 	= $.getAjaxRequestUrl();
		}

		var nonce 		= formData.get( 'nonce' );

		if( ! nonce ){
			nonce = streamtube.nonce;
		}

		var button = form.find( 'button[type=submit]' );

		if( button.length === 0 ){
			button = $( 'button[form='+formId+']' );
		}

		formData.append( 'submit', button.val() );

		var jqxhr = $.ajax({
			url 			: requestUrl,
			data 			: formData,
			processData 	: false,
			contentType 	: false,
			type 			: method,
          	xhr 			: function() {
                var jqXHR = new XMLHttpRequest();

                jqXHR.upload.addEventListener( 'progress', function( event ) {
                	if( event.lengthComputable ){
                		var percentComplete = Math.round( (event.loaded / event.total) * 100 );
                		form.find( '.progress-bar' )
                		.css( 'width', percentComplete + '%' )
                		.attr( 'aria-valuenow', percentComplete )
                		.html( percentComplete  + '%' )
                	}

                }, false );

                return jqXHR;
            },			
			beforeSend: function( jqXHR ) {

				jqXHR.setRequestHeader( 'X-WP-Nonce', nonce );

				$( document.body ).trigger( 'streamtube_ajax_form_before_send', [ form, formData, jqXHR ] );

				$( document.body ).trigger( action + '_before_send', [ form, formData, jqXHR ] );

				button
				.addClass( 'disabled' )
				.attr( 'disabled', 'disabled' )
				.append( $.getSpinner(false) );

				if( formData.get('has_spinner') && streamtube.ajax_spinner ){
					$.displaySpinner();
				}
			}
		});

		jqxhr.fail( function( jqXHR, textStatus, errorThrown ){

			$( document.body ).trigger( 'streamtube_ajax_form_failed', [ errorThrown, jqXHR, form, formData ] );

			$( document.body ).trigger( action + '_failed', [ errorThrown, jqXHR, form, formData ] );

			let errorMessage = jqXHR.responseJSON?.message || errorThrown; 

			if( ! errorMessage && jqXHR.responseJSON?.data ){
				errorMessage = jqXHR.responseJSON.data;
			}

			button
			.removeClass( 'disabled' )
			.removeAttr( 'disabled' )
			.find( '.spinner-border' ).remove();		

			$.showToast( errorMessage, 'danger' );	

		})

		jqxhr.done( function( data, textStatus, jqXHR ){
			$( document.body ).trigger( 'streamtube_ajax_form_done', [ data, textStatus, jqXHR, formData, form ] );
			$( document.body ).trigger( action, [ data, textStatus, jqXHR, formData, form ] );

			if( data.success == false ){
				$.showToast( data.data, 'danger' );
			}else{
				if( data.data?.message ){
					$.showToast( data.data.message, 'success' );
				}
			}
		})

		jqxhr.always( function( jqXHR, textStatus ){
			$( document.body ).trigger( 'streamtube_ajax_form_completed', [ jqXHR, textStatus, formData, form ] );
			$( document.body ).trigger( action + '_complete', [ jqXHR, textStatus, formData, form ] );

			button
			.removeClass( 'disabled' )
			.removeAttr( 'disabled' )
			.find( '.spinner-border' ).remove();

			$.removeSpinner();
		});
	}

	/**
	 *
	 * Call AJAX on Element's event
	 *
	 * @since  1.0.0
	 * 
	 */
	$.ajaxElementOnEventRequest = function( event = null ){

		if( event !== null ){
			event.preventDefault();	
		}

        var element =   $(this);

        if( element.hasClass( 'waiting' ) ){
        	return false;
        }

        var	endpoint 	=   element.attr( 'data-endpoint' );
        var	data		=   element.attr( 'data-params' );
        var action		=	element.attr( 'data-action' );
        var method		=	element.attr( 'data-method' );
        var nonce		=	element.attr( 'data-nonce' );
        var isConfirm	=	element.attr( 'data-confirm' );
        var isPrompt 	=	element.attr( 'data-prompt' );
        var promptValue =	element.attr( 'data-prompt-value' );

        if( isConfirm && confirm( isConfirm.replace(/\\n/g, '\n') ) !== true ){
        	return;
        }

        if( isPrompt && prompt( isPrompt ) !== promptValue ){
        	return;
        }

        var formData = new FormData();

        formData.append( 'action', action );

        if( method === undefined ){
        	method = 'POST';
        }

        if( ! endpoint ){
        	endpoint = $.getAjaxRequestUrl();
	        if( nonce ){
	        	formData.append( '_wpnonce', nonce );	
	        }else{
	        	formData.append( '_wpnonce', streamtube._wpnonce );	
	        }     

	        formData.append( 'data', data );   	
        }else{
        	// Rest
        	if( $.isJson( data ) ){
        		const jsonData = $.parseJSON( data );

        		if (Array.isArray(jsonData)) { 
					for (const item of jsonData) { 
						for (const key in item) {
							if (item.hasOwnProperty(key)) {
							formData.append(key, item[key]);
							}
						}
					}
				}else{
					for (const key in jsonData) {
						if (jsonData.hasOwnProperty(key)) {
							formData.append(key, jsonData[key]);
						}
					}					
				}
        	}
        	else{
        		formData.append( 'data', data );
        	}
        }    

        var jqxhr = $.ajax({
			url: endpoint,
			data: formData,
			processData: false,
			contentType: false,
			type: method,
			beforeSend: function( jqXHR ) {

				jqXHR.setRequestHeader( 'X-WP-Nonce', streamtube.nonce );

				var inputs = ['text', 'checkbox', 'radio', 'email', 'search', 'password'];

				var elementType = element.attr( 'type' );

	        	element.addClass( 'active waiting' );

				element
				.addClass( 'disabled' )
				.attr( 'disabled', 'disabled' );

				if( elementType && $.inArray( elementType, inputs ) == -1 ){
					element.append( $.getSpinner(false) );
				}

				$( document.body ).trigger( 'streamtube_ajax_element_before_send', [ element, formData ] );

	        	$( document.body ).trigger( action + '_before_send', [ element, formData ] );
			}
        })

        .done( function( response, textStatus, jqXHR ){

        	element
        	.removeClass( 'active waiting disabled' )
        	.removeAttr( 'disabled' );

			if( response.success == false ){
				$.showToast( response.data, 'danger' );
			}else{
				if( response.data?.message ){
					$.showToast( response.data.message, 'success' );
				}
			}        	

        	$( document.body ).trigger( 'streamtube_ajax_element_done', [ response, textStatus, jqXHR, element ] );
        	/**
        	 *
        	 * Action trigger
        	 *
        	 * @since 1.0.0
        	 * 
        	 */
            $( document.body ).trigger( action, [ response, textStatus, jqXHR, element ] );
        })
        
		.fail(function(jqXHR, textStatus, errorThrown) {

			const errorMessage = jqXHR.responseJSON?.message || errorThrown; 
			
			$.showToast( errorMessage, 'danger' );

			$(document.body).trigger('streamtube_ajax_element_failed', [jqXHR, textStatus, errorThrown, element]);

			$( document.body ).trigger( action + '_failed', [ errorThrown, jqXHR, formData, element ] );
		})

        .always( function( jqXHR, textStatus ){

        	$( document.body ).trigger( 'streamtube_ajax_element_completed', [ jqXHR, textStatus, element ] );
        	element.find( '.spinner-border' ).remove();
        } );
	}

	/**
	 *
	 * Upload Big File Size
	 *
	 * @since 1.0.0
	 * 
	 */
	$.uploadBigFile = function( file, sliceSize = 10240, form = null ){
		var size = parseInt( file.size );

		sliceSize = parseInt( sliceSize );

		if( ! sliceSize ){
			sliceSize = 10240;
		}

		if( sliceSize >= size ){
			sliceSize = size/2;
		}

		//sliceSize = Math.min( sliceSize, size );

		var start = 0;

		// First chunk size.
		var end = sliceSize;

		// First chunk index
		var chunkIndex = 0;

		$.uploadChunk( file, start, end, chunkIndex, sliceSize, form );
	}	

	/**
	 *
	 * Upload Chunk
	 *
	 * @since 1.0.0
	 * 
	 */
	$.uploadChunk = function( file, start, end, chunkIndex, sliceSize, form ){

	    var formData 	= new FormData();

	    const fileId 	= $.generateFileId(file);
	    const progress = $(`div[data-file-id="${fileId}"]`);

	    var jqXHR 		= new XMLHttpRequest();

	    var fileSize	= file.size;

	    var chunkTotal	= Math.ceil( fileSize/sliceSize );
	    
	    formData.append( 'action' , 'upload_video_chunk');

	    formData.append( 'size' , fileSize );
	    formData.append( 'type' , file.type );

	    formData.append( 'name' , file.name );

	    formData.append( 'chunks' , chunkTotal );
	    formData.append( 'chunk' , chunkIndex );

	    formData.append( 'async-upload' , file.slice( start, end ) );

	    // Regular upload without rest.
	    formData.append( '_wpnonce' , streamtube.media_form );
	 

	    if (fileSize - end < 0) {
	        end = fileSize;
	    }		

	    if (end < fileSize) {
	        jqXHR.onreadystatechange = function() {
	            if ( jqXHR.readyState == XMLHttpRequest.DONE ) {

					if ( jqXHR.status === 200 ) {
		                chunkIndex++;
		                $.uploadChunk( file, start + sliceSize, start + (sliceSize * 2), chunkIndex, sliceSize, form );
					}else{
		                // Handle errors
		                
		                let errorResponse = jqXHR.responseText;

		                if( errorResponse ){
		                	errorResponse = $.parseJSON( errorResponse );
		                }

		                if ( errorResponse && errorResponse?.data) {
		                    $(document.body).trigger('upload_video', [ errorResponse, jqXHR.statusText, jqXHR, file, form]);
		                } else {
		                    $.showToast( jqXHR.statusText || 'An error occurred', 'danger');
		                }

		                return;
					}
	            }
	        }
	    }

		jqXHR.upload.onprogress = function( event ) { 

		    var percentComplete = parseInt( ( chunkIndex * 100)/( chunkTotal - 1 ) );

    		progress.find( '.progress-bar' )
    		.css( 'width', percentComplete + '%' )
    		.attr( 'aria-valuenow', percentComplete )
    		.html( percentComplete  + '%' );
		}

		jqXHR.onload = function() {
			if( jqXHR.readyState === 4 && jqXHR.responseText ){

				var response = $.parseJSON( jqXHR.responseText );

	            if ( jqXHR.status !== 200) {
	                if (response?.data) {
	                    $(document.body).trigger('upload_video', [ response, jqXHR.statusText, jqXHR, file, form]);
	                } else {
	                    $.showToast( jqXHR.statusText || 'An error occurred', 'danger');
	                }
	                return;
	            }

				if( response.data.id ){
					$.post( $.getAjaxRequestUrl(), {
						action : 'upload_video_chunks',
						nonce : streamtube.chunks_nonce,
						attachment_id : response.data.id,
						post_status: form.find( 'select[name=post_status]' ).val(),
						'_wpnonce' : streamtube._wpnonce
					} )
					.done( function( data, textStatus, jqXHR ){
						$( document.body ).trigger( 'upload_video', [ data, textStatus, jqXHR, file, form ] );
					} );
				}
			}
		}


	    jqXHR.open( 'POST', $.getAjaxRequestUrl(), true );

	    jqXHR.setRequestHeader( 'X-WP-Nonce', streamtube.nonce );

	    jqXHR.send( formData );
	}

	$.uploadFile = function( file, form ){

	    var formData 	= new FormData();

	    const fileId 	= $.generateFileId(file);
	    const progress = $(`div[data-file-id="${fileId}"]`);;

	    const action 	= form.find( 'input[name=action]' ).val();
	    var jqXHR 		= new XMLHttpRequest();

		formData.append( 'file', file );

		let postStatus = form.find( 'select[name=post_status]' );

		if( postStatus.length !== 0 ){
			formData.append( 'post_status', postStatus.val() );
		}

		var jqxhr = $.ajax({
			url 			: $.getRestUrl( '/upload' ),
			data 			: formData,
			processData 	: false,
			contentType 	: false,
			type 			: 'POST',
          	xhr 			: function() {
                var jqXHR = new XMLHttpRequest();

                jqXHR.upload.addEventListener( 'progress', function( event ) {
                	if( event.lengthComputable ){
                		var percentComplete = Math.round( (event.loaded / event.total) * 100 );
                		progress.find( '.progress-bar' )
                		.css( 'width', percentComplete + '%' )
                		.attr( 'aria-valuenow', percentComplete )
                		.html( percentComplete  + '%' )
                	}

                }, false );

                return jqXHR;
            },
			beforeSend: function( jqXHR ) {
				jqXHR.setRequestHeader( 'X-WP-Nonce', streamtube.nonce );
			}            
		});		

		jqxhr.fail( function( jqXHR, textStatus, errorThrown ){
			if( jqXHR?.responseJSON?.data ){
				$(document.body).trigger(action, [jqXHR.responseJSON, textStatus, jqXHR, file, form]);
			}else{
				const errorMessage = jqXHR?.responseJSON?.data ?? errorThrown ?? textStatus;
				$.showToast( errorMessage, 'danger');
			}
		});

		jqxhr.done( function( data, textStatus, jqXHR ){
			$( document.body ).trigger( 'streamtube_ajax_form_done', [ data, textStatus, jqXHR, formData, form ] );
			$( document.body ).trigger( action, [ data, textStatus, jqXHR, file, form ] );			
		});
	}

	window.streamTubeUploadFile = $.uploadFile;

	$.getCartTotal = function(){
		const url = $.getAjaxRequestUrl() + '?action=get_cart_total&_wpnonce=' + streamtube._wpnonce;
		$.get( url , function( data ){
			if( data.success ){

				var output = '<span class="cart-total bg-danger text-white small p-1 px-2 ms-2 rounded">'+ data.data.item_count_text + ' (' +  data.data.total + ')</span>';

				var elm = $( '.header-user__dropdown #nav-cart a' );

				elm.find( '.cart-total' ).remove();
				elm.append( output );

				if( data.data.item_count > 0 ){
					$( '.cart-count' ).html( data.data.item_count ).removeClass( 'd-none' );
				}else{
					$( '.cart-count' ).html('').addClass( 'd-none' );
				}

				$( document.body ).trigger( 'get_cart_total_loaded', [ data ] );
			}
		} );
	}

	$.displaySpinner = function(){
		var spinner = '<div class="body-overlay-spinner bg-black opacity-25 position-fixed w-100 h-100 top-0 start-0 z-3">';
			spinner += '<div class="position-absolute top-50 start-50 translate-middle">';
				spinner += '<div class="spinner-border text-success" role="status">';
					spinner += '<span class="visually-hidden">Loading...</span>';
				spinner += '</div>';
			spinner += '</div>';			
		spinner += '</div>';
		$( 'body' ).addClass( 'body-spinner' ).append( spinner );
	}

	$.removeSpinner = function(){
		$( 'body' ).removeClass( 'body-spinner' ).find( '.body-overlay-spinner' ).remove();
	}

})(jQuery);