<?php
/**
 * Define the Patreon functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.1
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class StreamTube_Core_Patreon {

	public $metabox = false;

	/**
	 * Constructor for the Streamtube_Core_Patreon class.
	 *
	 * This constructor includes the 'class-streamtube-core-patreon-metabox.php' file
	 * which is required for the functionality of the Streamtube Core Patreon plugin.
	 *
	 * @since 1.0.0
	 */
	function __construct() {
		require_once trailingslashit( plugin_dir_path( __FILE__ ) ) . 'class-streamtube-core-patreon-metabox.php';

		if ( class_exists( 'StreamTube_Core_Patreon_Metabox' ) ) {
			$this->metabox = new StreamTube_Core_Patreon_Metabox();
		}
	}

	/**
	 * Renders a block of content wrapped in a div with specific CSS classes.
	 *
	 * This method takes the provided content and wraps it in a div element with
	 * the classes "require-membership", "require-pmpro-levels", and "require-patreon".
	 * Additionally, it positions the content absolutely in the center of the parent
	 * element using Bootstrap utility classes.
	 *
	 * @param string $content The content to be wrapped and rendered.
	 * @return string The HTML string with the wrapped content.
	 */
	public function render_block( $content ) {
		$output = '<div class="require-membership require-patreon">'
			. '<div class="top-50 start-50 translate-middle position-absolute">'
			. $content
			. '</div>'
			. '</div>';

		return $output;
	}

	/**
	 * Check if given post is protected
	 * 
	 */
	private function is_post_protected( $post_id = 0 ) {

		$post_id = absint( $post_id );

		if (
			! class_exists( 'Patreon_Frontend' ) ||
			! method_exists( 'Patreon_Frontend', 'protectContentFromUsers' ) ||
			! class_exists( 'Patreon_Wordpress' ) ||
			! is_callable( array( 'Patreon_Wordpress', 'lock_or_not' ) ) ||
			! is_callable( array( 'Patreon_Frontend', 'protectContentFromUsers' ) )
		) {
			return new WP_Error(
				'no_handler',
				esc_html__( 'Handler was not found.', 'streamtube-core' )
			);
		}

		$maybe_blocked = Patreon_Wordpress::lock_or_not( $post_id );

		if ( isset( $maybe_blocked['lock'] ) && $maybe_blocked['lock'] ) {
			return Patreon_Frontend::protectContentFromUsers( '', $post_id );
		}

		return $post_id;
	}

	/**
	 * Filters the player output to protect content using Patreon.
	 *
	 * This function checks if the `Patreon_Frontend` class and its method `protectContentFromUsers` exist.
	 * If they do, it uses the method to protect the player content from users.
	 * Finally, it renders the block with the player content.
	 *
	 * @param string $player The player content to be filtered.
	 * @param array $setup Optional. An array of setup parameters. Default is an empty array.
	 * @return string The filtered player content.
	 */
	public function filter_player_output( $player, $setup = array() ) {

		$maybe_protected_content = $this->is_post_protected( $setup['mediaid'] );

		if ( is_wp_error( $maybe_protected_content ) || is_int( $maybe_protected_content ) ) {
			return $player;
		}

		$this->unblock_post_content();

		$maybe_protected_content = $this->render_block( $maybe_protected_content );

		$maybe_protected_content .= sprintf(
			'<div class="player-poster bg-cover" style="background-image:url(%s)"></div>',
			$setup['poster2'] ? $setup['poster2'] : $setup['poster']
		);

		return $maybe_protected_content;
	}

	/**
	 * Filters the player embed output.
	 *
	 * This function is a callback for filtering the player embed output. It takes the embed code and arguments,
	 * and returns the filtered player output.
	 *
	 * @param string $embed The embed code for the player.
	 * @param array  $args  Additional arguments for the embed.
	 * @return string The filtered player output.
	 */
	public function filter_player_embed_output( $embed, $setup ) {
		return $this->filter_player_output( $embed, $setup );
	}

	/**
	 * Filters the download permission for a post.
	 *
	 * This function checks if the post is protected and if the current user has the necessary permissions
	 * to download the content. If the post is protected or the user has moderation permissions, it returns
	 * the original permission status. Otherwise, it denies the download permission.
	 *
	 * @param bool $can The current permission status for downloading the content.
	 * @return bool The modified permission status for downloading the content.
	 */
	public function filter_download_permission( $can ) {
		$maybe_protected_content = $this->is_post_protected( get_the_ID() );

		if (
			is_wp_error( $maybe_protected_content ) ||
			is_int( $maybe_protected_content ) ||
			Streamtube_Core_Permission::moderate_posts( get_current_user_id(), 'video' ) ) {
			return $can;
		}

		return false;
	}

	/**
	 * Filters the video REST source field.
	 *
	 * This function is used to protect the video source if it requires Patreon levels.
	 *
	 * @param string $source The current video source.
	 * @param array $post The post data array.
	 * @return string The filtered video source.
	 */
	public function filter_rest_video_source_field( $source = '', $post = array() ) {

		$maybe_protected_content = $this->is_post_protected( $post['id'] );

		if (
			is_wp_error( $maybe_protected_content ) ||
			is_int( $maybe_protected_content ) ||
			Streamtube_Core_Permission::moderate_posts( get_current_user_id(), 'video' ) ) {
			return $source;
		}

		$level = (int) get_post_meta( $post['id'], 'patreon-level', true );

		return array(
			'protected' => true,
			'service'   => 'patreon',
			'levels'    => array(
				$level
			)
		);
	}

	/**
	 * Unblock post content since we have blocked the player.
	 *
	 * This function removes content that has been blocked due to the player being blocked.
	 *
	 * @return void
	 */
	private function unblock_post_content() {
		if ( apply_filters( 'streamtube/core/patreon/unblock_post_content', true ) ) {
			add_filter( 'ptrn/override_content_filtering', function ($content, $post_id) {
				return array_merge(
					array( 'override' => true ),
					compact( 'content' )
				);
			}, 10, 2 );
		}
	}
	/**
	 * Filters the membership badge for a given post.
	 *
	 * This function checks if a post is protected and, if so, loads a premium badge template.
	 *
	 * @param string $badge   The current badge HTML. Default is an empty string.
	 * @param int    $post_id The ID of the post being checked.
	 * @return string The filtered badge HTML.
	 */
	public function add_membership_badge( $badge = '', $post_id = 0 ) {

		$maybe_protected_content = $this->is_post_protected( $post_id );

		if ( is_wp_error( $maybe_protected_content ) || is_int( $maybe_protected_content ) ) {
			return $badge;
		}

		ob_start();
		streamtube_core_load_template( 'misc/premium-badge.php', false, array(
			'service' => 'patreon'
		) );
		return ob_get_clean();
	}
	/**
	 * Adds Patreon support to specific post types.
	 *
	 * This method enables the 'patreon' feature for the specified post types,
	 * allowing integration with Patreon functionality. By default, it adds
	 * support for the 'page', 'post', and 'video' post types.
	 *
	 * @return void
	 */
	public function add_post_type_supports() {
		$builtin = array( 'page', 'post', 'video' );

		foreach ( $builtin as $post_type ) {
			add_post_type_support( $post_type, 'patreon' );
		}
	}

	/**
	 * Enqueues the necessary stylesheets for the Patreon integration.
	 */
	public function enqueue_scripts() {

		global $wp_query;

		if ( isset( $wp_query->query_vars['dashboard'] ) && defined( 'PATREON_PLUGIN_ASSETS' ) ) {
			wp_enqueue_script(
				'patreon-admin',
				PATREON_PLUGIN_ASSETS . '/js/admin.js',
				array( 'jquery' ),
				defined( 'PATREON_WORDPRESS_VERSION' ) ? PATREON_WORDPRESS_VERSION : '1.0.0',
				true
			);
		}

		if ( is_singular() || is_embed() || isset( $wp_query->query_vars['dashboard'] ) ) {
			wp_enqueue_style(
				'streamtube-core-patreon',
				untrailingslashit( plugin_dir_url( __FILE__ ) ) . '/assets/style.css',
				array(),
				filemtime( untrailingslashit( plugin_dir_path( __FILE__ ) ) . '/assets/style.css' )
			);
		}
	}
}