<?php
/**
 * Define the Sell Capability functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.2
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      2.2
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

Class StreamTube_Core_Woocommerce_Sell_Capability{ 

    /**
     * This feature can be enabled by addon
     * @return boolean
     */
    public static function is_active(){
        return apply_filters( 'streamtube/core/woocommerce/sell_capability', true );
    }

    /**
     *
     * Check if can update product capabilities
     * 
     */
    public static function can_update(){
        return apply_filters( 
            'streamtube/core/woocommerce/sell_capability/can_update', 
            current_user_can( 'administrator' ) 
        );
    }

    public function get_capabilities( $post_id = 0 ){
        $capabilities = (array)get_post_meta( $post_id, '_xcapabilities', true );

        return array_unique( array_filter( $capabilities ) );
    }    

    /**
     *
     * Add "Capabilities" tab
     * 
     * @param array $tabs
     * 
     */
    public function add_tab( $tabs ){
        if( self::can_update() ){
            $tabs['capabilities'] = array(
                'label'     => esc_html__( 'xCapabilities', 'woocreds-exchange' ),
                'target'    => 'xcapabilities',
                'class'     => array('show_if_simple')
            );
        }

        return $tabs;
    }    

    /**
     * Capability tab content
     */
    public function tab_content(){

        global $post;

        if( ! self::can_update() ){
            return;
        }

        $xcapabilities = $this->get_capabilities( $post->ID );
        ?>
        <div id="xcapabilities" class="panel woocommerce_options_panel">

            <p class="badge bg-danger text-white w-100">
                <?php
                esc_html_e( 'Purchasing this product will grant the selected capabilities to the buyer. DO NOT add any capabilities unless you fully understand their purpose.', 'streamtube-core' );
                ?>
            </p>

            <?php
            woocommerce_wp_text_input( array(
                'label'         =>  esc_html__( 'Search capabilities', 'streamtube-core' ),
                'id'            =>  'search-capabilities'
            ) );
            ?>

            <p>
                <?php 
                printf(
                    '<button type="button" class="%s" id="search-product-capabilities">',
                    is_admin() ? 'button button-secondary' : 'btn btn-primary'
                );
                ?>
                    <?php esc_html_e( 'Search', 'streamtube-core' );?>
                </button>
            </p>

            <div class="p-4">
                <div class="about__section has-2-columns row row-cols-2">

                    <div class="column is-vertically-aligned-center">
                        <ul id="xcapabilities-search-list" class="capabilities-list border list-unstyled">
                            
                        </ul>       
                    </div>     

                    <div class="column is-vertically-aligned-center">
                        <ul id="xcapabilities-list" class="capabilities-list granted-capabilities border list-unstyled">
                            <?php 
                            if( $xcapabilities ){
                                for ( $i=0;  $i < count( $xcapabilities );  $i++) { 
                                    ?>
                                    <?php printf( '<li class="capability border-bottom" id="%s">', esc_attr( $xcapabilities[$i] ) ); ?>
                                        <p class="form-field">
                                            <?php printf( '<label>%s</label>', $xcapabilities[$i] ); ?>

                                            <?php
                                            printf(
                                                '<input type="hidden" value="%s" name="xcapabilities[]">',
                                                esc_attr( $xcapabilities[$i] )
                                            );
                                            ?>

                                            <?php
                                            printf(
                                                '<button data-value="%s" type="button" class="%s">%s</button>',
                                                 esc_attr( $xcapabilities[$i] ),
                                                'button button-small button-secondary btn button-secondary button-remove-capability bg-danger p-1',
                                                '<span class="dashicons dashicons-minus text-white"></span>'
                                            );
                                            ?>
                                        </p>
                                    </li>
                                    <?php
                                }
                            }
                            ?>
                        </ul>            
                    </div>

                </div>
            </div>
            <?php
            wp_nonce_field( 'save_xcapabilities', 'save_xcapabilities' );
            ?>

        </div>
        <?php
    }

    /**
     * Save meta fields
     * 
     */
    public function save_fields( $post_id ){

        $http_data = wp_parse_args( $_POST, array(
            'save_xcapabilities'    =>  '',
            'xcapabilities'         =>  ''
        ) );

        extract( $http_data );

        $xcapabilities = (array)$xcapabilities;

        if( ! wp_verify_nonce( $save_xcapabilities, 'save_xcapabilities' ) ){
            return;
        }

        if( ! self::can_update() ){
            return;
        }

        return update_post_meta( $post_id, '_xcapabilities', $xcapabilities );
    }

    private function add_user_capabilities( $user_id = 0, $capabilities = array() ){

        $user = new WP_User( $user_id );

        for ( $i = 0; $i < count( $capabilities ); $i++) { 
            if( ! StreamTube_Core_Role::is_admin_capability( $capabilities[$i] ) ){

                /**
                 * Fires before add cap
                 *
                 * @param WP_User
                 * @param string $capability
                 */
                do_action( 
                    'streamtube/core/woocommerce/xcapabilities/before_add_capability', 
                    $user, 
                    $capabilities[$i] 
                );

                $user->add_cap( $capabilities[$i] );

                /**
                 * Fires after added capability
                 *
                 * @param WP_User
                 * @param string $capability
                 */
                do_action( 
                    'streamtube/core/user/added_capability', 
                    $user, 
                    $capabilities[$i]
                );                 

                /**
                 * Fires after add cap
                 *
                 * @param WP_User
                 * @param string $capability
                 */
                /**
                 * Fires after add cap
                 *
                 * @param WP_User
                 * @param string $capability
                 */
                do_action( 
                    'streamtube/core/woocommerce/xcapabilities/after_add_capability', 
                    $user, 
                    $capabilities[$i] 
                );                
            }
        }
    }

    private function remove_user_capabilities( $user_id = 0, $capabilities = array() ){

        $user = new WP_User( $user_id );

        for ( $i = 0; $i < count( $capabilities ); $i++) { 
            /**
             * Fires before remove capability
             *
             * @param WP_User
             * @param string $capability
             */
            do_action( 
                'streamtube/core/woocommerce/xcapabilities/before_remove_capability', 
                $user, 
                $capabilities[$i] 
            );            

            $user->remove_cap( $capabilities[$i] ); 

            /**
             * Fires after removed capability
             *
             * @param WP_User
             * @param string $capability
             */
            do_action( 
                'streamtube/core/user/removed_capability', 
                $user, 
                $capabilities[$i]
            );

            /**
             * Fires after remove capability
             *
             * @param WP_User
             * @param string $capability
             */
            do_action( 
                'streamtube/core/woocommerce/xcapabilities/after_remove_capability', 
                $user, 
                $capabilities[$i] 
            );                
        }
    }   

    /**
     *
     * Remove assigned capabilities if any refund was made.
     * 
     */
    public function remove_capabilities( $order_id ){
        $order = wc_get_order( $order_id );

        $customer_id = $order->get_user_id();

        foreach ( $order->get_items() as $order_line_item ) {
            $product_id = $order_line_item->get_product_id();

            $capabilities = $this->get_capabilities( $product_id );

            if( $capabilities ){
                $this->remove_user_capabilities( $customer_id, $capabilities );

                /**
                 *
                 * Fires after updating (remove|add) capabilities
                 *
                 * @param int $customer_id
                 * @param array $capabilities
                 * @param object $order_line_item
                 * @param WP_Order $order
                 * 
                 */
                do_action( 
                    'streamtube/core/woocommerce/xcapabilities/removed_capabilities', 
                    $customer_id,
                    $capabilities,
                    $order_line_item,
                    $order
                );                
            }
        }                
    }

    /**
     *
     * update user capabilities once Order completed
     * 
     * @param integer $order_id
     */
    public function update_user_capabilities( $order_id = 0 ){

        $order  = wc_get_order( $order_id );
        $status = $order->get_status();

        $customer_id = $order->get_user_id();

        foreach ( $order->get_items() as $order_line_item ) {
            $product_id = $order_line_item->get_product_id();

            $capabilities = $this->get_capabilities( $product_id );

            if( $capabilities ){
                if( $status === 'completed' ){
                    $this->add_user_capabilities( $customer_id, $capabilities );
                }
                else{
                    $this->remove_user_capabilities( $customer_id, $capabilities );
                }          

                /**
                 *
                 * Fires after updating (remove|add) capabilities
                 *
                 * @param int $customer_id
                 * @param array $capabilities
                 * @param object $order_line_item
                 * @param WP_Order $order
                 * 
                 */
                do_action( 
                    'streamtube/core/woocommerce/xcapabilities/updated_capabilities', 
                    $customer_id,
                    $capabilities,
                    $order_line_item,
                    $order
                );
            }
        }
    }

    /**
     *
     * Enqueue scripts
     * 
     */
    public function enqueue_scripts(){

        global $post_type_screen;

        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;

        $screen = $screen ? $screen->id : $post_type_screen;

        if( $screen && in_array( $screen , array( 'product', 'edit-product' )) ){
            wp_enqueue_script( 
                'woocommerce-xcapability', 
                plugin_dir_url( __FILE__ ) . 'public/xcapability.js', 
                array( 'jquery' ), 
                filemtime( plugin_dir_path( __FILE__ ) . 'public/xcapability.js' ), 
                true
            );
        }
    }
}