<?php
/**
 * Define the WP Statistics functionality
 *
 * https://wordpress.org/plugins/wp-statistics/
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'STREAMTUBE_CORE_WP_STATISTICS_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );

define( 'STREAMTUBE_CORE_WP_STATISTICS_URL', untrailingslashit( plugin_dir_url( __FILE__ ) ) );

/**
 *  
 */
class StreamTube_Core_WP_Statistics {

	const SLUG = 'statistics';

	const AFFID = 3647;

	/**
	 * Holds metaboxes list
	 * @var array
	 */
	protected $metaboxes = array();

	function __construct() {
		$this->load_dependencies();

		add_action( 'wp', array( $this, 'load_statistics' ) );
	}

	/**
	 * Short fn to include file
	 */
	private function include_file( $file ) {
		require_once( untrailingslashit( STREAMTUBE_CORE_WP_STATISTICS_PATH ) . $file );
	}
	/**
	 * Load dependencies
	 */
	private function load_dependencies() {
		$this->include_file( '/class-streamtube-core-statistics-permission.php' );
		$this->include_file( '/helper.php' );
	}
	/**
	 * Checks if the current page is a dashboard page and the user is an author.
	 *
	 * @global WP_Query $wp_query The global WP_Query instance.
	 *
	 * @return bool True if the current page is a dashboard page and the user is an author, false otherwise.
	 */
	public static function is_dashboard() {
		global $wp_query;
		return isset( $wp_query->query_vars['dashboard'] ) && is_author();
	}

	/**
	 * Check if current page is in statistics
	 * @return boolean
	 */
	public static function is_statistics() {
		global $wp_query;

		// Ensure we're on the dashboard and the correct slug exists
		if ( ! self::is_dashboard() ) {
			return false;
		}

		if ( empty( $wp_query->query_vars['dashboard'] ) ) {
			return 'user_dashboard';
		}

		$dashboardPath = untrailingslashit( $wp_query->query_vars['dashboard'] );

		// Return 'overview' if the exact slug matches
		if ( $dashboardPath === self::SLUG ) {
			return 'overview';
		}

		// Extract the second segment of the path, defaulting to 'overview'
		$segments = explode( '/', $dashboardPath );

		return $segments[1] ?? $segments[0];
	}

	public function load_statistics() {
		if ( self::is_dashboard() ) {
			// List of classes to instantiate
			$classes = [ 
				'WP_Statistics\Service\Admin\ContentAnalytics\ContentAnalyticsManager',
				'WP_Statistics\Service\Admin\AuthorAnalytics\AuthorAnalyticsManager',
				'WP_Statistics\Service\Admin\Geographic\GeographicManager',
				'WP_Statistics\Service\Admin\Devices\DevicesManager',
				'WP_Statistics\Service\Admin\CategoryAnalytics\CategoryAnalyticsManager',
				'WP_Statistics\Service\Admin\PageInsights\PageInsightsManager',
				'WP_Statistics\Service\Admin\VisitorInsights\VisitorInsightsManager',
				'WP_Statistics\Service\Admin\Overview\OverviewManager',
			];

			// Instantiate classes if they exist
			foreach ( $classes as $class ) {
				if ( class_exists( $class ) ) {
					new $class();
				}
			}
		}
	}

	/**
	 *
	 * Register custom metaboxes
	 * 
	 */
	public function register_custom_meta_boxes( $metaboxes ) {

		// Models
		$this->include_file( '/model/class-streamtube-core-wp-statistics-visitorsmodel.php' );

		// Providers
		$this->include_file( '/provider/class-streamtube-core-wp-statistics-searchrnginechartdataprovider.php' );
		$this->include_file( '/provider/class-streamtube-core-wp-statistics-socialmediaChartdataprovider.php' );
		$this->include_file( '/provider/class-streamtube-core-wp-statistics-sourcecategorychartdataprovider.php' );
		$this->include_file( '/provider/class-streamtube-core-wp-statistics-trafficchartdataprovider.php' );
		$this->include_file( '/provider/class-streamtube-core-wp-statistics-devicesdataprovider.php' );

		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-mostvisitedpages.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-dailytraffictrend.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-trafficsummary.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-trafficoverview.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-mostactivevisitors.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-latestvisitor.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-browserusage.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-operatingsystems.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-deviceusagebreakdown.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-topdevicemodel.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-topcountries.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-topreferring.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-searchengines.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-currentlyonline.php' );
		$this->include_file( '/metabox/class-streamtube-core-statistics-metabox-globalvisitordistribution.php' );

		$metaboxes[] = StreamTube_Core_WP_Statistics_MostVisitedPages::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_DailyTrafficTrend::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_TrafficSummary::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_TrafficOverview::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_MostActiveVisitors::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_LatestVisitor::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_BrowserUsage::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_OperatingSystems::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_DeviceUsageBreakdown::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_TopDeviceModel::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_TopCountries::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_TopReferring::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_SearchEngines::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_CurrentlyOnline::class;
		$metaboxes[] = StreamTube_Core_WP_Statistics_GlobalVisitorDistribution::class;

		return $metaboxes;
	}

	/**
	 *
	 * Get post visitors (unique views)
	 *
	 * @param  array  $args
	 * @return int
	 * 
	 */
	public function get_post_visitors( $args = array() ) {

		if ( ! class_exists( 'WP_Statistics\Models\VisitorsModel' ) ) {
			return 0;
		}

		$args = wp_parse_args( $args, array(
			'post_id' => 0,
			'date'    => 'total'
		) );

		$args['post_type'] = get_post_type( $args['post_id'] );

		$visitorsModel = new WP_Statistics\Models\VisitorsModel();

		return apply_filters(
			'streamtube/core/wpstatistics/get_post_visitors',
			$visitorsModel->countVisitors( $args ),
			$args
		);
	}

	/**
	 *
	 * Get post hits
	 * 
	 * @param  array  $args
	 * @return int
	 * 
	 */
	public function get_post_views( $args = array() ) {

		$postViews = 0;

		if ( ! class_exists( 'WP_Statistics\Models\ViewsModel' ) ) {
			return 0;
		}

		$args = wp_parse_args( $args, array(
			'post_id' => 0,
			'date'    => 'total'
		) );

		$args['post_id'] = (int) $args['post_id'];

		$viewsModel = new WP_Statistics\Models\ViewsModel();

		if ( is_callable( array( $viewsModel, 'countViewsFromPagesOnly' ) ) ) {
			$postViews = $viewsModel->countViewsFromPagesOnly( array_merge( $args, array(
				'date'       => array(
					'from' => get_post_time( 'Y-m-d', false, $args['post_id'] ),
					'to'   => date( 'Y-m-d' )
				),
				'historical' => true
			) ) );
		} else {
			$args['resource_type'] = get_post_type( $args['post_id'] );
			$postViews             = $viewsModel->countViews( $args );
		}

		return apply_filters(
			'streamtube/core/wpstatistics/get_post_views',
			$postViews,
			$args
		);
	}

	/**
	 *
	 * Get page views or visitors
	 * 
	 * @param  array  $args
	 * @return int|string
	 */
	public function get_page_views( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'metric' => streamtube_core_get_statistics_metric(),
			'format' => true
		) );

		extract( $args );

		$views = 0;

		if ( $metric === 'views' ) {
			$views = $this->get_post_views( $args );
		} else {
			$views = $this->get_post_visitors( $args );
		}

		if ( $format ) {
			return apply_filters( 'streamtube_core_format_page_views', $views );
		}

		return $views;
	}

	public function get_term_visitors( $args = array() ) {

		if ( ! class_exists( 'WP_Statistics\Models\VisitorsModel' ) ) {
			return 0;
		}

		$args = wp_parse_args( $args, array(
			'term'       => 0,
			'taxonomy'   => '',
			'date'       => 'total',
			'historical' => true
		) );

		if ( ! $args['taxonomy'] ) {
			$term = get_term( absint( $args['term'] ) );
			if ( $term ) {
				$args['taxonomy'] = $term->taxonomy;
			}
		}

		$args['resource_id']   = $args['term'];
		$args['resource_type'] = ! in_array( $args['taxonomy'], array( 'category', 'post_tag' ) ) ? 'tax' : $args['taxonomy'];

		unset( $args['term'] );
		unset( $args['taxonomy'] );

		$visitorsModel = new WP_Statistics\Models\VisitorsModel();

		return apply_filters(
			'streamtube/core/wpstatistics/get_term_visitors',
			$visitorsModel->countVisitors( $args ),
			$args
		);
	}

	public function get_term_views( $args = array() ) {

		if ( ! class_exists( 'WP_Statistics\Models\ViewsModel' ) ) {
			return 0;
		}

		$args = wp_parse_args( $args, array(
			'term'       => 0,
			'taxonomy'   => '',
			'date'       => 'total',
			'historical' => true
		) );

		if ( ! $args['taxonomy'] ) {
			$term = get_term( absint( $args['term'] ) );
			if ( $term ) {
				$args['taxonomy'] = $term->taxonomy;
			}
		}

		$args['post_id']       = $args['term'];
		$args['uri']           = get_term_link( intval( $term->term_id ), $term->taxonomy );
		$args['resource_type'] = ! in_array( $args['taxonomy'], array( 'category', 'post_tag' ) ) ? 'tax' : $args['taxonomy'];

		$viewsModel = new WP_Statistics\Models\ViewsModel();

		return apply_filters(
			'streamtube/core/wpstatistics/get_term_views',
			$viewsModel->countViewsFromPagesOnly( $args ),
			$args
		);
	}

	public function get_term_page_views( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'format'           => true,
			'ignore_post_type' => true
		) );

		if ( streamtube_core_get_statistics_metric() === 'visitors' ) {
			$views = $this->get_term_visitors( $args );
		} else {
			$views = $this->get_term_views( $args );
		}

		if ( $args['format'] ) {
			return apply_filters( 'streamtube_core_format_page_views', $views );
		}

		return $views;
	}
	public function get_rest_statistics_field( $object ) {
		return array(
			'visitors' => $this->get_post_visitors( array(
				'post_id' => $object['id']
			) ),
			'views'    => $this->get_post_views( array(
				'post_id' => $object['id']
			) )
		);
	}

	/**
	 *
	 * Get all registered metaboxes
	 * 
	 * @return array
	 */
	public function get_active_meta_boxes() {
		if ( class_exists( 'WP_Statistics\Service\Admin\Metabox\MetaboxHelper' ) ) {
			return WP_Statistics\Service\Admin\Metabox\MetaboxHelper::getActiveMetaboxes();
		}
		return array();
	}

	/**
	 * Determines the screens for a metabox.
	 *
	 * @param object $metabox The metabox object.
	 * @return array The list of screens for the metabox.
	 */
	protected function get_metabox_screens( $metabox ) {

		$screens = $metabox->getScreen();

		if ( is_string( $screens ) ) {
			$screens = array( $screens );
		}

		if ( in_array( 'toplevel_page_wps_overview_page', $screens ) ) {
			array_diff( $screens, [ 'toplevel_page_wps_overview_page' ] );
			$screens[] = 'dashboard_statistics';
		}

		if ( in_array( 'dashboard', $screens ) && ! WP_STATISTICS\Option::get( 'disable_dashboard' ) ) {
			array_diff( $screens, [ 'dashboard' ] );
			$screens[] = 'user_dashboard';
		}

		return array_filter( array_values( $screens ) );
	}

	/**
	 *
	 * Don't display those boxes
	 * 
	 * @return array
	 */
	protected function get_exclude_metaboxes() {
		return apply_filters( 'streamtube/core/wpstatistics/exclude_metaboxes', array(
			'wp-statistics-about-metabox',
			'wp-statistics-go-premium-widget',
			'wp-statistics-post-visitors-locked-widget'
		) );
	}

	/**
	 *
	 * Check if metabox is public
	 * 
	 * @param   $metabox 
	 * 
	 */
	protected function is_public_metabox( $metabox ) {

		if ( in_array( $metabox->getKey(), $this->get_exclude_metaboxes() ) ) {
			return false;
		}

		if ( ! method_exists( $metabox, 'isPublic' ) || method_exists( $metabox, 'isPublic' ) && $metabox->isPublic() ) {
			return true;
		}

		return StreamTube_Core_WP_Statistics_Permission::view_all_statistics();
	}

	/**
	 * Adds metaboxes to the dashboard.
	 *
	 * This method registers metaboxes based on the active metaboxes list.
	 * It ensures that metaboxes are only added to the appropriate screens.
	 */
	public function add_meta_boxes() {
		$metaboxes = $this->get_active_meta_boxes();

		// Exit early if in the admin area
		if ( is_admin() ) {
			return $metaboxes;
		}

		foreach ( $metaboxes as $metabox ) {

			if ( $this->is_public_metabox( $metabox ) ) {
				// Register the metabox
				streamtube_add_meta_box(
					$metabox->getKey(),
					$metabox->getName(),
					function () use ($metabox) {
						$metabox->render();
					},
					$this->get_metabox_screens( $metabox ),
					$metabox->getContext(),
					$metabox->getPriority()
				);
			}
		}
	}

	/**
	 *
	 * Register `dashboard_statistics` screen
	 * 
	 */
	public function register_meta_box_screens( $screen = 'normal' ) {
		Streamtube_Core_MetaBox::_add( 'dashboard_statistics', $screen );
	}

	/**
	 *
	 * Add additional statistics classes
	 * 
	 */
	public function filter_body_classes( $classes ) {

		$is_statistics = self::is_statistics();

		if ( $is_statistics ) {
			$classes[] = 'wps_page';

			if ( is_string( $is_statistics ) ) {
				$classes[] = 'statistics_page_wps_' . $is_statistics . '_page';
			}
		}

		return $classes;
	}

	/**
	 *
	 * Filter statistics admin script
	 * 
	 */
	public function filter_admin_script_assets( $args = array() ) {

		global $streamtube;

		$is_statistics = self::is_statistics();

		$post_id = isset( $_REQUEST['post_id'] ) ? absint( $_REQUEST['post_id'] ) : $streamtube->get()->post->get_edit_post_id();

		if ( in_array( $is_statistics, array( 'dashboard_statistics', 'overview', 'user_dashboard' ) ) ) {
			$args['options']['overview_page'] = 1;

			$metaboxes = $this->get_active_meta_boxes();

			$args['meta_boxes'] = array();

			foreach ( $metaboxes as $metabox ) {
				if ( $this->is_public_metabox( $metabox ) ) {
					$args['meta_boxes'][ $metabox->getContext()][] = $metabox->getKey();
				}
			}
		}

		$args['request_params']['page'] = $is_statistics;
		$args['request_params']['tab']  = isset( $_GET['tab'] ) ? $_GET['tab'] : '';

		if ( $this->is_dashboard() && $post_id && Streamtube_Core_Permission::can_edit_post( $post_id ) ) {
			$args['request_params']['page'] = 'content-analytics';
			$args['post_creation_date']     = get_the_date( WP_Statistics\Components\DateTime::$defaultDateFormat, $post_id );
			$args['initial_post_date']      = $args['post_creation_date'];
			$args['active_post_type']       = get_post_type_object( get_post_type( $post_id ) )->labels->name;
		} else {
			$args['post_creation_date'] = get_the_date( WP_Statistics\Components\DateTime::$defaultDateFormat );
		}

		return $args;
	}

	/**
	 *
	 * Filter Overview callback
	 * 
	 * @param  array $menu_items
	 * 
	 */
	public function filter_overview_views( $menu_items ) {
		if ( self::is_dashboard() ) {
			if ( isset( $menu_items['parent'] ) ) {
				$this->include_file( '/class-streamtube-core-wp-statistics-overview.php' );
				$menu_items['parent']['callback'] = StreamTube_Core_WP_Statistics_OverviewPage::class;
			}
		}
		return $menu_items;
	}

	/**
	 *
	 * Filter `Visitor Insights ` views
	 * 
	 */
	public function filter_visitors_views( $views ) {
		if ( self::is_statistics() ) {
			$this->include_file( '/class-streamtube-core-wp-statistics-visitors-tabview.php' );
			$this->include_file( '/class-streamtube-core-wp-statistics-visitors-singleview.php' );

			$views['tabs']           = StreamTube_Core_WP_Statistics_Visitors_TabsView::class;
			$views['single-visitor'] = StreamTube_Core_WP_Statistics_Visitors_SingleView::class;
		}

		return $views;
	}

	/**
	 *
	 * Filter the Content Analytics views
	 * Only apply to frontend dashboard
	 * 
	 */
	public function filter_content_analytics_views( $views ) {

		if ( self::is_statistics() ) {

			$this->include_file( '/class-streamtube-core-wp-statistics-content-analytics-tabviews.php' );
			$this->include_file( '/class-streamtube-core-wp-statistics-content-analytics-singleview.php' );

			if (
				! class_exists( 'StreamTube_Core_WP_Statistics_ContentAnalytics_TabsView' ) ||
				! class_exists( 'StreamTube_Core_WP_Statistics_ContentAnalytics_SingleView' ) ) {
				return $views;
			}

			$views = array(
				'tabs'   => StreamTube_Core_WP_Statistics_ContentAnalytics_TabsView::class,
				'single' => StreamTube_Core_WP_Statistics_ContentAnalytics_SingleView::class
			);
		}

		return $views;
	}

	public function filter_pages_insight_views( $views ) {

		if ( self::is_statistics() ) {

			$this->include_file( '/class-streamtube-core-wp-statistics-page-insights-tabviews.php' );

			$views['tabs'] = StreamTube_Core_WP_Statistics_Page_Insights_TabsView::class;
		}

		return $views;
	}

	/**
	 *
	 * Referrals View
	 * 
	 */
	public function filter_referrals_views( $views ) {
		if ( self::is_statistics() && ! StreamTube_Core_WP_Statistics_Permission::view_all_statistics() ) {

			$this->include_file( '/class-streamtube-core-wp-statistics-referrals-tabviews.php' );

			$views['tabs'] = StreamTube_Core_WP_Statistics_Referrals_TabsView::class;
		}

		return $views;
	}

	/**
	 *
	 * Referrals View
	 * 
	 */
	public function filter_author_analytics_views( $views ) {
		if ( self::is_statistics() ) {
			$this->include_file( '/class-streamtube-core-wp-statistics-authors-analytics-views.php' );
			$this->include_file( '/class-streamtube-core-wp-statistics-authors-performance-views.php' );

			$views['performance'] = StreamTube_Core_WP_Statistics_AuthorsPerformance_View::class;
			$views['authors']     = StreamTube_Core_WP_Statistics_AuthorsAnalytics_View::class;
		}

		return $views;
	}

	public function filter_category_analytics_views( $views ) {
		if ( self::is_statistics() ) {

			$this->include_file( '/class-streamtube-core-wp-statistics-category-analytics-tabviews.php' );
			$this->include_file( '/class-streamtube-core-wp-statistics-category-analytics-singleview.php' );

			$views['tab']    = StreamTube_Core_WP_Statistics_CategoryAnalytics_TabView::class;
			$views['single'] = StreamTube_Core_WP_Statistics_CategoryAnalytics_SingleView::class;
		}

		return $views;
	}

	/**
	 *
	 * Geographic views
	 * 
	 */
	public function filter_geographic_views( $views ) {
		if ( self::is_statistics() ) {
			$this->include_file( '/class-streamtube-core-wp-statistics-geographic-tabviews.php' );
			$views['tabs'] = StreamTube_Core_WP_Statistics_Geographic_TabView::class;
		}
		return $views;
	}

	public function filter_devices_views( $views ) {
		if ( self::is_statistics() ) {
			$this->include_file( '/class-streamtube-core-wp-statistics-devices-tabviews.php' );

			$views['tabs'] = StreamTube_Core_WP_Statistics_Devices_TabView::class;
		}

		return $views;
	}

	/**
	 * Modifies query clauses when terms are sorted by hits column.
	 *
	 * @param array $clauses Clauses for the query.
	 * @param array $taxonomies Taxonomy names.
	 * @param array $args Term query arguments.
	 *
	 * @return array Updated clauses for the query.
	 *
	 * @hooked filter: `terms_clauses` - 10
	 */
	public function filter_orderby_tax_views_clauses( $clauses, $taxonomies, $args ) {

		// If order-by.
		if ( ! isset( $args['orderby'] ) || $args['orderby'] !== 'hits' ) {
			return $clauses;
		}

		// Add date condition if needed
		$dateCondition = '';

		$historicalSubQuery = '';

		if ( streamtube_core_get_statistics_metric() === 'visitors' ) {
			if ( ! empty( $dateCondition ) ) {
				$dateCondition = "AND `visitor_relationships`.`date` $dateCondition";
			}
			$clauses['fields'] .= ', (SELECT COUNT(DISTINCT `visitor_id`) FROM ' . WP_STATISTICS\DB::table( 'visitor_relationships' ) . ' AS `visitor_relationships` LEFT JOIN ' . WP_STATISTICS\DB::table( 'pages' ) . ' AS `pages` ON `visitor_relationships`.`page_id` = `pages`.`page_id` WHERE `pages`.`type` IN ("category", "post_tag", "tax") AND `t`.`term_id` = `pages`.`id` ' . $dateCondition . ') AS `tax_hits_sortable` ';
		} else {

			if ( ! empty( $dateCondition ) ) {
				$dateCondition = "AND `pages`.`date` $dateCondition";
			} else {
				// Consider historical for total views
				$historicalSubQuery = ' + IFNULL((SELECT SUM(`historical`.`value`) FROM ' . WP_STATISTICS\DB::table( 'historical' ) . ' AS `historical` WHERE `historical`.`page_id` = `t`.`term_id` AND `historical`.`uri` LIKE CONCAT("%/", `t`.`slug`, "/")), 0)';
			}

			$clauses['fields'] .= ', ((SELECT SUM(`pages`.`count`) FROM ' . WP_STATISTICS\DB::table( 'pages' ) . ' AS `pages` WHERE `pages`.`type` IN ("category", "post_tag", "tax") AND `t`.`term_id` = `pages`.`id` ' . $dateCondition . ')' . $historicalSubQuery . ') AS `tax_hits_sortable` ';
		}

		// Order by `tax_hits_sortable`
		$clauses['orderby'] = " ORDER BY coalesce(`tax_hits_sortable`, 0)";

		return $clauses;
	}

	/**
	 * Filter posts by view count while securing against SQL injection.
	 *
	 * @param  array  $clauses   SQL query clauses.
	 * @param  object $wp_query  WP_Query instance.
	 * @return array             Modified SQL query clauses.
	 */
	public function filter_orderby_post_views_clauses( $clauses, $wp_query ) {
		global $wpdb, $widget_instance;

		$from = $to = '';

		if ( ! class_exists( 'WP_STATISTICS\DB' ) ) {
			return $clauses;
		}

		$widget_instance = wp_parse_args( (array) $widget_instance, array(
			'custom_orderby'    => '',
			'statistics_date'   => '',
			'statistics_from'   => '',
			'statistics_to'     => '',
			'statistics_metric' => ''
		) );

		if ( empty( $widget_instance['statistics_metric'] ) || $widget_instance['statistics_metric'] === 'default' ) {
			$widget_instance['statistics_metric'] = streamtube_core_get_statistics_metric();
		}

		$q = wp_parse_args( wp_unslash( $_REQUEST ), array(
			'from'       => '',
			'to'         => '',
			'date_range' => ''
		) );

		if ( $q['date_range'] ) {
			$q['date_range'] = str_replace( 'ago', '', $q['date_range'] );
		}

		// Sanitize user input
		$from             = $q['from'];
		$to               = $q['to'];
		$query_date_range = $q['date_range'];

		if ( $widget_instance['statistics_date'] ) {

			if ( is_string( $widget_instance['statistics_date'] ) ) {
				$widget_instance['statistics_date'] = str_replace( 'ago', '', strtolower( $widget_instance['statistics_date'] ) );
				$date_range                         = WP_Statistics\Components\DateRange::get( $widget_instance['statistics_date'] );
			}

			if ( is_array( $widget_instance['statistics_date'] ) ) {
				$date_range = $widget_instance['statistics_date'];
			}

			if ( is_array( $date_range ) ) {
				$from = isset( $date_range['from'] ) ? $date_range['from'] : $from;
				$to   = isset( $date_range['to'] ) ? $date_range['to'] : $to;
			}
		}

		if ( ! empty( $widget_instance['statistics_from'] ) ) {
			$from = date( 'Y-m-d', strtotime( $widget_instance['statistics_from'] ) );
		}

		if ( ! empty( $widget_instance['statistics_to'] ) ) {
			$to = date( 'Y-m-d', strtotime( $widget_instance['statistics_to'] ) );
		}

		if ( $widget_instance['custom_orderby'] && $query_date_range ) {
			$query_date_range = WP_Statistics\Components\DateRange::get( $query_date_range );

			if ( is_array( $query_date_range ) ) {
				$from = isset( $query_date_range['from'] ) ? $query_date_range['from'] : $from;
				$to   = isset( $query_date_range['to'] ) ? $query_date_range['to'] : $to;
			}
		}

		if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $from ) ) {
			$from = ''; // Invalid date format
		}

		if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $to ) ) {
			$to = date( 'Y-m-d' ); // Default to today if invalid
		}

		// Apply final filtering to the range for further refinement
		$date_range = apply_filters(
			'streamtube/core/orderby_post_views_date_range',
			compact( 'from', 'to' )
		);

		if ( $date_range['from'] && $date_range['to'] ) {
			$widget_instance['statistics_date_range'] = array(
				'from' => $date_range['from'],
				'to'   => $date_range['to']
			);

			$widget_instance['callback_params'] = array(
				'orderby'           => 'post_view',
				'orderby_post_view' => true,
				'statistics_date'   => $widget_instance['statistics_date_range']
			);

		}

		// Get order direction safely
		$order = $widget_instance['order'] ?? $wp_query->query_vars['order'];
		$order = in_array( strtoupper( $order ), [ 'ASC', 'DESC' ], true ) ? strtoupper( $order ) : 'DESC';

		// Date condition for filtering
		$dateCondition = '';

		if ( $from && $to ) {
			$dateCondition = $wpdb->prepare( 'BETWEEN %s AND %s', $from, $to );
		}

		if ( $widget_instance['statistics_metric'] === 'visitors' ) {

			if ( ! empty( $dateCondition ) ) {
				$dateCondition = "AND `visitor_relationships`.`date` $dateCondition";
			}
			$clauses['fields'] .= ', (SELECT COUNT(DISTINCT `visitor_id`) FROM ' . WP_STATISTICS\DB::table( 'visitor_relationships' ) . ' AS `visitor_relationships` LEFT JOIN ' . WP_STATISTICS\DB::table( 'pages' ) . ' AS `pages` ON `visitor_relationships`.`page_id` = `pages`.`page_id` WHERE (`pages`.`type` IN ("page", "post", "product") OR `pages`.`type` LIKE "post_type_%") AND ' . $wpdb->posts . '.`ID` = `pages`.`id` ' . $dateCondition . ') AS `post_hits_sortable` ';

		} else {

			$historicalSubQuery = '';

			if ( ! empty( $dateCondition ) ) {
				$dateCondition = "AND `pages`.`date` $dateCondition";
			} else {
				// Consider historical data for total views
				$historicalSubQuery = ' + IFNULL((
                    SELECT SUM(`historical`.`value`) 
                    FROM ' . WP_STATISTICS\DB::table( 'historical' ) . ' AS `historical` 
                    WHERE `historical`.`page_id` = ' . $wpdb->posts . '.`ID` 
                    AND `historical`.`uri` LIKE CONCAT("%/", ' . $wpdb->posts . '.`post_name`, "/")
                ), 0)';
			}

			// Modify SQL query securely
			$clauses['fields'] .= ', (
                (SELECT SUM(`pages`.`count`) 
                FROM ' . WP_STATISTICS\DB::table( 'pages' ) . ' AS `pages`
                WHERE (`pages`.`type` IN ("page", "post", "product") OR `pages`.`type` LIKE "post_type_%") 
                AND ' . $wpdb->posts . '.`ID` = `pages`.`id` 
                ' . $dateCondition . ') 
                ' . $historicalSubQuery . ') AS `post_hits_sortable` ';
		}

		$dateCondition = '';
		if ( $from && $to ) {
			$dateCondition = " AND `pages`.`date` " . $wpdb->prepare( 'BETWEEN %s AND %s', $from, $to );
		}

		$clauses['where'] .= " AND (
            SELECT COALESCE(SUM(`pages`.`count`), 0) 
            FROM " . WP_STATISTICS\DB::table( 'pages' ) . " AS `pages`
            WHERE `pages`.`id` = " . $wpdb->posts . ".ID
            " . $dateCondition . "
        ) > 0 ";

		// Secure ORDER BY clause
		$clauses['orderby'] = " COALESCE(`post_hits_sortable`, 0) $order";

		return $clauses;
	}

	/**
	 *
	 * Hooks into `posts_clauses` to filter posts by views
	 * Apply to Post List and Flat widgets
	 * 
	 */
	public function handler_orderby_views( $query_args ) {
		if ( isset( $query_args['orderby_post_view'] ) && $query_args['orderby_post_view'] ) {
			add_filter( 'posts_clauses', array( $this, 'filter_orderby_post_views_clauses' ), 10, 2 );
		}
	}

	/**
	 * Remove `posts_clauses` filter
	 */
	public function remove_handler_orderby_views( $query_args ) {
		if ( isset( $query_args['orderby_post_view'] ) && $query_args['orderby_post_view'] ) {
			remove_filter( 'posts_clauses', array( $this, 'filter_orderby_post_views_clauses' ), 10, 2 );
		}
	}

	/**
	 *
	 * Filter and display the Views column content
	 * 
	 */
	public function render_post_views_column( $column, $post_id ) {

		if ( $column === 'wp-statistics-post-views' ) {

			$hitColumnHandler = new WP_Statistics\Service\Admin\Posts\HitColumnHandler();

			if ( ! WP_STATISTICS\Helper::isAddOnActive( 'mini-chart' ) ) {
				printf(
					'<a href="%s"><span class="icon-chart-bar text-muted me-1"></span>%s</a>',
					esc_url(
						add_query_arg(
							array(
								'post_id' => $post_id,
								'type'    => 'single',
								'from'    => get_the_date( 'Y-m-d', $post_id ),
								'to'      => current_time( 'Y-m-d' )
							),
							streamtube_core_statistics_page_url( 'content-analytics' )
						)
					),
					$this->get_page_views( array(
						'post_id' => $post_id,
						'format'  => true
					) )
				);
			} else {
				$hitColumnHandler->renderHitColumn( 'wp-statistics-post-hits', $post_id );
			}
		}
	}

	public function render_tax_views_column( $output, $column, $term_id ) {
		if ( $column === 'wp-statistics-post-views' ) {

			$hitColumnHandler = new WP_Statistics\Service\Admin\Posts\HitColumnHandler();

			if ( ! WP_STATISTICS\Helper::isAddOnActive( 'mini-chart' ) ) {
				$output = sprintf(
					'<a href="%s"><span class="icon-chart-bar text-muted me-1"></span>%s</a>',
					esc_url(
						add_query_arg(
							array(
								'term_id' => $term_id,
								'type'    => 'single'
							),
							streamtube_core_statistics_page_url( 'category-analytics' )
						)
					),
					$this->get_term_page_views( array(
						'term'   => $term_id,
						'format' => true
					) )
				);
			} else {
				$output = $hitColumnHandler->renderTaxHitColumn( null, 'wp-statistics-post-hits', $term_id );
			}
		}

		return $output;
	}

	public function modify_sortable_columns( $columns ) {
		$columns['wp-statistics-post-views'] = 'hits';

		return $columns;
	}

	public function add_views_column( $columns ) {

		unset( $columns['wp-statistics-post-hits'] );
		unset( $columns['wp-statistics-tax-hits'] );

		return array_merge( $columns, array(
			'wp-statistics-post-views' => esc_html__( 'Views', 'streamtube-core' )
		) );
	}

	/**
	 *
	 * Add Views column content for all public post types
	 * 
	 */
	public function add_post_table_columns() {

		if ( class_exists( 'WP_Statistics\Service\Admin\Posts\HitColumnHandler' ) && streamtube_core_is_frontend_request() ) {

			$hitColumnHandler = new WP_Statistics\Service\Admin\Posts\HitColumnHandler();

			$post_types = array_keys( streamtube_core_get_statistics_post_types() );

			foreach ( $post_types as $post_type ) {

				add_filter( "manage_{$post_type}_posts_columns", array( $this, 'add_views_column' ), 50, 1 );

				add_action( "manage_{$post_type}_posts_custom_column", [ $this, 'render_post_views_column' ], 50, 2 );

				add_filter( "manage_edit-{$post_type}_sortable_columns", [ $this, 'modify_sortable_columns' ] );
			}

			foreach ( get_taxonomies( array( 'public' => true ) ) as $tax => $name ) {
				add_filter( "manage_edit-{$tax}_columns", [ $this, 'add_views_column' ], 50, 2 );
				add_filter( "manage_{$tax}_custom_column", [ $this, 'render_tax_views_column' ], 50, 3 );
				add_filter( "manage_edit-{$tax}_sortable_columns", [ $this, 'modify_sortable_columns' ] );
			}

			if ( isset( $_REQUEST['orderby'] ) && in_array( $_REQUEST['orderby'], array( 'hits', 'post_view' ) ) ) {
				add_filter( 'posts_clauses', [ $this, 'filter_orderby_post_views_clauses' ], 50, 2 );

				add_filter( 'terms_clauses', [ $this, 'filter_orderby_tax_views_clauses' ], 50, 3 );
			}
		}

	}

	/**
	 * 
	 * Get user nav items
	 * 
	 * @return array
	 *
	 * @since  1.0.0
	 * 
	 */
	public function add_post_nav_item( $menu_items, $post ) {

		if ( post_type_supports( $post->post_type, 'wpstatistics' ) ) {

			$this->include_file( '/class-streamtube-core-wp-statistics-content-analytics-singleview.php' );

			$menu_items['statistics'] = array(
				'title'    => esc_html__( 'Statistics', 'streamtube-core' ),
				'icon'     => 'icon-chart-bar',
				'template' => streamtube_core_get_template( 'post/statistics.php' ),
				'priority' => 100
			);
		}

		return $menu_items;
	}

	public function filter_post_row_actions( $actions, $post ) {
		if (
			! streamtube_core_is_frontend_request() ||
			! in_array( $post->post_type, array_keys( streamtube_core_get_statistics_post_types() ) ) ) {
			return $actions;
		}

		$actions['statistics'] = sprintf(
			'<a class="text-success" href="%s">%s%s</a>',
			esc_url( trailingslashit(
				remove_query_arg(
					array_keys( $_REQUEST ),
					wp_doing_ajax() ? wp_get_referer() : $_SERVER['REQUEST_URI']
				),
			) . $post->ID . '/statistics' ),
			'<span class="icon-chart-bar"></span>',
			esc_html__( 'Statistics', 'streamtube-core' )
		);

		return $actions;
	}

	/**
	 *
	 * Display component content for single view with custom content
	 * such as category, posts ...
	 *
	 * Only apply to frontend dashboard
	 * 
	 */
	public function add_header_component_single_type() {
		load_template( STREAMTUBE_CORE_WP_STATISTICS_PATH . '/public/components/header.php' );
	}

	/**
	 *
	 * The Statistics view template
	 * 
	 */
	private function the_statistics_view( $menu ) {

		/**
		 *
		 * Fires before loading view
		 * 
		 */
		do_action( 'streamtube/core/wpstatistics/before_load_view', $menu );

		load_template( STREAMTUBE_CORE_WP_STATISTICS_PATH . '/public/statistics.php', true, $menu );

		/**
		 *
		 * Fires after loading view
		 * 
		 */
		do_action( 'streamtube/core/wpstatistics/after_load_view', $menu );
	}

	/**
	 * Builds a parent menu item.
	 *
	 * @param array $menu The menu item data.
	 * @return array The formatted parent menu item.
	 */
	protected function build_parent_menu_item( $menu ) {
		return array_merge( $menu, [ 
			'alt_title' => esc_html__( 'Overview', 'streamtube-core' ),
			'icon'      => 'icon-chart-bar',
			'cap'       => function () {
				return StreamTube_Core_WP_Statistics_Permission::view_statistics();
			},
			'callback'  => function () use ($menu) {
				$this->the_statistics_view( $menu );
			},
			'priority'  => 1
		] );
	}

	/**
	 * Builds a submenu item.
	 *
	 * @param array $menu The submenu item data.
	 * @return array The formatted submenu item.
	 */
	protected function build_submenu_item( $menu ) {
		return array_merge( $menu, [ 
			'icon'     => 'icon-chart-area',
			'cap'      => function () {
				return StreamTube_Core_WP_Statistics_Permission::view_statistics();
			},
			'callback' => function () use ($menu) {
				$this->the_statistics_view( $menu );
			}
		] );
	}

	/**
	 * Determines whether a submenu item should be skipped.
	 *
	 * @param array $menu The submenu item data.
	 * @return bool Whether the submenu item should be skipped.
	 */
	protected function should_skip_submenu_item( $menu ) {
		// Skip if callback is not set, class doesn't exist, or page URL is 'overview'
		if ( ! isset( $menu['callback'] ) || ! class_exists( $menu['callback'] ) || $menu['page_url'] === 'overview' ) {
			return true;
		}

		// Skip if the page is not public and the user doesn't have permission to view all statistics
		if ( ! streamtube_core_is_statistics_public_pages( $menu['page_url'] ) && ! StreamTube_Core_WP_Statistics_Permission::view_all_statistics() ) {
			return true;
		}

		return false;
	}

	/**
	 * Registers the `Statistics` dashboard menu.
	 *
	 * This method adds menu items and submenu items to the dashboard based on the
	 * WP Statistics menu list. It ensures proper permissions and dynamically loads
	 * the required templates and classes.
	 *
	 * @param array $menu_items The existing menu items.
	 * @return array The updated menu items.
	 */
	public function register_dashboard_menu( $menu_items ) {
		// Check if WP Statistics Menus class and method exist
		if ( ! class_exists( 'WP_STATISTICS\Menus' ) || ! method_exists( 'WP_STATISTICS\Menus', 'get_menu_list' ) ) {
			return $menu_items;
		}

		// Loop through the WP Statistics menu list
		foreach ( WP_STATISTICS\Menus::get_menu_list() as $key => $menu ) {
			// Handle parent menu items
			if ( ! isset( $menu['sub'] ) ) {
				$menu_items[ self::SLUG ] = $this->build_parent_menu_item( $menu );
			} else {
				// Handle submenu items
				if ( $this->should_skip_submenu_item( $menu ) ) {
					continue;
				}

				$menu_items[ self::SLUG ]['submenu'][ $menu['page_url'] ] = $this->build_submenu_item( $menu );
			}
		}

		return $menu_items;
	}

	/**
	 *
	 * Ensures that post views are enabled.
	 * 
	 */
	public function enable_display_post_views( $boolean = null ) {
		return true;
	}

	/**
	 *
	 * Display post views
	 * 
	 */
	public function display_post_view_count( $pageviews, $post_id ) {

		global $widget_instance;

		$widget_instance = wp_parse_args( $widget_instance, array(
			'statistics_hits_range' => '',
			'statistics_date_range' => '',
			'statistics_metric'     => streamtube_core_get_statistics_metric()
		) );

		$date = 'total';

		if ( wp_validate_boolean( $widget_instance['statistics_hits_range'] ) && $widget_instance['statistics_date_range'] ) {
			$date = $widget_instance['statistics_date_range'];
		}

		return $this->get_page_views( array(
			'metric'  => $widget_instance['statistics_metric'],
			'post_id' => $post_id,
			'format'  => false,
			'date'    => $date
		) );
	}

	/**
	 *
	 * Load single video post views 
	 * 
	 * @since 1.0.8
	 * 
	 */
	public function display_single_post_view_count() {
		if ( $this->enable_display_post_views() === true ) {
			get_template_part( 'template-parts/post-views' );
		}
	}

	/**
	 *
	 * Display the statistics button on frontend single page
	 * 
	 */
	public function display_single_post_statistics_button() {
		if ( current_user_can( 'edit_post', get_the_ID() ) ) {
			load_template( STREAMTUBE_CORE_WP_STATISTICS_PATH . '/public/components/statistics-button.php' );
		}
	}

	/**
	 *
	 * Display the Date Range dropdown
	 * 
	 */
	public function display_sortby_date_range() {
		load_template( STREAMTUBE_CORE_WP_STATISTICS_PATH . '/public/components/date-range.php', true );
	}

	/**
	 *
	 * Remove statistics header
	 * Only apply to regular authors
	 * 
	 */
	public function remove_view_header( $output ) {

		if (
			class_exists( 'WP_STATISTICS\Option' ) &&
			current_user_can( WP_STATISTICS\Option::get( 'manage_capability', 'manage_options' ) ) ) {
			return $output;
		}

		$options = array(
			'wpstatistics_hide_header' => '.wps-adminHeader',
			'wpstatistics_hide_footer' => '.wps-adminFooter'
		);

		foreach ( $options as $key => $value ) {
			if ( get_option( $key ) ) {
				$output = streamtube_core_remove_html_elements( $output, $value );
			}
		}

		return $output;
	}

	/**
	 *
	 * The Post List Statistics tab callback
	 * 
	 */
	public function widget_settings_tab_callback( $instance, $widget ) {
		$instance = wp_parse_args( $instance, array(
			'statistics_date'       => '',
			'statistics_from'       => '',
			'statistics_to'         => '',
			'statistics_hits_range' => '',
			'statistics_metric'     => ''
		) );

		ob_start();

		?>

		<p class="alert bg-info" style="padding: .5rem;">
			<?php
			esc_html_e( 'These options are utilized in combination with the \'Order by number of views\' functionality.', 'streamtube-core' )
				?>
		</p>

		<h3><?php esc_html_e( 'Custom date range', 'streamtube-core' ); ?></h3>

		<div class="field-control">
			<?php printf(
				'<label for="%s">%s</label>',
				esc_attr( $widget->get_field_id( 'statistics_from' ) ),
				esc_html__( 'From Date', 'streamtube-core' )
			); ?>

			<?php printf(
				'<input type="date" class="widefat" id="%s" name="%s" value="%s"/>',
				esc_attr( $widget->get_field_id( 'statistics_from' ) ),
				esc_attr( $widget->get_field_name( 'statistics_from' ) ),
				esc_attr( $instance['statistics_from'] )
			); ?>

			<span class="field-help">
				<?php esc_html_e( 'The starting date to retrieve posts based on view count (hits).', 'streamtube-core' ); ?>
			</span>

		</div>

		<div class="field-control">
			<?php printf(
				'<label for="%s">%s</label>',
				esc_attr( $widget->get_field_id( 'statistics_to' ) ),
				esc_html__( 'To Date', 'streamtube-core' )
			); ?>

			<?php printf(
				'<input type="date" class="widefat" id="%s" name="%s" value="%s"/>',
				esc_attr( $widget->get_field_id( 'statistics_to' ) ),
				esc_attr( $widget->get_field_name( 'statistics_to' ) ),
				esc_attr( $instance['statistics_to'] )
			); ?>

			<span class="field-help">
				<?php esc_html_e( 'The end date for retrieving posts based on view count (hits).', 'streamtube-core' ); ?>
			</span>

		</div>

		<hr />

		<h3><?php esc_html_e( 'OR specify the defined date range', 'streamtube-core' ); ?></h3>
		<p>
			<?php esc_html_e( 'The defined date range will be ignored if a custom date range (From and To) is provided.', 'streamtube-core' ); ?>
		</p>
		<div class="field-control">
			<?php printf(
				'<label for="%s">%s</label>',
				esc_attr( $widget->get_field_id( 'statistics_date' ) ),
				esc_html__( 'Date Range', 'streamtube-core' )
			); ?>

			<?php printf(
				'<input type="text" class="widefat" id="%s" name="%s" value="%s" placeholder="%s"/>',
				esc_attr( $widget->get_field_id( 'statistics_date' ) ),
				esc_attr( $widget->get_field_name( 'statistics_date' ) ),
				esc_attr( $instance['statistics_date'] ),
				esc_attr__( 'e.g: 7daysago', 'streamtube-core' )
			); ?>

			<span class="field-help">
				<?php printf(
					esc_html__( 'The date range to retrieve posts from, defined values are: %s', 'streamtube-core' ),
					'<strong>' . join( ', ', array_keys( Streamtube_Core_Widget_Filter_Post_Date::get_options() ) ) . '</strong>'
				) ?>
			</span>
		</div>

		<hr />

		<div class="field-control">
			<?php printf(
				'<label for="%s">%s</label>',
				esc_attr( $widget->get_field_id( 'statistics_metric' ) ),
				esc_html__( 'Metric', 'streamtube-core' )

			); ?>

			<?php printf(
				'<select class="widefat" id="%s" name="%s"/>',
				esc_attr( $widget->get_field_id( 'statistics_metric' ) ),
				esc_attr( $widget->get_field_name( 'statistics_metric' ) )

			); ?>

			<?php

			$metrics = array_merge(
				array(
					'' => esc_html__( 'Default', 'streamtube-core' )
				),
				streamtube_core_get_statistics_metrics()
			);

			foreach ( $metrics as $type => $text ) {
				printf(
					'<option value="%s" %s>%s</option>',
					esc_attr( $type ),
					selected( $type, $instance['statistics_metric'] ),
					esc_html( $text )
				);
			}

			?>

			</select>

		</div>

		<div class="field-control">
			<?php printf(
				'<input type="checkbox" class="widefat" id="%s" name="%s" %s/>',
				esc_attr( $widget->get_field_id( 'statistics_hits_range' ) ),
				esc_attr( $widget->get_field_name( 'statistics_hits_range' ) ),
				checked( 'on', $instance['statistics_hits_range'], false )

			); ?>
			<?php printf(
				'<label for="%s">%s</label>',
				esc_attr( $widget->get_field_id( 'statistics_hits_range' ) ),
				esc_html__( 'Display post views based on the configured date range, otherwise, all-time views will be shown.', 'streamtube-core' )
			); ?>
		</div>
		<?php

		return ob_get_clean();
	}

	/**
	 *
	 * Permission screen
	 * 
	 */
	public function register_permission_module() {
		streamtube_core_register_module(
			'wp-statistics',
			esc_html__( 'WP Statistics', 'streamtube-core' ),
			(array) StreamTube_Core_WP_Statistics_Permission::get_capabilities()
		);
	}

	/**
	 * Redirects specific WP Statistics admin requests to the user dashboard.
	 *
	 * Prevents direct access to certain WP Statistics admin pages from within the user dashboard,
	 * ensuring a consistent user experience.
	 * 
	 */
	public function filter_admin_request() {

		$referer     = wp_get_referer();
		$current_uri = $_SERVER['REQUEST_URI'];
		$user_uri    = untrailingslashit( get_author_posts_url( get_current_user_id() ) ) . '/dashboard';

		if ( ! str_contains( $referer, $user_uri ) || ! str_contains( $current_uri, '/admin.php?page=wps_' ) || wp_doing_ajax() ) {
			return;
		}

		// Parse URL and extract query parameters
		$parsed_url = parse_url( $current_uri );
		if ( empty( $parsed_url['query'] ) ) {
			return;
		}

		parse_str( $parsed_url['query'], $params );

		$page = $params['page'] ?? '';

		if ( preg_match( '/wps_(.*?)_page/', $page, $matches ) ) {
			$subpage = $matches[1];

			if ( ! WP_STATISTICS\Menus::in_page( $subpage ) ) {
				return;
			}

			$restricted_pages = [ 'settings', 'optimization', 'privacy-audit', 'plugins' ];

			if ( ! in_array( $subpage, $restricted_pages, true ) ) {
				$redirect_uri = $user_uri . '/statistics/' . ( $subpage !== 'overview' ? $subpage : '' );
				unset( $params['page'] );

				wp_redirect( add_query_arg( $params, $redirect_uri ) );
				exit;
			}
		}
	}

	public function rest_api_init() {
		register_rest_field(
			array_keys( streamtube_core_get_public_post_types() ),
			'statistics',
			array(
				'get_callback' => array( $this, 'get_rest_statistics_field' )
			)
		);
	}

	/**
	 *
	 * Add `wpstatistics` support for all public post types
	 * 
	 */
	public function add_post_type_support() {
		foreach ( streamtube_core_get_public_post_types() as $post_type ) {
			add_post_type_support( $post_type, 'wpstatistics' );
		}
	}

	public function add_widget_settings_tab( $tabs, $widget ) {
		$tabs['wpstatistics'] = array(
			'title'    => esc_html__( 'Statistics', 'streamtube-core' ),
			'callback' => array( $this, 'widget_settings_tab_callback' )
		);

		return $tabs;
	}

	/**
	 *
	 * Enqueue statistics css and scripts
	 * 
	 */
	public function enqueue_scripts() {

		$file = '/includes/admin/class-wp-statistics-admin-assets.php';

		if (
			! self::is_dashboard() ||
			! defined( 'WP_STATISTICS_DIR' ) ||
			! file_exists( untrailingslashit( WP_STATISTICS_DIR ) . $file ) ) {
			return;
		}

		$hook = 'dashboard';

		require_once untrailingslashit( WP_STATISTICS_DIR ) . $file;

		if ( ! class_exists( 'WP_STATISTICS\Admin_Assets' ) ) {
			return;
		}

		$Admin_Assets = new WP_STATISTICS\Admin_Assets();

		$is_statistics = self::is_statistics();

		$pages = array_merge( array_keys( WP_STATISTICS\Menus::$pages ), array( 'dashboard', 'user_dashboard', 'overview' ) );

		if ( $is_statistics && in_array( $is_statistics, $pages ) || get_post_type( $is_statistics ) ) {

			set_current_screen( $hook );

			$_GET['page'] = isset( $_REQUEST['page'] ) ? $_REQUEST['page'] : 'wps_' . $is_statistics . '_page';

			add_filter( 'wp_statistics_admin_assets', array( $this, 'filter_admin_script_assets' ) );

			$Admin_Assets->admin_styles();
			$Admin_Assets->admin_scripts( $hook );

			unset( $GLOBALS['current_screen'] );

			if ( WP_STATISTICS\Option::get( 'disable_dashboard' ) ) {

				wp_enqueue_script(
						$Admin_Assets::$prefix,
					$Admin_Assets::url( 'admin.min.js' ),
					array( 'jquery' ),
					$Admin_Assets::version(),
					[ 'in_footer' => true ]
				);

				wp_localize_script( $Admin_Assets::$prefix, 'wps_global', $Admin_Assets::wps_global( $hook ) );
			}

			if ( ! wp_script_is( $Admin_Assets::$prefix . '-jqvmap', 'enqueued' ) ) {
				wp_enqueue_script(
						$Admin_Assets::$prefix . '-jqvmap',
					$Admin_Assets::url( 'jqvmap/jquery.vmap.min.js' ),
					array( 'jquery' ), "1.5.1",
					[ 'in_footer' => true ]
				);

				wp_enqueue_script(
						$Admin_Assets::$prefix . '-jqvmap-world',
					$Admin_Assets::url( 'jqvmap/jquery.vmap.world.min.js' ),
					array( 'jquery' ),
					"1.5.1",
					[ 'in_footer' => true ]
				);
			}

			wp_enqueue_style(
				'streamtube-core-wp-statistics',
				STREAMTUBE_CORE_WP_STATISTICS_URL . '/public/assets/style.css',
				array( $Admin_Assets::$prefix ),
				filemtime( STREAMTUBE_CORE_WP_STATISTICS_PATH . '/public/assets/style.css' )
			);

			wp_enqueue_script(
				'streamtube-core-wp-statistics',
				STREAMTUBE_CORE_WP_STATISTICS_URL . '/public/assets/scripts.js',
				array( $Admin_Assets::$prefix ),
				filemtime( STREAMTUBE_CORE_WP_STATISTICS_PATH . '/public/assets/scripts.js' ),
				[ 'in_footer' => true ]
			);

			wp_localize_script( 'streamtube-core-wp-statistics', 'streamtube_wpstatistics_jsvars', array(
				'ref' => self::AFFID
			) );

			remove_filter( 'wp_statistics_admin_assets', array( $this, 'filter_admin_script_assets' ) );
		}
	}

	/**
	 * Enqueues the embed scripts for the frontend.
	 *
	 * @return void
	 */
	public function enqueue_embed_scripts() {
		$this->include_file( '/class-streamtube-core-statistics-frontend.php' );

		if ( class_exists( 'StreamTube_Core_WP_Statistics_Frontend' ) ) {
			$Frontend = new StreamTube_Core_WP_Statistics_Frontend();

			$Frontend->enqueue_embed_scripts();
		}
	}
}