<?php
/**
 * Define the Traffic Overview metabox functionality
 *
 * https://wordpress.org/plugins/wp-statistics/
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

if( ! defined('ABSPATH' ) ){
    return;
}
use WP_STATISTICS\Menus;
use WP_Statistics\Models\OnlineModel;
use WP_Statistics\Models\VisitorsModel;
use WP_Statistics\Models\ViewsModel;
use WP_Statistics\Components\View;
use WP_Statistics\Components\DateRange;
use WP_Statistics\Service\Charts\ChartDataProviderFactory;
use WP_Statistics\Service\Admin\Metabox\Metaboxes\TrafficOverview;

class StreamTube_Core_WP_Statistics_TrafficOverview extends TrafficOverview{

    protected $visitorsModel;

    protected $viewsModel;

    protected $onlineModel;

    public function __construct(){

        if( $this->canViewAll() ){
            parent::__construct();
        }else{
            $this->visitorsModel    = new VisitorsModel();
            $this->viewsModel       = new ViewsModel();
            $this->onlineModel      = new OnlineModel(); 
        }
    }

    public function getScreen(){
        return [Menus::get_action_menu_slug('overview'), 'dashboard'];
    }

    public function canViewAll(){
        return StreamTube_Core_WP_Statistics_Permission::view_all_statistics();
    }

    public function getData(){

        $args = [
            'ignore_post_type'  => false,
            'include_total'     => true
        ];

        if( ! $this->canViewAll() ){
            $args = array_merge( $args, array(
                'author_id'     =>  get_current_user_id()
            ) );

            $TrafficChartDataProvider = new StreamTube_Core_WP_Statistics_TrafficChartDataProvider(
                array_merge( 
                    $args,
                    [
                        'date' => DateRange::get('30days'), 
                        'prev_data' => true
                    ]
                )
            );

            $chartData = $TrafficChartDataProvider->getData();

            $views = $this->viewsModel->getViewsSummary($args);

            if( $views ){
                foreach ($views as &$item) {
                    if (isset($item['views'])) {
                        $item['hits'] = $item['views']; // Add the new key
                        unset($item['views']); // Remove the old key
                    }
                }
            }
            
            $data = [
                'online'    => $this->onlineModel->countOnlines($args),
                'visitors'  => $this->visitorsModel->getVisitorsSummary($args),
                'hits'      => $views
            ];              
        }else{
            $chartData  = $this->dataProvider->getTrafficChartData(
                array_merge(
                    $args, 
                    [
                        'date' => DateRange::get('30days'), 'prev_data' => true
                    ]
                )
            );
            $data       = $this->dataProvider->getTrafficOverviewData($args);            
        }

        // Merge chart data with template data
        $data = array_merge($data, [
            'total' => [
                'visitors'  => [
                    'current'   => array_sum($chartData['data']['datasets'][0]['data']),
                    'prev'      => array_sum($chartData['previousData']['datasets'][0]['data'])
                ],
                'views'     => [
                    'current'   => array_sum($chartData['data']['datasets'][1]['data']),
                    'prev'      => array_sum($chartData['previousData']['datasets'][1]['data'])
                ]
            ]
        ]);

        // Prevent previous data from being sent to the js
        unset($chartData['previousData']);

        $output = View::load('metabox/traffic-overview', ['data' => $data], true);

        return [
            'data'      => $chartData,
            'output'    => $output
        ];
    }    
}