<?php
if( ! defined('ABSPATH') ){
    exit;
}

use WP_Statistics\Components\DateRange;
use WP_STATISTICS\Helper;
use WP_Statistics\Models\ViewsModel;
use WP_Statistics\Models\VisitorsModel;
use WP_Statistics\Service\Charts\AbstractChartDataProvider;
use WP_Statistics\Service\Charts\Traits\LineChartResponseTrait;
use WP_STATISTICS\TimeZone;
use WP_Statistics\Service\Charts\DataProvider\TrafficChartDataProvider;

class StreamTube_Core_WP_Statistics_TrafficChartDataProvider extends TrafficChartDataProvider{

    protected function setThisPeriodData() {
        // Get the current period and list of dates
        $currentPeriod = $this->args['date'] ?? DateRange::get();
        $currentDates = array_keys(TimeZone::getListDays($currentPeriod));

        // Fetch daily visitors data
        $visitorsData = $this->visitorsModel->countDailyVisitors(array_merge($this->args, ['include_hits' => true]));
        $visitorsData = $visitorsData ? array_map(fn($item) => (array) $item, $visitorsData) : [];

        // Fetch daily views data
        $viewsData = $this->viewsModel->countDailyViews(array_merge($this->args, ['date' => $currentPeriod]));
        $viewsData = $viewsData ? array_map(fn($item) => (array) $item, $viewsData) : [];
        $views = $viewsData ? array_column($viewsData, 'views', 'date') : [];

        // Create a normal indexed array
        $data = [];

        foreach ($currentDates as $date) {
            // Initialize values in case of no data
            $visitorData = current(array_filter($visitorsData, fn($visitor) => $visitor['date'] === $date));
            $viewsData = $views[$date] ?? 0;  // Default to 0 if no views data

            $data[] = [
                'date'     => $date,
                'visitors' => $visitorData['visitors'] ?? 0,  // Default to 0 if no visitor data
                'views'    => $viewsData,
            ];
        }

        // Parse the merged data
        $data = $this->parseData($currentDates, $data);

        // Set chart labels and datasets
        $this->setChartLabels($data['labels']);
        $this->addChartDataset(esc_html__('Visitors', 'wp-statistics'), $data['visitors']);
        $this->addChartDataset(esc_html__('Views', 'wp-statistics'), $data['views']);
    }

    protected function setPrevPeriodData() {
        $currentPeriod = $this->args['date'] ?? DateRange::get();
        $prevPeriod = DateRange::getPrevPeriod($currentPeriod);
        $prevDates = array_keys(TimeZone::getListDays($prevPeriod));

        // Fetch daily visitors data for the previous period
        $visitorsData = $this->visitorsModel->countDailyVisitors(array_merge($this->args, ['include_hits' => true, 'date' => $prevPeriod]));
        $visitorsData = $visitorsData ? array_map(fn($item) => (array) $item, $visitorsData) : [];

        // Fetch daily views data for the previous period
        $viewsData = $this->viewsModel->countDailyViews(array_merge($this->args, ['date' => $prevPeriod]));
        $viewsData = $viewsData ? array_map(fn($item) => (array) $item, $viewsData) : [];
        $views = $viewsData ? array_column($viewsData, 'views', 'date') : [];

        // Create a normal indexed array
        $data = [];

        foreach ($prevDates as $date) {
            // Initialize values in case of no data
            $visitorData = current(array_filter($visitorsData, fn($visitor) => $visitor['date'] === $date));
            $viewsData = $views[$date] ?? 0;  // Default to 0 if no views data

            $data[] = [
                'date'     => $date,
                'visitors' => $visitorData['visitors'] ?? 0,  // Default to 0 if no visitor data
                'views'    => $viewsData,
            ];
        }

        // Parse the merged data
        $data = $this->parseData($prevDates, $data);

        // Set previous period chart labels and datasets
        $this->setChartPreviousLabels($data['labels']);
        $this->addChartPreviousDataset(esc_html__('Visitors', 'wp-statistics'), $data['visitors']);
        $this->addChartPreviousDataset(esc_html__('Views', 'wp-statistics'), $data['views']);
    }

    protected function parseData($dates, $data){
        $visitors = wp_list_pluck($data, 'visitors', 'date');
        $views    = wp_list_pluck($data, 'views', 'date');

        $parsedData = [];
        foreach ($dates as $date) {
            $parsedData['labels'][]   = [
                'formatted_date'    => date_i18n(Helper::getDefaultDateFormat(false, true, true), strtotime($date)),
                'date'              => date_i18n('Y-m-d', strtotime($date)),
                'day'               => date_i18n('l', strtotime($date))
            ];
            $parsedData['visitors'][] = isset($visitors[$date]) ? intval($visitors[$date]) : 0;
            $parsedData['views'][]    = isset($views[$date]) ? intval($views[$date]) : 0;
        }

        return $parsedData;
    }	
}