<?php
/**
 * YITH WooCommerce Affiliates
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.2
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      2.2
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class StreamTube_Core_WooCommerce_YITH_Affiliates {
	/**
	 * The slug for the affiliate dashboard.
	 *
	 * This constant defines the URL slug used for the affiliate dashboard in the YITH WooCommerce Affiliates integration.
	 */
	const DASHBOARD_SLUG = 'affiliate-dashboard';

	/**
	 * Retrieves the location of the affiliate dashboard.
	 *
	 * This function constructs the URL for the affiliate dashboard by combining the author's posts URL,
	 * a predefined dashboard slug, and an endpoint.
	 *
	 * @return string The URL of the affiliate dashboard.
	 */
	private static function get_affiliate_dashboard_url( $endpoint = '' ) {
		return sprintf(
			'%s/dashboard/shopping/%s/%s/',
			untrailingslashit( get_author_posts_url( get_current_user_id() ) ),
			self::DASHBOARD_SLUG,
			$endpoint
		);
	}

	/**
	 * Checks if the current query is for the affiliate dashboard.
	 *
	 * This function determines if the current query is for the WooCommerce Affiliates dashboard
	 * by checking the 'dashboard' query variable and matching it against a specific pattern.
	 *
	 * @param WP_Query|null $wp_query The current query object. If null, the global $wp_query will be used.
	 * 
	 * @return bool|string Returns the matched pattern if the query is for the affiliate dashboard, 
	 *                     true if the 'dashboard' query variable is set and contains the dashboard slug,
	 *                     or false otherwise.
	 */
	private static function is_affiliate_dashboard( $wp_query = null ) {

		if ( $wp_query === null ) {
			global $wp_query;
		}

		if ( isset( $wp_query->query_vars['dashboard'] ) && strpos( $wp_query->query_vars['dashboard'], self::DASHBOARD_SLUG ) !== false ) {
			/**
			 * Checks if the current query matches a specific pattern for the WooCommerce Affiliates dashboard.
			 *
			 * @return bool Returns true if the pattern matches, false otherwise.
			 */
			if ( preg_match( '/affiliate-dashboard\/(.+?)\/?$/', $wp_query->query_vars['dashboard'], $matches ) ) {
				return $matches[1];
			}

			return true;
		}

		return false;
	}

	/**
	 * Retrieves the dashboard location setting.
	 *
	 * This function fetches the dashboard location setting from the options table.
	 * If the setting is not found, it defaults to 'page'.
	 *
	 * @return string The dashboard location setting.
	 */
	private function get_dashboard_location() {
		return get_option( 'yith_wcaf_dashboard_location', 'page' );
	}

	/**
	 * Parses the request to handle custom dashboard endpoints.
	 *
	 * This function checks if the request contains the custom dashboard endpoint and processes it accordingly.
	 * It extracts the endpoint value from the URL and sets it as a query variable.
	 *
	 * @param WP $wp The current WordPress environment instance.
	 */
	public function parse_request( $wp_query ) {
		if ( $this->get_dashboard_location() === 'my-account' ) {

			$endpoint = self::is_affiliate_dashboard( $wp_query );

			if ( is_string( $endpoint ) ) {
				$wp_query->query_vars[ $endpoint ] = true;
			}
		}
	}

	/**
	 * Filters the URL for a specific YITH WooCommerce Affiliates endpoint.
	 *
	 * This function allows customization of the URL for a given endpoint in the YITH WooCommerce Affiliates plugin.
	 *
	 * @param string $url The original URL.
	 * @param string $endpoint The endpoint name.
	 * @param string $value The value to be added to the endpoint URL.
	 * @param string $permalink The permalink structure.
	 * @return string The filtered URL.
	 */
	public function filter_yith_wcaf_get_endpoint_url( $url, $endpoint, $value, $permalink ) {

		if ( $this->get_dashboard_location() === 'my-account' ) {
			return self::get_affiliate_dashboard_url( $endpoint );
		}

		return $url;
	}

	/**
	 * Filters the redirection URL for becoming an affiliate.
	 *
	 * This function modifies the redirection URL when a user becomes an affiliate
	 * by returning the affiliate dashboard location.
	 *
	 * @param string $url The original redirection URL.
	 * @return string The modified redirection URL pointing to the affiliate dashboard.
	 */
	public function filter_yith_wcaf_become_an_affiliate_redirection( $url ) {
		return self::get_affiliate_dashboard_url();
	}
	/**
	 * Filters the page heading for the affiliate dashboard.
	 *
	 * This function checks if the current page is the affiliate dashboard.
	 * If it is, it returns the translated string 'Affiliate' as the heading.
	 * Otherwise, it returns the original heading.
	 *
	 * @param string $heading The original heading.
	 * @return string The filtered heading.
	 */
	public function filter_page_heading( $heading = '' ) {
		if ( self::is_affiliate_dashboard() ) {
			return esc_html__( 'Affiliate', 'streamtube-core' );
		}
		return $heading;
	}

	/**
	 * Add `Shopping > Affiliates` dashboard menu.
	 *
	 * This function modifies the WooCommerce dashboard menu to include an "Affiliates" submenu under the "Shopping" menu.
	 * It checks if the WooCommerce and YITH WooCommerce Affiliates plugins are active before making changes.
	 *
	 * @param array $menu_items The existing dashboard menu items.
	 * @return array The modified dashboard menu items.
	 */
	public function add_dashboard_menu( $menu_items ) {
		if ( ! function_exists( 'WC' ) || ! function_exists( 'yith_affiliates_constructor' ) ) {
			return $menu_items;
		}

		if ( isset( $menu_items['shopping']['submenu'][ self::DASHBOARD_SLUG ] ) ) {
			$menu_items['shopping']['submenu'][ self::DASHBOARD_SLUG ]['title'] = esc_html__( 'Affiliates', 'streamtube-core' );
		}

		return $menu_items;
	}

	/**
	 * Enqueue the necessary styles for the Streamtube Core YITH WooCommerce Affiliates plugin.
	 *
	 * This function registers and enqueues the stylesheet for the Streamtube Core YITH WooCommerce Affiliates plugin.
	 * It ensures that the stylesheet is loaded with the correct dependencies and version based on the file modification time.
	 *
	 * @return void
	 */
	public function enqueue_scripts() {
		wp_enqueue_style(
			'streamtube-core-yith-affiliates',
			untrailingslashit( plugin_dir_url( __FILE__ ) ) . '/assets/style.css',
			array( 'yith-wcaf' ),
			filemtime( untrailingslashit( plugin_dir_path( __FILE__ ) ) . '/assets/style.css' )
		);
	}
}