<?php
/**
 * Define the Youtube Importer functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class StreamTube_Core_Youtube_Importer {

	/**
	 *
	 * Holds the admin object
	 * 
	 * @var object
	 *
	 * @since 2.0
	 * 
	 */
	public $admin;

	/**
	 *
	 * Holds the post_type object
	 * 
	 * @var object
	 *
	 * @since 2.0
	 * 
	 */
	public $post_type;

	/**
	 *
	 * Holds the options object
	 * 
	 * @var object
	 *
	 * @since 2.0
	 * 
	 */
	public $options;

	/**
	 *
	 * Holds the Youtube API object
	 * 
	 * @var object
	 *
	 * @since 2.0
	 * 
	 */
	public $api;

	public $oauth;

	/**
	 *
	 * Class contructor
	 *
	 * @since 2.0
	 * 
	 */
	public function __construct() {

		$this->api = new stdClass();

		$this->create_oauth();

		$this->load_dependencies();
	}

	/**
	 *
	 * Include file
	 * 
	 * @param  string $file
	 *
	 * @since 2.0
	 * 
	 */
	private function include_file( $file ) {
		require_once plugin_dir_path( __FILE__ ) . $file;
	}

	/**
	 *
	 * Load dependencies
	 *
	 * @since 2.0
	 * 
	 */
	public function load_dependencies() {

		$this->include_file( 'class-streamtube-core-youtube-api.php' );

		$this->include_file( 'class-streamtube-core-youtube-api-search.php' );

		$this->include_file( 'class-streamtube-core-youtube-api-videos.php' );

		$this->api->search = new StreamTube_Core_Youtube_API_Search();

		$this->api->video = new StreamTube_Core_Youtube_API_Videos();

		$this->include_file( 'class-streamtube-core-youtube-admin.php' );

		$this->admin = new StreamTube_Core_Youtube_Importer_Admin();

		$this->include_file( 'class-streamtube-core-youtube-post-type.php' );

		$this->post_type = new StreamTube_Core_Youtube_Importer_Post_Type();

		$this->include_file( 'class-streamtube-core-youtube-options.php' );

		$this->options = new StreamTube_Core_Youtube_Importer_Options();
	}

	/**
	 *
	 * Get custom schedules
	 * 
	 * @return array
	 */
	public function get_schedules() {
		$schedules = array(
			'hourly'     => array(
				'interval' => HOUR_IN_SECONDS,
				'display'  => esc_html__( 'Once Hourly', 'streamtube-core' ),
			),
			'twicedaily' => array(
				'interval' => 12 * HOUR_IN_SECONDS,
				'display'  => esc_html__( 'Twice Daily', 'streamtube-core' ),
			),
			'daily'      => array(
				'interval' => DAY_IN_SECONDS,
				'display'  => esc_html__( 'Once Daily', 'streamtube-core' ),
			),
			'weekly'     => array(
				'interval' => WEEK_IN_SECONDS,
				'display'  => esc_html__( 'Once Weekly', 'streamtube-core' )
			)
		);

		/**
		 *
		 * Filter $schedules
		 * 
		 */
		return apply_filters( 'streamtube/core/youtube_importer/get_schedules', $schedules );
	}

	/**
	 *
	 * Set video thumbnail
	 * 
	 * @param int $post_id
	 * @param array $item
	 *
	 * @since 2.0
	 * 
	 */
	private function set_post_thumbnail( $post_id, $thumbnail_url ) {
		if ( ! function_exists( 'media_sideload_image' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/media.php' );
			require_once( ABSPATH . 'wp-admin/includes/file.php' );
			require_once( ABSPATH . 'wp-admin/includes/image.php' );
		}

		if ( apply_filters( 'streamtube/core/hash_youtube_image_file_name', true ) ) {
			add_filter(
				'sanitize_file_name',
				'streamtube_core_hash_file_name',
				10,
				1
			);
		}

		$thumbnail_id = media_sideload_image( $thumbnail_url, $post_id, null, 'id' );

		if ( is_int( $thumbnail_id ) ) {
			set_post_thumbnail( $post_id, $thumbnail_id );
		}

		return $thumbnail_id;
	}

	/**
	 *
	 * Set video terms
	 * 
	 * @param int $post_id
	 * @param int $importer_id
	 *
	 * @since 2.0
	 * 
	 */
	private function set_post_terms( $post_id, $importer_id ) {
		$taxonomies = get_object_taxonomies( Streamtube_Core_Post::CPT_VIDEO, 'object' );

		foreach ( $taxonomies as $tax => $object ) {

			$fields = is_taxonomy_hierarchical( $tax ) ? 'ids' : 'slugs';

			$terms = wp_get_post_terms( $importer_id, $tax, array(
				'fields' => $fields
			) );

			if ( $terms ) {
				wp_set_post_terms( $post_id, $terms, $tax, true );
			}
		}
	}

	/**
	 *
	 * Check if video imported
	 * 
	 * @param  string  $yt_id
	 * @return true|false
	 *
	 * @since 2.0
	 * 
	 */
	public function is_existed( $yt_id, $importer_id = '' ) {
		global $wpdb;

		// Define post statuses to check
		$post_statuses = array( 'publish', 'pending', 'private', 'unlist', 'future', 'draft' );

		if ( function_exists( 'get_post_stati' ) ) {
			$post_statuses = array_keys( get_post_stati() );
		}

		/**
		 *
		 * Filter the post statuses
		 * 
		 * @var array
		 */
		$post_statuses = apply_filters(
			'streamtube/core/youtube_importer/check_exist_post_statuses',
			$post_statuses,
			$yt_id,
			$importer_id
		);

		$settings = $this->admin->get_settings( $importer_id );

		// Default to the custom post type if not set
		$post_type = ! empty( $settings['post_type'] ) ? $settings['post_type'] : Streamtube_Core_Post::CPT_VIDEO;

		$sql = "
            SELECT posts.ID 
            FROM {$wpdb->postmeta} AS meta
            INNER JOIN {$wpdb->posts} AS posts ON posts.ID = meta.post_id
            WHERE posts.post_type = %s 
                AND meta.meta_key = 'video_url' 
                AND meta.meta_value LIKE %s 
                AND posts.post_status IN (" . implode( ',', array_fill( 0, count( $post_statuses ), '%s' ) ) . ")
        ";

		// Prepare and execute the SQL statement
		$placeholders = array_merge( [ $post_type, "%{$yt_id}%" ], $post_statuses );
		$prepared_sql = $wpdb->prepare( $sql, ...$placeholders );

		$results = $wpdb->get_var( $prepared_sql );

		// Return true if a matching post was found, false otherwise
		return ! empty( $results );
	}

	/**
	 *
	 * Search Youtube content
	 * 
	 * @param  integer $importer_id
	 * @param  array   $settings
	 * @return Wp_Error|Array
	 *
	 * @since 2.0
	 * 
	 */
	public function search_content( $importer_id = 0, $settings = array() ) {

		$post = get_post( $importer_id );

		$post_type_cap = get_post_type_object( StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE )->cap;

		$settings = wp_parse_args( $settings, $this->admin->get_settings( $importer_id ) );

		if ( $settings['publishedAfter'] ) {
			$settings['publishedAfter'] = date( 'Y-m-d\TH:i:s\Z', strtotime( $settings['publishedAfter'] ) );
		}

		if ( $settings['publishedBefore'] ) {
			$settings['publishedBefore'] = date( 'Y-m-d\TH:i:s\Z', strtotime( $settings['publishedBefore'] ) );
		}

		$settings = wp_parse_args( $settings, array(
			'channelId' => ''
		) );

		$settings['channelId'] = trim( $settings['channelId'] );

		if ( preg_match( '/^(PLV-|PL)/i', $settings['channelId'] ) ) {
			$settings['searchIn'] = 'playlist';
		} else {
			$settings['searchIn'] = 'channel';
		}

		/**
		 *
		 * Filter the searchIn value
		 * Maybe for pre-filter?
		 * 
		 */
		$settings['searchIn'] = apply_filters(
			'streamtube/core/youtube_importer/search_content/settings/search_in',
			$settings['searchIn'],
			$settings
		);

		if ( $settings['searchIn'] == 'playlist' ) {
			$settings['playlistId'] = $settings['channelId'];
			unset( $settings['channelId'] );

			$this->api->search->set_api_endpoint( '/playlistItems' );
		}

		if ( wp_validate_boolean( $settings['forMine'] ) ) {
			$settings['type'] = 'video';

			$videoAttributes = array(
				"videoDefinition",
				"videoDimension",
				"videoDuration",
				"videoEmbeddable",
				"videoLicense",
				"videoPaidProductPlacement",
				"videoSyndicated",
				"videoType"
			);

			for ( $i = 0; $i < count( $videoAttributes ); $i++ ) {
				if ( array_key_exists( $videoAttributes[ $i ], $settings ) ) {
					unset( $settings[ $videoAttributes[ $i ] ] );
				}
			}
		}

		foreach ( $settings as $key => $value ) {
			if ( ! $value || empty( $value ) ) {
				unset( $settings[ $key ] );
			}
		}
		;

		if ( ! user_can( $post->post_author, $post_type_cap->search_videos ) ) {

			if ( ! isset( $settings['channelId'] ) && ! isset( $settings['playlistId'] ) && ! $settings['forMine'] ) {
				return new WP_Error(
					'no_search_videos_permission',
					sprintf(
						esc_html__( '%s does not have permission to search for videos across YouTube. Please select a channel or playlist, or choose to search your own videos.', 'streamtube-core' ),
						get_userdata( $post->post_author )->display_name
					)
				);
			}
		}

		$permissions = array(
			'channelId'  => array(
				'capability'    => $post_type_cap->import_others_channels,
				'error_code'    => 'no_import_others_yt_channels_permission',
				'error_message' => esc_html__( '%s does not have permission to import videos from other YouTube channels.', 'streamtube-core' ),
			),
			'playlistId' => array(
				'capability'    => $post_type_cap->import_others_playlists,
				'error_code'    => 'no_import_others_yt_playlists_permission',
				'error_message' => esc_html__( '%s does not have permission to import videos from other YouTube playlists.', 'streamtube-core' ),
			)
		);

		foreach ( $permissions as $key => $details ) {
			if ( isset( $settings[ $key ] ) && ! empty( $settings[ $key ] ) ) {
				if ( ! user_can( $post->post_author, $details['capability'] ) ) {
					$lists = $this->get_importable_lists( $post->post_author );

					if ( ! array_key_exists( $settings[ $key ], $lists ) ) {
						return new WP_Error(
							$details['error_code'],
							sprintf(
								$details['error_message'],
								get_userdata( $post->post_author )->display_name
							)
						);
					}
				}
			}
		}

		/**
		 *
		 * Pre filter the $settings before sending to API
		 *
		 * @param array $settings
		 * @param WP_Post $importer post
		 * 
		 */
		$settings = apply_filters(
			'streamtube/core/youtube_importer/search_content/settings',
			$settings,
			$post
		);

		$response = $this->api->search->get_data( $settings['apikey'], $settings );

		if ( ! is_wp_error( $response ) ) {
			update_post_meta( $importer_id, '_total', $this->api->search->get_total_results( $response ) );
		}

		return $response;
	}

	/**
	 *
	 * Import Youtube content
	 * 
	 * @param  array $yt_video_ids
	 * @param  int   $importer_id
	 * @return Wp_Error|Array
	 *
	 * @since 2.0
	 * 
	 */
	public function import_content( $yt_video_ids, $importer_id = 0 ) {
		$errors = new WP_Error();

		$posts                   = array();
		$settings                = $this->admin->get_settings( $importer_id );
		$youtube_importer_post   = get_post( $importer_id );
		$youtube_importer_object = get_post_type_object( $youtube_importer_post->post_type );

		if ( ! $settings['post_author'] ) {
			$settings['post_author'] = $youtube_importer_post->post_author;
		}

		if ( empty( $settings['post_type'] ) ) {
			$settings['post_type'] = Streamtube_Core_Post::CPT_VIDEO;
		}

		if ( ! user_can( $youtube_importer_post->post_author, get_post_type_object( $settings['post_type'] )->cap->edit_posts ) ) {
			$errors->add(
				'no_permission',
				sprintf(
					esc_html__( 'Sorry, you do not have permission to import Youtube videos into %s type.', 'streamtube-core' ),
					get_post_type_object( $settings['post_type'] )->labels->name
				)
			);
		}

		if ( ! user_can( $youtube_importer_post->post_author, $youtube_importer_object->cap->edit_post, $importer_id ) ) {
			$errors->add(
				'no_permission',
				sprintf(
					esc_html__( 'Sorry, you do not have permission to import videos from %s', 'streamtube-core' ),
					$youtube_importer_post->post_title
				)
			);
		}

		if ( ! $yt_video_ids ) {
			$errors->add(
				'no_yt_video_id',
				esc_html__( 'No Youtube Video ID', 'streamtube-core' )
			);
		}

		/**
		 * Filter WP_Errors
		 */
		$errors = apply_filters(
			'streamtube/core/youtube_importer/import_content/errors',
			$errors,
			$yt_video_ids,
			$importer_id,
			$settings
		);

		if ( $errors->get_error_codes() ) {
			return $errors;
		}

		$response = $this->api->video->get_data( $settings['apikey'], array(
			'id' => is_array( $yt_video_ids ) ? join( ',', $yt_video_ids ) : $yt_video_ids
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( $response['items'] && count( $response['items'] ) > 0 ) {
			for ( $i = 0; $i < count( $response['items'] ); $i++ ) {

				if ( ! $this->is_existed( $this->api->video->get_item_id( $response['items'][ $i ] ), $importer_id ) ) {

					$video_url = $this->api->video->get_item_url( $response['items'][ $i ] );

					$statistics      = $this->api->video->get_item_statistics( $response['items'][ $i ] );
					$content_details = $this->api->video->get_item_content_details( $response['items'][ $i ] );

					if ( array_key_exists( 'duration', $content_details ) ) {
						$content_details['_length'] = streamtube_convert_youtube_duration( $content_details['duration'] );
					}

					$post_args = array(
						'post_type'    => $settings['post_type'],
						'post_title'   => $this->api->video->get_item_title( $response['items'][ $i ] ),
						'post_content' => $this->api->video->get_item_description( $response['items'][ $i ] ),
						'post_status'  => $settings['post_status'] ? $settings['post_status'] : 'pending',
						'post_author'  => $settings['post_author'],
						'meta_input'   => array_merge( $statistics, $content_details, compact( 'video_url' ) )
					);

					if ( is_post_type_viewable( $settings['post_type'] ) ) {
						$post_args['post_type'] = trim( $settings['post_type'] );
					}

					if ( $post_args['post_type'] != Streamtube_Core_Post::CPT_VIDEO ) {
						if ( ! empty( $settings['post_meta_field'] ) ) {
							$meta_key                = trim( sanitize_key( $settings['post_meta_field'] ) );
							$post_args['meta_input'] = array_merge( $post_args['meta_input'], array(
								$meta_key => $video_url
							) );
						} else {
							$post_args['post_content'] = $video_url . '<br/>' . $post_args['post_content'];
						}
					}

					/**
					 *
					 * Filter post args
					 * 
					 */
					$post_args = apply_filters(
						'streamtube/core/youtube_importer/post_args',
						$post_args,
						$response['items'][ $i ],
						$response,
						$settings
					);

					$post_id = wp_insert_post( $post_args, true );

					if ( ! is_wp_error( $post_id ) ) {

						if ( $maybe_thumbnail_url = $this->api->video->get_item_thumbnail_url( $response['items'][ $i ] ) ) {
							$this->set_post_thumbnail( $post_id, $maybe_thumbnail_url );
						}

						$this->set_post_terms( $post_id, $importer_id );

						if ( $settings['post_tags'] ) {
							$tags = $this->api->video->get_item_tags( $response['items'][ $i ] );

							$taxonomy_tag = $settings['post_type'] == Streamtube_Core_Post::CPT_VIDEO ? 'video_tag' : 'post_tag';

							$taxonomy_tag = apply_filters(
								'streamtube/core/youtube_importer/taxonomy_tag',
								$taxonomy_tag,
								$importer_id,
								$response['items'][ $i ],
								$response,
								$settings
							);

							if ( $tags ) {

								$max_tags_items = (int) get_option( sprintf( 'taxonomy_%s_%s_max_items', $settings['post_type'], $taxonomy_tag ), 0 );

								if ( Streamtube_Core_Permission::moderate_posts( $post_args['post_author'], $settings['post_type'] ) ) {
									$max_tags_items = 0;
								}

								if ( $max_tags_items > 0 ) {
									$tags = array_slice( $tags, 0, $max_tags_items );
								}

								wp_set_post_terms( $post_id, $tags, $taxonomy_tag, true );
							}
						}

						if ( $importer_id ) {
							update_post_meta( $post_id, 'yt_importer_id', $importer_id );
						}

						/**
						 *
						 * Fires after video imported
						 *
						 * @since 2.0
						 * 
						 */
						do_action(
							'streamtube/core/youtube_importer/imported',
							$post_id,
							$importer_id,
							$response['items'][ $i ],
							$response,
							$settings
						);

						$posts[ $this->api->video->get_item_id( $response['items'][ $i ] ) ] = get_permalink( $post_id );
					}
				}
			}
		}

		return compact( 'response', 'posts' );
	}

	/**
	 *
	 * Get imported videos from given importer ID
	 * 
	 * @param  int $importer_id
	 * @return false|array
	 *
	 * @since 2.0
	 * 
	 */
	public function get_imported_videos( $importer_id, $number = 5 ) {
		global $wpdb;

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT post_id FROM $wpdb->postmeta where meta_key = %s and meta_value = %d ORDER BY post_id DESC LIMIT %d",
				'yt_importer_id',
				$importer_id,
				$number
			),
			OBJECT
		);

		return $results;
	}

	/**
	 * AJAX search content
	 *
	 * @since 2.0
	 */
	public function ajax_search_content() {

		if ( ! isset( $_POST['post_ID'] ) ) {
			wp_send_json_error( new WP_Error(
				'post_id_not_found',
				esc_html__( 'Post ID was not found', 'streamtube-core' )
			) );
		}

		$post_type_object = get_post_type_object( StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE );

		if ( ! current_user_can( $post_type_object->cap->edit_post, $_POST['post_ID'] ) || ! is_array( $_POST['yt_importer'] ) ) {
			wp_send_json_error( new WP_Error(
				'no_permission',
				sprintf(
					esc_html__( 'Sorry, you do not have permission to edit %s', 'streamtube-core' ),
					$post_type_object->labels->name
				)
			) );
		}

		$params = wp_parse_args( $_POST['yt_importer'], array(
			'forMine'             => '',
			'order_playlist_date' => '',
			'order_asc'           => ''
		) );

		if ( isset( $_POST['next_page_token'] ) ) {
			$params['pageToken'] = wp_unslash( $_POST['next_page_token'] );
		}

		$response = $this->search_content( $_POST['post_ID'], $params );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		if ( $this->api->search->get_item_ids( $response ) ) {

			$items = $response['items'];

			ob_start();

			for ( $i = 0; $i < count( $items ); $i++ ) {
				load_template( plugin_dir_path( __FILE__ ) . 'admin/item-loop.php', false, array_merge( $items[ $i ], array(
					'importer_id' => $_POST['post_ID']
				) ) );
			}

			$li = ob_get_clean();

			$output = sprintf(
				'<ul class="yt-video-list border">%s</ul>',
				$li
			);

			if ( $this->api->search->get_next_page_token( $response ) ) {
				$output .= sprintf(
					'<button type="button" class="btn btn-danger d-block w-100 button button-primary button-search-youtube button-yt-next-page" data-next-page-token="%s">%s</button>',
					esc_attr( $this->api->search->get_next_page_token( $response ) ),
					esc_html__( 'Load more', 'streamtube-core' )
				);
			}

			wp_send_json_success( $output );

		}

		wp_send_json_error( new WP_Error(
			'no_content',
			esc_html__( 'No content was found', 'streamtube-core' )
		) );
	}

	/**
	 * AJAX import content
	 *
	 * @since 2.0
	 */
	public function ajax_import_content() {

		$item_id     = isset( $_POST['item_id'] ) ? $_POST['item_id'] : '';
		$importer_id = isset( $_POST['importer_id'] ) ? (int) $_POST['importer_id'] : 0;
		$nonce       = isset( $_POST['nonce'] ) ? $_POST['nonce'] : '';

		if ( ! wp_verify_nonce( $nonce, $item_id . $importer_id ) ) {
			wp_send_json_error(
				new WP_Error(
					'invalid_nonce',
					esc_html__( 'Invalid nonce. Please refresh the page or try again later.', 'streamtube-core' )
				)
			);
		}

		$response = $this->import_content( $item_id, $importer_id );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( array_merge( $response, array(
			'message' => esc_html__( 'Imported', 'streamtube-core' )
		) ) );
	}

	/**
	 *
	 * AJAX bulk import
	 * 
	 */
	public function ajax_bulk_import_content() {

		$http_data = wp_parse_args( $_REQUEST, array(
			'nonce'  => '',
			'yt_ids' => array()
		) );

		extract( $http_data );

		check_ajax_referer( 'bulk_import_youtube', 'nonce' );

		$post_type_cap = get_post_type_object( StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE )->cap;

		if ( ! current_user_can( $post_type_cap->bulk_import ) ) {
			wp_send_json_error( new WP_Error(
				'no_permission',
				esc_html__( 'You do not have permission to bulk import YouTube content.', 'streamtube-core' )
			) );
		}

		if ( is_string( $yt_ids ) || is_array( $yt_ids ) && count( $yt_ids ) == 0 ) {
			wp_send_json_error( new WP_Error(
				'no_content',
				esc_html__( 'No content were submitted', 'streamtube-core' )
			) );
		}

		$response = $this->import_content( $_POST['yt_ids'], $_POST['post_ID'] );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( array_merge( $response, array(
			'message' => esc_html__( 'Imported', 'streamtube-core' )
		) ) );
	}

	/**
	 *
	 * Run cron job bulk import content
	 * 
	 * @since 2.0
	 */
	public function run_bulk_import_content( $importer_id, $settings = array() ) {

		$last_check = current_time( 'timestamp' );

		if ( ! $settings ) {
			$settings = $this->admin->get_settings( $importer_id );
		}

		if ( ! $settings['enable'] || get_post_status( $importer_id ) !== 'publish' ) {
			return new WP_Error(
				'disabled_importer',
				esc_html__( 'This importer is either disabled or not published yet.', 'streamtube-core' )
			);
		}

		if ( (int) $settings['update_number'] == 0 ) {
			return new WP_Error(
				'invalid_number',
				esc_html__( 'Invalid Number', 'streamtube-core' )
			);
		}

		$settings['maxResults'] = (int) $settings['update_number'];

		if ( "" != $next_page_token = get_post_meta( $importer_id, 'next_page_token', true ) ) {
			$settings['pageToken'] = $next_page_token;
		}

		$response = $this->search_content( $importer_id, $settings );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		update_post_meta( $importer_id, 'last_check', $last_check );

		$item_ids = $this->api->search->get_item_ids( $response );

		if ( ! $item_ids ) {
			return new WP_Error(
				'no_content',
				esc_html__( 'No content was found', 'streamtube-core' )
			);
		}

		$results = $this->import_content( $item_ids, $importer_id );

		if ( is_wp_error( $results ) ) {
			return $results;
		}

		$next_page_token = $this->api->search->get_next_page_token( $response );

		if ( is_string( $next_page_token ) ) {
			update_post_meta( $importer_id, 'next_page_token', $next_page_token );
		}

		/**
		 *
		 * Fires after importing completed
		 *
		 * @param int $importer_id
		 * @param array $results
		 *
		 * @since 2.0
		 * 
		 */
		do_action( 'streamtube/core/youtube_importer/bulk_imported', $importer_id, $results );

		return compact( 'response', 'results', 'last_check' );
	}

	/**
	 *
	 * AJAX bulk import button handler
	 * 
	 * @since 2.0
	 */
	public function ajax_run_bulk_import_content() {

		$http_data = wp_parse_args( $_REQUEST, array(
			'importer_id' => 0,
			'key'         => '',
			'nonce'       => ''
		) );

		extract( $http_data );

		if ( ! $importer_id || ! wp_verify_nonce( $nonce, 'run_schedule_' . $importer_id ) ) {
			wp_send_json_error( new WP_Error(
				'invalid_nonce',
				esc_html__( 'Invalid nonce. Please refresh the page or try again later.', 'streamtube-core' )
			) );
		}

		$results = $this->run_bulk_import_content( $importer_id );

		if ( is_wp_error( $results ) ) {
			wp_send_json_error( $results );
		}

		$count = count( $results['results']['posts'] );

		$message = sprintf(
			_n( '%s post has been imported.', '%s posts have been imported.', $count, 'streamtube-core' ),
			number_format_i18n( $count )
		);

		$last_check = sprintf(
			esc_html__( '%s ago', 'streamtube-core' ),
			human_time_diff( $results['last_check'], current_time( 'timestamp' ) )
		);

		wp_send_json_success( compact( 'message', 'last_check' ) );
	}

	/**
	 *
	 * Cron job task
	 *
	 * @since 2.0
	 * 
	 */
	public function template_run_bulk_import_content( $template ) {

		global $post;

		if ( ! is_singular( StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE ) ) {
			return $template;
		}

		if ( ! isset( $_GET['key'] ) ) {
			return locate_template( array( '404.php' ) );
		}

		$key = wp_unslash( $_GET['key'] );

		$post_type_cap = get_post_type_object( StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE )->cap;

		if ( ! user_can( $post->post_author, $post_type_cap->view_schedule_url ) ) {
			printf(
				'<p>%s</p>',
				esc_html__( 'Owner does not have permission to schedule this importer.', 'streamtube-core' )
			);
		}

		$settings = $this->admin->get_settings( $post->ID );

		if ( ! $key || $key != $settings['cron_tag_key'] ) {
			printf(
				'<p>%s</p>',
				esc_html__( 'Invalid Cron Tab Key.', 'streamtube-core' )
			);
		}

		$results = $this->run_bulk_import_content( $post->ID, $settings );

		if ( is_wp_error( $results ) ) {
			echo $results->get_error_message();
			exit;
		}

		$count = count( $results['results']['posts'] );

		printf(
			_n( '%s post has been imported.', '%s posts have been imported.', $count, 'streamtube-core' ),
			number_format_i18n( $count )
		) . '<br/>';

		printf(
			esc_html__( 'Next Page: %s', 'streamtube-core' ),
			get_post_meta( $post->ID, 'next_page_token', true )
		) . '<br/>';

		echo '<ol>';

		foreach ( $results['results']['posts'] as $key => $value ) {
			printf(
				'<li><a target="_blank" href="%s"><strong>%s</strong> ==> %s</a></li>',
				esc_url( $value ),
				$key,
				$value
			);
		}

		echo '</ol>';
		exit;

	}

	public function ajax_get_tax_terms() {

		$results = array();

		$data = wp_parse_args( $_GET, array(
			'search' => '',
			'tax'    => ''
		) );

		if ( ! $data['search'] || ! $data['tax'] ) {
			exit;
		}

		$terms = get_terms( array(
			'taxonomy'   => $data['tax'],
			'hide_empty' => false,
			'name__like' => sanitize_text_field( $data['search'] ),
			'number'     => 20
		) );

		if ( $terms ) {
			foreach ( $terms as $term ) {
				$results[] = array(
					'id'   => $term->slug,
					'text' => $term->name
				);
			}
		}

		wp_send_json_success( compact( 'results' ) );
	}

	/**
	 *
	 * Handles errors for YouTube embed imports.
	 *
	 * If the embed already exists and is a YouTube video, it returns a WP_Error
	 * 
	 */
	public function import_youtube_embed_error_handler( $errors, $source ) {
		$maybe_youtube_id = Streamtube_Core_oEmbed::get_youtube_id( $source );

		if ( ! $maybe_youtube_id ) {
			return $errors;
		}

		if ( $this->is_existed( $maybe_youtube_id ) ) {
			$errors->add( 'video_exists', esc_html__( 'This video already exists.', 'streamtube-core' ) );
		}

		return $errors;
	}

	/**
	 * Import YouTube Embed using API
	 *
	 * @param int|WP_Post $post
	 * @param string $source
	 * 
	 */
	public function import_youtube_embed( $post, $source ) {

		$maybe_youtube_id = Streamtube_Core_oEmbed::get_youtube_id( $source );

		if ( ! $maybe_youtube_id ) {
			return $post;
		}

		$response = $this->api->video->get_data( get_option( 'youtube_api_key' ), array(
			'id' => $maybe_youtube_id
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( ! is_object( $post ) ) {
			$post = get_post( $post );
		}

		$statistics      = $this->api->video->get_item_statistics( $response['items'][0] );
		$content_details = $this->api->video->get_item_content_details( $response['items'][0] );
		$tags            = $this->api->video->get_item_tags( $response['items'][0] );

		$metadata = array_merge( $statistics, $content_details );

		if ( array_key_exists( 'duration', $metadata ) ) {
			$metadata['_length'] = streamtube_convert_youtube_duration( $metadata['duration'] );
		}

		foreach ( $metadata as $key => $value ) {
			update_post_meta( $post->ID, $key, $value );
		}

		if ( $tags && apply_filters( 'streamtube/core/import_embed_youtube_tags', true ) === true ) {
			wp_set_post_terms( $post->ID, $tags, 'video_tag' );
		}

		wp_update_post( array(
			'ID'           => $post->ID,
			'post_content' => wpautop( $this->api->video->get_item_description( $response['items'][0] ) )
		) );

		/**
		 *
		 * Fires after importing youtube url
		 *
		 * @param object WP_Post $post
		 * @param string $source
		 * 
		 */
		do_action( 'streamtube/core/youtube_importer/imported_youtube_embed', $post, $source );
	}

	/**
	 *
	 * Filter importer post settings
	 * Auto retrieve access_token if found
	 * 
	 * @param  array $settings
	 * @param  int $importer_id
	 */
	public function filter_importer_settings( $settings, $importer_id ) {
		if ( $this->oauth ) {
			$token = $this->oauth->_get_user_token( get_post( $importer_id )->post_author );
			if ( $this->oauth->_is_valid_token( $token ) ) {
				$settings['apikey'] = $token['access_token'];
				$settings['oauth']  = true;
			}
		}

		return $settings;
	}

	/**
	 *
	 * Clear all schedules of given post
	 *
	 * Hooked into `deleted_post_youtube_importer` to deletes the associated import schedule.
	 * 
	 */
	public function clear_import_schedules( $post_id = 0, $post = null ) {
		wp_clear_scheduled_hook( 'streamtube_youtube_imports', array( $post_id ) );
	}

	/**
	 *
	 * Add a WP Cron to execute the cron job for importing videos
	 * Hook into `save_post_youtube_importer` action
	 * 
	 */
	public function add_import_schedule( $post_id, $post, $update ) {

		$settings = $this->admin->get_settings( $post_id );

		$schedules = $this->get_schedules();

		$post_type_cap = get_post_type_object( $post->post_type )->cap;

		if ( ! user_can( $post->post_author, $post_type_cap->add_schedule ) || $post->post_status !== 'publish' || ! $settings['enable'] ) {
			return $this->clear_import_schedules( $post_id );
		}

		// Clear all schedule once updating
		$this->clear_import_schedules( $post_id );

		$args = array( $post_id );

		// Add new schedule
		if ( in_array( $settings['cron_interval'], array_keys( $schedules ) ) ) {
			if ( ! wp_next_scheduled( 'streamtube_youtube_imports', $args ) ) {
				wp_schedule_event( time(), $settings['cron_interval'], 'streamtube_youtube_imports', $args );
			}
		}
	}

	/**
	 *
	 * Execute cron jobs
	 * 
	 * @param  array $args
	 */
	public function execute_import_schedule( $post_id, $interval = 'hourly' ) {

		$settings = $this->admin->get_settings( $post_id );

		// If post settings not exists, delete the schedule
		if ( ! $settings ) {
			$args = array( $post_id );
			return wp_unschedule_event(
				wp_next_scheduled( 'streamtube_youtube_imports', $args ),
				'streamtube_youtube_imports',
				$args
			);
		}

		$post = get_post( $post_id );

		$post_type_cap = get_post_type_object( $post->post_type )->cap;

		if ( user_can( $post->post_author, $post_type_cap->add_schedule ) ) {
			return $this->run_bulk_import_content( $post_id, $settings );
		}

		return false;
	}

	/**
	 *
	 * Create Google OAuth object
	 * 
	 */
	public function create_oauth() {
		global $streamtube_google_oauth;

		if ( $streamtube_google_oauth && $streamtube_google_oauth->_can_auth() ) {

			$this->oauth = clone $streamtube_google_oauth;

			$this->oauth->set_scopes( array( 'https://www.googleapis.com/auth/youtube' ) );

			$this->oauth->set_service( 'youtube' );
		}
	}

	/**
	 * Create the OAuth url
	 */
	public function create_oauth_url() {

		if ( ! $this->oauth ) {
			return false;
		}

		$url = '';

		if ( is_admin() ) {

			$args         = array( 'post_type' => StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE );
			$redirect_url = admin_url( 'post-new.php' );

			if ( isset( $_REQUEST['post'] ) ) {
				$redirect_url   = admin_url( 'post.php' );
				$args['post']   = absint( $_REQUEST['post'] );
				$args['action'] = 'edit';
			}

			$url = $this->oauth->_create_oauth_url( urlencode( add_query_arg( $args, $redirect_url ) ) );
		} else {
			global $wp;
			$url = $this->oauth->_create_oauth_url( urlencode( home_url( $wp->request ) ) );
		}

		/**
		 *
		 * Filter url
		 *
		 * @param string $url
		 * @param object $oauth
		 * 
		 */
		return apply_filters( 'streamtube/core/youtube_importer/oauth_url', $url, $this->oauth );
	}

	/**
	 * The revoke token URL
	 */
	public function create_revoke_token_url() {
		return $this->oauth ? $this->oauth->_revoke_token_url() : false;
	}

	/**
	 *
	 * Fetch current user channels after authenticated
	 * 
	 */
	public function fetch_mine_channels( $user_id, $token = array() ) {
		$response = wp_remote_get( add_query_arg(
			array(
				'mine'       => true,
				'maxResults' => 50,
				'part'       => 'snippet'
			),
			'https://www.googleapis.com/youtube/v3/channels'
		), array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $token['access_token']
			)
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( is_array( $response ) && array_key_exists( 'items', $response ) && $response['items'] ) {
			update_user_meta( $user_id, '_yt_channels', $response['items'] );
		}
	}

	/**
	 *
	 * Fetch current user playlists after authenticated
	 * 
	 */
	public function fetch_mine_playlists( $user_id, $token = array() ) {
		$response = wp_remote_get( add_query_arg(
			array(
				'mine'       => true,
				'maxResults' => 50,
				'part'       => 'snippet'
			),
			'https://www.googleapis.com/youtube/v3/playlists'
		), array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $token['access_token']
			)
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( is_array( $response ) && array_key_exists( 'items', $response ) && $response['items'] ) {
			update_user_meta( $user_id, '_yt_playlists', $response['items'] );
		}
	}

	/**
	 *
	 * Get mine channels based on meta value
	 * 
	 */
	public function get_mine_channels( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$channels     = array();
		$_yt_channels = get_user_meta( $user_id, '_yt_channels', true );

		if ( $_yt_channels ) {
			for ( $i = 0; $i < count( $_yt_channels ); $i++ ) {
				$channels[ $_yt_channels[ $i ]['id'] ] = $_yt_channels[ $i ]['snippet']['title'];
			}
		}
		return $channels;
	}

	public function get_mine_playlists( $user_id = 0 ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$playlists     = array();
		$_yt_playlists = get_user_meta( $user_id, '_yt_playlists', true );

		if ( $_yt_playlists ) {
			for ( $i = 0; $i < count( $_yt_playlists ); $i++ ) {
				$playlists[ $_yt_playlists[ $i ]['id'] ] = $_yt_playlists[ $i ]['snippet']['title'];
			}
		}
		return $playlists;
	}

	/**
	 *
	 * Merge channels and playlist if found
	 * 
	 * @return array
	 */
	public function get_importable_lists( $user_id = 0 ) {
		$lists = array_merge( $this->get_mine_channels( $user_id ), $this->get_mine_playlists( $user_id ) );

		return $lists ?? array();
	}

	/**
	 *
	 * Hook into `init` action
	 * Request token
	 * 
	 */
	public function request_oauth_token() {

		if ( ! is_user_logged_in() ) {
			return;
		}

		$http_data = wp_parse_args( $_REQUEST, array(
			'code'  => '',
			'error' => '',
			'state' => '',
			'oauth' => '',
		) );

		extract( $http_data );

		if ( $state ) {
			$state = urldecode( $state );
		}

		if ( ! $oauth || $oauth !== 'google' || ! $this->oauth ) {
			return;
		}

		$user_id = get_current_user_id();

		if ( ! $code ) {
			return;
		}

		$token = $this->oauth->_request_token( $code );

		if ( is_wp_error( $token ) ) {
			wp_die( $token->get_error_message(), $token->get_error_message() );
		}

		if ( $this->oauth->_is_valid_token( $token ) ) {
			$this->oauth->_update_user_token( $user_id, $token );
			$this->oauth->_set_user_token_transient( $user_id, $token );

			// Delete user channels and playlists
			// New lists will be fetched via `streamtube/core/youtube_importer/oauth/authenticated` action hook
			delete_user_meta( $user_id, '_yt_channels' );
			delete_user_meta( $user_id, '_yt_playlists' );

			/**
			 *
			 * Fires after authenticated
			 *
			 * @param int $user_id
			 * @param array $token
			 * 
			 */
			do_action( 'streamtube/core/youtube/oauth/authenticated', $user_id, $token );

			/**
			 *
			 * Fires after authenticated
			 *
			 * @param int $user_id
			 * @param array $token
			 * 
			 */
			do_action( 'streamtube/core/youtube_importer/oauth/authenticated', $user_id, $token );

			if ( wp_http_validate_url( $state ) ) {
				wp_redirect( $state );
				exit;
			}
		}
	}

	/**
	 * Revoke user token
	 */
	public function revoke_oauth_token() {

		if ( ! is_user_logged_in() ) {
			return;
		}

		$http_data = wp_parse_args( $_REQUEST, array(
			'oauth'        => '',
			'revoke_token' => ''
		) );

		extract( $http_data );

		if ( wp_verify_nonce( $revoke_token, 'revoke_token_' . get_current_user_id() ) ) {
			if ( $oauth == 'google' ) {
				$this->oauth->_revoke_user_token( get_current_user_id() );
			}
		}
	}
}