<?php
/**
 * Define the Youtube Importer functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class StreamTube_Core_Youtube_Importer_Options {

	public static function get_types() {
		return array(
			'video' => esc_html__( 'Video', 'streamtube-core' )
		);
	}

	/**
	 * Retrieves the available video types for YouTube importer.
	 *
	 * This method returns an array of video types that can be used to filter
	 * YouTube search results. The available types include:
	 * - 'any': Retrieve any type of video.
	 * - 'episode': Retrieve only episodes of shows.
	 * - 'movie': Retrieve only movies.
	 *
	 * The returned array is filterable using the 'streamtube/core/youtube_importer/get_search_ins' filter.
	 *
	 * @return array An associative array of video types with keys as identifiers
	 *               and values as their corresponding labels.
	 */
	public static function get_video_types() {
		$r = array(
			'any'     => esc_html__( 'Any', 'streamtube-core' ),
			'episode' => esc_html__( 'Only retrieve episodes of shows', 'streamtube-core' ),
			'movie'   => esc_html__( 'Only retrieve movies', 'streamtube-core' )
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_search_ins', $r );
	}

	/**
	 * Retrieves the search input options for the YouTube importer.
	 *
	 * This method returns an array of search input types, such as 'channel' and 'playlist',
	 * with their corresponding localized labels. The returned array can be filtered using
	 * the 'streamtube/core/youtube_importer/get_search_ins' filter hook.
	 *
	 * @return array An associative array of search input options with keys as input types
	 *               (e.g., 'channel', 'playlist') and values as their localized labels.
	 */
	public static function get_search_ins() {
		$r = array(
			'channel'  => esc_html__( 'Channel', 'streamtube-core' ),
			'playlist' => esc_html__( 'Playlist', 'streamtube-core' ),
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_search_ins', $r );
	}


	/**
	 * Retrieves an array of available order options for YouTube importer.
	 *
	 * This method returns a list of order options that can be used to sort
	 * YouTube search results. Each option is represented as a key-value pair,
	 * where the key is the order parameter and the value is its human-readable label.
	 *
	 * The returned array can be filtered using the 'streamtube/core/youtube_importer/get_orders' filter.
	 *
	 * @return array Associative array of order options with keys as order parameters
	 *               and values as their corresponding labels.
	 *               Example:
	 *               [
	 *                   'date'      => 'Date',
	 *                   'rating'    => 'Rating',
	 *                   'relevance' => 'Relevance',
	 *                   'title'     => 'Title',
	 *                   'viewCount' => 'View Count'
	 *               ]
	 */
	public static function get_orders() {
		$r = array(
			'date'      => esc_html__( 'Date', 'streamtube-core' ),
			'rating'    => esc_html__( 'Rating', 'streamtube-core' ),
			'relevance' => esc_html__( 'Relevance', 'streamtube-core' ),
			'title'     => esc_html__( 'Title', 'streamtube-core' ),
			'viewCount' => esc_html__( 'View Count', 'streamtube-core' )
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_orders', $r );
	}

	/**
	 * Retrieves the safe search options for the YouTube importer.
	 *
	 * This method returns an array of safe search levels that can be applied
	 * when importing content from YouTube. The available options include:
	 * - 'moderate': Moderate filtering of restricted content.
	 * - 'none': No filtering of restricted content.
	 * - 'strict': Exclude all restricted content.
	 *
	 * The returned array can be filtered using the 'streamtube/core/youtube_importer/get_safe_search' filter.
	 *
	 * @return array An associative array of safe search options with keys as the option values
	 *               and values as their corresponding descriptions.
	 */
	public static function get_safe_search() {
		$r = array(
			'moderate' => esc_html__( 'Moderate', 'streamtube-core' ),
			'none'     => esc_html__( 'None', 'streamtube-core' ),
			'strict'   => esc_html__( 'Exclude all restricted content', 'streamtube-core' )
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_safe_search', $r );
	}

	/**
	 * Retrieves the video definition options for YouTube importer.
	 *
	 * This method returns an array of video definition options that can be used
	 * to filter YouTube videos based on their resolution. The available options
	 * include:
	 * - 'any': Retrieve videos of any resolution.
	 * - 'high': Retrieve only HD videos.
	 * - 'standard': Retrieve only standard definition videos.
	 *
	 * The returned array is filterable using the 'streamtube/core/youtube_importer/get_video_definition' filter.
	 *
	 * @return array An associative array of video definition options with keys as
	 *               the option identifiers and values as their respective labels.
	 */
	public static function get_video_definition() {
		$r = array(
			'any'      => esc_html__( 'Any', 'streamtube-core' ),
			'high'     => esc_html__( 'Only retrieve HD videos', 'streamtube-core' ),
			'standard' => esc_html__( 'Only retrieve videos in standard definition', 'streamtube-core' )
		);
		return apply_filters( 'streamtube/core/youtube_importer/get_video_definition', $r );
	}

	/**
	 * Retrieves the available video dimensions for YouTube importer.
	 *
	 * This method returns an array of video dimension options, including:
	 * - 'any': Any video dimension.
	 * - '2d': 2D video dimension.
	 * - '3d': 3D video dimension.
	 *
	 * The returned array can be filtered using the 'streamtube/core/youtube_importer/get_video_dimension' filter.
	 *
	 * @return array An associative array of video dimension options with keys as dimension types
	 *               and values as their corresponding labels.
	 */
	public static function get_video_dimension() {
		$r = array(
			'any' => esc_html__( 'Any', 'streamtube-core' ),
			'2d'  => esc_html__( '2D', 'streamtube-core' ),
			'3d'  => esc_html__( '3D', 'streamtube-core' )
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_video_dimension', $r );
	}

	/**
	 * Retrieves the available video duration options for YouTube video imports.
	 *
	 * This method returns an array of video duration options, each with a label
	 * describing the duration criteria. The options include:
	 * - 'any': Includes videos of any duration.
	 * - 'long': Includes only videos longer than 20 minutes.
	 * - 'medium': Includes only videos between 4 and 20 minutes long.
	 * - 'short': Includes only videos shorter than 4 minutes.
	 *
	 * The returned array can be filtered using the 'streamtube/core/youtube_importer/get_video_duration' filter.
	 *
	 * @return array An associative array of video duration options, where the keys
	 *               are the duration identifiers ('any', 'long', 'medium', 'short')
	 *               and the values are their respective labels.
	 */
	public static function get_video_duration() {
		$r = array(
			'any'    => esc_html__( 'Any', 'streamtube-core' ),
			'long'   => esc_html__( 'Only include videos longer than 20 minutes', 'streamtube-core' ),
			'medium' => esc_html__( 'Only include videos that are between four and 20 minutes long', 'streamtube-core' ),
			'short'  => esc_html__( 'Only include videos that are less than four minutes long', 'streamtube-core' )
		);
		return apply_filters( 'streamtube/core/youtube_importer/get_video_duration', $r );
	}

	/**
	 * Retrieves the available event types for YouTube broadcasts.
	 *
	 * This method returns an array of event types that can be used to filter
	 * YouTube broadcasts based on their status (e.g., completed, live, or upcoming).
	 * The returned array can be modified using the 'streamtube/core/youtube_importer/get_event_type' filter.
	 *
	 * @return array An associative array where the keys represent the event type
	 *               identifiers and the values are their corresponding labels.
	 *               Possible keys include:
	 *               - ''          : No specific event type (None).
	 *               - 'completed' : Only include completed broadcasts.
	 *               - 'live'      : Only include active broadcasts.
	 *               - 'upcoming'  : Only include upcoming broadcasts.
	 */
	public static function get_event_type() {
		$r = array(
			''          => esc_html__( 'None', 'streamtube-core' ),
			'completed' => esc_html__( 'Only include completed broadcasts.', 'streamtube-core' ),
			'live'      => esc_html__( 'Only include active broadcasts.', 'streamtube-core' ),
			'upcoming'  => esc_html__( 'Only include upcoming broadcasts.', 'streamtube-core' )
		);
		return apply_filters( 'streamtube/core/youtube_importer/get_event_type', $r );
	}

	/**
	 * Retrieves the available video license options for YouTube videos.
	 *
	 * This method returns an array of video license options, including:
	 * - 'any': Any license type.
	 * - 'creativeCommon': Creative Commons license.
	 * - 'youtube': Standard YouTube License.
	 *
	 * The returned array can be filtered using the 'streamtube/core/youtube_importer/get_video_license' filter.
	 *
	 * @return array Associative array of video license options with keys as license types
	 *               and values as their corresponding labels.
	 */
	public static function get_video_license() {
		$r = array(
			'any'            => esc_html__( 'Any', 'streamtube-core' ),
			'creativeCommon' => esc_html__( 'Creative Commons', 'streamtube-core' ),
			'youtube'        => esc_html__( 'Standard YouTube License', 'streamtube-core' )
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_video_license', $r );
	}

	/**
	 * Retrieves the frequency units for the YouTube importer.
	 *
	 * This method returns an array of frequency units (minutes, hours, days) 
	 * that can be used for scheduling purposes. The returned array can be 
	 * filtered using the 'streamtube/core/youtube_importer/get_frequency_unit' filter.
	 *
	 * @return array Associative array of frequency units with keys as unit identifiers 
	 *               ('minutes', 'hours', 'days') and values as their localized labels.
	 */
	public static function get_frequency_unit() {
		$r = array(
			'minutes' => esc_html__( 'Minutes', 'streamtube-core' ),
			'hours'   => esc_html__( 'Hours', 'streamtube-core' ),
			'days'    => esc_html__( 'Days', 'streamtube-core' )
		);

		return apply_filters( 'streamtube/core/youtube_importer/get_frequency_unit', $r );
	}

	/**
	 * Retrieves the list of post statuses.
	 *
	 * This method applies the 'streamtube/core/youtube_importer/get_post_statuses' filter
	 * to the list of post statuses retrieved using the `get_post_statuses()` function.
	 *
	 * @return array Filtered list of post statuses.
	 */
	public static function get_post_statuses() {
		return apply_filters( 'streamtube/core/youtube_importer/get_post_statuses', get_post_statuses() );
	}

	/**
	 * Retrieves a list of post types that meet specific criteria.
	 *
	 * This method fetches post types that are public and have the 'show_ui_frontend' argument set to true.
	 * The arguments can be filtered using the 'streamtube/core/youtube_importer/get_post_types' filter.
	 *
	 * @return array An array of post type names that match the specified criteria.
	 */
	public static function get_post_types() {
		$args = array(
			'public'           => true,
			'show_ui_frontend' => true
		);

		$post_types = get_post_types( $args );

		return apply_filters( 'streamtube/core/youtube_importer/get_post_types', $post_types, $args );
	}
}