<?php
/**
 * Define Cloudflare Stream API 
 *
 * @since      1.0.0
 * @package    WP_Cloudflare_Stream
 * @subpackage WP_Cloudflare_Stream/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WP_Cloudflare_Stream_API {

	/**
	 *
	 * Holds the account ID
	 * 
	 * @var string
	 *
	 * @since 1.0.0
	 * 
	 */
	protected $account_id = '';

	/**
	 *
	 * Holds the API Token
	 * 
	 * @var string
	 *
	 * @since 1.0.0
	 * 
	 */
	protected $api_token = '';

	/**
	 *
	 * Holds the Ingest Domain
	 * 
	 * @var string
	 *
	 * @since 1.0.0
	 * 
	 */
	protected $ingest_domain = '';

	/**
	 *
	 * Holds the base URL
	 *
	 * @since 1.0.0
	 * 
	 */
	protected $api_url = 'https://api.cloudflare.com/client/v4/accounts/$ACCOUNT/stream';

	/**
	 *
	 * Holds the cloudflarestream URL
	 *
	 * @since 1.0.0
	 * 
	 */
	protected $cloudflarestream_url = 'https://cloudflarestream.com';

	/**
	 *
	 * Holds the subdomain
	 * 
	 * @var string
	 */
	protected $subdomain = '';

	/**
	 *
	 * Holds the creator
	 * 
	 * @var integer
	 */
	protected $creator = 0;

	/**
	 *
	 * Holds the allowedOrigins
	 * 
	 * @var array
	 */
	protected $allowedOrigins = array();

	/**
	 *
	 * Holds the requireSignedURLs
	 * 
	 * @var boolean
	 */
	protected $requireSignedURLs = false;

	/**
	 *
	 * Holds the watermark
	 * 
	 * @var array
	 */
	protected $watermark = array();

	protected $preferLowLatency = false;

	protected $timeoutSeconds = 0;

	protected $deleteRecordingAfterDays = null;

	protected $maxDurationSeconds = -1;

	protected $scheduledDeletion = '';

	/**
	 * Holds the file size threshold to decide whether to use direct or resumable upload.
	 *
	 * @var integer
	 */
	protected $big_file_size = 209715200; // 200MB
	/**
	 *
	 * Class contructor
	 * 
	 * @since 1.0.0
	 */
	public function __construct( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'account_id'               => '',
			'api_token'                => '',
			'ingest_domain'            => '',
			'subdomain'                => '',
			'creator'                  => get_current_user_id(),
			'allowedOrigins'           => array(),
			'requireSignedURLs'        => false,
			'watermark'                => array(),
			'preferLowLatency'         => false,
			'timeoutSeconds'           => 0,
			'deleteRecordingAfterDays' => null,
			'maxDurationSeconds'       => -1,
			'scheduledDeletion'        => '',
			'big_file_size'            => 0
		) );

		$this->account_id = trim( $args['account_id'] );

		$this->api_token = trim( $args['api_token'] );

		$this->api_url = str_replace( '$ACCOUNT', $this->account_id, $this->api_url );

		$this->subdomain = $this->filter_subdomain( $args['subdomain'] );

		$this->creator = strval( $args['creator'] );

		$this->allowedOrigins = $args['allowedOrigins'];

		$this->requireSignedURLs = $args['requireSignedURLs'];

		$this->watermark = $args['watermark'];

		$this->preferLowLatency = $args['preferLowLatency'];

		$this->timeoutSeconds = absint( $args['timeoutSeconds'] );

		$this->deleteRecordingAfterDays = absint( $args['deleteRecordingAfterDays'] ) > 0 ? $args['deleteRecordingAfterDays'] : null;

		$this->maxDurationSeconds = (int) $args['maxDurationSeconds'];

		$this->big_file_size = self::parse_big_file_size( $args['big_file_size'] );

		$this->set_scheduled_deletion( $args['scheduledDeletion'] );
	}

	/**
	 * Encodes data using base64url encoding.
	 *
	 * @param string $data The data to be encoded.
	 *
	 * @return string The base64url encoded data.
	 */
	protected function base64Url( $data ) {
		return str_replace( [ '+', '/', '=' ], [ '-', '_', '' ], base64_encode( $data ) );
	}

	public static function parse_big_file_size( $size = '' ) {
		$size = strtolower( $size );

		if ( strpos( $size, 'mb' ) ) {
			$size = (int) $size * 1024 * 1024;
		}

		if ( strpos( $size, 'gb' ) ) {
			$size = (int) $size * 1024 * 1024 * 1024;
		}

		return (int) $size;
	}

	public function set_scheduled_deletion( $scheduledDeletion = 0 ) {

		$scheduledDeletion = absint( $scheduledDeletion );

		if ( $scheduledDeletion > 0 ) {
			$this->scheduledDeletion = gmdate( 'Y-m-d\TH:i:s\Z', current_time( 'timestamp' ) + $scheduledDeletion * 86400 );
		} else {
			$this->scheduledDeletion = null;
		}
	}

	/**
	 *
	 * Get scheduledDeletion
	 * 
	 * @return string|null
	 * 
	 */
	public function get_scheduled_deletion() {
		return $this->scheduledDeletion;
	}

	/**
	 *
	 * Call API
	 *
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	protected function call_api( $url, $args = array() ) {

		$args = wp_parse_args( $args, array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $this->api_token,
				'Content-Type'  => 'application/json'
			)
		) );

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( is_array( $body ) && array_key_exists( 'success', $body ) && wp_validate_boolean( $body['success'] ) != true ) {

			$wp_errors = new WP_Error();

			$errors   = array_key_exists( 'errors', $body ) && $body['errors'] ? $body['errors'] : false;
			$messages = array_key_exists( 'messages', $body ) && $body['messages'] ? $body['messages'] : false;

			if ( $errors ) {
				for ( $i = 0; $i < count( $errors ); $i++ ) {
					$wp_errors->add(
						$errors[ $i ]['code'],
						$errors[ $i ]['message']
					);
				}
			}

			if ( $messages ) {
				for ( $i = 0; $i < count( $messages ); $i++ ) {
					$wp_errors->add(
						$messages[ $i ]['code'],
						$messages[ $i ]['message']
					);
				}
			}

			if ( $wp_errors->get_error_codes() ) {
				return $wp_errors;
			}
		}

		if ( wp_remote_retrieve_response_code( $response ) === 201 ) {
			return array(
				'uploadURL' => wp_remote_retrieve_header( $response, 'Location' ),
				'uid'       => wp_remote_retrieve_header( $response, 'stream-media-id' )
			);
		}

		return is_array( $body ) && array_key_exists( 'result', $body ) ? $body['result'] : $body;
	}

	/**
	 *
	 * Filter subdomain
	 * 
	 */
	private function filter_subdomain( $subdomain = '' ) {
		$subdomain = str_replace( 'http://', '', $subdomain );
		$subdomain = str_replace( 'https://', '', $subdomain );

		return $subdomain;
	}

	/**
	 * Signs a URL token for stream reproduction.
	 *
	 */
	public function get_sign_token( $uid, $key = array(), $nbf = null ) {

		$key = wp_parse_args( $key, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		$exp = $key['exp'];

		unset( $key['exp'] );

		$privateKey = base64_decode( $key['pem'] );

		$header  = [ 'alg' => 'RS256', 'kid' => $key['id'] ];
		$payload = [ 'sub' => $uid, 'kid' => $key['id'] ];

		if ( $exp ) {
			$payload['exp'] = floor( microtime( true ) * 1000 ) + $exp;
		}

		if ( $nbf ) {
			$payload['nbf'] = $nbf;
		}

		$encodedHeader  = $this->base64Url( json_encode( $header ) );
		$encodedPayload = $this->base64Url( json_encode( $payload ) );

		$signature = '';

		if ( ! openssl_sign( "$encodedHeader.$encodedPayload", $signature, $privateKey, OPENSSL_ALGO_SHA256 ) ) {
			return new WP_Error(
				'failed_sign_token',
				esc_html__( 'Failed to sign the token.', 'wp-cloudflare-stream' )
			);
		}

		$encodedSignature = $this->base64Url( $signature );

		return "$encodedHeader.$encodedPayload.$encodedSignature";
	}

	/**
	 *
	 * Enable mp4 download
	 * 
	 * @param  $uid 
	 *
	 * @return call_api()
	 */
	public function enable_download( $uid ) {
		return $this->call_api( $this->api_url . "/{$uid}/downloads", array(
			'method' => 'POST'
		) );
	}

	/**
	 *
	 * Disable mp4 download
	 * 
	 * @param  string $uid
	 * @return call_api()
	 */
	public function disable_download( $uid = '' ) {
		return $this->call_api( $this->api_url . "/{$uid}/downloads", array(
			'method' => 'DELETE'
		) );
	}

	private function _build_resumable_upload_metadata( $args = array() ) {
		$_metadata = array();

		$metadata = array(
			'name', 'requiresignedurls', 'allowedorigins', 'thumbnailtimestamppct', 'watermark'
		);

		foreach ( $args as $key => $value ) {

			if ( ! in_array( $key, $metadata ) ) {
				continue;
			}

			$value = is_array( $value ) ? implode( ',', $value ) : $value;

			$_metadata[] = sprintf(
				'%s %s',
				$key,
				base64_encode( $value )
			);
		}

		$_metadata['name'] = $args['name'];

		return $_metadata;
	}

	/**
	 *
	 * Create one-time direct upload url
	 * 
	 * @param  array $args
	 *
	 * @return WP_Error|array
	 * 
	 */
	public function create_direct_upload( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'size'               => 0,
			'expiry'             => gmdate( 'Y-m-d\TH:i:s\Z', current_time( 'timestamp' ) + 3600 ),
			'name'               => '',
			'maxDurationSeconds' => $this->maxDurationSeconds,
			'allowedOrigins'     => $this->allowedOrigins,
			'creator'            => $this->creator,
			'requireSignedURLs'  => $this->requireSignedURLs,
			'scheduledDeletion'  => $this->scheduledDeletion
		) );

		$meta = array();

		extract( $args );

		if ( $name ) {
			$meta = compact( 'name' );
		}

		$call_api_args = compact(
			'maxDurationSeconds',
			'allowedOrigins',
			'creator',
			'expiry',
			'meta',
			'requireSignedURLs',
			'scheduledDeletion'
		);

		if ( $this->watermark ) {
			$call_api_args['watermark'] = $this->watermark;
		}

		if ( ! $this->big_file_size || ( $this->big_file_size > 0 && $this->big_file_size >= $size ) ) {
			return $this->call_api( untrailingslashit( $this->api_url ) . '/direct_upload', array(
				'method' => 'POST',
				'body'   => json_encode( $call_api_args )
			) );
		}

		$call_api_args['name'] = $name;

		return $this->call_api( trailingslashit( $this->api_url ) . '?direct_user=true', array(
			'method'  => 'POST',
			'headers' => array(
				'Authorization'   => 'Bearer ' . $this->api_token,
				'Content-Type'    => 'application/json',
				'Tus-Resumable'   => '1.0.0',
				'Upload-Length'   => $size,
				'Upload-Creator'  => get_current_user_id(),
				'Upload-Metadata' => implode( ',', $this->_build_resumable_upload_metadata( $call_api_args ) )
			)
		) );
	}

	/**
	 *
	 * Fetch Video
	 * 
	 * @param  string $url
	 * @param  string $name
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function fetch_video( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'url'               => '',
			'name'              => '',
			'allowedOrigins'    => $this->allowedOrigins,
			'requireSignedURLs' => $this->requireSignedURLs,
			'scheduledDeletion' => $this->scheduledDeletion,
			'creator'           => $this->creator
		) );

		extract( $args );

		$meta = compact( 'name' );

		$body = compact( 'url', 'meta', 'allowedOrigins', 'requireSignedURLs', 'scheduledDeletion' );

		if ( $creator ) {
			$body['creator'] = $creator;
		}

		if ( $this->watermark ) {
			$body['watermark'] = $this->watermark;
		}

		return $this->call_api( $this->api_url . "/copy", array(
			'method' => 'POST',
			'body'   => json_encode( $body )
		) );
	}

	/**
	 *
	 * Delete Video
	 * 
	 * @param  string $uid
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function delete_video( $uid = '' ) {
		return $this->call_api( trailingslashit( $this->api_url ) . $uid, array(
			'method' => 'DELETE'
		) );
	}

	/**
	 *
	 * Update Video
	 * 
	 * @param  string $uid
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function update_video( $uid, $data = array() ) {

		$data = wp_parse_args( $data, array(
			'name'              => '',
			'meta'              => array(),
			'allowedOrigins'    => $this->allowedOrigins,
			'requireSignedURLs' => $this->requireSignedURLs,
			'creator'           => $this->creator,
			'scheduledDeletion' => $this->scheduledDeletion
		) );

		if ( $data['name'] ) {
			$data['meta'] = array(
				'name' => $data['name']
			);

			unset( $data['name'] );
		}

		return $this->call_api( trailingslashit( $this->api_url ) . $uid, array(
			'method' => 'POST',
			'body'   => json_encode( $data )
		) );
	}

	/**
	 *
	 * Get Video
	 * 
	 * @param  string $uid
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function get_video( $uid = '' ) {

		if ( ! is_string( $uid ) || empty( $uid ) ) {
			return new WP_Error(
				'empty_uid',
				esc_html__( 'UID is required.', 'wp-cloudflare-stream' )
			);
		}

		return $this->call_api( trailingslashit( $this->api_url ) . $uid, array(
			'method' => 'GET'
		) );
	}

	/**
	 *
	 * Get Videos
	 * 
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function get_videos( $args = array() ) {
		$response = wp_remote_get( add_query_arg( $args, $this->api_url ), array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $this->api_token,
				'Content-Type'  => 'application/json'
			)
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		return json_decode( wp_remote_retrieve_body( $response ), true );
	}

	/**
	 *
	 * Download video
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.1
	 * 
	 */
	public function get_download_video_url( $uid ) {
		$response = $this->call_api( trailingslashit( $this->api_url ) . $uid . '/downloads', array(
			'method' => 'GET'
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( is_array( $response ) && array_key_exists( 'default', $response ) ) {
			return $response['default']['url'];
		}

		return false;
	}

	/**
	 *
	 * Start live stream
	 * 
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function create_live_stream( $args = array() ) {
		$args = wp_parse_args( $args, array(
			'uid'                      => '',
			'name'                     => '',
			'description'              => '',
			'mode'                     => 'automatic',
			'creator'                  => $this->creator,
			'preferLowLatency'         => $this->preferLowLatency,
			'timeoutSeconds'           => $this->timeoutSeconds,
			'deleteRecordingAfterDays' => $this->deleteRecordingAfterDays,
			'allowedOrigins'           => $this->allowedOrigins,
			'requireSignedURLs'        => $this->requireSignedURLs
		) );

		extract( $args );

		$body = $meta = $recording = array();

		if ( ! empty( $name ) ) {
			$meta['name'] = $name;
		}

		if ( ! empty( $description ) ) {
			$meta['description'] = $description;
		}

		if ( $creator ) {
			$body['creator'] = $creator;
		}

		$body['preferLowLatency']         = $preferLowLatency;
		$body['deleteRecordingAfterDays'] = $deleteRecordingAfterDays;

		if ( ! empty( $mode ) ) {
			$recording['mode'] = $mode;
		}

		if ( is_integer( $timeoutSeconds ) ) {
			$recording['timeoutSeconds'] = absint( $timeoutSeconds );
		}

		$recording['requireSignedURLs'] = $requireSignedURLs;

		if ( $allowedOrigins && is_array( $allowedOrigins ) ) {
			$recording['allowedOrigins'] = $allowedOrigins;
		}

		if ( $meta ) {
			$body['meta'] = $meta;
		}

		if ( $recording ) {
			$body['recording'] = $recording;
		}

		$api_url = $this->api_url . "/live_inputs";

		if ( $uid ) {
			$api_url = trailingslashit( $api_url ) . $uid;
		}

		$response = $this->call_api( $api_url, array(
			'method' => $uid ? 'PUT' : 'POST',
			'body'   => json_encode( $body )
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		return $response;
	}

	/**
	 *
	 * Update live stream
	 * 
	 * @param  array  $args
	 * 
	 */
	public function update_live_stream( $args = array() ) {
		return $this->create_live_stream( $args );
	}

	/**
	 *
	 * Close live stream
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function close_live_stream( $data = array() ) {

		$data = array_merge( $data, array(
			'mode' => 'off'
		) );

		return $this->create_live_stream( $data );
	}

	/**
	 *
	 * Open live stream
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function open_live_stream( $data = array() ) {

		$data = array_merge( $data, array(
			'mode' => 'automatic'
		) );

		return $this->create_live_stream( $data );
	}

	/**
	 *
	 * Delete all playback of given live stream input
	 * 
	 * @param  string $uid
	 * 
	 */
	public function delete_live_stream_playbacks( $uid ) {

		$recorded_videos = $this->get_recorded_videos( $uid );

		if ( is_wp_error( $recorded_videos ) ) {
			return $recorded_videos;
		}

		$recorded_video_uids = wp_list_pluck( $recorded_videos, 'uid' );

		if ( is_array( $recorded_video_uids ) ) {
			for ( $i = 0; $i < count( $recorded_video_uids ); $i++ ) {
				$this->delete_video( $recorded_video_uids[ $i ] );
			}
		}
	}

	/**
	 *
	 * Delete live stream
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function delete_live_stream( $uid = '' ) {

		if ( ! $uid ) {
			return new WP_Error(
				'empty_uid',
				esc_html__( 'UID is required.', 'wp-cloudflare-stream' )
			);
		}

		return $this->call_api( $this->api_url . "/live_inputs/" . $uid, array(
			'method' => 'DELETE'
		) );
	}

	/**
	 *
	 * Get live stream
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function get_live_stream( $uid ) {

		if ( ! $uid ) {
			return new WP_Error(
				'empty_uid',
				esc_html__( 'UID is required.', 'wp-cloudflare-stream' )
			);
		}

		return $this->call_api( $this->api_url . "/live_inputs/" . $uid, array(
			'method' => 'GET'
		) );
	}

	/**
	 *
	 * Get recorded videos of given live stream ID
	 *
	 * @see https://developers.cloudflare.com/stream/stream-live/watch-live-stream/
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function get_recorded_videos( $uid ) {
		return $this->call_api( $this->api_url . "/live_inputs/" . $uid . '/videos', array(
			'method' => 'GET'
		) );
	}

	/**
	 *
	 * Request live stream status
	 *
	 * @example https://videodelivery.net/34036a0695ab5237ce757ac53fd158a2/lifecycle
	 * 
	 * 
	 * @param  string $uid
	 * @return WP_Error|array
	 *
	 * @since 1.0.0
	 * 
	 */
	public function poll_live_status( $uid, $sign_token = array() ) {

		$sign_token = wp_parse_args( $sign_token, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		$url = sprintf( 'https://videodelivery.net/%s/lifecycle', $uid );

		if ( $this->subdomain ) {
			$url = sprintf( 'https://%s/%s/lifecycle', $this->subdomain, $uid );
		}

		if ( $sign_token['pem'] ) {
			$url = str_replace( $uid, $this->get_sign_token(
				$uid,
				$sign_token
			), $url );
		}

		$response = wp_remote_get( $url );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response = json_decode( wp_remote_retrieve_body( $response ), true );

		return $response;
	}

	/**
	 *
	 * Add live output
	 * 
	 * @param string $uid  live UID
	 * @param array  $data
	 */
	public function add_live_output( $uid, $data = array() ) {

		$data = wp_parse_args( $data, array(
			'url'       => '',
			'streamkey' => '',
			'enabled'   => true
		) );

		return $this->call_api( trailingslashit( $this->api_url ) . 'live_inputs/' . $uid . '/outputs', array(
			'method' => 'POST',
			'body'   => json_encode( $data )
		) );
	}

	public function update_live_output( $uid, $output_uid = '', $data = array() ) {
		return $this->call_api( trailingslashit( $this->api_url ) . 'live_inputs/' . $uid . '/outputs/' . $output_uid, array(
			'method' => 'PUT',
			'body'   => json_encode( $data )
		) );
	}

	/**
	 *
	 * Delete output
	 * 
	 * @param  string $output_uid
	 * 
	 */
	public function delete_live_output( $uid, $output_uid = '' ) {
		return $this->call_api( trailingslashit( $this->api_url ) . 'live_inputs/' . $uid . '/outputs/' . $output_uid, array(
			'method' => 'DELETE'
		) );
	}

	/**
	 *
	 * Live destinations
	 * 
	 * @param  string $uid live uid
	 * @return WP_Error/array
	 * 
	 */
	public function get_live_destinations( $uid ) {
		return $this->call_api( $this->api_url . "/live_inputs/" . $uid . '/destinations', array(
			'method' => 'GET'
		) );
	}

	/**
	 *
	 * Get HLS playback URL
	 * 
	 * @param  string $uid
	 *
	 * @since 1.0.0
	 * 
	 */
	public function get_playback_url( $uid, $sign_token = array(), $hls = true ) {

		$sign_token = wp_parse_args( $sign_token, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		$url = sprintf(
			'%s/%s/manifest/video.%s',
			untrailingslashit( $this->cloudflarestream_url ),
			$uid,
			$hls ? 'm3u8' : 'mpd'
		);

		if ( $this->subdomain ) {
			$url = sprintf(
				'https://%s/%s/manifest/video.%s',
				$this->subdomain,
				$uid,
				$hls ? 'm3u8' : 'mpd'
			);
		}

		if ( $sign_token['pem'] ) {
			$url = str_replace( $uid, $this->get_sign_token(
				$uid,
				$sign_token
			), $url );
		}

		return $url;
	}

	/**
	 *
	 * Get preview url
	 * 
	 * @param  string $uid
	 * @param  array  $sign_token
	 * @return string
	 * 
	 */
	public function get_preview_url( $uid, $sign_token = array() ) {

		$sign_token = wp_parse_args( $sign_token, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		$url = sprintf(
			'https://%s/%s/watch',
			$this->subdomain,
			$uid
		);

		if ( $sign_token['pem'] ) {
			$url = str_replace( $uid, $this->get_sign_token(
				$uid,
				$sign_token
			), $url );
		}

		return $url;
	}

	/**
	 *
	 * Get thumbnail image URL
	 * 
	 * @param  array  $args
	 * @return string
	 *
	 * @since 1.0.0
	 * 
	 */
	public function get_thumbnail_url( $args = array(), $sign_token = array() ) {

		$sign_token = wp_parse_args( $sign_token, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		$args = wp_parse_args( $args, array(
			'uid'    => '',
			'time'   => '5s',
			'height' => 360,
			'ext'    => 'jpg'
		) );

		$url = sprintf(
			'https://%s/%s/thumbnails/thumbnail.%s?height=%s',
			$this->subdomain ? $this->subdomain : 'videodelivery.net',
			$args['uid'],
			$args['ext'],
			$args['height']
		);

		if ( $args['ext'] == 'gif' ) {
			$url = add_query_arg( array(
				'time' => $args['time']
			), $url );
		}

		if ( $sign_token['pem'] ) {
			$url = str_replace( $args['uid'], $this->get_sign_token(
				$args['uid'],
				$sign_token
			), $url );
		}

		return $url;
	}

	/**
	 *
	 * Get download URL
	 * 
	 * @param  string $uid
	 * @param  string $name
	 * @return string
	 */
	public function get_download_url( $uid, $name = '', $sign_token = array() ) {

		$sign_token = wp_parse_args( $sign_token, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		if ( ! $this->subdomain ) {
			return false;
		}

		$url = sprintf(
			'https://%s/%s/downloads/default.mp4',
			$this->subdomain,
			$uid
		);

		if ( $name ) {
			$url = add_query_arg( array(
				'filename' => $name
			), $url );
		}

		if ( $sign_token['pem'] ) {
			$url = str_replace( $uid, $this->get_sign_token(
				$uid,
				$sign_token
			), $url );
		}

		return $url;
	}

	/**
	 *
	 * Get cloudflare iframe
	 * 
	 */
	public function get_iframe( $uid = '', $sign_token = array() ) {

		$sign_token = wp_parse_args( $sign_token, array(
			'pem' => '',
			'id'  => '',
			'exp' => 3600 * 12
		) );

		$base_url = untrailingslashit( $this->cloudflarestream_url );

		if ( $this->subdomain ) {
			$base_url = 'https://' . $this->subdomain;
		}

		$iframe = sprintf(
			'<iframe src="%s/%s/iframe" allow="accelerometer; gyroscope; autoplay; encrypted-media; picture-in-picture;" allowfullscreen="true"></iframe>',
			$base_url,
			$uid
		);

		if ( $sign_token['pem'] ) {
			$iframe = str_replace( $uid, $this->get_sign_token(
				$uid,
				$sign_token
			), $iframe );
		}

		return $iframe;
	}

	/**
	 *
	 * Generate stream key for creating Sign Token
	 * 
	 * @return array|WP_Error
	 * 
	 */
	public function generate_stream_key() {
		return $this->call_api( trailingslashit( $this->api_url ) . 'keys', array(
			'method' => 'POST'
		) );
	}

	/**
	 *
	 * Delete stream key
	 * 
	 * @param  string $key_id
	 * 
	 */
	public function delete_stream_key( $key_id = '' ) {
		return $this->call_api( trailingslashit( $this->api_url ) . 'keys/' . $key_id, array(
			'method' => 'DELETE'
		) );
	}

	/**
	 *
	 * Subscribe Webhook
	 * 
	 * @param  string $notificationUrl
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function subscribe_webhook( $notificationUrl ) {
		return $this->call_api( $this->api_url . "/webhook", array(
			'method' => 'PUT',
			'body'   => json_encode( compact( 'notificationUrl' ) )
		) );
	}

	public function get_webhooks() {
		return $this->call_api( $this->api_url . "/webhook", array(
			'method' => 'GET'
		) );
	}

	/**
	 *
	 * Upload watermark
	 * 
	 * @param  array  $args
	 * @return call_api()
	 *
	 * @since 1.0.0
	 * 
	 */
	public function upload_watermark( $args = array() ) {
		$args = wp_parse_args( $args, array(
			'url'      => '',
			'name'     => esc_html__( 'Watermark', 'wp-cloudflare-stream' ),
			'opacity'  => 1.0,
			'padding'  => 0.0,
			'scale'    => 0.15,
			'position' => 'upperRight'
		) );

		return $this->call_api( $this->api_url . "/watermarks", array(
			'method' => 'POST',
			'body'   => json_encode( $args )
		) );
	}
}