<?php
/**
 *
 * Rest API
 *
 * @link       https://1.envato.market/mgXE4y
 * @since      1.0.0
 *
 * @package    WP_Cloudflare_Stream
 * @subpackage WP_Cloudflare_Stream/includes
 */

/**
 *
 * @package    WP_Cloudflare_Stream
 * @subpackage WP_Cloudflare_Stream/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

if( ! class_exists( 'StreamTube_Core_Rest_API' ) ){
    return;
}

class WP_Cloudflare_Stream_Rest_API extends StreamTube_Core_Rest_API{

    protected $path     =   '/cloudflare';

    /**
     * @since 1.0.6
     */
    public function rest_api_init(){

        register_rest_route(
            "{$this->namespace}{$this->version}",
            $this->path . '/endpoint', 
            array(
                'methods'   =>  WP_REST_Server::CREATABLE,
                'callback'  =>  array( $this , 'create_endpoint' ),
                'args'      =>  array(
                    'name'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return ( is_string( $param ) || is_number( $param ) ) && ! empty( $param );
                        }
                    ),                    
                    'size'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return ( is_string( $param ) || is_number( $param ) ) && ! empty( $param );
                        }
                    ),
                    'type'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    )                    
                ), 
                'permission_callback'   =>  array( $this , 'create_upload_endpoint_permission_check' )
            )
        );      

        register_rest_route(
            "{$this->namespace}{$this->version}",
            $this->path . '/import', 
            array(
                'methods'   =>  WP_REST_Server::CREATABLE,
                'callback'  =>  array( $this , 'create_import' ),
                'args'      =>  array(
                    'name'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    ),
                    'type'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    ),                    
                    'uid'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    ),
                    'status'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    )                    
                ), 
                'permission_callback'   =>  array( $this , 'create_import_permission_check' )
            )
        );

        register_rest_route(
            "{$this->namespace}{$this->version}",
            $this->path . '/live', 
            array(
                'methods'   =>  WP_REST_Server::CREATABLE,
                'callback'  =>  array( $this , 'create_live' ),
                'args'      =>  array(
                    'name'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param ) && ! empty( $param );
                        }
                    ),                    
                    'description'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    ),
                    'publish'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param );
                        }
                    )                    
                ), 
                'permission_callback'   =>  array( $this , 'create_live_permission_check' )
            )
        );        

        register_rest_route(
            "{$this->namespace}{$this->version}",
            $this->path . '/delete', 
            array(
                'methods'   =>  WP_REST_Server::CREATABLE,
                'callback'  =>  array( $this , 'create_delete' ),
                'args'      =>  array(                  
                    'uid'  =>  array(
                        'validate_callback' =>  function( $param, $request, $key ){
                            return is_string( $param ) && ! empty( $param );
                        }
                    )
                ), 
                'permission_callback'   =>  array( $this, 'delete_item_permission_check' )
            )
        );                  
    } 

    /**
     *
     * Get plugin instance
     * 
     * @return plugin instance
     */
    private function get_instance(){
        return $GLOBALS['wp_cloudflare_stream']->get();
    }  

    /**
     *
     * Get StreamTube core instance
     * 
     * @return StreamTube core instance
     */
    private function get_streamtube(){
        return $GLOBALS['streamtube']->get();
    }

    /**
     *
     * Check `can_upload` permission
     *
     */
    public function can_upload(){
        return  class_exists( 'Streamtube_Core_Permission' ) && 
                method_exists( 'Streamtube_Core_Permission', 'can_upload' ) &&
                class_exists( 'Streamtube_Core_Upload' ) &&
                $this->get_instance()->post->is_enabled();
    } 

    /**
     *
     * Create upload stream permission check
     * 
     */
    public function create_upload_endpoint_permission_check( $request ){
        return $this->can_upload();
    }

    /**
     *
     * Delete stream permission check
     * 
     */
    public function create_import_permission_check( $request ){
        return $this->can_upload();
    }

    /**
     *
     * Crreate live stream permission check
     * 
     */
    public function create_live_permission_check( $request ){       
        return  $this->get_instance()->post->settings['enable'] && 
                $this->get_instance()->post->settings['live_stream_enable'] && 
                WP_Cloudflare_Stream_Permission::create_live();
    }

    /**
     *
     * Delete stream permission check
     * 
     */
    public function delete_item_permission_check( $request ){

        if( ! WP_Cloudflare_Stream_Permission::delete_videos() ){
            return new WP_Error(
                'rest_cannot_delete',
                esc_html__( 'Sorry, you are not allowed to delete this stream.', 'wp-cloudflare-stream' ),
                array( 'status' => rest_authorization_required_code() )
            );            
        }

        $video = $this->get_instance()->post->cloudflare_api->get_video( $request['uid'] );

        if( is_wp_error( $video ) ){
            return $video;
        }

        if( absint( $video['creator'] ) !== get_current_user_id() && ! WP_Cloudflare_Stream_Permission::delete_others_videos() ){
            return new WP_Error(
                'rest_cannot_delete',
                esc_html__( 'Sorry, you are not allowed to delete this stream.', 'wp-cloudflare-stream' ),
                array( 'status' => rest_authorization_required_code() )
            );
        }

        return true;
    }

    /**
     *
     * Request one-time upload url
     * 
     * @param  WP_Rest_Request $request
     * 
     */
    public function create_endpoint( $request ){
        $response = $this->get_instance()->post->create_direct_upload( array(
            'name'  =>  $request['name'],
            'size'  =>  $request['size'],
            'type'  =>  $request['type']
        ) );

        if ( is_wp_error( $response )) {
            wp_send_json_error( $response, 403 );  
        }

        wp_send_json_success( $response );
    }

    /**
     *
     * Import video
     * 
     * @param  WP_Rest_Request $request
     * 
     */
    public function create_import( $request ){

        $post_status = $request['status'] ?? 'pending';

        if( $post_status === 'publish' && ! current_user_can( get_post_type_object( 'video' )->cap->publish_posts ) ){
            $post_status = 'pending';
        }

        $post_args = array(
            'name'          =>  $request['name'],
            'type'          =>  $request['type'],
            'uid'           =>  $request['uid'],
            'post_status'   =>  $post_status,
            'post_author'   =>  get_current_user_id()
        );

        /**
         *
         * Filter post args
         * 
         * @param array $post_args
         * @param WP_Rest_Request $request 
         */
        $post_args = apply_filters( 
            'wp_cloudflare_stream/rest/import/post_args', 
            $post_args, 
            $request 
        );

        $response  = $this->get_instance()->post->_import_video( $post_args );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response, 403 );
        }

        $post = $this->get_streamtube()->post->get_post( $response['video_id'] );

        wp_send_json_success( compact( 'post' ) );
    }

    /**
     *
     * Create live stream
     * 
     */
    public function create_live( $request ){

        $response = $this->get_instance()->post->start_live_stream( array(
            'name'          =>  $request['name'],
            'description'   =>  $request['description'],
            'post_status'   =>  $request['publish'] ? 'publish' : 'pending'
        ) );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        $response = (object)$response;

        $post = $this->get_streamtube()->post->get_post( $response->video_id );

        $response->post = $post;

        wp_send_json_success( $response );
    }

    /**
     *
     * Delete video
     * 
     * @param  WP_Rest_Request $request
     * 
     */
    public function create_delete( $request ){

        $uid = $request['uid'];

        $attachment_id = $this->get_instance()->post->get_attachment_id_from_uid( $uid );

        if( $attachment_id ){

            $parent = get_post_parent( $attachment_id );

            if( $parent ){
                $response = wp_delete_post( $parent->ID, true );
            }else{
                $response = $this->get_instance()->post->_delete_attachment( $attachment_id );    
            }
            
        }else{
            $response = $this->get_instance()->post->delete_cloudflare_video( $uid );
        }

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        wp_send_json_success( $response );
    }
}