<?php
/**
 * iMi License Helper (Safe version)
 * Functions are wrapped with function_exists to avoid redeclaration errors.
 */

if ( ! function_exists( 'imi_is_localhost' ) ) {
	function imi_is_localhost() {
		$domain = preg_replace( '|https?://([^/]+)|', '$1', home_url() );
		$patterns = [ 'localhost', '127.0.0.1', '.local', '.test', '.dev', '.mamp', '.localwp' ];
		foreach ( $patterns as $pattern ) {
			if ( strpos( $domain, $pattern ) !== false || preg_match( '|^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$|', $domain ) ) {
				return true;
			}
		}
		return false;
	}
}

if ( ! function_exists( 'imi_get_current_domain' ) ) {
	function imi_get_current_domain() {
		return urlencode( site_url() );
	}
}

if ( ! function_exists( 'imi_get_purchase_code' ) ) {
	function imi_get_purchase_code( $option_key ) {
		if ( is_multisite() ) {
			return get_option( $option_key . '_' . get_current_blog_id() );
		}
		return get_option( $option_key );
	}
}

if ( ! function_exists( 'imi_save_purchase_code' ) ) {
	function imi_save_purchase_code( $option_code_key, $option_status_key, $code, $status ) {
		if ( is_multisite() ) {
			$blog_id = get_current_blog_id();
			update_option( $option_code_key . '_' . $blog_id, $code );
			update_option( $option_status_key . '_' . $blog_id, $status );
		} else {
			update_option( $option_code_key, $code );
			update_option( $option_status_key, $status );
		}
	}
}

if ( ! function_exists( 'imi_delete_purchase_code' ) ) {
	function imi_delete_purchase_code( $option_code_key, $option_status_key, $support_key ) {
		if ( is_multisite() ) {
			$blog_id = get_current_blog_id();
			delete_option( $option_code_key . '_' . $blog_id );
			delete_option( $option_status_key . '_' . $blog_id );
			delete_option( $support_key . '_' . $blog_id );
		} else {
			delete_option( $option_code_key );
			delete_option( $option_status_key );
			delete_option( $support_key );
		}
	}
}

if ( ! function_exists( 'imi_update_support_until' ) ) {
	function imi_update_support_until( $support_key, $date ) {
		if ( is_multisite() ) {
			update_option( $support_key . '_' . get_current_blog_id(), $date );
		} else {
			update_option( $support_key, $date );
		}
	}
}

if ( ! function_exists( 'imi_register_auth_ajax_handler' ) ) {
	function imi_register_auth_ajax_handler( $action_slug, $option_code_key, $option_status_key, $support_key ) {
		add_action( 'wp_ajax_' . $action_slug, function () use ( $option_code_key, $option_status_key, $support_key ) {
			$status    = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
			$authCode  = isset($_GET['authCode']) ? sanitize_text_field($_GET['authCode']) : '';
			$remove    = isset($_GET['remove']) ? absint($_GET['remove']) : 0;

			if ( $remove === 1 ) {
				imi_delete_purchase_code( $option_code_key, $option_status_key, $support_key );
				wp_send_json_success([
					'message' => '<div class="notice notice-success is-dismissible">Purchase code has been unregistered successfully.</div>'
				]);
			} else {
				imi_save_purchase_code( $option_code_key, $option_status_key, $authCode, $status );
				wp_send_json_success([
					'message' => '<div class="notice notice-success is-dismissible">Purchase code has been saved successfully.</div>'
				]);
			}
			wp_die();
		});
	}
}

if ( ! function_exists('imi_get_purchase_code_by_option') ) {
    /**
     * Get the stored purchase code for this site/blog using a product-specific option key.
     * Multisite-aware: appends _{blog_id} automatically if needed.
     */
    function imi_get_purchase_code_by_option( string $option_key_base ) : string {
        if ( is_multisite() ) {
            $blog_id = get_current_blog_id();
            return (string) get_option( $option_key_base . '_' . $blog_id );
        }
        return (string) get_option( $option_key_base );
    }
}

if ( ! function_exists('imi_register_license_guard') ) {
    /**
     * Enforce license presence on restricted admin pages for a product.
     *
     * @param array $args {
     *   @type string   purchase_option   Required. Base option key for the purchase code (e.g. 'nativechurch_auth_code').
     *   @type string   welcome_page      Required. Slug of the product’s welcome/registration page (e.g. 'imi-admin-welcome').
     *   @type string[] restricted_pages  Required. Page slugs to restrict unless licensed (e.g. ['_options','imi-admin-demo-importer','imi-admin-plugins']).
     *   @type bool     skip_on_local     Optional. Defaults true. Skip enforcement on localhost/dev domains.
     * }
     */
    function imi_register_license_guard( array $args ) : void {
        $defaults = [
            'purchase_option'  => '',
            'welcome_page'     => '',
            'restricted_pages' => [],
            'skip_on_local'    => true,
        ];
        $cfg = array_merge( $defaults, $args );

        // Validate required
        if ( ! $cfg['purchase_option'] || ! $cfg['welcome_page'] || empty( $cfg['restricted_pages'] ) ) {
            return;
        }

        add_action( 'admin_init', function () use ( $cfg ) {

            // Only for admins, and not during AJAX
            if ( ! current_user_can( 'manage_options' ) ) return;
            if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) return;

            // Optional: skip on local/dev environments
            if ( function_exists( 'imi_is_localhost' ) && imi_is_localhost() && $cfg['skip_on_local'] ) {
                return;
            }

            // If any code (e.g., your theme) declares this product as entitled, skip enforcement.
            $is_entitled = apply_filters(
                'imi_license_entitled',
                false,
                [
                    // Pass minimal product context so themes/plugins can decide:
                    'purchase_option'  => $cfg['purchase_option'],
                    'welcome_page'     => $cfg['welcome_page'],
                    'restricted_pages' => (array) $cfg['restricted_pages'],
                ]
            );
            if ( $is_entitled ) {
                return;
            }

            // Run on both admin.php and edit.php (many CPT subpages use edit.php)
            global $pagenow;
            if ( $pagenow !== 'admin.php' && $pagenow !== 'edit.php' ) return;

            // Targeted admin subpage
            $page = isset( $_REQUEST['page'] ) ? sanitize_key( $_REQUEST['page'] ) : '';
            if ( ! $page ) return;

            // Never redirect if already on the welcome/registration page
            if ( $page === $cfg['welcome_page'] ) return;

            // Only enforce on restricted pages
            if ( ! in_array( $page, (array) $cfg['restricted_pages'], true ) ) return;

            // Fetch the stored purchase code for this product
            if ( ! function_exists( 'imi_get_purchase_code_by_option' ) ) return;
            $purchase_code = imi_get_purchase_code_by_option( $cfg['purchase_option'] );

            if ( empty( $purchase_code ) ) {
                // Preserve post_type context if present (e.g., edit.php?post_type=eventer)
                $post_type = isset( $_REQUEST['post_type'] ) ? sanitize_key( $_REQUEST['post_type'] ) : '';
                if ( $pagenow === 'edit.php' && $post_type ) {
                    $url = add_query_arg(
                        [
                            'post_type' => $post_type,
                            'page'      => $cfg['welcome_page'],
                        ],
                        admin_url( 'edit.php' )
                    );
                } else {
                    // Fallback to admin.php?page=welcome_page
                    $url = add_query_arg( 'page', $cfg['welcome_page'], admin_url( 'admin.php' ) );
                }

                wp_safe_redirect( $url );
                exit;
            }
        } );
    }
}

if ( ! function_exists( 'imi_schedule_license_check' ) ) {
	function imi_schedule_license_check( $product_id, $option_code_key, $option_status_key, $support_key ) {

		// Schedule daily check for this product
		add_action( 'init', function () use ( $product_id ) {
			$hook = 'imi_check_purchase_code_' . $product_id;
			if ( ! wp_next_scheduled( $hook ) ) {
				wp_schedule_event( time(), 'daily', $hook );
			}
		} );

		// The checker
		add_action( 'imi_check_purchase_code_' . $product_id, function () use ( $product_id, $option_code_key, $option_status_key, $support_key ) {

			if ( function_exists( 'imi_is_localhost' ) && imi_is_localhost() ) return;

			$code = imi_get_purchase_code( $option_code_key ); // your generic helper
			if ( ! $code ) return;

			$domain_param = imi_get_current_domain(); // returns normalized domain (e.g., example.com)
			$endpoint = add_query_arg(
				array(
					'purchase' => rawurlencode( $code ),
					'domain'   => rawurlencode( $domain_param ),
					'item'     => rawurlencode( $product_id ),
				),
				'https://envato.api.imithemes.com/wp-json/imi/check-purchase'
			);

			$response = wp_remote_get( $endpoint, array( 'timeout' => 15 ) );
			if ( is_wp_error( $response ) ) return;

			$http = (int) wp_remote_retrieve_response_code( $response );
			if ( $http !== 200 ) {
				// treat as temporary
				$fail_key = 'imi_license_failures_' . $product_id;
				$failures = (int) get_option( $fail_key, 0 );
				update_option( $fail_key, min( $failures + 1, 100 ) );
				return;
			}

			$data = json_decode( wp_remote_retrieve_body( $response ), true );
			if ( ! is_array( $data ) ) return;

			$status = isset( $data['status'] ) ? (int) $data['status'] : 0;

			if ( $status === 1 ) {
				// Valid
				// Optionally store support_until locally if provided
				if ( ! empty( $data['support_until'] ) ) {
					$support_until = sanitize_text_field( $data['support_until'] );
					imi_update_support_until( $support_key, $support_until ); // generic helper per product/site
				}

				// Reset failure counter for this product
				update_option( 'imi_license_failures_' . $product_id, 0 );
				return;
			}

			if ( $status === -1 ) {
				// Permanently invalid → remove locally, product-specific keys only
				if ( is_multisite() ) {
					$blog_id = get_current_blog_id();
					delete_option( $option_code_key   . '_' . $blog_id );
					delete_option( $option_status_key . '_' . $blog_id );
					// Clear product-local support key
					imi_update_support_until( $support_key, '' );
				} else {
					delete_option( $option_code_key );
					delete_option( $option_status_key );
					imi_update_support_until( $support_key, '' );
				}

				set_transient( 'imi_license_invalid_notice', true, 5 * MINUTE_IN_SECONDS );
				return;
			}

			// status === 0 (temporary) or unknown → DO NOT delete
			$fail_key = 'imi_license_failures_' . $product_id;
			$failures = (int) get_option( $fail_key, 0 );
			update_option( $fail_key, min( $failures + 1, 100 ) );

			// If API still returned a support date, we can store it (non-destructive)
			if ( ! empty( $data['support_until'] ) ) {
				$support_until = sanitize_text_field( $data['support_until'] );
				imi_update_support_until( $support_key, $support_until );
			}
		} );
	}
}

// How long to keep local copies (match your banner transient, e.g. 12 hours)
if ( ! defined('IMI_BANNER_IMG_TTL') ) {
    define('IMI_BANNER_IMG_TTL', 12 * HOUR_IN_SECONDS);
}

/**
 * Cache a remote image locally under uploads/imi-banners and return the local URL.
 * Falls back to the original URL on failure.
 */
if ( ! function_exists( 'imi_cache_remote_image' ) ) {
    function imi_cache_remote_image( string $remote_url, string $cache_key ) : string {
        if ( empty($remote_url) ) return '';

        // Only http/https
        $scheme = wp_parse_url($remote_url, PHP_URL_SCHEME);
        if ( ! in_array( strtolower( (string) $scheme ), ['http','https'], true ) ) {
            return esc_url_raw($remote_url);
        }

        // Uploads dir
        $uploads = wp_upload_dir();
        if ( ! empty($uploads['error']) ) {
            return esc_url_raw($remote_url);
        }

        $subdir  = trailingslashit( $uploads['basedir'] ) . 'imi-banners/';
        $suburl  = trailingslashit( $uploads['baseurl'] ) . 'imi-banners/';

        if ( ! file_exists( $subdir ) ) {
            wp_mkdir_p( $subdir );
        }

        // Build a stable filename: <cache_key>-<hash>.<ext>
        $path_part = wp_parse_url($remote_url, PHP_URL_PATH);
        $ext       = pathinfo( (string) $path_part, PATHINFO_EXTENSION );
        $ext       = $ext ? strtolower($ext) : 'jpg'; // default if none
        $hash      = md5( strtolower( $remote_url ) );
        $filename  = sanitize_file_name( $cache_key . '-' . $hash . '.' . $ext );

        $local_path = $subdir . $filename;
        $local_url  = $suburl . $filename;

        // If exists and fresh, use it
        if ( file_exists( $local_path ) ) {
            $age = time() - (int) filemtime( $local_path );
            if ( $age < IMI_BANNER_IMG_TTL ) {
                return esc_url_raw( $local_url );
            }
        }

        // (Re)download
        $resp = wp_remote_get( $remote_url, [ 'timeout' => 10 ] );
        if ( is_wp_error($resp) ) {
            // fallback to remote if download failed
            return esc_url_raw( $remote_url );
        }

        $code = (int) wp_remote_retrieve_response_code( $resp );
        $body = wp_remote_retrieve_body( $resp );
        if ( $code !== 200 || ! $body ) {
            return esc_url_raw( $remote_url );
        }

        // If content-type present, adjust extension
        $ctype = wp_remote_retrieve_header( $resp, 'content-type' );
        if ( $ctype ) {
            $map = [
                'image/jpeg' => 'jpg',
                'image/jpg'  => 'jpg',
                'image/png'  => 'png',
                'image/gif'  => 'gif',
                'image/webp' => 'webp',
                'image/svg+xml' => 'svg',
            ];
            $maybe_ext = $map[ strtolower($ctype) ] ?? $ext;
            if ( $maybe_ext && $maybe_ext !== $ext ) {
                $ext      = $maybe_ext;
                $filename = sanitize_file_name( $cache_key . '-' . $hash . '.' . $ext );
                $local_path = $subdir . $filename;
                $local_url  = $suburl . $filename;
            }
        }

        // Write file
        $written = file_put_contents( $local_path, $body );
        if ( false === $written ) {
            return esc_url_raw( $remote_url );
        }

        // Touch for fresh mtime
        @touch( $local_path, time() );

        return esc_url_raw( $local_url );
    }
}

if ( ! function_exists( 'imi_fetch_and_display_banners' ) ) {
	function imi_fetch_and_display_banners( $product_id, $option_code_key ) {
		if ( ! current_user_can( 'manage_options' ) ) return;

		$cache_key = 'imi_banner_cache_' . $product_id;
		$banners   = get_transient( $cache_key );

		if ( false === $banners ) {
			$domain   = imi_get_current_domain();
			$purchase = imi_get_purchase_code( $option_code_key );

			$response = wp_remote_get( "https://envato.api.imithemes.com/wp-json/imi/banners?product=$product_id&domain=$domain&purchase=$purchase" );
			if ( is_wp_error( $response ) ) return;

			$banners = json_decode( wp_remote_retrieve_body( $response ), true );
			if ( ! is_array( $banners ) ) return;

			set_transient( $cache_key, $banners, 12 * HOUR_IN_SECONDS );
		}

		foreach ( $banners as $banner ) {
            static $imi_printed_banner_ids = []; // persists across multiple calls in the same request

            $id = esc_attr( $banner['id'] ?? '' );
            if ( $id && isset( $imi_printed_banner_ids[ $id ] ) ) {
                continue; // already printed by theme or plugin
            }
            $imi_printed_banner_ids[ $id ] = true;
            
			$type    = esc_attr( $banner['type'] ?? 'info' );
			$title   = esc_html( $banner['title'] ?? '' );
			$message = wp_kses_post( $banner['message'] ?? '' );
			$url     = ! empty( $banner['url'] ) ? ' <a href="' . esc_url( $banner['url'] ) . '" class="button button-primary" target="_blank" rel="noopener">Learn more</a>' : '';
			$cached_image_url = '';
            if ( ! empty( $banner['image'] ) ) {
                // Use the banner's ID as the cache key so duplicates share the same file
                $cached_image_url = imi_cache_remote_image( $banner['image'], $banner['id'] ?? md5($banner['image']) );
            }
            $image = $cached_image_url
                ? '<div class="imi-notice-logo-wrapper"><img src="' . esc_url( $cached_image_url ) . '" style="max-width:80px; height:auto;" /></div>'
                : '';
			$id      = esc_attr( $banner['id'] ?? '' );

			echo '<div class="notice notice-' . $type . ' is-dismissible imi-notice" data-banner-id="' . $id . '">'
				. $image .
				'<div class="imi-notice-content-wrapper">
					<h3>' . $title . '</h3>
					<p>' . $message . '</p>' . $url . '
				</div>
			</div>';
		}
	}
}

// Schedule once
add_action('init', function () {
    if ( ! wp_next_scheduled('imi_cleanup_cached_banner_images') ) {
        wp_schedule_event( time(), 'daily', 'imi_cleanup_cached_banner_images' );
    }
});

add_action('imi_cleanup_cached_banner_images', function () {
    $uploads = wp_upload_dir();
    if ( ! empty($uploads['error']) ) return;

    $dir = trailingslashit( $uploads['basedir'] ) . 'imi-banners/';
    if ( ! is_dir( $dir ) ) return;

    $ttl_hard = 7 * DAY_IN_SECONDS; // delete files older than 7 days
    $now = time();

    foreach ( glob( $dir . '*' ) as $file ) {
        if ( is_file( $file ) ) {
            $age = $now - (int) filemtime( $file );
            if ( $age > $ttl_hard ) {
                @unlink( $file );
            }
        }
    }
});
