<?php
/* API TO CREATE A NEW EVENTER POST
** INCLUDES CUSTOM FIELDS, TICKETS, WOO TICKETS, TIME SLOTS
*/
add_action('rest_api_init', function () {
    register_rest_route('eventer/v1', '/create-update-event', [
        'methods' => 'POST',
        'callback' => 'eventer_create_update_event',
        'permission_callback' => 'eventer_api_permission',
    ]);
});

// Permission callback for authentication
function eventer_api_permission() {
    return current_user_can('edit_posts');
}

// Callback to handle the request
function eventer_create_update_event(WP_REST_Request $request) {
    $params = $request->get_json_params();

    // Validate required fields
    if (empty($params['title'])) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('The "title" field is required.','eventer'),
        ], 400);
    }

    // Check if updating an existing event
    $event_id = isset($params['id']) ? absint($params['id']) : 0;

    // Prepare event data
    $event_data = [
        'post_type'    => 'eventer',
        'post_title'   => sanitize_text_field($params['title']),
        'post_content' => isset($params['description']) ? sanitize_textarea_field($params['description']) : '',
        'post_status'  => isset($params['status']) ? sanitize_text_field($params['status']) : 'publish',
    ];

    // Insert or update the post
    if ($event_id) {
        $event_data['ID'] = $event_id;
        $event_id = wp_update_post($event_data);
    } else {
        $event_id = wp_insert_post($event_data);
    }

    // Check for errors
    if (is_wp_error($event_id)) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('Failed to save the event.','eventer'),
        ], 500);
    }

    // Save basic meta fields
    $meta_fields = [
        'event_start_dt', 'event_end_dt', 'event_each_day_time', 'event_all_day',
        'event_featured', 'event_virtual', 'event_specific_venue', 'event_additional_info',
        'event_frequency_type', 'event_day_month', 'event_weekly_repeat', 'event_week_day',
        'event_frequency', 'event_frequency_count', 'event_multiple_dt_exc',
        'event_multiple_dt_inc', 'event_custom_permalink', 'event_custom_permalink_target',
        'event_custom_permalink_btn', 'event_individual_registrants_fields', 'event_custom_status',
        'event_registration_swtich', 'event_registration_form', 'event_custom_registration_url',
        'event_registration_target', 'event_email_additional_info', 'event_ticket_email',
        'event_ticket_image_notes', 'event_ticket_pdf_notes',
    ];

    foreach ($meta_fields as $field) {
        if (isset($params[$field])) {
            update_post_meta($event_id, 'eventer_' . $field, sanitize_text_field($params[$field]));
        }
    }

    // Save time slots
    if (isset($params['time_slots']) && is_array($params['time_slots'])) {
        $time_slots = array_map(function ($slot) {
            return [
                'title' => sanitize_text_field($slot['title']),
                'start' => sanitize_text_field($slot['start']),
                'end'   => sanitize_text_field($slot['end']),
                'desc'  => sanitize_textarea_field($slot['desc']),
            ];
        }, $params['time_slots']);

        update_post_meta($event_id, 'eventer_time_slot', $time_slots);
    }

    // Save time slot mandatory option
    if (isset($params['time_slots_mandatory'])) {
        $mandatory_value = $params['time_slots_mandatory'] === 'on' ? 'on' : 'off';
        update_post_meta($event_id, 'eventer_time_slots_mandatory', $mandatory_value);
    }

    // Save taxonomies
    $taxonomies = [
        'eventer-category',
        'eventer-tag',
        'eventer-venue',
        'eventer-organizer'
    ];

    foreach ($taxonomies as $taxonomy) {
        if (isset($params[$taxonomy]) && is_array($params[$taxonomy])) {
            $terms = array_map('sanitize_text_field', $params[$taxonomy]);
            wp_set_object_terms($event_id, $terms, $taxonomy);
        }
    }

    // Check WooCommerce ticketing settings
    $woocommerce_ticketing = eventer_get_settings('eventer_enable_woocommerce_ticketing');

    if ($woocommerce_ticketing === 'yes') {
        // Save WooCommerce tickets
        if (isset($params['woocommerce_tickets']) && is_array($params['woocommerce_tickets'])) {
            $tickets = array_map(function ($ticket) {
                return [
                    'pid'       => absint($ticket['product_id']),
                    'name'      => sanitize_text_field($ticket['name']),
                    'number'    => absint($ticket['number']),
                    'price'     => floatval($ticket['price']),
                    'enabled'   => sanitize_text_field($ticket['enabled']),
                    'action'    => sanitize_text_field($ticket['action']), // Add/Update/Delete
                ];
            }, $params['woocommerce_tickets']);

            update_post_meta($event_id, 'eventer_tickets', $tickets);
        }

        // Save WooCommerce-specific fields
        if (isset($params['disable_booking_before'])) {
            update_post_meta($event_id, 'eventer_disable_booking_before', absint($params['disable_booking_before']));
        }
        if (isset($params['common_ticket_count'])) {
            update_post_meta($event_id, 'eventer_common_ticket_count', absint($params['common_ticket_count']));
        }
        if (isset($params['common_tickets_management'])) {
            update_post_meta($event_id, 'eventer_common_tickets_management', sanitize_text_field($params['common_tickets_management']));
        }

    } else {
        // Save regular tickets if WooCommerce ticketing is disabled
        if (isset($params['tickets']) && is_array($params['tickets'])) {
            $tickets = array_map(function ($ticket) {
                return [
                    'name'      => sanitize_text_field($ticket['name']),
                    'number'    => absint($ticket['number']),
                    'price'     => floatval($ticket['price']),
                    'restrict'  => isset($ticket['restrict']) && $ticket['restrict'] === '1' ? '1' : '0',
                    'id'        => uniqid('ticket_', true), // Unique ticket ID
                ];
            }, $params['tickets']);

            update_post_meta($event_id, 'eventer_tickets', $tickets);
        }
    }

    return new WP_REST_Response([
        'success' => true,
        'message' => __('Event saved successfully.','eventer'),
        'event_id' => $event_id,
    ]);
}

/* API TO CREATE A NEW EVENTER POST TAXONOMY
** MULTI API FOR ALL THE TAXONOMIES
*/
add_action('rest_api_init', function () {
    register_rest_route('eventer/v1', '/create-taxonomy', [
        'methods' => 'POST',
        'callback' => 'eventer_create_taxonomy',
        'permission_callback' => 'eventer_create_taxonomy_permission',
    ]);
});

// Permission callback for authentication
function eventer_create_taxonomy_permission() {
    return current_user_can('manage_categories');
}

// Callback function to create taxonomy terms
function eventer_create_taxonomy(WP_REST_Request $request) {
    $params = $request->get_json_params();

    // Validate required fields
    if (empty($params['taxonomy'])) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('The "taxonomy" field is required.','eventer'),
        ], 400);
    }

    if (empty($params['name'])) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('The "name" field is required.','eventer'),
        ], 400);
    }

    // Supported taxonomies
    $supported_taxonomies = ['eventer-category', 'eventer-tag', 'eventer-venue', 'eventer-organizer'];

    $taxonomy = sanitize_text_field($params['taxonomy']);

    // Check if the taxonomy is supported
    if (!in_array($taxonomy, $supported_taxonomies, true)) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('Invalid taxonomy. Supported taxonomies are: ','eventer') . implode(', ', $supported_taxonomies),
        ], 400);
    }

    $term_name = sanitize_text_field($params['name']);
    $term_slug = isset($params['slug']) ? sanitize_title($params['slug']) : '';
    $term_description = isset($params['description']) ? sanitize_textarea_field($params['description']) : '';

    // Create or update the term
    $result = wp_insert_term($term_name, $taxonomy, [
        'slug'        => $term_slug,
        'description' => $term_description,
    ]);

    if (is_wp_error($result)) {
        return new WP_REST_Response([
            'success' => false,
            'message' => $result->get_error_message(),
        ], 500);
    }

    return new WP_REST_Response([
        'success' => true,
        'message' => __('Taxonomy term created successfully.','eventer'),
        'term_id' => $result['term_id'],
        'taxonomy' => $taxonomy,
    ]);
}
