<?php
namespace BooklyInvoices\Backend\Components\Invoice;

use Bookly\Lib as BooklyLib;
use Bookly\Lib\Payment\Proxy;
use Bookly\Backend\Modules\Notifications\Lib\Codes;
use BooklyInvoices\Backend\Modules\Settings\Lib\Helper;

class Invoice extends BooklyLib\Base\Component
{
    /**
     * Render invoice content.
     *
     * @param BooklyLib\Entities\Payment $payment
     * @return string
     */
    public static function render( BooklyLib\Entities\Payment $payment )
    {
        $payment_data = $payment->getPaymentData();
        $company_logo = BooklyLib\Utils\Common::getAttachmentUrl( get_option( 'bookly_co_logo_attachment_id' ), 'full' );

        $data = $payment_data['payment'];
        $created_at = BooklyLib\Slots\DatePoint::fromStr( $data['created_at'] );
        $helper = new Helper();
        $codes = array(
            'company_address' => nl2br( get_option( 'bookly_co_address' ) ),
            'company_logo' => $company_logo ? sprintf( '<img src="%s"/>', esc_attr( $company_logo[0] ) ) : '',
            'company_name' => get_option( 'bookly_co_name' ),
            'company_phone' => get_option( 'bookly_co_phone' ),
            'company_website' => get_option( 'bookly_co_website' ),
            'client_email' => '',
            'client_first_name' => strtok( $data['customer'], ' ' ),
            'client_last_name' => strtok( '' ),
            'client_name' => $data['customer'],
            'client_phone' => '',
            'client_note' => '',
            'client_address' => '',
            'invoice_number' => $data['invoice_number'],
            'invoice_link' => $data['id'] ? admin_url( 'admin-ajax.php?action=bookly_invoices_download&token=' . $data['token'] ) : '',
            'invoice_date' => BooklyLib\Utils\DateTime::formatDate( $created_at->format( 'Y-m-d' ) ),
            'invoice_due_date' => BooklyLib\Utils\DateTime::formatDate( $created_at->modify( get_option( 'bookly_invoices_due_days' ) * DAY_IN_SECONDS )->format( 'Y-m-d' ) ),
            'invoice_due_days' => get_option( 'bookly_invoices_due_days' ),
        );

        $time_zone_offset = null;
        $time_zone = null;

        /** @var BooklyLib\Entities\CustomerAppointment $ca */
        $first_ca = true;
        foreach ( BooklyLib\Entities\CustomerAppointment::query( 'ca' )->where( 'ca.payment_id', $data['id'] )->find() as $ca ) {
            $ca->customer = BooklyLib\Entities\Customer::find( $ca->getCustomerId() );
            if ( $first_ca ) {
                $time_zone_offset = $ca->getTimeZoneOffset();
                $time_zone = $ca->getTimeZone();
                $codes['client_email'] = $ca->customer->getEmail();
                $codes['client_first_name'] = $ca->customer->getFirstName();
                $codes['client_last_name'] = $ca->customer->getLastName();
                $codes['client_name'] = $ca->customer->getFullName();
                $codes['client_phone'] = $ca->customer->getPhone();
                $codes['client_address'] = $ca->customer->getAddress();
                $codes['client_birthday'] = $ca->customer->getBirthday() ? BooklyLib\Utils\DateTime::formatDate( $ca->customer->getBirthday() ) : '';
                $codes['client_note'] = $ca->customer->getNotes();
            }
            $first_ca = false;

            $codes = BooklyLib\Proxy\Shared::prepareCustomerAppointmentCodes( $codes, $ca, 'text' );
        }

        if ( $payment->getPaidType() === BooklyLib\Entities\Payment::PAY_DEPOSIT ) {
            $show_deposit = BooklyLib\Config::depositPaymentsActive();
            if ( ! $show_deposit ) {
                foreach ( $data['items'] as $item ) {
                    if ( array_key_exists( 'deposit_format', $item ) && $item['deposit_format'] ) {
                        $show_deposit = true;
                        break;
                    }
                }
            }
        } else {
            $show_deposit = false;
        }

        switch ( $data['type'] ) {
            case BooklyLib\Entities\Payment::TYPE_PAYPAL:
                $price_correction = get_option( 'bookly_paypal_increase' ) != 0
                    || get_option( 'bookly_paypal_addition' ) != 0;
                break;
            case BooklyLib\Entities\Payment::TYPE_CLOUD_STRIPE:
                $price_correction = get_option( 'bookly_cloud_stripe_increase' ) != 0
                    || get_option( 'bookly_cloud_stripe_addition' ) != 0;
                break;
            default:
                $price_correction = Proxy\Shared::paymentSpecificPriceExists( $data['type'] ) === true;
                break;
        }

        $show = array(
            'coupons' => isset( $data['coupon'] ) && ( $data['coupon']['discount'] > 0 || $data['coupon']['deduction'] > 0 ),
            'customer_groups' => BooklyLib\Config::customerGroupsActive() && $data['group_discount'],
            'deposit' => (int) $show_deposit,
            'price_correction' => $price_correction,
            'manual_adjustments' => get_option( 'bookly_invoices_show_adjustments', 0 ),
            'taxes' => (int) ( BooklyLib\Config::taxesActive() || $data['tax_total'] > 0 ),
        );
        $adjustments = isset( $payment_data['adjustments'] ) ? $payment_data['adjustments'] : array();

        $data['child_payments'] = BooklyLib\Entities\Payment::query( 'p' )
            ->select( 'p.type, p.paid, p.tax' )
            ->where( 'p.parent_id', $payment->getId() )
            ->where( 'p.status', BooklyLib\Entities\Payment::STATUS_COMPLETED )
            ->fetchArray();

        $content = self::renderTemplate( 'invoice', array(
            'helper' => $helper,
            'codes' => '',
            'payment' => $data,
            'adjustments' => $adjustments,
            'show' => $show,
            'time_zone_offset' => $time_zone_offset,
            'time_zone' => $time_zone,
        ), false );

        return BooklyLib\Utils\Codes::replace( $content, $codes, false );
    }

    /**
     * Render editable template for invoice.
     *
     * @return string
     */
    public static function appearance()
    {
        wp_enqueue_media();

        self::enqueueScripts( array(
            'module' => array(
                'js/invoice-appearance.js' => array( 'jquery' ),
            ),
        ) );

        wp_localize_script( 'bookly-invoice-appearance.js', 'BooklyInvoicesL10n', array(
            'invalid_due_days' => __( 'Invoice due days: Please enter value in days from 1 to 365', 'bookly' ),
        ) );

        $helper = new Helper( 'editable' );
        $codes_list = new Codes( 'text' );
        $codes = json_encode( $codes_list->getGroups( array( 'customer', 'invoice', 'company', 'customer_appointment' ) ) );

        return self::renderTemplate( 'invoice', compact( 'helper', 'codes' ), false );
    }
}