<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       http://wordpress-multisite-sync.welaunch.io
 * @since      1.0.0
 *
 * @package    Wordpress_Multisite_Sync
 * @subpackage Wordpress_Multisite_Sync/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Wordpress_Multisite_Sync
 * @subpackage Wordpress_Multisite_Sync/admin
 * @author     Daniel Barenkamp <support@welaunch.io>
 */
class Wordpress_Multisite_Sync_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	private $cron;
	protected $crons;

	private $currentSite;
	private $sourceSite;
	private $destinationSite;
	private $destinationSites;
	protected $cronRunning;
	protected $currentSourceBlogID;
	protected $childSourcePost;
	
	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	private $post_fields = array(
		'post_author',
		'post_date',
		'post_date_gmt',
		'post_content',
		'post_title',
		'post_excerpt',
		'post_status',
		'post_password',
		'post_name',
		'post_modified',
		'post_modified_gmt',
		'post_content_filtered',
		'post_parent',
		'post_type',
		'post_mime_type',
		'comment_status',
		'comment_count',
		'ping_status',
		'to_ping',
		'pinged',
		'guid',
		'menu_order',
		);

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

    	$this->crons = array(
	        1 => 'first-cron-',
	        2 => 'second-cron-',
	        3 => 'third-cron-',
	        4 => 'fourth-cron-',
	        5 => 'fifth-cron-',
	        6 => 'sixth-cron-',
	        7 => 'seventh-cron-',
	        8 => 'eight-cron-',
	        9 => 'ninth-cron-',
	        10 => 'tenth-cron-',
        );

	}

	public function load_redux()
	{
	    // Load the theme/plugin options
	    if ( file_exists( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/options-init.php' ) ) {
	        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/options-init.php';
	    }
	}

	public function init() {
		global $wordpress_multisite_sync, $wpdb;

		$this->options = $wordpress_multisite_sync;

		if (!$this->get_option('enable')) {
			return FALSE;
		}

		$this->cronRunning = false;

		$this->currentSourceBlogID = get_current_blog_id();

        add_action( 'save_post', array($this, 'run_cronjob_on_publish_update_post'), 10, 3  );
        add_action( 'woocommerce_after_product_object_save', array($this, 'woo_product_sync'), 10, 2);
        add_action( 'wp_trash_post', array($this, 'run_cronjob_on_trash_post'), 10, 1  );
        add_action( 'before_delete_post', array($this, 'run_cronjob_on_delete_post'), 10, 1  );

		if ( isset($_GET['multisite-sync'])) {
			$count = 0;
			foreach ($this->crons as $key => $cron) {

				if ($this->get_option($cron . 'enable')) {
					$func = 'run_cronjob_' . $key;
					$this->$func();
					$count++;
				}
			}
			wp_redirect( admin_url('admin.php?page=wordpress_multisite_sync_options&multisite-sync-success=' . $count));
		}

		if ( isset($_GET['multisite-sync-success'])) {
			add_action( 'admin_notices', array($this, 'multisite_sync_success') );
		}
			

    	foreach ($this->crons as $key => $cron) {

			if (!$this->get_option($cron . 'enable')) {
				wp_clear_scheduled_hook('run_wordpress_multisite_sync_cronjob' . $key);
				continue;
			}

			if (! wp_next_scheduled ( 'run_wordpress_multisite_sync_cronjob' . $key )) {
				$recurrence = $this->get_option($cron . 'recurrence');
				if(empty($recurrence)) {
					$recurrence = 'twicedaily';
				}
				wp_schedule_event(time(), $recurrence, 'run_wordpress_multisite_sync_cronjob' . $key);
			}

			add_action( 'run_wordpress_multisite_sync_cronjob' . $key, array($this, 'run_cronjob_' . $key) );

		}
	}

	public function run_cronjob_on_publish_update_post($post_id, $post, $update)
	{
		$notAllowedPostTypes = array(
			'product_variation',
			'nav_menu_item',
			'revision',
		);

		// if($post->post_type )

		if( in_array($post->post_type, $notAllowedPostTypes) ) {
			return;
		}
		
		if($this->currentSourceBlogID !== get_current_blog_id()) {
			return;
		}

		foreach ($this->crons as $key => $cron) {


			update_post_meta( $post_id, '_wordpress_multisite_has_not_changed_site_' . $cron, '0' );

			if ( !$this->get_option($cron . 'enable')) {
				continue;
			}

			if ( !$this->get_option($cron . 'runOnUpdate') && ($update === true) ) {
				continue;
			}

			if( !$this->get_option($cron . 'runOnPublish') ) {
				continue;
			}

			$sourcePostTypes = $this->get_option($cron . 'sourcePostTypes');
			if(!in_array($post->post_type, $sourcePostTypes)) {
				continue;
			}


			$func = 'run_cronjob_' . $key;
			$this->$func();
		}
	}

	public function woo_product_sync($product, $data)
    {
        if(!$product) {
            return;
        }

        $id = $product->get_id();
 
        $this->run_cronjob_on_publish_update_post($id, get_post($id), true);
    }

	public function run_cronjob_on_trash_post($post_id)
	{
		$post = get_post($post_id);

		$notAllowedPostTypes = array(
			'product_variation',
			'nav_menu_item',
			'revision',
		);
		if( in_array($post->post_type, $notAllowedPostTypes) ) {
			return;
		}
		
		if($this->currentSourceBlogID !== get_current_blog_id()) {
			return;
		}

		foreach ($this->crons as $key => $cron) {

			update_post_meta( $post_id, '_wordpress_multisite_has_not_changed_site_' . $cron, '0' );

			if ( !$this->get_option($cron . 'enable')) {
				continue;
			}

			if ( !$this->get_option($cron . 'runOnTrash') ) {
				continue;
			}

			$func = 'run_cronjob_' . $key;
			$this->$func();
		}
	}


	public function run_cronjob_on_delete_post($post_id)
	{
		$post = get_post($post_id);

		$notAllowedPostTypes = array(
			'product_variation',
			'nav_menu_item',
			'revision',
		);

		if( in_array($post->post_type, $notAllowedPostTypes) ) {
			return;
		}

		if($this->currentSourceBlogID !== get_current_blog_id()) {
			return;
		}

		foreach ($this->crons as $key => $cron) {

			update_post_meta( $post_id, '_wordpress_multisite_has_not_changed_site_' . $cron, '0' );

			if ( !$this->get_option($cron . 'enable')) {
				continue;
			}

			if ( !$this->get_option($cron . 'runOnDelete') ) {
				continue;
			}

			$this->cron = $cron;

			$sourceSite = $this->get_option($cron.'sourceSite');
			$destinationSites = $this->get_option($cron.'destinationSites');
			$sourcePostTypes = $this->get_option($cron.'sourcePostTypes');

			if(!empty($sourcePostTypes)) {
				if(!in_array($post->post_type, $sourcePostTypes)) {
					continue;
				}
			}

			if(empty($destinationSites) || empty($sourceSite)) {
				continue;
			}

			foreach ($destinationSites as $destinationSite) {

				switch_to_blog($destinationSite);

				global $wpdb;

				$id = $post->ID;
				$post_title = $post->post_title;
				$post_name = $post->post_name;
				$post_type = $post->post_type;

				$deleteKey = $this->get_option($this->cron.'deleteKey');
				if(!$deleteKey) {
					die('Delete key missing from plugin settings.');
				}

				if($deleteKey === "sku") {

					switch_to_blog($sourceSite);
					$deleteKeyValue = get_post_meta($id, '_sku', true);
					switch_to_blog($destinationSite);

					$sql = "SELECT $wpdb->posts.* 
				    FROM $wpdb->posts, $wpdb->postmeta
				    WHERE $wpdb->posts.ID = $wpdb->postmeta.post_id 
				    AND $wpdb->postmeta.meta_value = %s 
				    AND $wpdb->posts.post_type = %s
				    AND $wpdb->postmeta.meta_key = '_sku' 
				    AND $wpdb->posts.post_status <> 'revision'";

				} else {

					$deleteKeyValue = $post->$deleteKey;
					$sql = "SELECT *
						FROM $wpdb->posts 
						WHERE " . $deleteKey. " = %s 
						AND $wpdb->posts.post_type = %s 
						AND $wpdb->posts.post_type <> 'revision';";
				}

				if(empty($deleteKeyValue)) {
					return FALSE;
				}

				$sqlCheck = $wpdb->get_var( $wpdb->prepare($sql, 
					$deleteKeyValue, $post_type
				));
				
				if(!$sqlCheck) {
					continue;
				}
				wp_delete_post($sqlCheck);
			}

			switch_to_blog($sourceSite);
		}

	}

// run_cronjob_on_trash_post

	public function add_custom_cron_schedules( $schedules ) {

		$schedules['minute'] = array(
			'interval' => 60,
			'display' => __( 'Every Minute', 'wordpress-multisite-sync' )
		);

		$schedules['10minutes'] = array(
			'interval' => 600,
			'display' => __( 'Every 10 Minutes', 'wordpress-multisite-sync' )
		);

		$schedules['30minutes'] = array(
			'interval' => 1800,
			'display' => __( 'Every 30 Minutes', 'wordpress-multisite-sync' )
		);

		$schedules['weekly'] = array(
			'interval' => 604800,
			'display' => __( 'Every Week', 'wordpress-multisite-sync' )
		);

		$schedules['monthly'] = array(
			'interval' => 2592000,
			'display' => __( 'Every Month', 'wordpress-multisite-sync' )
		);

		return $schedules;
	}

	public function multisite_sync_success()
	{
		$crons_runned = $_GET['multisite-sync-success'];
    ?>

	    <div class="notice notice-success is-dismissible">
	        <p><?php _e( $crons_runned . ' multisite cronjobs manually done', 'wordpress-multisite-sync' ); ?></p>
	    </div>
    
    <?php
	}

	public function run_cronjob_1() {
		$this->run_cronjob('first-cron-');
	}

	public function run_cronjob_2() {
		$this->run_cronjob('second-cron-');
	}

	public function run_cronjob_3() {
		$this->run_cronjob('third-cron-');
	}

	public function run_cronjob_4() {
		$this->run_cronjob('fourth-cron-');
	}

	public function run_cronjob_5() {
		$this->run_cronjob('fifth-cron-');
	}

	public function run_cronjob_6() {
		$this->run_cronjob('sixth-cron-');
	}

	public function run_cronjob_7() {
		$this->run_cronjob('seventh-cron-');
	}

	public function run_cronjob_8() {
		$this->run_cronjob('eight-cron-');
	}

	public function run_cronjob_9() {
		$this->run_cronjob('ninth-cron-');
	}

	public function run_cronjob_10() {
		$this->run_cronjob('tenth-cron-');
	}


	public function run_cronjob($job)
	{
		global $wpdb;

		if($this->cronRunning) {
			return;
		}

		$this->cronRunning = true;

    	$log = "<h1>Wordpress Multisite Cronjobs run report:</h1>";
        $cron = $job;

		$this->cron = $cron;
		$log .= '<b>' . strtoupper( str_replace('-', ' ', $job) ) . " Job :</b><br/>";

		$sourceSite = $this->get_option($this->cron.'sourceSite');
		$destinationSites = $this->get_option($this->cron.'destinationSites');
		$sourcePostTypes = $this->get_option($this->cron.'sourcePostTypes');
		$sourcePostStatus = $this->get_option($this->cron.'sourcePostStatus');

		$this->currentSite = get_current_blog_id();
		$this->sourceSite = $sourceSite;
		$this->destinationSites = $destinationSites;

		if(empty($destinationSites) || empty($sourceSite)) {
			return false;
		}

		switch_to_blog($sourceSite);

		$args = array(
				'posts_per_page'   => 500,
				'offset'           => 0,
				'category'         => '',
				'category_name'    => '',
				'orderby'          => 'id',
				'order'            => 'DESC',
				'include'          => '',
				'exclude'          => '',
				'post_mime_type'   => '',
				'post_parent'      => '',
				'author'			=> '',
				'suppress_filters' => true,
				'tax_query'			=> array(
					'relation' => 'OR',
				),
			);

		$meta_query = array();

		if(!isset($_GET['multisite-sync-ignore-changed'])) {

			$meta_query = array(
		        'relation'      => 'OR',
		        array(
		            'key'       => '_wordpress_multisite_has_not_changed_site_' . $cron,
		            'compare'   => 'EXISTS'
		        ),
		        // array(
		        //     'key'       => '_wordpress_multisite_has_not_changed_site_' . $cron,
		        //     'value'     => '0',
		        //     'compare'   => '='
	            // )
			);
		}

		$args['meta_query'] = $meta_query;

		if(!empty($sourcePostTypes)) {
			$args['post_type'] = $sourcePostTypes;
		} else {
			$args['post_type'] = 'any';
		}

		if(!empty($sourcePostStatus) && $sourcePostStatus !== "any") {
			$args['post_status'] = $sourcePostStatus;
		} else {
			$args['post_status'] = get_post_stati();//'any';
			unset($args['post_status']['auto-draft']);
			unset($args['post_status']['inherit']);
		}

		$excludePosts = $this->get_option($this->cron.'excludePosts');
		if(!empty($excludePosts))
		{
			$excludePosts = explode(",", $excludePosts);
		} else {
			$excludePosts = array();
		}

		if($this->get_option('syncOnlyCustomTaxonomies')) {
		    $taxonomies = $wpdb->get_results("SELECT Distinct taxonomy FROM `{$wpdb->prefix}term_taxonomy`;", ARRAY_A);
		    foreach ($taxonomies as $taxonomy) {
		    	$taxonomy = $taxonomy['taxonomy'];
		    	$terms = $this->get_option($this->cron.'postTaxonomy' . $taxonomy);
		    	if(!$terms) {
		    		continue;
		    	}

		    	$tmp = array();
		    	foreach ($terms as $term) {
		    		$term_split = explode('_', $term);
		    		if(empty($term_split[0]) || empty($term_split[1])) {
		    			continue;
	    			}

		    		$term_blog_id = $term_split[0];
		    		if($sourceSite !== $term_blog_id) {
		    			continue;
		    		}

		    		$term = $term_split[1];

		    		$tmp[] = $term;
		    	}
		    	
		    	$terms = $tmp;
				if(!$terms) {
		    		continue;
		    	}

		    	$args['tax_query'][] = array(
					'taxonomy' => $taxonomy,
					'field' => 'term_id', 
					'terms' => $terms, /// Where term_id of Term 1 is "1".
					'include_children' => false
		    	);
		    }
	    }

		$sourcePosts = get_posts( $args );

		if(empty($sourcePosts)) {
			return;
		}

		$productLog = array();
		foreach ($sourcePosts as $post) {

			$this->sourcePost = get_post( $post->ID );
			if(is_array($excludePosts)) {
				if(in_array($this->sourcePost->ID, $excludePosts)) {
					continue;
				}
			}

			if(!isset($_GET['multisite-sync-ignore-changed'])) {
				$hasNotChanged = get_post_meta($post->ID, '_wordpress_multisite_has_not_changed_site_' . $cron, true );
				if($hasNotChanged == "1") {
					continue;
				}
			}

			if($this->get_option('syncOnlySpecificPosts')) {

				$syncPost = get_post_meta($post->ID, 'wordpress_multisite_sync_sync_post', true );
				if($syncPost != "1") {

					if(isset($_POST['wordpress_multisite_sync_sync_post']) && $_POST['wordpress_multisite_sync_sync_post'] == "1") {

					} else {
						continue;
					}
				}
			}
			
			delete_post_meta( $post->ID, '_wordpress_multisite_has_not_changed_site_' . $cron );

			foreach ($destinationSites as $destinationSite) {

				if($this->get_option('syncOnlySpecificPostsPerSite')) {

					if(isset($_POST['wordpress_multisite_sync_sync_post_sites']) && !empty($_POST['wordpress_multisite_sync_sync_post_sites'])) {
						$syncPostSites = $_POST['wordpress_multisite_sync_sync_post_sites'];
					} else {
						$syncPostSites = get_post_meta($post->ID, 'wordpress_multisite_sync_sync_post_sites', true );	
					}
					
					if(!empty($syncPostSites) && is_array($syncPostSites)) {
						if(!in_array($destinationSite, $syncPostSites)) {
							continue;
						}
					} else {
						continue;
					}
				}

				$this->destinationSite = $destinationSite;

				switch_to_blog($this->destinationSite);

				$check = $this->check_post_exists($this->sourcePost);

				if($check === FALSE) {

					$productLog[$destinationSite][$this->sourcePost->ID] = array(
					'post_id' => $this->sourcePost->ID, 
					'post_title' => $this->sourcePost->post_title,
					'status' => "Empty Update Key!");

					switch_to_blog($this->sourceSite);
					continue;
				// Create
				} elseif( $check === NULL) {

					$createIfNotExists = $this->get_option($this->cron.'createIfNotExists');
					if($createIfNotExists) {

						$newPostId = $this->process_post($post);
						$productLog[$destinationSite][$this->sourcePost->ID] = array(
							'post_id' => $this->sourcePost->ID, 
							'post_title' => $this->sourcePost->post_title, 
							'status' => "Created new Post: " . $newPostId
						);

					} else {
						$productLog[$destinationSite][$this->sourcePost->ID] = array(
							'post_id' => $this->sourcePost->ID, 
							'post_title' => $this->sourcePost->post_title, 
							'status' => "Post not found, but will not be created.",
						);
					}

				// UPDATE
				} else {

					$updateIfExists = $this->get_option($this->cron.'updateIfExists');
					if($updateIfExists) {

						$newPostId = $this->process_post($post, $check);
						$productLog[$destinationSite][$this->sourcePost->ID] = array(
							'post_id' => $this->sourcePost->ID, 
							'post_title' => $post->post_title, 
							'status' => "Updated Post: " . $check);

					} else {
						
						$productLog[$destinationSite][$this->sourcePost->ID] = array(
							'post_id' => $this->sourcePost->ID, 
							'post_title' => $post->post_title, 
							'status' => "Post found, but will not be updated: " . $check);
					}
				}

				switch_to_blog($this->sourceSite);

				$children_products = get_children( 'post_parent=' . $this->sourcePost->ID . '&post_type=product_variation' );
				if(!empty($children_products)){
					foreach ( $children_products as $child ) {
						if (in_array($child->ID, $excludePosts)) {
							continue;
						}

						$this->childSourcePost = $child;

						switch_to_blog($this->destinationSite);
						
						$check_child = $this->check_post_exists($child);
						$child->post_parent = $newPostId;
						
						// CREATE
						if($check_child === NULL){

							$productLog[$destinationSite][$this->childSourcePost->ID] = array(
								'post_id' => $this->childSourcePost->ID, 
								'post_title' => $child->post_title, 
								'status' => "Child found and will be created: " . $check_child);
							$this->process_post($child, NULL, true);
						// UPDATE
						} else { 

							$productLog[$destinationSite][$this->childSourcePost->ID] = array(
							'post_id' => $this->childSourcePost->ID, 
							'post_title' => $child->post_title, 
							'status' => "Child found and will be updated: " . $check_child);
							$this->process_post($child, $check_child, true);
						}
					}
				}
			}
	    }	

	    switch_to_blog($this->currentSite);
		unset ( $GLOBALS['_wp_switched_stack'] );
		$GLOBALS['switched'] = false; 

		$log .= "Source Site: " . $this->sourceSite . '</br>';
		$log .= "Destination Sites: " . implode(', ', $this->destinationSites) . '</br>';
		$log .= "Log:<br/>";
		$log .= "<pre>" . var_export($productLog, true) . '</pre>';

        $mailLogs = $this->get_option('mailLogs');
        if($mailLogs && !empty($productLog))
		{
			$to = $this->get_option('mailLogsEmail');
			$subject = 'Wordpress Multisite Sync';
			$body = $log;
			$headers = array('Content-Type: text/html; charset=UTF-8');

			wp_mail( $to, $subject, $body, $headers);
		}

		$this->cronRunning = false;
	}

	private function check_post_exists($post)
	{
		global $wpdb;

		$id = $post->ID;
		$post_title = $post->post_title;
		$post_name = $post->post_name;
		$post_type = $post->post_type;

		$updateKey = $this->get_option($this->cron.'updateKey');
		if(!$updateKey) {
			die('Update key missing from plugin settings.');
		}

		if($updateKey === "sku") {

			switch_to_blog($this->sourceSite);
			$updateKeyValue = get_post_meta($id, '_sku', true);
			switch_to_blog($this->destinationSite);

			$sql = "SELECT $wpdb->posts.* 
		    FROM $wpdb->posts, $wpdb->postmeta
		    WHERE $wpdb->posts.ID = $wpdb->postmeta.post_id 
		    AND $wpdb->postmeta.meta_value = %s 
		    AND $wpdb->posts.post_type = %s
		    AND $wpdb->postmeta.meta_key = '_sku' 
		    AND $wpdb->posts.post_status <> 'revision'";

		} else {

			$updateKeyValue = $post->$updateKey;
			$sql = "SELECT *
				FROM $wpdb->posts 
				WHERE " . $updateKey. " = %s 
				AND $wpdb->posts.post_type = %s 
				AND $wpdb->posts.post_type <> 'revision';";
		}

		if(empty($updateKeyValue)) {
			return FALSE;
		}

		$sqlCheck = $wpdb->get_var( $wpdb->prepare($sql, 
			$updateKeyValue, $post_type
		));

		return $sqlCheck;
	}

	private function process_post($post, $id = NULL, $isChild = false)
	{
		global $wpdb;

		// only update setted fields
		foreach ($this->post_fields as $post_field) {
			$temp = $this->get_option($post_field);
			if($temp == FALSE){
				unset($post->$post_field);
			}
		}

		if($isChild) {
			$sourcePostId = $this->childSourcePost->ID;
		} else {
			$sourcePostId = $this->sourcePost->ID;
		}
		
		if($id != NULL){
			$post->ID = $id;
		} else {
			unset($post->ID);
		}

		$postStatus = $this->get_option($this->cron . 'postStatus');
		if(!empty($postStatus)) {
			$post->post_status = $postStatus;
		}

		remove_all_filters("content_save_pre");
		$newPostId = wp_insert_post($post);


		// Sync Post Meta
		$syncPostMeta = $this->get_option($this->cron . 'syncPostMeta');
		if($syncPostMeta) {

			$sourceSite = $this->get_option($this->cron . 'sourceSite');
			switch_to_blog($this->sourceSite);

			$sourcePostMetas = $wpdb->get_results( $wpdb->prepare("
				SELECT *
				FROM $wpdb->postmeta 
				WHERE post_id = %s", 
				$sourcePostId
			));
			switch_to_blog($this->destinationSite);

			$ignorePostMetas = array('_product_image_gallery', '_thumbnail_id');

			$excludePostmeta =  $this->get_option($this->cron . 'excludePostmeta');
			if(!empty($excludePostmeta)) {
				$excludePostmeta = explode(",", $excludePostmeta);
				$ignorePostMetas = array_merge($ignorePostMetas, $excludePostmeta);
			}

			$includePostMetas = $this->get_option($this->cron . 'includePostMetas');
			if(!empty($includePostMetas)) {
				$includePostMetas = explode(",", $includePostMetas);
			}

			foreach ($sourcePostMetas as $sourcePostMeta) {
				
				if(in_array($sourcePostMeta->meta_key, $ignorePostMetas)) {
					continue;
				}

				if(!empty($includePostMetas) && !in_array($sourcePostMeta->meta_key, $includePostMetas)) {
					continue;
				}

				$value = maybe_unserialize($sourcePostMeta->meta_value);
				$updatePostMeta = update_post_meta($newPostId, $sourcePostMeta->meta_key, $value);
				// FALSE = fail or same value
				// TRUE = success
				// or it return meta_id
				
			}
		}
		
		// Sync Taxonomies
		$syncTaxonomies = $this->get_option($this->cron . 'syncTaxonomies');
		if($syncTaxonomies) {
			switch_to_blog($this->sourceSite);

			$taxonomies = get_object_taxonomies( $post->post_type );
			$updateKey = $this->get_option($this->cron.'updateKey');

			foreach ( $taxonomies as $taxonomy ) {

				switch_to_blog($this->sourceSite);
				$post_terms       = wp_get_object_terms( $sourcePostId, $taxonomy );
				$post_terms_count = sizeof( $post_terms );
				switch_to_blog($this->destinationSite);

				wp_remove_object_terms($newPostId, 1, $taxonomy);
					
				// clear all existing
				wp_set_object_terms( $newPostId, array(), $taxonomy);

				for ( $i = 0; $i < $post_terms_count; $i++ ) {

					if($updateKey == "post_name") {
						wp_set_object_terms( $newPostId, $post_terms[ $i ]->slug, $taxonomy, true );
					} else {
						wp_set_object_terms( $newPostId, $post_terms[ $i ]->name, $taxonomy, true );
					}
				}

				
			}
			
			if($post->post_type == "product" || $post->post_type == "product_variation")
			{
				$this->duplicate_attributes($sourcePostId, $newPostId);
			}
		}

		// Sync Tags
		$syncTags = $this->get_option($this->cron . 'syncTags');
		if($syncTags) 
		{
			switch_to_blog($this->sourceSite);

			$originalTags = wp_get_post_tags( $sourcePostId);
			
			$tags = array();
			foreach ($originalTags as $originalTag) {
				$tags[] = $originalTag->name;
			}

			$tags = implode(',', $tags);
			switch_to_blog($this->destinationSite);
			wp_set_post_tags( $newPostId, $tags, true );
		}

		// Duplicate the images | FIRST ONLY FOR NON EXISITING POST
		$syncImages = $this->get_option($this->cron . 'syncImages');
		if($syncImages)
		{
			switch_to_blog($this->sourceSite);
			$duplicateImages = new Wordpress_Multisite_Sync_Images();

			$duplicateImages->set_original_blog_id($this->sourceSite);
			$duplicateImages->set_to_duplicate_blog_id($this->destinationSite);
			$duplicateImages->duplicate_media($sourcePostId, $newPostId);
		}

		if($post->post_type == "product" || $post->post_type == "product_variation") {
			$product = wc_get_product($newPostId);
			if($product) {
				$product->save();
			}
		}

		return $newPostId;
	}

	/**
	 * Duplicate the attribute
	 *
	 * @param mixed $id
	 * @param mixed $new_id
	 * @since 1.0.0
	 */
	private function duplicate_attributes($post_id, $new_id)
	{
		global $wpdb;
		switch_to_blog( $this->sourceSite );

		$product = wc_get_product( $post_id );

		if($product === false) {
			switch_to_blog( $this->sourceSite );
			return false;
		}

		$product_attributes = $product->get_attributes();

		switch_to_blog( $this->destinationSite );

		$new_product_attributes = $product_attributes;

		if ( ! empty( $product_attributes ) ) {
			$attribute_keys  = array_keys( $product_attributes );
			$attribute_total = sizeof( $attribute_keys );

			for ( $i = 0; $i < $attribute_total; $i ++ ) {

				$attribute = $product_attributes[ $attribute_keys[ $i ] ];

				if ( (isset($attribute['is_taxonomy'] ) && isset($attribute['name']) && !empty($attribute['name'])) || (isset($attribute['id']) && $attribute['id'] <> 0)) {

					$taxonomy = $attribute['name'];
					if ( ! taxonomy_exists( $taxonomy ) ) {
						register_taxonomy( $taxonomy, 'product' );
					}

					$taxonomy = wc_sanitize_taxonomy_name( preg_replace( "/^pa\_/", '', $taxonomy ) );

					switch_to_blog( $this->sourceSite );

					$source_attribute_taxonomy = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}woocommerce_attribute_taxonomies WHERE attribute_name = %s;", $taxonomy ) );
					if ( ! $source_attribute_taxonomy ) {
						$attribute_label = wc_attribute_label( $taxonomy );
					} else {
						$attribute_label = $source_attribute_taxonomy->attribute_label;
					}
					// OLD FUNCTION FOR GETTING ATTRIBUTE TERMS (e.g. color -> blue, black ...) 
					// Problem: All values were copied
					// $args            = array(
					// 	'hide_empty' => false
					// );
					// $attribute_terms = get_terms( array( $attribute['name'] ), $args );
					// print_r($attribute_terms);

					$args2            = array(
						'hide_empty' => false,
						'fields' => 'ids',
					);
					$attribute_terms_test = wc_get_product_terms( $product->get_id(), $attribute['name'], $args2 );

					$temp = array();
					foreach ($attribute_terms_test as $term_id) {
						$temp[] = get_term($term_id, $attribute['name']);
					}
					$attribute_terms = $temp;

					// switch_to_blog( $this->sourceSite );
					switch_to_blog( $this->destinationSite );

					// Let's check if the attribute taxonomy exists in the destination blog
					$destination_attribute_taxonomy = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}woocommerce_attribute_taxonomies WHERE attribute_name = %s;", $taxonomy ) );
					if ( ! $destination_attribute_taxonomy && $source_attribute_taxonomy ) {

						// Insert term
						$result = $wpdb->insert(
							$wpdb->prefix . 'woocommerce_attribute_taxonomies',
							array(
								'attribute_name'    => $taxonomy,
								'attribute_label'   => $attribute_label,
								'attribute_type'    => $source_attribute_taxonomy->attribute_type,
								'attribute_orderby' => $source_attribute_taxonomy->attribute_orderby,
								'attribute_public'  => $source_attribute_taxonomy->attribute_public
							),
							array( '%s', '%s', '%s', '%s', '%d' )
						);

						$new_attribute_taxonomy_id = false;
						if ( $result ) {
							$new_attribute_taxonomy_id = $wpdb->insert_id;

							$new_attribute_product_terms = array();
							foreach ( $attribute_terms as $term ) {
								if ( term_exists( $term->name, $term->taxonomy ) ) {
									$new_term = get_term_by( 'name', $term->name, $term->taxonomy, ARRAY_A );
								} else {
									$new_term = wp_insert_term( $term->name, $term->taxonomy, array( 'slug' => $term->slug ) );
								}

								if ( ! empty( $new_term ) && ! is_wp_error( $new_term ) ) {
									$new_attribute_product_terms[] = $new_term['term_id'];
								}
							}

							// Assign to product
							wp_set_post_terms( $new_id, $new_attribute_product_terms, $attribute['name'], true );

						}

						if ( $new_attribute_taxonomy_id ) {
							$new_product_attributes[ $attribute_keys[ $i ] ] = $attribute;
						}
					} elseif ( $destination_attribute_taxonomy && $source_attribute_taxonomy ) {

						$new_attribute_product_terms = array();
						foreach ( $attribute_terms as $term ) {
							if ( term_exists( $term->name, $term->taxonomy ) ) {
								$new_term = get_term_by( 'name', $term->name, $term->taxonomy, ARRAY_A );
							} else {
								$new_term = wp_insert_term( $term->name, $term->taxonomy, array( 'slug' => $term->slug ) );
							}

							if ( ! empty( $new_term ) && ! is_wp_error( $new_term ) ) {
								$new_attribute_product_terms[] = $new_term['term_id'];
							}

							// Assign to product
							wp_set_post_terms( $new_id, $new_attribute_product_terms, $attribute['name'], true );

						}

					} else {
						// Exists
						$new_product_attributes[ $attribute_keys[ $i ] ] = $attribute;
					}
				} else {
					// Custom product attribute
					$new_product_attributes[ $attribute_keys[ $i ] ] = $attribute;
				}
			}

			delete_transient( 'wc_attribute_taxonomies' );
			// update_post_meta( $new_id, '_product_attributes', $new_product_attributes );
		}
		switch_to_blog( $this->sourceSite );
	}

	public function update_post_updated_status($post_ID, $post_after, $post_before)
	{
		foreach ($this->crons as $key => $cron) {
			if (!$this->get_option($cron . 'enable')) {
				return;
			}
			update_post_meta( $post_ID, '_wordpress_multisite_has_not_changed_site_' . $cron, '0' );
		}
	}

	public function update_post_meta_updated_status( $meta_id, $object_id, $meta_key, $_meta_value ) 
	{
		foreach ($this->crons as $key => $cron) {
			if (!$this->get_option($cron . 'enable')) {
				return;
			}
			update_post_meta( $object_id, '_wordpress_multisite_has_not_changed_site_' . $cron, '0' );
		}
	}


    /**
     * Add custom ticket metaboxes
     * @author Daniel Barenkamp
     * @version 1.0.0
     * @since   1.0.0
     * @link    https://www.welaunch.io
     * @param   [type]                       $post_type [description]
     * @param   [type]                       $post      [description]
     */
    public function add_custom_metaboxes($post_type, $post)
    {
    	if($this->get_option('syncOnlySpecificPosts') || $this->get_option('syncOnlySpecificPostsPerSite') ) {
        	add_meta_box('wordpress-multisite-sync-connected', esc_html__('Sync Post', 'wordpress-multisite-sync'), array($this, 'meta_box'), '', 'side', 'high');
    	}
    }

    /**
     * Display Metabox metabox
     * @author Daniel Barenkamp
     * @version 1.0.0
     * @since   1.0.0
     * @link    https://www.welaunch.io
     * @return  [type]                       [description]
     */
    public function meta_box()
    {
        global $post;

        wp_nonce_field(basename(__FILE__), 'wordpress_multisite_sync_meta_nonce');

        echo '<div class="wordpress-multisite-sync-container">';

            if($this->get_option('syncOnlySpecificPosts')) {

                $checked = "";
                $syncThisPost = get_post_meta($post->ID, 'wordpress_multisite_sync_sync_post', true);
                if($syncThisPost == "1") {
                    $checked = 'checked="checked"';
                }

                echo '<label for="wordpress-multisite-sync-sync-post" style="font-size: 12px; margin-bottom: 10px; display:block; margin-top: 10px; border-bottom: 1px solid #eaeaea; padding-bottom: 5px;">
                <input id="wordpress-multisite-sync-sync-post" type="checkbox" name="wordpress_multisite_sync_sync_post" value="1" ' . $checked . '>
                ' . esc_html__('Sync this post', 'wordpress-multisite-sync') . '
                </label>';

            }

			if($this->get_option('syncOnlySpecificPostsPerSite')) {

				$syncThisPostSites = get_post_meta($post->ID, 'wordpress_multisite_sync_sync_post_sites', true);
				if(!is_array($syncThisPostSites)) {
					$syncThisPostSites = array();
				}

				$sites = get_sites();
				foreach($sites as $site) {

	                $checked = "";
	                if(in_array($site->blog_id, $syncThisPostSites)) {
	                    $checked = 'checked="checked"';
	                }

	                echo '<label for="wordpress-multisite-sync-sync-post-sites" style="font-size: 12px; margin-bottom: 10px; display:block; margin-top: 10px; border-bottom: 1px solid #eaeaea; padding-bottom: 5px;">
	                <input id="wordpress-multisite-sync-sync-post-sites" type="checkbox" name="wordpress_multisite_sync_sync_post_sites[]" value="' . $site->blog_id . '" ' . $checked . '>
	                ' . sprintf( esc_html__('Sync to %s', 'wordpress-multisite-sync'), $site->domain . $site->path) . '
	                </label>';
				}

			}

        echo '</div>';
    }

    /**
     * Save Custom Metaboxes
     * @author Daniel Barenkamp
     * @version 1.0.0
     * @since   1.0.0
     * @link    https://www.welaunch.io
     * @param   [type]                       $post_id [description]
     * @param   [type]                       $post    [description]
     * @return  [type]                                [description]
     */
    public function save_custom_metaboxes($post_id, $post, $update)
    {
        global $post, $sync_running;

        if(!$this->get_option('syncOnlySpecificPosts') && !$this->get_option('syncOnlySpecificPostsPerSite')) {
        	return false;
    	}

        if(!is_a($post, 'WP_Post')) {
            return false;
        }

        if (!current_user_can('edit_post', $post->ID)) {
            return false;
        }

        if (!isset($_POST['wordpress_multisite_sync_meta_nonce']) || !wp_verify_nonce($_POST['wordpress_multisite_sync_meta_nonce'], basename(__FILE__))) {
            return false;
        }

        if(isset($_POST['wordpress_multisite_sync_sync_post']) && !empty($_POST['wordpress_multisite_sync_sync_post'])) {
            update_post_meta($post->ID, 'wordpress_multisite_sync_sync_post', esc_html($_POST['wordpress_multisite_sync_sync_post']) );
        } else {
            delete_post_meta($post->ID, 'wordpress_multisite_sync_sync_post' );
        }

        if(isset($_POST['wordpress_multisite_sync_sync_post_sites']) && !empty($_POST['wordpress_multisite_sync_sync_post_sites'])) {
            update_post_meta($post->ID, 'wordpress_multisite_sync_sync_post_sites', $_POST['wordpress_multisite_sync_sync_post_sites']);
        }     
    }

	/**
	 * Gets options
	 *
	 * @since    1.0.0
	 */
    private function get_option($option)
    {
		if(!is_array($this->options)) {
    		return false;
    	}
    	
    	
    	if(!array_key_exists($option, $this->options))
    	{
    		return false;
    	}
    	return $this->options[$option];
    }

}
