<?php

/**
 * Duplicate product images
 *
 * @link       http://woocommerce-multisite-duplicate.db-dzine.de
 * @since      1.0.0
 *
 * @package    WooCommerce_Multisite_Duplicate
 * @subpackage WooCommerce_Multisite_Duplicate/admin
 * @author     Daniel Barenkamp <contact@db-dzine.de>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Wordpress_Multisite_Sync_Images {

	/**
	 * options of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options
	 */
	private $options;

	/**
	 * The original Blog ID from duplicate to
	 *
	 * @since  	1.0.0
	 * @access 	private
	 * @var  	int 		$original_blog_id
	 */
	private $original_blog_id;

	/**
	 * The to duplicate blog ID from duplicate to
	 *
	 * @since  	1.0.0
	 * @access 	private
	 * @var  	int 		$to_duplicate_blog_id
	 */
	private $to_duplicate_blog_id;

	/**
	 * Constructor
	 */
	public function __construct() {
		global $wordpress_multisite_sync;

		$this->options = $wordpress_multisite_sync;
	}

	public function set_original_blog_id($id)
	{
		$this->original_blog_id = $id;
	}

	public function set_to_duplicate_blog_id($id)
	{
		$this->to_duplicate_blog_id = $id;
	}

	/**
	 * duplicate media
	 *
	 * @param  WP_Post $post Post object
	 * @return array
	 */
	public function duplicate_media( $post_id, $new_post_id ) {

		// Get all media
		$images = $this->get_all_images( $post_id );

		if(empty($images)){
			switch_to_blog($this->original_blog_id);
			return FALSE;
		}

		$new_gallery_images = array();
		foreach ( $images as $key => $image ) {

			switch_to_blog($this->original_blog_id);
			$image_alt_text = get_post_meta( $image->ID, '_wp_attachment_image_alt', true );
			$metadata = wp_get_attachment_metadata( $image->ID );

			// Is the image a thumbnail?
			$is_thumbnail = false;
			if ( ! empty( $image->is_thumbnail ) )
				$is_thumbnail = true;
			switch_to_blog( $this->to_duplicate_blog_id );
			
			$new_image = (array)$image;
            unset( $new_image['ID'] );
            unset( $new_image['guid'] );

			$upload = $this->fetch_remote_file( $image->guid );

			if(is_int($upload)){
				$new_image_id = $upload;
			} else {
				if ( is_wp_error( $upload ) ) {
					if($this->get_option('imageDebug')) {
						var_dump($upload);
						wp_die();
					}

	                continue;
				}

	            if ( $info = wp_check_filetype( $upload['file'] ) )
	                $new_image['post_mime_type'] = $info['type'];
	            else
	                continue;

	            $new_image['post_parent'] = $new_post_id;

	            $new_image['guid'] = $upload['url'];

	            require_once( ABSPATH . 'wp-admin/includes/image.php' );

	            $new_image_id = wp_insert_attachment( $new_image, $upload['file'] );
	            wp_update_attachment_metadata( $new_image_id, wp_generate_attachment_metadata( $new_image_id, $upload['file'] ) );

	            // alt text
	            update_post_meta( $new_image_id, '_wp_attachment_image_alt', $image_alt_text );
			}

            if ( $is_thumbnail ){
            	set_post_thumbnail( $new_post_id, $new_image_id );
            } else {
            	$new_gallery_images[] = $new_image_id;
            }

		}

		$new_gallery_images = implode(',', $new_gallery_images);
		update_post_meta( $new_post_id, '_product_image_gallery', $new_gallery_images);

		switch_to_blog($this->original_blog_id);
	}

	/**
	 * Get images
	 *
	 * @param  int $item_id
	 * @return array
	 */
	public function get_all_images( $item_id ) {

		switch_to_blog($this->original_blog_id);

		$orig_post = get_post( $item_id );
		$all_images = array();

		// get Thumbnail
		$thumbnail = get_post( get_post_thumbnail_id( $orig_post->ID ) );

		// get product gallery images
		$product_gallery_images = get_post_meta($orig_post->ID, '_product_image_gallery');
		if(!empty($product_gallery_images)){
			$product_gallery_images = explode(',', $product_gallery_images[0]);

			$args = array(
				'post_type' => 'attachment',
				'posts_per_page' => -1,
	    		'post__in' => $product_gallery_images
			);
			$product_gallery_images = get_posts($args);
		}

		if ( ! empty( $thumbnail ) ) {
			$thumbnail->is_thumbnail = true;
		}

		if(is_array($product_gallery_images) && !empty($product_gallery_images)){
			$all_images = array_merge( array( $thumbnail ), $product_gallery_images );
		} else {
			$all_images = array( $thumbnail );
		}

		switch_to_blog( $this->to_duplicate_blog_id );

		if(empty($all_images)){
			return FALSE;
		}

		// Removing duplicate attachments
		$all_images_no_duplicates = array();
		$temp = array();
		foreach ( $all_images as $attachment ) {
			if ( ! in_array( $attachment->ID, $temp ) ) {
				$temp[] = $attachment->ID;
				$all_images_no_duplicates[] = $attachment;
			}
		}

		$all_images = $all_images_no_duplicates;
		return $all_images;
	}

	/** 
     * Fetch an image and download it.
     *
     * @return WP_Error/Array Image properties/Error
     */
    public function fetch_remote_file( $url )  {
        $file = basename( $url );
        $file_parts = pathinfo($file);
        $file_name = $file_parts['filename'];
        $file_extension = $file_parts['extension'];

		$args = array(
			// 'name' => $file_name,
			's' => $file_name,
			'post_type' => 'attachment',
			'posts_per_page' => -1,
		);

		$check_file_exists = get_posts($args);
		if(!empty($check_file_exists)){
			$fileExists = $check_file_exists[0]->ID;
		  	return $fileExists;
		}

        $upload = wp_upload_bits( $file, null, 0 );

        if ( $upload['error'] )
            return new WP_Error( 'upload_dir_error', $upload['error'] );

		$response = wp_remote_get( $url, array(
			'stream' => true,
			'filename' => $upload['file']
		) );

        if ( ! $response ) {
            @unlink( $upload['file'] );
            return new WP_Error( 'import_file_error', sprintf( __('Remote server did not respond for file: %s', 'wordpress-multisite-sync' ), $url ) );
        }

        if( !is_array($response)) {
        	return false;
        }

        if ( $response['response']['code'] != '200' ) {
            @unlink( $upload['file'] );
            return new WP_Error( 'import_file_error', sprintf( __('Remote server returned error response %1$d %2$s', 'wordpress-multisite-sync' ), esc_html($response['response']['code']), get_status_header_desc($response['response']['code']) ) );
        }

        $filesize = filesize( $upload['file'] );

        if ( isset( $response['headers']['content-length'] ) && $filesize != $response['headers']['content-length'] ) {
            @unlink( $upload['file'] );
            return new WP_Error( 'import_file_error', __('Remote file is incorrect size', 'wordpress-multisite-sync' ) );
        }

        if ( 0 == $filesize ) {
            @unlink( $upload['file'] );
            return new WP_Error( 'import_file_error', __('Zero size file downloaded', 'wordpress-multisite-sync' ) );
        }

        $max_size = (int) apply_filters( 'mcc_attachment_size_limit', 0 );
        if ( ! empty( $max_size ) && $filesize > $max_size ) {
            @unlink( $upload['file'] );
            return new WP_Error( 'import_file_error', sprintf(__('Remote file is too large, limit is %s', 'wordpress-multisite-sync' ), size_format($max_size) ) );
        }

        return $upload;
    }

	/**
	 * Gets options
	 *
	 * @since    1.0.0
	 */
    private function get_option($option)
    {
		if(!is_array($this->options)) {
    		return false;
    	}
    	
    	if(!array_key_exists($option, $this->options))
    	{
    		return false;
    	}
    	return $this->options[$option];
    }
}