<?php
/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.4.12
 * @copyright       (C) 2018-2024 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

namespace Merkulove\Deblocker;

use Exception;
use Merkulove\Deblocker\Unity\Plugin;
use Merkulove\Deblocker\Unity\Settings;
use Merkulove\Deblocker\Unity\TabAssignments;
use Octha\Obfuscator\Factory;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class AlgorithmProgressive {

	/**
	 * Directory name in uploads folder
	 * @var string
	 */
	private static $dir_name;

	/**
	 * File name in uploads folder
	 * @var string
	 */
	private static $file_name;

	// CSS classes
	private static $blackout;
	private static $modal;
	private static $wrapper;
	private static $filter;

	/**
	 * Debug mode is enabled
	 * @var bool
	 */
	private static $is_debug = false;

	/**
	 * Create hashed website URL to create unique script name and folder
	 * @return void
	 */
	private static function create_hash() {

		// Create hash from site-url
		$hash = md5( self::keyword() ) . md5( site_url() );

		// Remove all non-alphanumeric characters
		$hash = preg_replace( '/[^a-zA-Z]/', '', $hash );

		// First 6 symbols start from a letter
		self::$dir_name = preg_replace( '/^[0-9]+/', 'a', substr( $hash, 0, 6 ) );

		// Last 6 symbols start from a letter
		self::$file_name = preg_replace( '/^[0-9]+/', 'a', substr( $hash, -6 ) );

		self::$blackout = preg_replace( '/^[0-9]+/', 'a', substr( $hash, 6, 6 ) );
		self::$modal = preg_replace( '/^[0-9]+/', 'a', substr( $hash, 12, 6 ) );
		self::$wrapper = preg_replace( '/^[0-9]+/', 'a', substr( $hash, 18, 6 ) );
		self::$filter = preg_replace( '/^[0-9]+/', 'a', substr( $hash, 4, 10 ) );

	}

	/**
	 * Get keyword.
	 * @return string
	 */
	private static function keyword(): string {

		$options = Settings::get_instance()->options ?? [];
		$is_random = isset( $options[ 'random_keyword' ] ) && $options[ 'random_keyword' ] === 'on';

		if ( $is_random ) {

			// Random keyword
			$random_keyword = get_transient( 'mdp_deblocker_random_keyword' );
			if ( ! empty( $random_keyword ) ) {

				$keyword = $random_keyword;

			} else {

				try {
					$keyword = Tools::random_string();
				} catch ( Exception $e ) {
					$keyword = 'DeBlocker';
				}

				// Save random keyword to transient
				set_transient( 'mdp_deblocker_random_keyword', $keyword, DAY_IN_SECONDS );

			}

		} else {

			// Default keyword
			$keyword = $options[ 'keyword' ] ?? '';

		}

		return $keyword;
	}

	/**
	 * Run Progressive algorithm.
	 * @return void
	 */
	static function run() {

		// Check if debug mode is enabled
		$options = Settings::get_instance()->options ?? [];
		if ( $options['debug'] === 'on' ) {
			self::$is_debug = true;
		}

		// Create hash
		self::create_hash();

		if ( $options['files_location'] === 'page_end' ) {

			// Create and add inline JS and CSS in body
			add_action( 'wp_footer', [ __CLASS__, 'add_to_body_end' ] );

		} elseif( $options['files_location'] === 'page_head' ) {

			// Create and add inline JS and CSS in the head
			add_action( 'wp_head', [ __CLASS__, 'add_to_head' ] );

		} else {

			// Create and add CSS for frontend
			$css_path = self::make_copy_css();
			add_action( 'wp_enqueue_scripts', [ __CLASS__, 'enqueue_algorithm_style' ] );

			// Create and add JS for frontend
			$js_path = self::make_copy_js();
			add_action( 'wp_enqueue_scripts', [ __CLASS__, 'enqueue_algorithm_script' ] );

			// Remove unused files
			self::remove_unused_files( [ $css_path, $js_path ] );

		}

	}

	/**
	 * Add a script to <body> end.
	 * @return void
	 */
	public static function add_to_body_end() {

		if ( ! TabAssignments::get_instance()->display() ) {
			return;
		}

		// Add CSS to <body> end
		echo wp_sprintf(
			'<style id="%s">%s</style>',
			esc_attr( self::$file_name ),
			Bricks::basic_css( self::$file_name, self::css_classes() )
		);

		// Add JS to <body> end
		if ( self::js() ) {
			/** @noinspection BadExpressionStatementJS */
			echo wp_sprintf(
				'<script id="%s">%s %s</script>',
				esc_attr( self::$file_name ),
				'window.Data' . self::$file_name . ' = ' . json_encode( self::frontend_settings() ) . ';',
				self::js()
			);
		}

	}

	/**
	 * Add script to <head>.
	 * @return void
	 */
	public static function add_to_head() {

		if ( ! TabAssignments::get_instance()->display() ) {
			return;
		}

		// Add CSS to <head>
		echo wp_sprintf(
			'<style id="%s">%s</style>',
			esc_attr( self::$file_name ),
			Bricks::basic_css( self::$file_name, self::css_classes() )
		);

		// Add JS to <head>
		if ( self::js() ) {
			/** @noinspection BadExpressionStatementJS */
			echo wp_sprintf(
				'<script id="%s">%s %s</script>',
				esc_attr( self::$file_name ),
				'window.Data' . self::$file_name . ' = ' . FrontEndSettings::array_to_js_object( self::frontend_settings() ) . ';',
				self::js()
			);
		}

	}

	/**
	 * Enqueue style for Progressive algorithm.
	 * @return void
	 */
	public static function enqueue_algorithm_style() {

		if ( ! TabAssignments::get_instance()->display() ) {
			return;
		}

		wp_enqueue_style(
			self::$file_name,
			self::url( self::$dir_name ) . '/' . self::$file_name . '.css',
			[],
			false,
			false
		);

	}

	/**
	 * Enqueue script for Progressive algorithm.
	 * @return void
	 */
	public static function enqueue_algorithm_script() {

		if ( ! TabAssignments::get_instance()->display() ) {
			return;
		}

		$options = Settings::get_instance()->options;
		$is_debug = isset( $options['debug'] ) && $options['debug'] === 'on';

		wp_enqueue_script(
			self::$file_name,
			self::url( self::$dir_name ) . '/' . self::$file_name . '.js',
			[],
			$is_debug ? rand( 1, 9999 ) : false,
			false
		);

		wp_localize_script(
			self::$file_name,
			'Data' . self::$file_name,
			self::frontend_settings()
		);

	}

	/**
	 * Get ads source.
	 * @param $options
	 *
	 * @return array|mixed|string
	 */
	private static function ads_src( $options ) {

		$check_url = $options['check_url'];
		$check_all = isset($options['check_all']) && $options['check_all'] === 'on';

		// Split by new line
		$urls = ( explode( PHP_EOL, $check_url ) );

		// Remove empty values
		$urls = array_filter( $urls );

		// Remove duplicates
		$urls = array_unique( $urls );

		// Remove spaces
		$urls = array_map( 'trim', $urls );

		if ( count( $urls ) > 1 ) {

			$ads_src = [];
			foreach ( $urls as $url ) {
				$ads_src[] = base64_encode( $url );
			}

			if ($check_all) {

				// Return all URLs
				return $ads_src;

			} else {

				// Return random URL
				shuffle( $ads_src );
				return $ads_src[0];

			}

		} else {

			// return first URL
			return base64_encode( $urls[0] ?? '' );

		}

	}

	/**
	 * Provide settings to front-end JS.
	 * @return array - base64 encoded string in array
	 */
	private static function frontend_settings(): array {

		$options = Settings::get_instance()->options ?? [];

		$frontend_options = [
			'src' => AlgorithmProgressive::ads_src( $options ),
			'bgColor' => $options['bg_color'],
			'blur' => $options['blur'],
			'button' => $options['button'],
			'buttonStyle' => $options[ 'button_style' ] ?? '',
			'caption' => esc_html__( $options['button_caption'], 'deblocker' ),
			'content' => self::content(),
			'cross' => $options['cross'],
			'debug' => $options['debug'] ?? 'off',
			'folder' => self::$dir_name,
			'guide' => $options[ 'guide' ],
			'guideTranslation' => self::guide_translation(),
			'loop' => $options[ 'is_loop' ] === 'on' ? $options['loop_timeout'] : '0',
			'modalColor' => $options['modal_color'],
			'overlay' => $options['overlay'] ?? 'off',
			'openLimit' => $options['open_limit'] ?? 0,
			'openLimitCount' => $options['open_limit_count'],
			'pluginSrc' => base64_encode( Plugin::get_url() ),
			'prefix' => self::$file_name,
			'redirect' => $options['is_redirect'] === 'on' ? $options['redirect'] : '',
			'rules' => $options['rules'] ?? [],
			'style' => $options['style'],
			'textColor' => $options['text_color'],
			'timeout' => $options['timeout'],
			'title' => esc_html__( $options['title'], 'deblocker' ),
			'translations' => $options[ 'translations' ],
			'langJSON' => self::multilang_json(),
			'blackout' => self::$blackout,
			'modal' => self::$modal,
			'wrapper' => self::$wrapper,
			'filter' => self::$filter
		];

		$string_frontend_options = json_encode( $frontend_options );

		return [base64_encode( $string_frontend_options )];

	}

	/**
	 * Get JS file from the plugin and save it to upload folder.
	 * @return string
	 */
	private static function make_copy_js(): string {

		// Get DeBlocker JS file
		$js = self::js();

		// Save JS file to uploads folder
		$path = self::path( self::$dir_name ) . '/' . self::$file_name . '.js';
		if ( self::$is_debug || ! file_exists( $path ) ) {
			file_put_contents( $path, $js );
			self::save_created_paths( $path );
		}

		return $path;

	}

	/**
	 * Get CSS file from the plugin and save it to upload folder.
	 * @return string
	 */
	private static function make_copy_css(): string {

		$css = Bricks::basic_css( self::$file_name, self::css_classes() );

		$path = self::path( self::$dir_name ) . '/' . self::$file_name . '.css';
		if ( self::$is_debug || ! file_exists( $path ) ) {
			file_put_contents( $path, $css );
			self::save_created_paths( $path );
		}

		return $path;

	}

	/**
	 * Get JS file from the plugin.
	 * @return false|string
	 */
	private static function js() {

		// Obfuscate JS for PHP 8.0 and higher
		$is_obfuscate = Settings::get_instance()->options['obfuscate'] ?? 'off';

		// Obfuscate JS for PHP 8.0 and higher
		if ( version_compare( PHP_VERSION, '8.0', '>=' ) && $is_obfuscate === 'on' ) {
			require_once Plugin::get_path() . 'vendor/autoload.php';
		}

		// Get DeBlocker JS file
		$js = wp_remote_get(
			Plugin::get_url() . 'js/deblocker.min.js?nocache=' . time(),
			[
				'sslverify' => false,
			]
		);

		// Validate response
		if ( is_wp_error( $js ) ) {

			// Create transient with notice
			set_transient( 'mdp_deblocker_js_error', $js->get_error_message(), 60 * 60 * 24 ); // TODO: improve
			// TODO: add notice to admin
			return false;

		}
		set_transient( 'mdp_deblocker_js_error', '', 60 * 60 * 24 ); // TODO: improve

		// Get Body
		$js = wp_remote_retrieve_body( $js );
		if ( ! $js ) {
			// TODO: add notice to admin
			return false;
		}

		if ( version_compare( PHP_VERSION, '8.0', '>=' ) && $is_obfuscate === 'on' ) {
			$hunter = new Factory($js);
			return $hunter->Obfuscate();
		} else {
			return $js;
		}

	}

	/**
	 * Get JSON with translations.
	 */
	private static function multilang_json(): string {

		$translations_counter = intval( apply_filters( 'mdp_deblocker_advanced_translations_count', Config::$translations_count ) );
		$lang_json            = '{';
		for ( $i = 1; $i <= $translations_counter; $i ++ ) {

			if ( ! empty( $options[ 'translate_locale_' . $i ] ) ) {

				// Add a comma after the previous language
				if ( $i > 1 ) {
					$lang_json .= ',';
				}

				$caption = ! empty( $options[ 'translate_button_caption_' . $i ] ) ? $options[ 'translate_button_caption_' . $i ] : esc_html__( $options['button_caption'], 'deblocker' );

				$lang_json .= '"' . str_replace( '_', '-', strtolower( $options[ 'translate_locale_' . $i ] ) ) . '":';
				$lang_json .= '{';
				$lang_json .= '"title":"' . $options[ 'translate_title_' . $i ] ?? esc_html__( $options['title'], 'deblocker' );
				$lang_json .= '",';
				$lang_json .= '"content":"' . str_replace( '"', '\\\"', $options[ 'translate_content_' . $i ] ?? self::content() );
				$lang_json .= '",';
				$lang_json .= '"caption":"' . $caption;
				$lang_json .= '"';
				$lang_json .= '}';

			}
		}
		$lang_json .= '}';

		return json_encode( $lang_json );

	}

	/**
	 * Get PATH to script directory in uploads folder.
	 *
	 * @param $dir
	 *
	 * @return string
	 */
	static function path( $dir ): string {

		$options = Settings::get_instance()->options ?? [];

		switch ( $options[ 'files_location' ] ) {

			case 'root_directory':

				$upload_dir = ABSPATH . $dir;
				$upload_dir = rtrim( $upload_dir, '/' ); // remove last slash

				break;

			case 'root':

				$upload_dir = ABSPATH;

				break;

			case 'plugins':

				$plugins_dir = str_replace( 'wp-content/plugins/deblocker/', 'wp-content/plugins', Plugin::get_path() );
				$upload_dir = $plugins_dir;

				break;

			case 'plugin':

				$plugins_dir = str_replace( 'wp-content/plugins/deblocker/', 'wp-content/plugins', Plugin::get_path() );
				$upload_dir = $plugins_dir . '/' . $dir;

				break;

			case 'themes':

				$upload_dir = get_theme_root();

				break;

			case 'theme':

				$upload_dir = get_theme_root() . '/' . $dir;

				break;

			case 'uploads':
			default:

				$upload_dir = wp_upload_dir();
				$upload_dir = $upload_dir['basedir'];
				$upload_dir = $upload_dir . '/' . $dir;

				break;

		}


		// Create folder if not exists
		if ( !file_exists( $upload_dir ) ) {
			mkdir( $upload_dir );
		}

		return $upload_dir;

	}

	/**
	 * Get URL to script directory in uploads folder.
	 * @param $dir
	 *
	 * @return string
	 */
	static function url( $dir ): string {

		$options = Settings::get_instance()->options ?? [];

		switch ( $options[ 'files_location' ] ) {

			case 'root_directory':

				$upload_dir = site_url() . '/' . $dir;

				break;

			case 'root':

				$upload_dir = site_url();

				break;

			case 'plugins':

				$upload_dir = plugin_dir_url( $dir );
				$upload_dir = rtrim( $upload_dir, '/' ); // remove last slash

				break;

			case 'plugin':

				$upload_dir = plugin_dir_url( $dir );
				$upload_dir = rtrim( $upload_dir, '/' ); // remove last slash
				$upload_dir = $upload_dir . '/' . $dir;

				break;

			case 'themes':

				$upload_dir = get_theme_root_uri();

				break;

			case 'theme':

				$upload_dir = get_theme_root_uri() . '/' . $dir;

				break;

			case 'uploads':
			default:

				$upload_dir = wp_upload_dir();
				$upload_dir = $upload_dir['baseurl'];
				$upload_dir = $upload_dir . '/' . $dir;

				break;

		}

		return $upload_dir;

	}

	/**
	 * Get content for JS file.
	 *
	 * @return string
	 */
	static function content(): string {

		$options = Settings::get_instance()->options ?? [];

		if ( $options['content'] !== strip_tags( $options['content'] ) ){

			$content = wp_kses_post( $options['content'] ); // HTML

		} else {

			$content = esc_html__( $options['content'], 'deblocker' ); // Plain text

		}

		return $content;

	}

	/**
	 * Save created directories to options.
	 * @param $path
	 * @return void
	 */
	private static function save_created_paths( $path ) {

		$paths = get_option( 'mdp_deblocker_paths', [] );
		if ( in_array( $path, $paths ) ) { return; }

		$paths[] = $path;

		update_option( 'mdp_deblocker_paths', $paths );

	}

	/**
	 * Remove created directories from options.
	 * @return void
	 */
	public static function remove_unused_files( $used_paths ) {

		$all_paths = get_option( 'mdp_deblocker_paths', [] );
		if ( empty( $all_paths ) ) { return; }
		if ( $all_paths === $used_paths ) { return; }

		$unused_paths = array_diff( $all_paths, $used_paths );
		foreach ( $unused_paths as $path ) {
			if ( file_exists( $path ) ) {
				unlink( $path );
			}
		}

		update_option( 'mdp_deblocker_paths', $used_paths );

	}

	/**
	 * Set guide translations
	 * @return array
	 */
	private static function guide_translation(): array {

		return [
			'trigger' => esc_html__( 'How do I disable my ad blocker?', 'deblocker' ),
			'triggerOk' => esc_html__( 'OK. I understand.', 'deblocker' ),
			'guideTitle' => esc_html__( 'To disable ad blocker on this site:', 'deblocker' ),
			'guideList1' => esc_html__( 'Right click on the ad blocker extension icon at the top right corner of your browser', 'deblocker' ),
			'guideList2' => esc_html__( 'From the menu choose', 'deblocker' ) . ' <b>' . esc_html__( '"Disable on this site"', 'deblocker' ) . '</b> ' . esc_html__( 'or', 'deblocker' ) . ' <b>' . esc_html__( '"Pause on this site"', 'deblocker' ) . ' </b>',
			'guideList3' => esc_html__( 'Refresh the page if not automatically refreshed', 'deblocker' ) ,
		];

	}

	private static function css_classes() {
		return [
			'blackout' => self::$blackout,
			'modal' => self::$modal,
			'wrapper' => self::$wrapper,
			'filter' => self::$filter,
		];
	}

}
