<?php

/**
 * Class ESSB_Copy_Link_Actions_Generator
 *
 * Responsible for generating actions related to the "Copy Link" functionality
 * within the Easy Social Share Buttons plugin. This class handles the logic
 * required to create, manage, and execute copy link actions for sharing purposes.
 *
 * @package EasySocialShareButtons
 */
class ESSB_Copy_Link_Actions_Generator
{
    private static $initialized = false;

    /**
     * Initializes the Copy Link actions for the Easy Social Share Buttons plugin.
     * This static method sets up necessary hooks, filters, or other initialization
     * routines required for the Copy Link share button functionality.
     */
    public static function init()
    {
        add_action('wp_footer', array('ESSB_Copy_Link_Actions_Generator', 'generate_stucture'), 99);
        self::register_actions();
    }

    /**
     * Generates the structure for the copy link actions.
     *
     * This static method is responsible for creating and returning
     * the necessary structure used for handling copy link actions
     * within the Easy Social Share Buttons plugin.
     */
    public static function generate_stucture()
    {
        if (!self::$initialized) {
            if (essb_options_bool_value('copylink_direct')) {
                self::generate_direct_copy();
            } else {
                self::prepare_popup_styles();
                self::generate_popup_copy();
            }

            self::$initialized = true;
        }
    }

    /**
     * Generates a direct copy action for sharing links.
     *
     * This static method is responsible for creating the necessary logic or output
     * to enable users to directly copy a link, typically for sharing purposes.
     */
    public static function generate_direct_copy()
    {

        $output = '<div style="display: none;"><input type="text" id="essb_direct_copy_link_field" /></div>';
        $output .= '<div id="essb_direct_copy_link_message" style="background: rgba(0,0,0,0.7); color: #fff; z-index: 1100; position: fixed; display: none; align-items: center; justify-content: center; bottom: 20px; right: 20px; max-width: 90%; padding: 15px 25px; font-size: 13px; font-family: -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,\"Helvetica Neue\",sans-serif;">';
        if (essb_options_value('translate_copy_message2') != '') {
            $output .= essb_options_value('translate_copy_message2');
        } else {
            $output .= esc_html__('Copied to clipboard.', 'essb');
        }
        $output .= '</div>';

        echo $output;
    }

    /**
     * Prepares and returns the necessary styles for the copy link popup.
     *
     * This static method is responsible for generating or retrieving the CSS styles
     * required to properly display the copy link popup within the Easy Social Share Buttons plugin.     
     */
    public static function prepare_popup_styles()
    {
        ESSB_Runtime_CSS_Builder::init('essb-copylink-popup');
        ESSB_Runtime_CSS_Builder::register_multiple(
            'essb-copylink-popup',
            '.essb-copylink-shadow',
            array(
                'display' => 'none',
                'align-items' => 'center',
                'justify-content' => 'center',
                'position' => 'fixed',
                'top' => '0',
                'left' => '0',
                'z-index' => '1300',
                'background-color' => 'rgba(0,0,0,0.2)',
                'width' => '100%',
                'height' => '100%',
            )
        );

        ESSB_Runtime_CSS_Builder::register_multiple(
            'essb-copylink-popup',
            '.essb-copylink-window',
            array(
                'width' => '90%',
                'max-width' => '600px',
                'display' => 'none',
                'align-items' => 'center',
                'justify-content' => 'center',
                'flex-direction' => 'column',
                'overflow.' => 'hidden',
                'z-index' => '1310',
                'background-color' => '#fff',
                'border-radius' => '5px',
                'overflow' => 'hidden',
            )
        );

        ESSB_Runtime_CSS_Builder::register_multiple(
            'essb-copylink-popup',
            '.essb-copylink-window-header',
            array(
                'width' => '100%',
                'display' => 'flex',
                'align-items' => 'center',
                'justify-content' => 'center',
                'overflow' => 'hidden',
            )
        );

        ESSB_Runtime_CSS_Builder::register_multiple(
            'essb-copylink-popup',
            '.essb-copylink-window-header',
            array(
                'width' => '100%',
                'display' => 'flex',
                'align-items' => 'center',
                'justify-content' => 'space-between',
                'font-size' => '16px',
                'padding' => '10px',
                'background-color' => '#1b84ff',
                'color' => '#fff',
            )
        );

        ESSB_Runtime_CSS_Builder::register_multiple(
            'essb-copylink-popup',
            '.essb-copylink-window-content',
            array(
                'width' => '100%',
                'padding' => '20px'
            )
        );

        ESSB_Runtime_CSS_Builder::register('essb-copylink-popup', '.essb-copylink-window-close', 'fill', '#fff');
    }

    /**
     * Generates the HTML and JavaScript required to display a popup for copying a link.
     *
     * This static method is responsible for creating the necessary markup and logic
     * to allow users to copy a URL to their clipboard via a popup interface.
     *
     */
    public static function generate_popup_copy()
    {
        $output = '';

        $translate_copy_message1 = essb_options_value('translate_copy_message1');
        if (empty($translate_copy_message1)) {
            $translate_copy_message1 = esc_html__('Press to copy the link', 'essb');
        }

        $output .= ESSB_Dynamic_CSS_Builder::register_and_prepare_inline_code('essb-copylink-popup', ESSB_Runtime_CSS_Builder::compile('essb-copylink-popup'), true);
        $output .= '<div class="essb-copylink-shadow">';
        $output .= '<div class="essb-copylink-window">';
        $output .= '<div class="essb-copylink-window-header"> <span>&nbsp;</span> <a href="#" class="essb-copylink-window-close"><svg style="width: 24px; height: 24px; padding: 5px;" height="32" viewBox="0 0 32 32" width="32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32,25.7c0,0.7-0.3,1.3-0.8,1.8l-3.7,3.7c-0.5,0.5-1.1,0.8-1.9,0.8c-0.7,0-1.3-0.3-1.8-0.8L16,23.3l-7.9,7.9C7.6,31.7,7,32,6.3,32c-0.8,0-1.4-0.3-1.9-0.8l-3.7-3.7C0.3,27.1,0,26.4,0,25.7c0-0.8,0.3-1.3,0.8-1.9L8.7,16L0.8,8C0.3,7.6,0,6.9,0,6.3c0-0.8,0.3-1.3,0.8-1.9l3.7-3.6C4.9,0.2,5.6,0,6.3,0C7,0,7.6,0.2,8.1,0.8L16,8.7l7.9-7.9C24.4,0.2,25,0,25.7,0c0.8,0,1.4,0.2,1.9,0.8l3.7,3.6C31.7,4.9,32,5.5,32,6.3c0,0.7-0.3,1.3-0.8,1.8L23.3,16l7.9,7.9C31.7,24.4,32,25,32,25.7z"/></svg></a> </div>';
        $output .= '<div class="essb-copylink-window-content">';
        $output .= '<div class="essb_copy_internal" style="display: flex; align-items: center;">';
        $output .= '<div style="width: calc(100% - 50px); padding: 5px;"><input type="text" id="essb_copy_link_field" style="width: 100%;padding: 5px 10px;font-size: 15px;background: #f5f6f7;border: 1px solid #ccc;font-family: -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,\"Helvetica Neue\",sans-serif;" /></div>';
        $output .= '<div style="width:50px;text-align: center;"><a href="#" class="essb-copy-link" title="' . $translate_copy_message1 . '" style="color:#5867dd;background:#fff;padding:10px;text-decoration: none;"><svg style="width: 24px; height: 24px; fill: currentColor;" class="essb-svg-icon" aria-hidden="true" role="img" focusable="false" width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><path d="M25.313 28v-18.688h-14.625v18.688h14.625zM25.313 6.688c1.438 0 2.688 1.188 2.688 2.625v18.688c0 1.438-1.25 2.688-2.688 2.688h-14.625c-1.438 0-2.688-1.25-2.688-2.688v-18.688c0-1.438 1.25-2.625 2.688-2.625h14.625zM21.313 1.313v2.688h-16v18.688h-2.625v-18.688c0-1.438 1.188-2.688 2.625-2.688h16z"></path></svg></a></div>';
        $output .= '</div>';
        $output .= '<div class="essb-copy-message" style="font-size: 13px; font-family: -apple-system,BlinkMacSystemFont,\"Segoe UI\",Roboto,Oxygen-Sans,Ubuntu,Cantarell,\"Helvetica Neue\",sans-serif;"></div>';
        $output .= '</div>';
        $output .= '</div>';

        $output .= '</div>';

        echo $output;
    }

    
    /**
     * Registers the actions related to the "Copy Link" share button.
     *
     * This static method is responsible for setting up any hooks, filters, or other
     * actions required for the "Copy Link" functionality within the Easy Social Share Buttons plugin.
     */
    public static function register_actions()
    {
        if (essb_options_bool_value('copylink_direct')) {
            ESSB_Dynamic_JS_Builder::push_footer_code('essb-copylink-direct', 'function essb_copy_link_direct(e){document.querySelector("#essb_direct_copy_link_field")&&(document.querySelector("#essb_direct_copy_link_field").value=e,setTimeout(function(){var e=document.querySelector("#essb_direct_copy_link_field");try{e.focus(),e.select(),e.setSelectionRange(0,99999),document.execCommand("copy"),navigator.clipboard.writeText(e.value),document.querySelector("#essb_direct_copy_link_message").style.display="flex",setTimeout(function(){document.querySelector("#essb_direct_copy_link_message").style.display="none"},2e3)}catch(c){console.log("Error link copy to clipboard!")}},100))}');
        } else {
            ESSB_Dynamic_JS_Builder::push_footer_code('essb-copylink-popup', 'function essb_open_copy_link_window(e,o){let s=window.location.href;if(e&&document.querySelector(".essb_"+e)){let n=document.querySelector(".essb_"+e).getAttribute("data-essb-url")||"";""!=n&&(s=n)}if(o&&""!=o&&(s=o),essb_settings&&essb_settings.copybutton_direct){essb.copy_link_direct(s);return}document.querySelector(".essb-copylink-window")&&(document.querySelector("#essb_copy_link_field")&&(document.querySelector("#essb_copy_link_field").value=s),document.querySelector(".essb-copylink-window").style.display="flex",document.querySelector(".essb-copylink-shadow").style.display="flex",document.querySelector(".essb-copylink-window-close")&&(document.querySelector(".essb-copylink-window-close").onclick=function(e){e.preventDefault(),document.querySelector(".essb-copylink-window").style.display="none",document.querySelector(".essb-copylink-shadow").style.display="none"}),document.querySelector(".essb-copylink-window .essb-copy-link")&&(document.querySelector(".essb-copylink-window .essb-copy-link").onclick=function(e){e.preventDefault();let o=document.querySelector("#essb_copy_link_field");try{o.focus(),o.select(),o.setSelectionRange(0,99999),document.execCommand("copy"),navigator.clipboard.writeText(o.value),document.querySelector(".essb-copylink-window .essb-copylink-window-header span").innerHTML=essb_settings.translate_copy_message2?essb_settings.translate_copy_message2:"Copied to clipboard.",setTimeout(function(){document.querySelector(".essb-copylink-window .essb-copylink-window-header span").innerHTML="&nbsp;"},2e3)}catch(s){console.log(s.stack),document.querySelector(".essb-copylink-window .essb-copylink-window-header span").innerHTML=essb_settings.translate_copy_message3?essb_settings.translate_copy_message3:"Please use Ctrl/Cmd+C to copy the URL.",setTimeout(function(){document.querySelector(".essb-copylink-window .essb-copylink-window-header span").innerHTML="&nbsp;"},2e3)}}))}');
        }
    }
}
