<?php

class Leopard_Wordpress_Offload_Media_Upload_Handler extends Leopard_Wordpress_Offload_Media_Item_Handler {
	/**
	 * @var string
	 */
	protected static $item_handler_key = 'upload';

	/**
	 * Keep track of individual files we've already attempted to upload
	 *
	 * @var array
	 */
	protected $attempted_upload = array();

	protected $files_uploaded = array();

	/**
	 * The default options that should be used if none supplied.
	 *
	 * @return array
	 */
	public static function default_options() {
		return array(
			'offloaded_files' => array(),
		);
	}

	public function rebuild_key($Key, $custom_prefix=''){
        return leopard_offload_media_rebuild_key($Key, $custom_prefix);
    }

	/**
	 * Prepare item for uploading by running filters, updating
	 *
	 * @param Item  $leopard_item
	 * @param array $options
	 *
	 * @return Manifest|WP_Error
	 */
	protected function pre_handle( Leopard_Wordpress_Offload_Media_Item $leopard_item, array $options ) {
	
		$manifest         = new Leopard_Wordpress_Offload_Media_Manifest();
		$source_type_name = leopard_offload_media_get_source_type_name( $leopard_item->source_type() );
		$primary_key      = Leopard_Wordpress_Offload_Media_Item::primary_object_key();

		// Check for valid file path before attempting upload
		if ( empty( $leopard_item->source_path() ) ) {
			$error_msg = sprintf( __( '%s with id %d does not have a valid file path', 'leopard-wordpress-offload-media' ), $source_type_name, $leopard_item->source_id() );

			return $this->return_handler_error( $error_msg );
		}

		// Ensure path is a string
		if ( ! is_string( $leopard_item->source_path() ) ) {
			$error_msg = sprintf( __( '%s with id %d. Provided path is not a string', 'leopard-wordpress-offload-media' ), $source_type_name, $leopard_item->source_id() );

			return $this->return_handler_error( $error_msg );
		}

		// Ensure primary source file exists for new offload.
		if ( empty( $leopard_item->id() ) && ! file_exists( $leopard_item->full_source_path( $primary_key ) ) ) {
			$error_msg = sprintf( __( 'Primary file %s does not exist', 'leopard-wordpress-offload-media' ), $leopard_item->full_source_path( $primary_key ) );

			return $this->return_handler_error( $error_msg );
		}

		// Get primary file's stats.
		$file_name     = wp_basename( $leopard_item->source_path() );
		$file_type     = wp_check_filetype_and_ext( $leopard_item->source_path(), $file_name );
		$allowed_types = get_allowed_mime_types();

		// check mime type of file is in allowed provider mime types
		if ( ! in_array( $file_type['type'], $allowed_types, true ) ) {
			$error_msg = sprintf( __( 'Mime type %s is not allowed', 'leopard-wordpress-offload-media' ), $file_type['type'] );

			return $this->return_handler_error( $error_msg );
		}

		$default_acl = get_option('nou_leopard_offload_media_private_public_radio_button', 'public');
		$private_acl = 'private';

		list( $aws_s3_client, $Bucket, $Region, $basedir_absolute ) = leopard_offload_media_provider_info();

		$cacheControl = get_option('nou_leopard_offload_media_cache_control', 'public, max-age=31536000');

		foreach ( $leopard_item->objects() as $object_key => $object ) {
			// Avoid attempting uploading to an item that doesn't have the primary file in place.

			if ( $primary_key !== $object_key && empty( $leopard_item->id() ) && ! isset( $manifest->objects[ $primary_key ] ) ) {
				continue;
			}

			$source_path = $leopard_item->full_source_path( $object_key );
			$source_key = $leopard_item->source_key( $object_key );

			// If the file has already been offloaded,
			// don't try and (fail to) re-offload if the file isn't available.
			if ( $this->in_offloaded_files( $object['source_file'], $options ) && ! file_exists( $source_path ) ) {
				continue;
			}

			/**
			 * This filter allows you to change the public/private status of an individual file associated
			 * with an uploaded item before it's uploaded to the provider.
			 *
			 * @param bool   $is_private Should the object be private?
			 * @param string $object_key A unique file identifier for a composite item, e.g. image's "size" such as full, small, medium, large.
			 * @param Item   $leopard_item The item being uploaded.
			 *
			 * @return bool
			 */
			$is_private = apply_filters( 'leopard_upload_object_key_as_private', $leopard_item->is_private( $object_key ), $object_key, $leopard_item );
			$leopard_item->set_is_private( $is_private, $object_key );

			$object_acl = $leopard_item->is_private( $object_key ) ? $private_acl : $default_acl;

			$args = array(
				'Bucket'       => $leopard_item->bucket(),
				'Key'          => $source_key,
				'SourceFile'   => $source_path,
				'ContentType'  => Leopard_Wordpress_Offload_Media_Utils::get_mime_type( $object['source_file'] ),
				'CacheControl' => $cacheControl,
			);

			// TODO: Remove GZIP functionality.
			// Handle gzip on supported items
			if (
				$this->should_gzip_file( $source_path, $leopard_item->source_type() ) &&
				false !== ( $gzip_body = gzencode( file_get_contents( $source_path ) ) )
			) {
				unset( $args['SourceFile'] );

				$args['Body']            = $gzip_body;
				$args['ContentEncoding'] = 'gzip';
			}

			/**
			 * This filter allows you to change the arguments passed to the cloud storage SDK client when
			 * offloading a file to the bucket.
			 *
			 * Note: It is possible to change the destination 'Bucket' only while processing the primary object_key.
			 *       All other object_keys will use the same bucket as the item's primary object.
			 *       The 'Key' should be the "public" Key path. If a private prefix is configured
			 *       for use with signed CloudFront URLs or similar, that prefix will be added later.
			 *       A change to the 'Key' will only be handled when processing the primary object key.
			 *
			 * @param array  $args        Information to be sent to storage provider during offload (e.g. PutObject)
			 * @param int    $source_id   Original file's unique ID for its source type
			 * @param string $object_key  A unique file identifier for a composite item, e.g. image's "size" such as full, small, medium, large
			 * @param bool   $copy        True if the object is being copied between buckets
			 * @param array  $item_source Item source array containing source type and id
			 *
			 * @return array
			 */
			$args = apply_filters( 'leopard_object_meta', $args, $leopard_item->source_id(), $object_key, false, $leopard_item->get_item_source_array() );

			// If the bucket is changed by the filter while processing the primary object,
			// we should try and use that bucket for the item.
			// If the bucket name is invalid, revert to configured bucket but log it.
			// We don't abort here as ephemeral filesystems need to be accounted for,
			// and the configured bucket is at least known to work.
			if ( $primary_key === $object_key && $leopard_item->bucket() !== $args['Bucket'] && empty( $leopard_item->id() ) ) {

				$bucket = $Bucket;

				if ( $bucket ) {
					$region = $Region;

					if ( is_wp_error( $region ) ) {
						unset( $region );
					}
				}

				if ( empty( $bucket ) || empty( $region ) ) {
					$mesg = sprintf(
						__( 'Bucket name "%1$s" is invalid, using "%2$s" instead.', 'leopard-wordpress-offload-media' ),
						$args['Bucket'],
						$leopard_item->bucket()
					);
					error_log( $mesg );
					$args['Bucket'] = $leopard_item->bucket();
				} else {
					$args['Bucket'] = $bucket;
					$leopard_item->set_bucket( $bucket );
					$leopard_item->set_region( $region );
				}

				unset( $bucket, $region );
			} elseif ( $primary_key === $object_key && $leopard_item->bucket() !== $args['Bucket'] && ! empty( $leopard_item->id() ) ) {
				$args['Bucket'] = $leopard_item->bucket();
				error_log( __( 'The bucket may not be changed via filters for a previously offloaded item.', 'leopard-wordpress-offload-media' ) );
			} elseif ( $primary_key !== $object_key && $leopard_item->bucket() !== $args['Bucket'] ) {
				$args['Bucket'] = $leopard_item->bucket();
			}

			// If the Key has been changed for the primary object key, then that should be reflected in the item.
			if ( $primary_key === $object_key && $source_key !== $args['Key'] && empty( $leopard_item->id() ) ) {
				$prefix = Leopard_Wordpress_Offload_Media_Utils::trailingslash_prefix( dirname( $args['Key'] ) );

				if ( $prefix === '.' ) {
					$prefix = '';
				}

				$leopard_item->update_path_prefix( $prefix );

				// If the filter tried to use a different filename too, log it.
				if ( wp_basename( $args['Key'] ) !== wp_basename( $source_key ) ) {
					$mesg = sprintf(
						__( 'The offloaded filename must not be changed, "%1$s" has been used instead of "%2$s".', 'leopard-wordpress-offload-media' ),
						wp_basename( $source_key ),
						wp_basename( $args['Key'] )
					);
					error_log( $mesg );
				}
			} elseif ( $primary_key === $object_key && $source_key !== $args['Key'] && ! empty( $leopard_item->id() ) ) {
				$args['Key'] = $source_key;
				error_log( __( 'The key may not be changed via filters for a previously offloaded item.', 'leopard-wordpress-offload-media' ) );
			} elseif ( $primary_key !== $object_key && $source_key !== $args['Key'] ) {
				$args['Key'] = $source_key;
			}

			// If ACL has been set, does the object's is_private need updating?
			$is_private = ! empty( $args['ACL'] ) && $private_acl === $args['ACL'] || $leopard_item->is_private( $object_key );
			$leopard_item->set_is_private( $is_private, $object_key );

			// Adjust the actual Key to add the private prefix before uploading.
			if ( $leopard_item->is_private( $object_key ) ) {
				$args['Key'] = $leopard_item->provider_key( $object_key );
			}

			// If we've already attempted to offload this source file, leave it out of the manifest.
			if ( in_array( md5( serialize( $args ) ), $this->attempted_upload ) ) {
				continue;
			}

			if ( $primary_key === $object_key ) {
				/**
				 * Actions fires when an Item's primary file might be offloaded.
				 *
				 * This action gives notice that an Item is being processed for upload to a bucket,
				 * and the given arguments represent the primary file's potential offload location.
				 * However, if the current process is for picking up extra files associated with the item,
				 * the indicated primary file may not actually be offloaded if it does not exist
				 * on the server but has already been offloaded.
				 *
				 * @param Item  $leopard_item The Item whose files are being offloaded.
				 * @param array $args       The arguments that could be used to offload the primary file.
				 */
				do_action( 'leopard_pre_upload_object', $leopard_item, $args );
			}

			if(empty($args['Bucket'])){
				$args['Bucket'] = $Bucket;
			}

			if(empty($args['Region'])){
				$args['Region'] = $Region;
			}

			if(!empty($args['Key'])){
				$args['Key'] = $this->rebuild_key($source_key);
			}

			$manifest->objects[ $object_key ]['args'] = $args;
		}
		
		return $manifest;
	}

	/**
	 * Upload item files to remote storage provider
	 *
	 * @param Item     $leopard_item
	 * @param Manifest $manifest
	 * @param array    $options
	 *
	 * @return bool|WP_Error
	 */
	protected function handle_item( Leopard_Wordpress_Offload_Media_Item $leopard_item, Leopard_Wordpress_Offload_Media_Manifest $manifest, array $options ) {
		
		list( $aws_s3_client, $Bucket, $Region, $basedir_absolute ) = leopard_offload_media_provider_info();

		try {
			$provider_client = $aws_s3_client;
		} catch ( Exception $e ) {
			return false;
		}

		try {
			$bucketFolder = $provider_client->getBucketMainFolder();
			if($bucketFolder){
				$leopard_item->set_path($provider_client->rebuild_key($leopard_item->path()));
			}
		} catch (\Throwable $th) {}

		$urlOffloaded = '';

		$allowed_types = [];
		$accepted_filetypes = get_option('nou_leopard_offload_media_accepted_filetypes', '');
		if(!empty($accepted_filetypes)){
			$allowed_types = explode( ',', $accepted_filetypes );
		}

		try {
			$argsArr = [];
			foreach ( $manifest->objects as $object_key => &$object ) {
				$args = $object['args'];
	
				$args['Bucket'] = $Bucket;
				$args['Region'] = $Region;
	
				$object['upload_result'] = array(
					'status'  => null,
					'message' => null,
				);

				if(!empty($allowed_types) && is_array($allowed_types)){
					$path_parts = pathinfo($args['SourceFile']);
					if(isset($path_parts['extension'])){
						if(!in_array($path_parts['extension'], $allowed_types)){
							continue;
						}
					}
				}
				
				if ( ! file_exists( $args['SourceFile'] ) ) {
					$error_msg = sprintf( __( 'File %s does not exist', 'leopard-wordpress-offload-media' ), $args['SourceFile'] );
	
					$object['upload_result']['status']  = self::STATUS_FAILED;
					$object['upload_result']['message'] = $error_msg;
	
					// If the missing source file is the primary file, abort the whole process.
					if ( Leopard_Wordpress_Offload_Media_Item::primary_object_key() === $object_key ) {
						return false;
					}
	
					continue;
				}

				$argsArr[] = $args;

				$this->attempted_upload[] = md5( serialize( $args ) );
			}
			
			if(count($argsArr) > 0){
				$results = $provider_client->upload_objects_async( $argsArr );
				if(count($results['successful_uploads']) > 0){
					try {
						$settings = leopard_offload_media_provider_settings();
						$provider = isset($settings['provider']) ? $settings['provider'] : 'aws';
						$data = wp_get_attachment_metadata( $leopard_item->source_id() );
			
						$provider_object = array(
							'provider' => $provider,
							'region'   => $Region,
							'bucket'   => $Bucket,
							'key' 	   => $leopard_item->path(),
							'url' 	   => $results['successful_uploads'][0]['url'],
							'data'     => $data
						);
						$cacheKey = "_leopard_wom_attachment_{$leopard_item->source_id()}_provider_info";
						nou_leopard_offload_media_set_cache_item($cacheKey, $provider_object);
						update_post_meta( $leopard_item->source_id(), '_nou_leopard_wom_amazonS3_info', $provider_object );
						update_post_meta( $leopard_item->source_id(), '_wp_nou_leopard_wom_s3_wordpress_path', '1' );
						update_post_meta( $leopard_item->source_id(), '_wp_nou_leopard_wom_s3_path', $results['successful_uploads'][0]['url'] );
						
						$leopard_item->save();
					} catch (\Throwable $th) {}

					try {
						leopard_offload_media_build_webp_function($leopard_item->source_id());
					} catch (Exception $e) {
						error_log("Error convert webp: ". $e->getMessage());
					}

					do_action('leopard_offload_media_copy_file_to_provider', $leopard_item->source_id());

					return true;
				}
			}
		} catch ( Exception $e ) {
			return false;
		}

		return true;
	}

	/**
	 * Handle local housekeeping after uploads.
	 *
	 * @param Item     $leopard_item
	 * @param Manifest $manifest
	 * @param array    $options
	 *
	 * @return bool|WP_Error
	 */
	protected function post_handle( Leopard_Wordpress_Offload_Media_Item $leopard_item, Leopard_Wordpress_Offload_Media_Manifest $manifest, array $options ) {
		$item_objects = $leopard_item->objects();
		$errors       = new WP_Error;
		$i            = 1;

		// Reconcile the Item's objects with their manifest status.
		foreach ( $item_objects as $object_key => $object ) {
			// If there was no attempt made to offload the file,
			// then remove it from list of offloaded objects.
			// However, if the source file has previously been offloaded,
			// we should just skip any further processing of it
			// as the associated objects are still offloaded.
			if ( ! isset( $manifest->objects[ $object_key ]['upload_result']['status'] ) ) {
				if ( empty( $options['offloaded_files'][ $object['source_file'] ] ) ) {
					unset( $item_objects[ $object_key ] );
				}
				continue;
			}

			// If the upload didn't succeed, we need to remove the object/size from the item.
			// However, if the source file has previously been offloaded, we should just log the error.
			if ( $manifest->objects[ $object_key ]['upload_result']['status'] !== self::STATUS_OK ) {
				if ( empty( $options['offloaded_files'][ $object['source_file'] ] ) ) {
					unset( $item_objects[ $object_key ] );
				}
				$errors->add( 'upload-object-' . $i++, $manifest->objects[ $object_key ]['upload_result']['message'] );
			}
		}

		// Set the potentially changed list of offloaded objects.
		$leopard_item->set_objects( $item_objects );

		// Only save if we have the primary file uploaded.
		
		if ( isset( $item_objects[ Leopard_Wordpress_Offload_Media_Item::primary_object_key() ] ) ) {
			$leopard_item->save();
		}

		/**
		 * Fires action after uploading finishes
		 *
		 * @param Item $leopard_item The item that was just uploaded
		 */
		do_action( 'leopard_post_upload_item', $leopard_item );

		if ( count( $errors->get_error_codes() ) ) {
			return $errors;
		}

		return true;
	}

	/**
	 * Should gzip file
	 *
	 * @param string $file_path
	 * @param string $source_type
	 *
	 * @return bool
	 */
	protected function should_gzip_file( $file_path, $source_type ) {

		$gzip = get_option('nou_leopard_offload_media_gzip', '');
	
		if(empty($gzip)){
			return false;
		}

		$file_type = wp_check_filetype_and_ext( $file_path, $file_path );
		$mimes     = $this->get_mime_types_to_gzip( $source_type );

		if ( in_array( $file_type, $mimes ) && is_readable( $file_path ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Get mime types to gzip
	 *
	 * @param string $source_type
	 *
	 * @return array
	 */
	protected function get_mime_types_to_gzip( $source_type ) {
		/**
		 * Return array of mime types that needs to be gzipped before upload
		 *
		 * @param array  $mime_types    The array of mime types
		 * @param bool   $media_library If the uploaded file is part of the media library
		 * @param string $source_type   The source type of the uploaded item
		 */
		return leopard_offload_media_get_mime_types_to_gzip('media_library' === $source_type ? true : false);
	}

	/**
	 * Has the given file name already been offloaded?
	 *
	 * @param string $filename
	 * @param array  $options
	 *
	 * @return bool
	 */
	private function in_offloaded_files( $filename, $options ) {
		if ( empty( $options['offloaded_files'] ) ) {
			return false;
		}

		return array_key_exists( $filename, $options['offloaded_files'] );
	}
}