<?php
/**
 * Admin Ajax
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}
if ( ! class_exists( 'FP_Donation_Admin_Ajax' ) ) {

	/**
	 * FP_Donation_Admin_Ajax Class
	 */
	class FP_Donation_Admin_Ajax {

		/**
		 * FP_Donation_Admin_Ajax Class initialization
		 */
		public static function init() {

			$actions = array(
				'simple_product_search' => false,
				'create_new_product'    => false,
				'add_reward_rule'       => false,
				'reset_settings_data'   => false,
				'reward_product'        => true,
				'handle_anonymous'      => true,
					) ;

			foreach ( $actions as $action => $nopriv ) {
				add_action( 'wp_ajax_fp_donation_' . $action , array( __CLASS__, $action ) ) ;

				if ( $nopriv ) {
					add_action( 'wp_ajax_nopriv_fp_donation_' . $action , array( __CLASS__, $action ) ) ;
				}
			}
		}

		/**
		 * Simple product search functionality
		 */
		public static function simple_product_search() {
			check_ajax_referer( 'donation-search-nonce' , 'donation_security' ) ;

			try {
				if ( ! isset( $_REQUEST ) || ! isset( $_REQUEST[ 'term' ] ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				$listofproducts = array() ;
				$term           = wc_clean( wp_unslash( $_REQUEST[ 'term' ] ) ) ;

				if ( empty( $term ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				$args = array(
					'post_type'         => 'product',
					's'                 => $term,
					'posts_per_page'    => -1,
					'search_prod_title' => $term,
					'post_status'       => 'publish',
						) ;

				$simple_products = get_posts( $args ) ;

				if ( fp_donation_check_is_array( $simple_products ) ) {
					foreach ( $simple_products as $simple_product ) {
						if ( is_object( $simple_product ) ) {
							$listofproducts[ $simple_product->ID ] = esc_html( esc_html( $simple_product->post_title ) . '(#' . absint( $simple_product->ID ) . ')' ) ;
						}
					}
				}

				wp_send_json( $listofproducts ) ;
			} catch ( Exception $ex ) {
				wp_send_json_error( $ex->getMessage() ) ;
			}
		}

		/**
		 * Create a donation product.
		 * 
		 */
		public static function create_new_product() {
			check_ajax_referer( 'product-creation-nonce' , 'fp_donation_security' ) ;

			try {
				if ( ! isset( $_POST ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				$product_name = isset( $_POST[ 'product_name' ] ) ? wc_clean( wp_unslash( $_POST[ 'product_name' ] ) ) : '' ;
				$type         = isset( $_POST[ 'type' ] ) ? wc_clean( wp_unslash( $_POST[ 'type' ] ) ) : '' ;
				// Return if the product name is empty.
				if ( ! $product_name ) {
					throw new exception( esc_html__( 'Please enter the product name' , 'donationsystem' ) ) ;
				}

				// Return if the current user does not have permission.
				if ( ! current_user_can( 'edit_posts' ) ) {
					throw new exception( esc_html__( "You don't have permission to do this action" , 'donationsystem' ) ) ;
				}

				$product_id = FP_Donation_Common_Function::create_new_product( $product_name ) ;
				$product    = sumo_donation_get_product( $product_id ) ;

				if ( 'automatic' == $type ) {
					update_option( '_fp_donation_product_selection_automatic' , '1' ) ;
					update_option( 'ds_select_particular_products_automatic' , $product_id ) ;
				} else {
					update_option( '_fp_donation_product_selection' , '1' ) ;
					update_option( 'ds_select_particular_products' , $product_id ) ;
				}

				wp_send_json_success( array( 'product_id' => absint( $product_id ), 'product_name' => $product->get_formatted_name() ) ) ;
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Add a reward rule.
		 */
		public static function add_reward_rule() {
			check_ajax_referer( 'rule-nonce' , 'fp_donation_security' ) ;

			try {
				if ( ! isset( $_POST ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				$iteration = isset( $_POST[ 'uniq_id' ] ) ? wc_clean( wp_unslash( $_POST[ 'uniq_id' ] ) ) : '' ;
				// Return if the unique id is empty.
				if ( ! $iteration ) {
					throw new exception( esc_html__( 'Process cannot complete.' , 'donationsystem' ) ) ;
				}

				// Return if the current user does not have permission.
				if ( ! current_user_can( 'edit_posts' ) ) {
					throw new exception( esc_html__( "You don't have permission to do this action" , 'donationsystem' ) ) ;
				}

				ob_start() ;
				?>
				<tr>
					<td>
						<p class="form-fields"><input type="number" step="any" required="required" name="fp_donation_rewards_rule[<?php echo esc_attr( $iteration ) ; ?>][min]" min='0' value='' /></p>
					</td>
					<td>
						<p class="form-fields"><input type="number" step='any' required="required" name="fp_donation_rewards_rule[<?php echo esc_attr( $iteration ) ; ?>][max]" min='0' value='' /></p>
					</td>
					<td>
						<p class='form-fields'>
							<select id='fp_donation_rewards_rule<?php echo esc_attr( $iteration ) ; ?>'  data-key ='<?php echo esc_attr( $iteration ) ; ?>' name='fp_donation_rewards_rule[<?php echo esc_attr( $iteration ) ; ?>][type]'>
								<option value='1'><?php esc_html_e( 'Free Products' , 'donationsystem' ) ; ?></option>
							</select>
						</p>
					</td>
					<td class='fp_donation_type_selection<?php echo esc_attr( $iteration ) ; ?>'>
						<?php
						$name     = 'fp_donation_rewards_rule[' . $iteration . '][product]' ;
						$class    = '_fp_donation_existing_id' ;
						$multiple = true ;
						$selected = array() ;
						$id       = '' ;
						sumo_donation_product_select2( $name , $id , $class , $selected , $multiple , '2' , "required = 'required'" ) ;
						?>
					</td>
					<td class="column-columnname num">
						<span class="fp-donation-remove-reward-rule button-secondary"><?php esc_html_e( 'Delete Rule' , 'donationsystem' ) ; ?></span>
					</td>
				</tr>
				<?php
				$contents = ob_get_contents() ;
				ob_end_clean() ;

				wp_send_json_success( array( 'html' => $contents ) ) ;
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Reset the donation settings data.
		 */
		public static function reset_settings_data() {
			check_ajax_referer( 'reset-nonce' , 'fp_donation_security' ) ;

			try {
				if ( ! isset( $_POST ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				$master_log = isset( $_POST[ 'master_log' ] ) ? wc_clean( wp_unslash( $_POST[ 'master_log' ] ) ) : false ;
				$settings   = isset( $_POST[ 'settings' ] ) ? wc_clean( wp_unslash( $_POST[ 'settings' ] ) ) : false ;
				if ( ! $master_log && ! $settings ) {
					throw new exception( esc_html__( 'Please select any one settings' , 'donationsystem' ) ) ;
				}

				// Return if the current user does not have permission.
				if ( ! current_user_can( 'edit_posts' ) ) {
					throw new exception( esc_html__( "You don't have permission to do this action" , 'donationsystem' ) ) ;
				}

				if ( $master_log ) {
					update_option( '_fp_donated_order_ids' , array() ) ;
				}

				if ( $settings ) {
					FP_Donation_General_Tab::reset_option_to_donationsystem() ;
					FP_Donation_Labels_Tab::reset_option_to_donationsystem() ;
					FP_Donation_Messages_Tab::reset_option_to_donationsystem() ;
					FP_Donation_Rewards_Tab::reset_option_to_donationsystem() ;
					FP_Donation_Shortcode_Tab::reset_option_to_donationsystem() ;
					FP_Donation_Form_Tab::reset_option_to_donationsystem() ;
					FP_Donation_FlyBox_Tab::reset_option_to_donationsystem() ;
				}

				wp_send_json_success( array( 'msg' => esc_html__( 'Resetted Successfully' , 'donationsystem' ) ) ) ;
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Add donation reward product to cart functionality
		 */
		public static function reward_product() {
			check_ajax_referer( 'donation-reward-nonce' , 'donation_security' ) ;

			try {
				if ( ! isset( $_REQUEST ) || ! isset( $_REQUEST[ 'product_id' ] ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}
				$cart_item_data                    = array() ;
				$cart_item_data[ 'sumo_donation' ] = array( 'type' => 'free_product' ) ;

				$product_id = isset( $_REQUEST[ 'type' ] ) ? absint( $_REQUEST[ 'product_id' ] ) : '' ;
				FP_Donation_Rewards_Cart::add_product_automatically_to_cart( $product_id ) ;

				wp_send_json_success( true ) ;
			} catch ( Exception $ex ) {
				wp_send_json_error( $ex->getMessage() ) ;
			}
		}

		/**
		 * Handle donation anonymous.
		 */
		public static function handle_anonymous() {
			check_ajax_referer( 'donation-anonymous-nonce' , 'fp_donation_security' ) ;

			try {
				if ( ! isset( $_POST ) ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				$value = isset( $_POST[ 'anonymous' ] ) ? wc_clean( wp_unslash( $_POST[ 'anonymous' ] ) ) : '' ;
				if ( ! $value ) {
					throw new exception( esc_html__( 'Invalid Request' , 'donationsystem' ) ) ;
				}

				WC()->session->set( 'fp_donation_anonymous' , $value ) ;

				wp_send_json_success() ;
			} catch ( Exception $ex ) {
				wp_send_json_error( $ex->getMessage() ) ;
			}
		}
	}

	FP_Donation_Admin_Ajax::init() ;
}
