<?php
/**
 * Widget Name: Accordion/FAQ
 * Description: Toggle of faq/accordion.
 * Author: Theplus
 * Author URI: https://posimyth.com
 *
 * @package ThePlus
 */

namespace TheplusAddons\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use TheplusAddons\Theplus_Element_Load;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class ThePlus_Accordion.
 */
class ThePlus_Accordion extends Widget_Base {

	/**
	 * Document Link For Need help.
	 *
	 * @var tp_doc of the class.
	 */
	public $tp_doc = THEPLUS_TPDOC;

	/**
	 * Helpdesk Link For Need help.
	 *
	 * @var tp_help of the class.
	 */
	public $tp_help = THEPLUS_HELP;	

	/**
	 * Get Widget Name.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	public function get_name() {
		return 'tp-accordion';
	}

	/**
	 * Get Widget Title.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	public function get_title() {
		return esc_html__( 'Accordion', 'theplus' );
	}

	/**
	 * Get Widget Icon.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	public function get_icon() {
		return 'theplus-i-accordion tpae-editor-logo';
	}

	/**
	 * Get Custom URL.
	 *
	 * @since 1.0.0
	 * @version 5.6.5
	 */
	public function get_custom_help_url() {
		$help_url = $this->tp_help;

		return esc_url( $help_url );
	}

	/**
	 * Get Widget Category.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	public function get_categories() {
		return array( 'plus-essential' );
	}

	/**
	 * Get Widget Keywords.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	public function get_keywords() {
		return array( 'Accordion', 'Toggle', 'Collapsible', 'Expandable', 'Content Panel', 'FAQ', 'Q&A', 'Show/Hide', 'Dropdown', 'Tabbed Content', 'Vertical Tabs', 'Accordion Menu', 'Expand/Collapse', 'Content Slider' );
	}

	/**
	 * It is use for widget add in catch or not.
	 *
	 * @since 6.1.2
	 */
	// public function is_dynamic_content(): bool {
	// 	return false;
	// }

	/**
	 * Disable Elementor's default inner wrapper for custom HTML control.
	 *
	 * @since 6.3.2
	 */
	public function has_widget_inner_wrapper(): bool {
		return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}
	
	/**
	 * Accordion Render.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Content', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		
		$this->add_control(
            'tpae_preset_controller',
            array(
                'type'        => 'tpae_preset_button',
                'temp_id'     => 17409,
                'label_block' => true,
            )
        );
		$this->add_control(
			'repeater_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'You can add repeaters here and include the content inside each of them.', 'theplus' ),
						esc_url( $this->tp_doc . 'elementor-accordion-widget-settings-overview/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'tab_title',
			array(
				'label'       => esc_html__( 'Title', 'theplus' ),
				'ai' => false,
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Accordion Title', 'theplus' ),
				'dynamic'     => array(
					'active' => true,
				),
				'show_label' => true,
			)
		);
		$repeater->add_control(
			'content_source',
			array(
				'label'   => esc_html__( 'Type', 'theplus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'content',
				'options' => array(
					'content'       => esc_html__( 'Content', 'theplus' ),
					'page_template' => esc_html__( 'Page Template', 'theplus' ),
				),
			)
		);
		$repeater->add_control(
			'content_source_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'If you want to write text directly inside the tab, keep the type as Content. To display other widgets or designs, create an Elementor template, design it as you like, then select Page Template and choose that template here.', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);
		$repeater->add_control(
			'tab_content',
			array(
				'label'      => esc_html__( 'Content', 'theplus' ),
				'type'       => Controls_Manager::WYSIWYG,
				'default'    => esc_html__( 'Accordion Content', 'theplus' ),
				'show_label' => false,
				'ai' => false,
				'dynamic'    => array(
					'active' => true,
				),
				'condition'  => array(
					'content_source' => array( 'content' ),
				),
				'separator'   => 'before',
			)
		);
		$repeater->add_control(
			'content_template',
			array(
				'label'       => esc_html__( 'Templates', 'theplus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '0',
				'options'     => theplus_get_templates(),
				'classes'     => 'tp-template-create-btn',
				'show_label' => true,
				'condition'   => array( 'content_source' => 'page_template' ),
			)
		);
		$repeater->add_control(
			'liveeditor',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => '<a class="tp-live-editor" id="tp-live-editor-button">Edit Template</a>',
				'content_classes' => 'tp-live-editor-btn',
				'label_block'     => true,
				'condition'       => array(
					'content_source'    => 'page_template',
					'content_template!' => '0',
				),
			)
		);
		$repeater->add_control(
			'create',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => '<a class="tp-live-create" id="tp-live-create-button">Create Template</a>',
				'content_classes' => 'tp-live-create-btn',
				'label_block'     => true,
				'condition'       => array(
					'content_source'   => 'page_template',
					'content_template' => '0',
				),
			)
		);
		$repeater->add_control(
			'backend_preview_template',
			array(
				'label'       => esc_html__( 'Backend Visibility', 'theplus' ),
				'type'        => Controls_Manager::SWITCHER,
				'default'     => 'no',
				'label_on'    => esc_html__( 'Show', 'theplus' ),
				'label_off'   => esc_html__( 'Hide', 'theplus' ),
				'condition'   => array( 
					'content_source' => 'page_template' 
				),
			)
		);
		$repeater->add_control(
			'backend_Note',
			[
				'type' => \Elementor\Controls_Manager::RAW_HTML,
				'raw'  => wp_kses_post(
					sprintf(
						'<i><b>%s</b> %s</i>',
						esc_html__( 'Note:', 'theplus' ),
						esc_html__( 'If disabled, Template will not visible/load in the backend for better page loading performance.', 'theplus' ),
					)
				),
				'content_classes' => 'tp-controller-label-text',
				'condition'   => array( 
					'content_source' => 'page_template' 
				),
			]
		);
		$repeater->add_control(
			'display_icon',
			array(
				'label'     => esc_html__( 'Show Icon', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
			)
		);
		$repeater->add_control(
			'icon_style',
			array(
				'label'     => esc_html__( 'Icon Font', 'theplus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'font_awesome',
				'options'   => array(
					'font_awesome'   => esc_html__( 'Font Awesome', 'theplus' ),
					'font_awesome_5' => esc_html__( 'Font Awesome 5', 'theplus' ),
					'icon_mind'      => esc_html__( 'Icons Mind', 'theplus' ),
				),
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);
		$repeater->add_control(
			'icon_fs_popover_toggle',
			[
				'label' => esc_html__( 'Font Awesome', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'font_awesome',
				),
			]
		);
		$repeater->start_popover();
		$repeater->add_control(
			'icon_fs_options_rep',
			array(
				'label'     => esc_html__( 'Font Awesome', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'after',
			)
		);
		$repeater->add_control(
			'icon_fontawesome',
			array(
				'label'     => esc_html__( 'Icon Library', 'theplus' ),
				'type'      => Controls_Manager::ICON,
				'default'   => 'fa fa-download',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'font_awesome',
				),
			)
		);
		$repeater->end_popover();
		$repeater->add_control(
			'icon_f5_popover_toggle',
			[
				'label' => esc_html__( 'Font Awesome 5', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'font_awesome_5',
				),
			]
		);
		$repeater->start_popover();
		$repeater->add_control(
			'icon_f5_options',
			array(
				'label'     => esc_html__( 'Font Awesome 5', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'after',
			)
		);
		$repeater->add_control(
			'icon_fontawesome_5',
			array(
				'label'     => esc_html__( 'Icon Library', 'theplus' ),
				'type'      => Controls_Manager::ICONS,
				'default'   => array(
					'value'   => 'fas fa-plus',
					'library' => 'solid',
				),
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'font_awesome_5',
				),
			)
		);
		$repeater->end_popover();
		$repeater->add_control(
			'icon_mind_popover_toggle',
			[
				'label' => esc_html__( 'Icons Mind', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'icon_mind',
				),
			]
		);
		$repeater->start_popover();
		$repeater->add_control(
				'icon_mind_options',
				array(
					'label'     => esc_html__( 'Icons Mind', 'theplus' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'after',
				)
			);
		$repeater->add_control(
			'icons_mind',
			array(
				'label'       => esc_html__( 'Icon Library', 'theplus' ),
				'type'        => Controls_Manager::SELECT2,
				'default'     => 'iconsmind-Download-2',
				'label_block' => true,
				'options'     => theplus_icons_mind(),
				'condition'   => array(
					'display_icon' => 'yes',
					'icon_style'   => 'icon_mind',
				),
			)
		);
		$repeater->end_popover();
		$repeater->add_control(
			'tab_hashid',
			array(
				'label'       => esc_html__( 'Unique ID', 'theplus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'ai' => false,
				'dynamic'     => array(
					'active' => true,
				),
				'label_block' => false,
			)
		);
		$repeater->add_control(
			'tab_hashid_Note',
			[
				'type' => \Elementor\Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i><b>%s</b>%s<a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Note:', 'theplus' ),
						esc_html__( 'Use this option to give anchor id to individual accordion.', 'theplus' ),
						esc_url( $this->tp_doc . 'anchor-link-specific-accordion-item/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'content_classes' => 'tp-controller-label-text',
			]
		);

		$this->add_control(
			'tabs',
			array(
				'label'       => 'Accordions',
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(
						'tab_title'   => esc_html__( 'Accordion #1', 'theplus' ),
						'tab_content' => esc_html__( 'I am item content. Click edit button to change this text. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'theplus' ),
					),
					array(
						'tab_title'   => esc_html__( 'Accordion #2', 'theplus' ),
						'tab_content' => esc_html__( 'I am item content. Click edit button to change this text. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'theplus' ),
					),
				),
				'title_field' => '{{{ tab_title }}}',
			)
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'icon_content_section',
			array(
				'label' => esc_html__( 'Toggle Icon', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'display_icon',
			array(
				'label'     => esc_html__( 'Show Icon', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
			)
		);
		$this->add_control(
			'icon_align',
			array(
				'label'       => esc_html__( 'Alignment', 'theplus' ),
				'type'        => Controls_Manager::CHOOSE,
				'options'     => array(
					'left'  => array(
						'title' => esc_html__( 'Start', 'theplus' ),
						'icon'  => 'eicon-h-align-left',
					),
					'right' => array(
						'title' => esc_html__( 'End', 'theplus' ),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'default'     => is_rtl() ? 'right' : 'left',
				'toggle'      => false,
				'label_block' => false,
				'condition'   => array(
					'display_icon' => 'yes',
				),
			)
		);	
		$this->add_control(
			'icon_style',
			array(
				'label'     => esc_html__( 'Icon Font', 'theplus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'font_awesome',
				'options'   => array(
					'font_awesome'   => esc_html__( 'Font Awesome', 'theplus' ),
					'font_awesome_5' => esc_html__( 'Font Awesome 5', 'theplus' ),
					'icon_mind'      => esc_html__( 'Icons Mind', 'theplus' ),
				),
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);

		$this->add_control(
			'icon_fs_popover_toggle',
			[
				'label' => esc_html__( 'Font Awesome', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'font_awesome',
				),
			]
		);
		$this->start_popover();
			$this->add_control(
				'icon_fs_options',
				array(
					'label'     => esc_html__( 'Font Awesome', 'theplus' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'after',
				)
			);
			$this->add_control(
				'icon_fontawesome',
				array(
					'label'     => esc_html__( 'Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICON,
					'default'   => 'fa fa-plus',
					'condition' => array(
						'display_icon' => 'yes',
						'icon_style'   => 'font_awesome',
					),
				)
			);
			$this->add_control(
				'icon_fontawesome_active',
				array(
					'label'     => esc_html__( 'Active Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICON,
					'default'   => 'fa fa-minus',
					'condition' => array(
						'display_icon' => 'yes',
						'icon_style'   => 'font_awesome',
					),
				)
			);
		$this->end_popover();

		$this->add_control(
			'icon_f5_popover_toggle',
			[
				'label' => esc_html__( 'Font Awesome 5', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'font_awesome_5',
				),
			]
		);
		$this->start_popover();
			$this->add_control(
				'icon_f5_options',
				array(
					'label'     => esc_html__( 'Font Awesome 5', 'theplus' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'after',
				)
			);
			$this->add_control(
				'icon_fontawesome_5',
				array(
					'label'     => esc_html__( 'Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-plus',
						'library' => 'solid',
					),
					'condition' => array(
						'display_icon' => 'yes',
						'icon_style'   => 'font_awesome_5',
					),
				)
			);
			$this->add_control(
				'icon_fontawesome_5_active',
				array(
					'label'     => esc_html__( 'Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-minus',
						'library' => 'solid',
					),
					'condition' => array(
						'display_icon' => 'yes',
						'icon_style'   => 'font_awesome_5',
					),
				)
			);
		$this->end_popover();

		$this->add_control(
			'icon_mind_popover_toggle',
			[
				'label' => esc_html__( 'Icons Mind', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => array(
					'display_icon' => 'yes',
					'icon_style'   => 'icon_mind',
				),
			]
		);
		$this->start_popover();
			$this->add_control(
				'icon_mind_options',
				array(
					'label'     => esc_html__( 'Icons Mind', 'theplus' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'after',
				)
			);
			$this->add_control(
				'icons_mind',
				array(
					'label'       => esc_html__( 'Icon Library', 'theplus' ),
					'type'        => Controls_Manager::SELECT2,
					'default'     => 'iconsmind-Add',
					'show_label' => true,
					'options'     => theplus_icons_mind(),
					'condition'   => array(
						'display_icon' => 'yes',
						'icon_style'   => 'icon_mind',
					),
				)
			);
			$this->add_control(
				'icons_mind_active',
				array(
					'label'       => esc_html__( 'Active Icon Library', 'theplus' ),
					'type'        => Controls_Manager::SELECT2,
					'default'     => 'iconsmind-Add',
					'show_label' => true,
					'options'     => theplus_icons_mind(),
					'condition'   => array(
						'display_icon' => 'yes',
						'icon_style'   => 'icon_mind',
					),
				)
			);
		$this->end_popover();

		$this->add_control(
			'title_html_tag',
			array(
				'label'     => esc_html__( 'Title Tag', 'theplus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					'h1'  => 'H1',
					'h2'  => 'H2',
					'h3'  => 'H3',
					'h4'  => 'H4',
					'h5'  => 'H5',
					'h6'  => 'H6',
					'div' => 'div',
				),
				'default'   => 'div',
				'separator' => 'before',
			)
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'extra_content_section',
			array(
				'label' => esc_html__( 'Special Option', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'on_hover_accordion',
			array(
				'label'     => esc_html__( 'On Hover Accordion', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
			)
		);
		$this->add_control(
			'on_hover_accordion_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Enable this to open accordion items when users hover instead of clicking, perfect for creating smoother, faster interactions.', 'theplus' ),
						esc_url( $this->tp_doc . 'elementor-accordion-on-hover/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);
		$this->add_control(
			'horizontal_popover',
			[
				'label'     => esc_html__( 'Horizontal Accordion', 'theplus' ),
				'type'      => Controls_Manager::POPOVER_TOGGLE,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'condition' => array(
					'on_hover_accordion!' => 'yes',
				),
			]
		);
		$this->start_popover();
			$this->add_control(
				'horizontal_accordion',
				array(
					'label'     => esc_html__( 'Horizontal Accordion', 'theplus' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => 'no',
					'separator' => 'after',
					'label_on'  => esc_html__( 'Show', 'theplus' ),
					'label_off' => esc_html__( 'Hide', 'theplus' ),
					'condition' => array(
						'on_hover_accordion!' => 'yes',
					),
				)
			);
			$this->add_control(
				'horizontal_acc_label',
				array(
					'type'  => Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
							esc_html__( 'Turn this on to display accordion items side by side horizontally, giving your layout a more modern and unique look.', 'theplus' ),
							esc_url( $this->tp_doc . 'elementor-horizontal-accordion/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
							esc_html__( 'Learn More', 'theplus' ),
						)
					),
					'label_block' => true,
					'condition' => array(
						'on_hover_accordion!' => 'yes',
					),
				)
			);
			$this->add_control(
				'horizontal_accordion_layout',
				array(
					'label'     => esc_html__( 'Layout', 'theplus' ),
					'type'      => Controls_Manager::SELECT,
					'default'   => 'tp_hal_1',
					'options'   => array(
						'tp_hal_1' => esc_html__( 'Layout 1', 'theplus' ),
						'tp_hal_2' => esc_html__( 'Layout 2', 'theplus' ),
					),
					'condition' => array(
						'on_hover_accordion!'  => 'yes',
						'horizontal_accordion' => 'yes',
					),
				)
			);
			$this->add_responsive_control(
				'horizontal_accordion_min_height',
				array(
					'type'        => Controls_Manager::SLIDER,
					'label'       => esc_html__( 'Height', 'theplus' ),
					'size_units'  => array( 'px' ),
					'range'       => array(
						'px' => array(
							'min'  => 1,
							'max'  => 700,
							'step' => 1,
						),
					),
					'default'     => array(
						'unit' => 'px',
						'size' => 300,
					),
					'render_type' => 'ui',
					'selectors'   => array(
						'{{WRAPPER}} .theplus-accordion-wrapper.tp-acc-hori' => 'min-height: {{SIZE}}{{UNIT}};max-height: {{SIZE}}{{UNIT}}',
					),
					'condition'   => array(
						'on_hover_accordion!'  => 'yes',
						'horizontal_accordion' => 'yes',
					),
				)
			);
			$this->add_control(
				'horizontal_accordion_title_width',
				array(
					'label'     => esc_html__( 'Title Width', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 1,
					'max'       => 300,
					'step'      => 10,
					'default'   => 60,
					'condition' => array(
						'on_hover_accordion!'  => 'yes',
						'horizontal_accordion' => 'yes',
					),
				)
			);
			$this->add_control(
				'horizontal_accordion_open_speed',
				array(
					'label'     => esc_html__( 'Open Speed', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 1,
					'max'       => 3000,
					'step'      => 10,
					'default'   => 400,
					'condition' => array(
						'on_hover_accordion!'  => 'yes',
						'horizontal_accordion' => 'yes',
					),
				)
			);
			$this->add_control(
				'horizontal_accordion_close_speed',
				array(
					'label'     => esc_html__( 'Close Speed', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 1,
					'max'       => 3000,
					'step'      => 10,
					'default'   => 400,
					'condition' => array(
						'on_hover_accordion!'  => 'yes',
						'horizontal_accordion' => 'yes',
					),
				)
			);
		$this->end_popover();
		/*autoplay*/
		$this->add_control(
			'autoplay_popover',
			[
				'label'     => esc_html__( 'Autoplay', 'theplus' ),
				'type'      => Controls_Manager::POPOVER_TOGGLE,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'separator' => 'before',
			]
		);
		$this->start_popover();
			$this->add_control(
				'tabs_autoplay',
				array(
					'label'     => esc_html__( 'Autoplay', 'theplus' ),
					'type'      => Controls_Manager::SWITCHER,
					'label_on'  => esc_html__( 'Show', 'theplus' ),
					'label_off' => esc_html__( 'Hide', 'theplus' ),
					'default'   => 'no',
					'separator' => 'after',
				)
			);
			$this->add_control(
				'autoplay_label',
				array(
					'type'  => Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
							esc_html__( 'You can enable this to automatically open accordion items one after another. Ideal for showcasing highlights without user interaction. Note that the autoplay runs only once after each page load.', 'theplus' ),
							esc_url( $this->tp_doc . 'elementor-accordion-autoplay/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
							esc_html__( 'Learn More', 'theplus' ),
						)
					),
					'label_block' => true,
				)
			);
			$this->add_control(
				'tabs_autoplaypause',
				array(
					'label'     => esc_html__( 'Play & Pause Button', 'theplus' ),
					'type'      => Controls_Manager::SWITCHER,
					'label_on'  => esc_html__( 'Show', 'theplus' ),
					'label_off' => esc_html__( 'Hide', 'theplus' ),
					'default'   => 'no',
					'condition' => array(
						'tabs_autoplay' => 'yes',
					),
				)
			);
			$this->add_control(
				'tabs_autoplay_border_color',
				array(
					'label'     => esc_html__( 'Color', 'theplus' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '#000',
					'condition' => array(
						'tabs_autoplay' => 'yes',
					),
				)
			);
			$this->add_responsive_control(
				'tabs_autoplay_duration',
				array(
					'label'     => esc_html__( 'Duration', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 1,
					'max'       => 100,
					'step'      => 1,
					'default'   => 5,
					'selectors' => array(
						'{{WRAPPER}} .tp-tab-playloop .plus-tab-header.active:after' => 'transition: transform {{VALUE}}000ms ease-in;',
					),
					'condition' => array(
						'tabs_autoplay' => 'yes',
					),
				)
			);
			$this->add_responsive_control(
				'tabs_autoplay_border_size',
				array(
					'label'     => esc_html__( 'Border Size', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 1,
					'max'       => 10,
					'step'      => 1,
					'default'   => 3,
					'selectors' => array(
						'{{WRAPPER}} .tp-tab-playloop .plus-tab-header:after' => 'border-bottom: solid {{VALUE}}px {{tabs_autoplay_border_color.VALUE}};',
					),
					'condition' => array(
						'tabs_autoplay' => 'yes',
					),
				)
			);
			$this->add_control(
				'autoplayicon',
				array(
					'label'     => esc_html__( 'Play Icon', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-play',
						'library' => 'solid',
					),
					'condition' => array(
						'tabs_autoplay'      => 'yes',
						'tabs_autoplaypause' => 'yes',
					),
				)
			);
			$this->add_control(
				'autopauseicon',
				array(
					'label'     => esc_html__( 'Pause Icon', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-pause',
						'library' => 'solid',
					),
					'condition' => array(
						'tabs_autoplay'      => 'yes',
						'tabs_autoplaypause' => 'yes',
					),
				)
			);
		$this->end_popover();
		$this->add_control(
			'expand_collapse_popover',
			[
				'label'     => esc_html__( 'Expand & Collapse Button', 'theplus' ),
				'type'      => Controls_Manager::POPOVER_TOGGLE,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'condition'   => array(
						'horizontal_accordion!' => 'yes',
					),
			]
		);	
		$this->start_popover();
			$this->add_control(
				'expand_collapse_accordion',
				array(
					'label'     => esc_html__( 'Expand & Collapse Button', 'theplus' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => 'no',
					'separator' => 'after',
					'label_on'  => esc_html__( 'Show', 'theplus' ),
					'label_off' => esc_html__( 'Hide', 'theplus' ),
					'separator' => 'after',
					'condition' => array(
						'horizontal_accordion!' => 'yes',
					),
				)
			);
			$this->add_control(
				'expand_collapse_label',
				array(
					'type'  => Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
							esc_html__( 'Activate this to let users expand or collapse all accordion items at once from a button, great for FAQs or large content sections.', 'theplus' ),
							esc_url( $this->tp_doc . 'expand-close-elementor-accordion-button/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
							esc_html__( 'Learn More', 'theplus' ),
						)
					),
					'label_block' => true,
					'condition' => array(
						'horizontal_accordion!' => 'yes',
					),
				)
			);
			$this->add_control(
				'eca_position',
				array(
					'label'     => esc_html__( 'Position', 'theplus' ),
					'type'      => Controls_Manager::SELECT,
					'default'   => 'ecabefore',
					'options'   => array(
						'ecabefore' => esc_html__( 'Before', 'theplus' ),
						'ecaafter'  => esc_html__( 'After', 'theplus' ),
					),
					'condition' => array(
						'expand_collapse_accordion' => 'yes',
					),
				)
			);
			$this->add_control(
				'eca_colall',
				array(
					'label'     => esc_html__( 'Collapse', 'theplus' ),
					'type'      => Controls_Manager::TEXT,
					'default'   => esc_html__( 'Collapse All', 'theplus' ),
					'ai' => false,
					'condition' => array(
						'expand_collapse_accordion' => 'yes',
					),
					'selectors' => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .tp-toggle-accordion.active:before' => 'content: "{{VALUE}}";',
					),
				)
			);
			$this->add_control(
				'eca_expall',
				array(
					'label'     => esc_html__( 'Expand', 'theplus' ),
					'type'      => Controls_Manager::TEXT,
					'default'   => esc_html__( 'Expand All', 'theplus' ),
					'ai' => false,
					'condition' => array(
						'expand_collapse_accordion' => 'yes',
					),
					'selectors' => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .tp-toggle-accordion:before' => 'content: "{{VALUE}}";',
					),
				)
			);
			$this->add_responsive_control(
				'eca_align',
				array(
					'label'     => esc_html__( 'Alignment', 'theplus' ),
					'type'      => Controls_Manager::CHOOSE,
					'default'   => 'flex-start',
					'options'   => array(
						'flex-start' => array(
							'title' => esc_html__( 'Left', 'theplus' ),
							'icon'  => 'eicon-text-align-left',
						),
						'center'     => array(
							'title' => esc_html__( 'Center', 'theplus' ),
							'icon'  => 'eicon-text-align-center',
						),
						'flex-end'   => array(
							'title' => esc_html__( 'Right', 'theplus' ),
							'icon'  => 'eicon-text-align-right',
						),
					),
					'selectors' => array(
						'{{WRAPPER}} .tp-aec-button' => 'justify-content: {{VALUE}};',
					),
					'condition' => array(
						'expand_collapse_accordion' => 'yes',
					),
				)
			);
			$this->add_control(
				'expand_Note',
				[
					'type' => \Elementor\Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<i><b>%s</b> %s</i>',
							esc_html__( 'Note:', 'theplus' ),
							esc_html__( 'If enabled, You will get button option to expand/collapse content of all accordions together.', 'theplus' ),
						)
					),
					'content_classes' => 'tp-controller-label-text',
				]
			);
		$this->end_popover();
		$this->add_control(
			'search_bar_popover',
			[
				'label'     => esc_html__( 'Search Bar', 'theplus' ),
				'type'      => Controls_Manager::POPOVER_TOGGLE,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
			]
		);	
		$this->start_popover();
			$this->add_control(
				'search_accordion',
				array(
					'label'     => esc_html__( 'Search Bar', 'theplus' ),
					'type'      => Controls_Manager::SWITCHER,
					'separator'   => 'after',
					'default'   => 'no',
					'label_on'  => esc_html__( 'Show', 'theplus' ),
					'label_off' => esc_html__( 'Hide', 'theplus' ),
				)
			);
			$this->add_control(
				'search_bar_label',
				array(
					'type'  => Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
							esc_html__( 'Enable this option to add a search bar above your accordion, allowing visitors to quickly find the content they’re looking for across each accordion item.', 'theplus' ),
							esc_url( $this->tp_doc . 'elementor-accordion-search/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
							esc_html__( 'Learn More', 'theplus' ),
						)
					),
					'label_block' => true,
				)
			);
			$this->add_control(
				'search_text_highlight',
				array(
					'label'     => wp_kses_post( 'Text Highlight' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => '',
					'condition' => array(
						'search_accordion' => 'yes',
					),
					'label_on'  => esc_html__( 'Show', 'theplus' ),
					'label_off' => esc_html__( 'Hide', 'theplus' ),
					'separator' => 'before',
				)
			);

			$this->add_control(
				'search_accordion_length',
				array(
					'label'     => esc_html__( 'Length', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 1,
					'max'       => 15,
					'step'      => 1,
					'default'   => 3,
					'condition' => array(
						'search_accordion' => 'yes',
					),
				)
			);
			$this->add_control(
				'search_accordion_placeholder',
				array(
					'label'       => esc_html__( 'Placeholder', 'theplus' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => esc_html__( 'Enter Search', 'theplus' ),
					'ai' => false,
					'dynamic'     => array(
						'active' => false,
					),
					'condition'   => array(
						'search_accordion' => 'yes',
					),
					'show_label' => true,
				)
			);
			$this->add_control(
				'search_accordion_icon',
				array(
					'label'     => esc_html__( 'Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-search',
						'library' => 'solid',
					),
					'condition' => array(
						'search_accordion' => 'yes',
					),
				)
			);
		$this->end_popover();
		$this->add_control(
			'slider_accordion_popover',
			[
				'label'     => esc_html__( 'Slider & Pagination', 'theplus' ),
				'type'      => Controls_Manager::POPOVER_TOGGLE,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'condition'   => array(
					'horizontal_accordion!' => 'yes',
				),
			]
		);	
		$this->start_popover();
			$this->add_control(
				'slider_accordion',
				array(
					'label'     => esc_html__( 'Slider & Pagination', 'theplus' ),
					'type'        => Controls_Manager::SWITCHER,
					'default'     => 'no',
					'label_on'    => esc_html__( 'Show', 'theplus' ),
					'label_off'   => esc_html__( 'Hide', 'theplus' ),
					'separator'   => 'after',
					'condition'   => array(
						'horizontal_accordion!' => 'yes',
					),
				)
			);
			$this->add_control(
				'slider_accordion_label',
				array(
					'type'  => Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
							esc_html__( 'Use this feature to convert your accordion items into a paginated format for better navigation on long lists of accordion items.', 'theplus' ),
							esc_url( $this->tp_doc . 'elementor-accordion-pagination/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
							esc_html__( 'Learn More', 'theplus' ),
						)
					),
					'label_block' => true,
				)
			);
			$this->add_control(
				'slider_accordion_align',
				array(
					'label'       => esc_html__( 'Alignment', 'theplus' ),
					'type'        => Controls_Manager::CHOOSE,
					'options'     => array(
						'tpsaleft'   => array(
							'title' => esc_html__( 'Left', 'theplus' ),
							'icon'  => 'eicon-text-align-left',
						),
						'tpsacenter' => array(
							'title' => esc_html__( 'Center', 'theplus' ),
							'icon'  => 'eicon-text-align-center',
						),
						'tpsaright'  => array(
							'title' => esc_html__( 'Right', 'theplus' ),
							'icon'  => 'eicon-text-align-right',
						),
					),
					'default'     => 'tpsaleft',
					'condition'   => array(
						'horizontal_accordion!' => 'yes',
						'slider_accordion'      => 'yes',
					),
					'label_block' => false,
					'toggle'      => true,
				)
			);
			$this->add_control(
				'slider_accordion_show',
				array(
					'label'     => esc_html__( 'Slide Per Page', 'theplus' ),
					'type'      => \Elementor\Controls_Manager::NUMBER,
					'min'       => 3,
					'max'       => 20,
					'step'      => 1,
					'default'   => 3,
					'condition' => array(
						'horizontal_accordion!' => 'yes',
						'slider_accordion'      => 'yes',
					),
				)
			);
			$this->add_control(
				'slider_accordion_icon_prev',
				array(
					'label'     => esc_html__( 'Previous Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-arrow-left',
						'library' => 'solid',
					),
					'condition' => array(
						'horizontal_accordion!' => 'yes',
						'slider_accordion'      => 'yes',
					),
				)
			);
			$this->add_control(
				'slider_accordion_text_prev',
				array(
					'label'     => esc_html__( 'Previous Text', 'theplus' ),
					'type'      => Controls_Manager::TEXT,
					'ai' => false,
					'condition' => array(
						'horizontal_accordion!' => 'yes',
						'slider_accordion'      => 'yes',
					),
				)
			);
			$this->add_control(
				'slider_accordion_icon_nxt',
				array(
					'label'     => esc_html__( 'Next Icon Library', 'theplus' ),
					'type'      => Controls_Manager::ICONS,
					'default'   => array(
						'value'   => 'fas fa-arrow-right',
						'library' => 'solid',
					),
					'condition' => array(
						'horizontal_accordion!' => 'yes',
						'slider_accordion'      => 'yes',
					),
				)
			);
			$this->add_control(
				'slider_accordion_text_nxt',
				array(
					'label'     => esc_html__( 'Next Text', 'theplus' ),
					'type'      => Controls_Manager::TEXT,
					'ai' => false,
					'condition' => array(
						'horizontal_accordion!' => 'yes',
						'slider_accordion'      => 'yes',
					),
				)
			);
			$this->add_control(
				'slider_Note',
				[
					'type' => \Elementor\Controls_Manager::RAW_HTML,
					'raw'   => wp_kses_post(
						sprintf(
							'<i><b>%s</b> %s</i>',
							esc_html__( 'Note:', 'theplus' ),
							esc_html__( 'By enabling this option, You will be able to divide accordions in multiple slides/section with previous and next options.', 'theplus' ),
						)
					),
					'content_classes' => 'tp-controller-label-text',
					'condition'   => array(
						'horizontal_accordion!' => 'yes',
					),
				]
			);
		$this->end_popover();
		
		$this->end_controls_section();

		$this->start_controls_section(
			'extra_option_section',
			array(
				'label' => esc_html__( 'Extra Option', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'active_accordion',
			array(
				'label'   => esc_html__( 'Active Tab', 'theplus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '1',
				'options' => theplus_get_numbers(),
			)
		);
		$this->add_control(
			'accordion_scroll_top',
			array(
				'label'       => esc_html__( 'Scroll Top', 'theplus' ),
				'type'        => Controls_Manager::SWITCHER,
				'default'     => 'no',
				'label_on'    => esc_html__( 'Show', 'theplus' ),
				'label_off'   => esc_html__( 'Hide', 'theplus' ),
			)
		);
		$this->add_control(
			'scroll_top_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Enable this to automatically scroll the active accordion into view when opened, ensuring your content stays visible without manual scrolling.', 'theplus' )
					)
				),
				'label_block' => true,
			)
		);
		$this->add_control(
			'act_offset',
			array(
				'label'     => esc_html__( 'Offset', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'min'       => 0,
				'max'       => 500,
				'step'      => 10,
				'default'   => 0,
				'condition' => array(
					'accordion_scroll_top' => 'yes',
				),
			)
		);
		$this->add_control(
			'act_speed',
			array(
				'label'     => esc_html__( 'Speed', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'min'       => 1,
				'max'       => 2000,
				'step'      => 10,
				'default'   => 500,
				'condition' => array(
					'accordion_scroll_top' => 'yes',
				),
			)
		);
		$this->add_control(
			'accordion_note',
			[
				'type' => \Elementor\Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<i><b>%s</b> %s</i>',
						esc_html__( 'Note:', 'theplus' ),
						esc_html__( 'If Show, When you click on accordion, It will scroll to top automatically.', 'theplus' ),
					)
				),
				'content_classes' => 'tp-controller-label-text',
				'condition' => array(
					'accordion_scroll_top' => 'yes',
				),
			]
		);
		$this->add_control(
			'accordion_stager',
			array(
				'label'     => esc_html__( 'Stagger Animation', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'separator' => 'before',
			)
		);
		$this->add_control(
			'stager_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Turn this on to apply staggered entrance animations for a smooth, sequential reveal of accordion items, perfect for creating engaging reading flow.', 'theplus' )
					)
				),
				'label_block' => true,
			)
		);
		$this->add_control(
			'accordion_stager_visi_delay',
			array(
				'label'     => esc_html__( 'Visibility Delay', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'min'       => 1,
				'max'       => 10000,
				'step'      => 100,
				'default'   => 500,
				'condition' => array(
					'accordion_stager' => 'yes',
				),
			)
		);
		$this->add_control(
			'accordion_stager_gap',
			array(
				'label'     => esc_html__( 'Gap', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'min'       => 50,
				'max'       => 5000,
				'step'      => 100,
				'default'   => 500,
				'condition' => array(
					'accordion_stager' => 'yes',
				),
			)
		);
		$this->add_control(
			'schema_accordion',
			array(
				'label'     => esc_html__( 'SEO Schema Markup', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'separator' => 'before',
			)
		);
		$this->add_control(
			'schema_accordion_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Enable this to boost your page SEO using built-in FAQ Schema markup, making your accordion content eligible for rich snippets on Google.', 'theplus' ),
						esc_url( $this->tp_doc . 'elementor-accordion-schema-markup/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);
		$this->add_control(
			'seo_Note',
			[
				'type' => \Elementor\Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> <b>%s</b> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Note:', 'theplus' ),
						esc_html__( 'By enabling this option, Your content of Accordions will be converted in to Structured data based on Google Search engine. It will be considered as', 'theplus' ),
						esc_url( 'https://developers.google.com/search/docs/advanced/structured-data/faqpage' ),
						esc_html__( 'FAQ Schema', 'theplus' ),
					)
				),
			]
		);
		$this->add_control(
			'connection_unique_id',
			array(
				'label'     => esc_html__( 'Carousel Connection ID', 'theplus' ),
				'type'      => Controls_Manager::TEXT,
				'ai'        => false,
				'default'   => '',
				'separator' => 'before',
			)
		);
		$this->add_control(
			'unique_id_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Use this option to sync your accordions with the Anything Carousel widget. Enter the same Unique ID in both widgets for seamless connection.', 'theplus' ),
						esc_url( $this->tp_doc . 'carousel-widgets-remotesync-elementor/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'theplus_section_needhelp',
			array(
				'label' => esc_html__( 'Need Help?', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'theplus_help_requests',
			array(
				'label'   => __( 'Need Help', 'theplus' ),
				'type'    => 'tpae_need_help',
				'default' => array(
					array(
						'label' => __( 'Read Docs', 'theplus' ),
						'url'   => 'https://theplusaddons.com/help/accordion/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget',
					),
					array(
						'label' => __( 'Watch Video', 'theplus' ),
						'url'   => 'https://www.youtube.com/watch?v=S2fpN63Lnzk&pp=0gcJCcMJAYcqIYzv',
					),
				),
			)
		);
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_toggle_style_icon',
			array(
				'label'     => esc_html__( 'Icon', 'theplus' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);
		
		$this->add_control(
			'toggleicon',
			array(
				'label'     => __( 'Toggle Icon', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
			)
		);
		$this->add_responsive_control(
			'toggle_icon_size',
			array(
				'label'     => esc_html__( 'Icon Size', 'theplus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.elementor-accordion .elementor-tab-title .elementor-accordion-icon' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .theplus-accordion-wrapper.elementor-accordion .elementor-tab-title .elementor-accordion-icon svg' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}};',
				),
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);
		$this->add_responsive_control(
			'icon_space',
			array(
				'label'     => esc_html__( 'Gap', 'theplus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-accordion-icon.elementor-accordion-icon-left' => 'margin-right: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .elementor-accordion .elementor-accordion-icon.elementor-accordion-icon-right' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);
		$this->start_controls_tabs( 'tabs_color_style' );
		$this->start_controls_tab(
			'tab_color_normal',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);
		$this->add_control(
			'icon_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title .elementor-accordion-icon i:before' => 'color: {{VALUE}};',
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title .elementor-accordion-icon svg' => 'fill: {{VALUE}};',
				),
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);
		$this->add_control(
			'icon_fill_color',
			array(
				'label'     => esc_html__( 'Fill', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title .elementor-accordion-icon svg path' => 'fill: {{VALUE}} !important;; ',
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title .elementor-accordion-icon svg' => 'fill: {{VALUE}} !important;',
				),

				'condition' => array(
					'icon_style' => 'font_awesome_5',
				),
			)
		);
		$this->add_control(
			'icon_stroke_color',
			array(
				'label'     => esc_html__( 'Stroke', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title .elementor-accordion-icon svg path' => 'stroke: {{VALUE}} !important;; ',
					'{{WRAPPER}} .elementor-accordion+ .elementor-tab-title .elementor-accordion-icon svg' => 'stroke: {{VALUE}} !important;',
				),
				'condition' => array(
					'icon_style' => 'font_awesome_5',
				),
			)
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_color_active',
			array(
				'label' => esc_html__( 'Active', 'theplus' ),
			)
		);
		$this->add_control(
			'icon_active_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title.active .elementor-accordion-icon i:before' => 'color: {{VALUE}};',
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title.active .elementor-accordion-icon svg' => 'fill: {{VALUE}};',
				),
				'condition' => array(
					'display_icon' => 'yes',
				),
			)
		);
		$this->add_control(
			'icon_fill_color_active',
			array(
				'label'     => esc_html__( 'Active Fill ', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title.active .elementor-accordion-icon svg path' => 'fill: {{VALUE}} !important;; ',
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title.active .elementor-accordion-icon svg' => 'fill: {{VALUE}} !important;',
				),

				'condition' => array(
					'icon_style' => 'font_awesome_5',
				),
			)
		);
		$this->add_control(
			'icon_stroke_color_active',
			array(
				'label'     => esc_html__( 'Active Stroke', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title.active .elementor-accordion-icon svg path' => 'stroke: {{VALUE}} !important;; ',
					'{{WRAPPER}} .elementor-accordion .elementor-tab-title.active .elementor-accordion-icon svg' => 'stroke: {{VALUE}} !important;',
				),
				'condition' => array(
					'icon_style' => 'font_awesome_5',
				),
			)
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->add_control(
			'loop_icon_heading',
			array(
				'label'     => esc_html__( 'Title Icon', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);
		$this->add_responsive_control(
			'loop_icon_size',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Icon Size', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 0,
						'max'  => 200,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 15,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix' => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix svg' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}}',
				),
			)
		);
		$this->add_responsive_control(
			'loop_icon_width',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Icon Width', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 0,
						'max'  => 250,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 35,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}};line-height: {{SIZE}}{{UNIT}} ;text-align: center;',
				),
			)
		);
		$this->add_responsive_control(
			'loop_icon_indent',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Icon Space/Indent', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 0,
						'max'  => 50,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 8,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix,{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix svg' => 'margin-right: {{SIZE}}{{UNIT}}',
				),
			)
		);


		$this->start_controls_tabs( 'tabs_loop_icon_style' );
			$this->start_controls_tab(
				'tab_loop_icon_normal',
				array(
					'label' => esc_html__( 'Normal', 'theplus' ),
				)
			);
			$this->add_control(
				'loop_icon_color',
				array(
					'label'     => esc_html__( 'Color', 'theplus' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix' => 'color: {{VALUE}};-webkit-text-fill-color: {{VALUE}};',
						'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix svg' => 'fill: {{VALUE}};',
					),
				)
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				array(
					'name'     => 'loop_icon_background',
					'types'    => array( 'classic', 'gradient' ),
					'separator' => 'before',
					'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix',
				)
			);
			$this->add_responsive_control(
				'loop_icon_border_radius',
				array(
					'label'      => esc_html__( 'Border Radius', 'theplus' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', '%' ),
					'selectors'  => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
				)
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'loop_icon_box_shadow',
					'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header .accordion-icon-prefix',
				)
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'tab_loop_icon_hover',
				array(
					'label' => esc_html__( 'Active', 'theplus' ),
				)
			);
			$this->add_control(
				'loop_icon_hover_color',
				array(
					'label'     => esc_html__( 'Color', 'theplus' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header.active .accordion-icon-prefix' => 'color: {{VALUE}};-webkit-text-fill-color: {{VALUE}};',
						'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header.active .accordion-icon-prefix svg' => 'fill: {{VALUE}};',
					),
				)
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				array(
					'name'     => 'loop_icon_hover_background',
					'separator' => 'before',
					'types'    => array( 'classic', 'gradient' ),
					'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header.active .accordion-icon-prefix',
				)
			);
			$this->add_responsive_control(
				'loop_icon_hover_border_radius',
				array(
					'label'      => esc_html__( 'Border Radius', 'theplus' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', '%' ),
					'selectors'  => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header.active .accordion-icon-prefix' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
				)
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'loop_icon_hover_box_shadow',
					'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .plus-accordion-header.active .accordion-icon-prefix',
				)
			);
			$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();

		/*Title Section*/

		$this->start_controls_section(
            'section_title_style',
            [
                'label' => esc_html__( 'Title', 'theplus' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'Typography', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header',
			)
		);
		$this->add_control(
			'title_align',
			array(
				'label'       => esc_html__( 'Alignment', 'theplus' ),
				'type'        => Controls_Manager::CHOOSE,
				'options'     => array(
					'text-left'   => array(
						'title' => esc_html__( 'Left', 'theplus' ),
						'icon'  => 'eicon-text-align-left',
					),
					'text-center' => array(
						'title' => esc_html__( 'Center', 'theplus' ),
						'icon'  => 'eicon-text-align-center',
					),
					'text-right'  => array(
						'title' => esc_html__( 'Right', 'theplus' ),
						'icon'  => 'eicon-text-align-right',
					),
				),
				'default'     => 'text-left',
				'label_block' => false,
			)
		); 
		$this->add_responsive_control(
			'title_gap',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Gap', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 0,
						'max'  => 250,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 68,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'line-height: {{SIZE}}{{UNIT}};',
				),
				'condition'   => array(
					'horizontal_accordion' => 'yes',
				),
			)
		);

		$this->start_controls_tabs( 'tabs_title_style' );
			$this->start_controls_tab(
				'tab_title_normal',
				array(
					'label' => esc_html__( 'Normal', 'theplus' ),
				)
			);
			$this->add_control(
				'title_color_option',
				array(
					'label'       => esc_html__( 'Color', 'theplus' ),
					'type'        => Controls_Manager::CHOOSE,
					'options'     => array(
						'solid'    => array(
							'title' => esc_html__( 'Classic', 'theplus' ),
							'icon'  => 'eicon-paint-brush',
						),
						'gradient' => array(
							'title' => esc_html__( 'Gradient', 'theplus' ),
							'icon'  => 'eicon-barcode',
						),
					),
					'label_block' => false,
					'default'     => 'solid',
				)
			);
			$this->add_control(
				'title_color',
				array(
					'label'     => esc_html__( 'Color', 'theplus' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '#313131',
					'selectors' => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'color: {{VALUE}}',
					),
					'condition' => array(
						'title_color_option' => 'solid',
					),
				)
			);
			$this->add_control(
				'title_gradient_color1',
				array(
					'label'     => esc_html__( 'Color 1', 'theplus' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => 'orange',
					'condition' => array(
						'title_color_option' => 'gradient',
					),
					'of_type'   => 'gradient',
				)
			);
			$this->add_control(
				'title_gradient_color1_control',
				array(
					'type'        => Controls_Manager::SLIDER,
					'label'       => esc_html__( 'Color 1 Location', 'theplus' ),
					'size_units'  => array( '%' ),
					'default'     => array(
						'unit' => '%',
						'size' => 0,
					),
					'render_type' => 'ui',
					'condition'   => array(
						'title_color_option' => 'gradient',
					),
					'of_type'     => 'gradient',
				)
			);
			$this->add_control(
				'title_gradient_color2',
				array(
					'label'     => esc_html__( 'Color 2', 'theplus' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => 'cyan',
					'condition' => array(
						'title_color_option' => 'gradient',
					),
					'of_type'   => 'gradient',
				)
			);
			$this->add_control(
				'title_gradient_color2_control',
				array(
					'type'        => Controls_Manager::SLIDER,
					'label'       => esc_html__( 'Color 2 Location', 'theplus' ),
					'size_units'  => array( '%' ),
					'default'     => array(
						'unit' => '%',
						'size' => 100,
					),
					'render_type' => 'ui',
					'condition'   => array(
						'title_color_option' => 'gradient',
					),
					'of_type'     => 'gradient',
				)
			);
			$this->add_control(
				'title_gradient_style',
				array(
					'type'      => Controls_Manager::SELECT,
					'label'     => esc_html__( 'Gradient Style', 'theplus' ),
					'default'   => 'linear',
					'options'   => theplus_get_gradient_styles(),
					'condition' => array(
						'title_color_option' => 'gradient',
					),
					'of_type'   => 'gradient',
				)
			);
			$this->add_control(
				'title_gradient_angle',
				array(
					'type'       => Controls_Manager::SLIDER,
					'label'      => esc_html__( 'Gradient Angle', 'theplus' ),
					'size_units' => array( 'deg' ),
					'default'    => array(
						'unit' => 'deg',
						'size' => 180,
					),
					'range'      => array(
						'deg' => array(
							'step' => 10,
						),
					),
					'selectors'  => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{title_gradient_color1.VALUE}} {{title_gradient_color1_control.SIZE}}{{title_gradient_color1_control.UNIT}}, {{title_gradient_color2.VALUE}} {{title_gradient_color2_control.SIZE}}{{title_gradient_color2_control.UNIT}});-webkit-background-clip: text;-webkit-text-fill-color: transparent;',
					),
					'condition'  => array(
						'title_color_option'   => 'gradient',
						'title_gradient_style' => array( 'linear' ),
					),
					'of_type'    => 'gradient',
				)
			);
			$this->add_control(
				'title_gradient_position',
				array(
					'type'      => Controls_Manager::SELECT,
					'label'     => esc_html__( 'Position', 'theplus' ),
					'options'   => theplus_get_position_options(),
					'default'   => 'center center',
					'selectors' => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{title_gradient_color1.VALUE}} {{title_gradient_color1_control.SIZE}}{{title_gradient_color1_control.UNIT}}, {{title_gradient_color2.VALUE}} {{title_gradient_color2_control.SIZE}}{{title_gradient_color2_control.UNIT}});-webkit-background-clip: text;-webkit-text-fill-color: transparent;',
					),
					'condition' => array(
						'title_color_option'   => 'gradient',
						'title_gradient_style' => 'radial',
					),
					'of_type'   => 'gradient',
				)
			);
			$this->end_controls_tab();

			$this->start_controls_tab(
				'tab_title_active',
				array(
					'label' => esc_html__( 'Active', 'theplus' ),
				)
			);
				$this->add_control(
					'title_active_color_option',
					array(
						'label'       => esc_html__( 'Title Color', 'theplus' ),
						'type'        => Controls_Manager::CHOOSE,
						'options'     => array(
							'solid'    => array(
								'title' => esc_html__( 'Classic', 'theplus' ),
								'icon'  => 'eicon-paint-brush',
							),
							'gradient' => array(
								'title' => esc_html__( 'Gradient', 'theplus' ),
								'icon'  => 'eicon-barcode',
							),
						),
						'label_block' => false,
						'default'     => 'solid',
					)
				);
				$this->add_control(
					'title_active_color',
					array(
						'label'     => esc_html__( 'Color', 'theplus' ),
						'type'      => Controls_Manager::COLOR,
						'default'   => '#3351a6',
						'selectors' => array(
							'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active' => 'color: {{VALUE}}',
						),
						'condition' => array(
							'title_active_color_option' => 'solid',
						),
					)
				);
				$this->add_control(
					'title_active_gradient_color1',
					array(
						'label'     => esc_html__( 'Color 1', 'theplus' ),
						'type'      => Controls_Manager::COLOR,
						'default'   => 'orange',
						'condition' => array(
							'title_active_color_option' => 'gradient',
						),
						'of_type'   => 'gradient',
					)
				);
				$this->add_control(
					'title_active_gradient_color1_control',
					array(
						'type'        => Controls_Manager::SLIDER,
						'label'       => esc_html__( 'Color 1 Location', 'theplus' ),
						'size_units'  => array( '%' ),
						'default'     => array(
							'unit' => '%',
							'size' => 0,
						),
						'render_type' => 'ui',
						'condition'   => array(
							'title_active_color_option' => 'gradient',
						),
						'of_type'     => 'gradient',
					)
				);
				$this->add_control(
					'title_active_gradient_color2',
					array(
						'label'     => esc_html__( 'Color 2', 'theplus' ),
						'type'      => Controls_Manager::COLOR,
						'default'   => 'cyan',
						'condition' => array(
							'title_active_color_option' => 'gradient',
						),
						'of_type'   => 'gradient',
					)
				);
				$this->add_control(
					'title_active_gradient_color2_control',
					array(
						'type'        => Controls_Manager::SLIDER,
						'label'       => esc_html__( 'Color 2 Location', 'theplus' ),
						'size_units'  => array( '%' ),
						'default'     => array(
							'unit' => '%',
							'size' => 100,
						),
						'render_type' => 'ui',
						'condition'   => array(
							'title_active_color_option' => 'gradient',
						),
						'of_type'     => 'gradient',
					)
				);
				$this->add_control(
					'title_active_gradient_style',
					array(
						'type'      => Controls_Manager::SELECT,
						'label'     => esc_html__( 'Gradient Style', 'theplus' ),
						'default'   => 'linear',
						'options'   => theplus_get_gradient_styles(),
						'condition' => array(
							'title_active_color_option' => 'gradient',
						),
						'of_type'   => 'gradient',
					)
				);
				$this->add_control(
					'title_active_gradient_angle',
					array(
						'type'       => Controls_Manager::SLIDER,
						'label'      => esc_html__( 'Gradient Angle', 'theplus' ),
						'size_units' => array( 'deg' ),
						'default'    => array(
							'unit' => 'deg',
							'size' => 180,
						),
						'range'      => array(
							'deg' => array(
								'step' => 10,
							),
						),
						'selectors'  => array(
							'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{title_active_gradient_color1.VALUE}} {{title_active_gradient_color1_control.SIZE}}{{title_active_gradient_color1_control.UNIT}}, {{title_active_gradient_color2.VALUE}} {{title_active_gradient_color2_control.SIZE}}{{title_active_gradient_color2_control.UNIT}});-webkit-background-clip: text;-webkit-text-fill-color: transparent;',
						),
						'condition'  => array(
							'title_active_color_option'   => 'gradient',
							'title_active_gradient_style' => array( 'linear' ),
						),
						'of_type'    => 'gradient',
					)
				);
				$this->add_control(
					'title_active_gradient_position',
					array(
						'type'      => Controls_Manager::SELECT,
						'label'     => esc_html__( 'Position', 'theplus' ),
						'options'   => theplus_get_position_options(),
						'default'   => 'center center',
						'selectors' => array(
							'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{title_active_gradient_color1.VALUE}} {{title_active_gradient_color1_control.SIZE}}{{title_active_gradient_color1_control.UNIT}}, {{title_active_gradient_color2.VALUE}} {{title_active_gradient_color2_control.SIZE}}{{title_active_gradient_color2_control.UNIT}});-webkit-background-clip: text;-webkit-text-fill-color: transparent;',
						),
						'condition' => array(
							'title_active_color_option'   => 'gradient',
							'title_active_gradient_style' => 'radial',
						),
						'of_type'   => 'gradient',
					)
				);
			$this->end_controls_tab();
		$this->end_controls_tabs();

        $this->add_responsive_control(
            'section_accordion_styling',
            [
                'label' => esc_html__( 'Title Background', 'theplus' ),
                'type'  => Controls_Manager::HEADING,
				'separator'  => 'before',
            ]
        );

        $this->add_responsive_control(
            'accordion_title_padding',
            [
                'label'      => esc_html__( 'Inner Padding', 'theplus' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em' ],
                'selectors'  => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .theplus-accordion-wrapper.elementor-accordion .elementor-tab-title .elementor-accordion-icon.elementor-accordion-icon-right' => 'right: {{RIGHT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'accordion_space',
            [
                'label'      => esc_html__( 'Accordion Between Space', 'theplus' ),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min'  => 0,
                        'max'  => 200,
                        'step' => 2,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 15,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
            'box_border',
            [
                'label'     => esc_html__( 'Box Border', 'theplus' ),
                'type'      => Controls_Manager::SWITCHER,
                'label_on'  => esc_html__( 'Show', 'theplus' ),
                'label_off' => esc_html__( 'Hide', 'theplus' ),
                'default'   => 'no',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'border_style',
            [
                'label'     => esc_html__( 'Border Style', 'theplus' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'solid',
                'options'   => theplus_get_border_style(),
                'selectors' => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'border-style: {{VALUE}};',
                ],
                'condition' => [
                    'box_border' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'box_border_width',
            [
                'label'      => esc_html__( 'Border Width', 'theplus' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'default'    => [
                    'top'    => 1,
                    'right'  => 1,
                    'bottom' => 1,
                    'left'   => 1,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'box_border' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs( 'tabs_background_style' );

        $this->start_controls_tab(
            'tab_background_normal',
            [
                'label' => esc_html__( 'Normal', 'theplus' ),
            ]
        );

		$this->add_control(
            'box_border_color',
            [
                'label'     => esc_html__( 'Border Color', 'theplus' ),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#252525',
                'selectors' => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'border-color: {{VALUE}};',
                ],
                'condition' => [
                    'box_border' => 'yes',
                ],
            ]
        );
        $this->add_responsive_control(
            'border_radius',
            [
                'label'      => esc_html__( 'Border Radius', 'theplus' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'box_border' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'box_background',
                'types'    => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_background_active',
            [
                'label' => esc_html__( 'Active', 'theplus' ),
            ]
        );

		$this->add_control(
            'box_border_active_color',
            [
                'label'     => esc_html__( 'Border Color', 'theplus' ),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#252525',
                'selectors' => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active' => 'border-color: {{VALUE}};',
                ],
                'condition' => [
                    'box_border' => 'yes',
                ],
            ]
        );
        $this->add_responsive_control(
            'border_active_radius',
            [
                'label'      => esc_html__( 'Border Radius', 'theplus' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'box_border' => 'yes',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'box_active_background',
                'types'    => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'shadow_options',
            [
                'label'     => esc_html__( 'Box Shadow Options', 'theplus' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( 'tabs_shadow_style' );

        $this->start_controls_tab(
            'tab_shadow_normal',
            [
                'label' => esc_html__( 'Normal', 'theplus' ),
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'box_shadow',
                'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_shadow_active',
            [
                'label' => esc_html__( 'Active', 'theplus' ),
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'box_active_shadow',
                'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header.active',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'filter_popover',
            [
                'label'     => esc_html__( 'Backdrop Filter', 'theplus' ),
                'type'      => Controls_Manager::POPOVER_TOGGLE,
                'default'   => 'no',
                'label_on'  => esc_html__( 'Show', 'theplus' ),
                'label_off' => esc_html__( 'Hide', 'theplus' ),
            ]
        );
		$this->start_popover();
			$this->add_control(
				'mbbf',
				array(
					'label'        => esc_html__( 'Backdrop Filter', 'theplus' ),
					'type'         => Controls_Manager::POPOVER_TOGGLE,
					'label_off'    => __( 'Default', 'theplus' ),
					'label_on'     => __( 'Custom', 'theplus' ),
					'return_value' => 'yes',
				)
			);
			$this->add_control(
				'mbbf_blur',
				array(
					'label'      => esc_html__( 'Blur', 'theplus' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px' ),
					'range'      => array(
						'px' => array(
							'max'  => 100,
							'min'  => 1,
							'step' => 1,
						),
					),
					'default'    => array(
						'unit' => 'px',
						'size' => 10,
					),
					'condition'  => array(
						'mbbf' => 'yes',
					),
				)
			);
			$this->add_control(
				'mbbf_grayscale',
				array(
					'label'      => esc_html__( 'Grayscale', 'theplus' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => array( 'px' ),
					'range'      => array(
						'px' => array(
							'max'  => 1,
							'min'  => 0,
							'step' => 0.1,
						),
					),
					'default'    => array(
						'unit' => 'px',
						'size' => 0,
					),
					'selectors'  => array(
						'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-header' => '-webkit-backdrop-filter:grayscale({{mbbf_grayscale.SIZE}})  blur({{mbbf_blur.SIZE}}{{mbbf_blur.UNIT}}) !important;backdrop-filter:grayscale({{mbbf_grayscale.SIZE}})  blur({{mbbf_blur.SIZE}}{{mbbf_blur.UNIT}}) !important;',
					),
					'condition'  => array(
						'mbbf' => 'yes',
					),
				)
			);
			$this->end_popover();
		$this->end_controls_section();

		/*Title Section end*/


		$this->start_controls_section(
			'section_content_styling',
			array(
				'label' => esc_html__( 'Content', 'theplus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);
		$this->add_responsive_control(
			'content_accordion_padding',
			array(
				'label'      => esc_html__( 'Padding', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_responsive_control(
			'content_accordion_margin',
			array(
				'label'      => esc_html__( 'Margin', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'desc_typography',
				'label'    => esc_html__( 'Typography', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content .plus-content-editor',
			)
		);
		$this->add_control(
			'desc_color',
			array(
				'label'     => esc_html__( 'Text Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content .plus-content-editor,{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content .plus-content-editor p' => 'color: {{VALUE}}',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'      => 'content_box_background',
				'types'     => array( 'classic', 'gradient' ),
				'selector'  => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content',

			)
		);
		$this->add_control(
			'content_box_border',
			array(
				'label'     => esc_html__( 'Box Border', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
				'default'   => 'no',
			)
		);

		$this->add_control(
			'content_border_style',
			array(
				'label'     => esc_html__( 'Border Style', 'theplus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'solid',
				'options'   => theplus_get_border_style(),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => 'border-style: {{VALUE}};',
				),
				'condition' => array(
					'content_box_border' => 'yes',
				),
			)
		);
		$this->add_responsive_control(
			'content_box_border_width',
			array(
				'label'      => esc_html__( 'Border Width', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'default'    => array(
					'top'    => 1,
					'right'  => 1,
					'bottom' => 1,
					'left'   => 1,
				),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'content_box_border' => 'yes',
				),
			)
		);

		$this->add_control(
			'content_box_border_color',
			array(
				'label'     => esc_html__( 'Border Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#252525',
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => 'border-color: {{VALUE}};',
				),
				'condition' => array(
					'content_box_border' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'content_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'content_box_border' => 'yes',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'content_box_shadow',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content',
			)
		);
		$this->add_control(
			'cbf',
			array(
				'label'        => esc_html__( 'Backdrop Filter', 'theplus' ),
				'type'         => Controls_Manager::POPOVER_TOGGLE,
				'label_off'    => __( 'Default', 'theplus' ),
				'label_on'     => __( 'Custom', 'theplus' ),
				'return_value' => 'yes',
			)
		);
		$this->add_control(
			'cbf_blur',
			array(
				'label'      => esc_html__( 'Blur', 'theplus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'max'  => 100,
						'min'  => 1,
						'step' => 1,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 10,
				),
				'condition'  => array(
					'cbf' => 'yes',
				),
			)
		);
		$this->add_control(
			'cbf_grayscale',
			array(
				'label'      => esc_html__( 'Grayscale', 'theplus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'max'  => 1,
						'min'  => 0,
						'step' => 0.1,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 0,
				),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .theplus-accordion-item .plus-accordion-content' => '-webkit-backdrop-filter:grayscale({{cbf_grayscale.SIZE}})  blur({{cbf_blur.SIZE}}{{cbf_blur.UNIT}}) !important;backdrop-filter:grayscale({{cbf_grayscale.SIZE}})  blur({{cbf_blur.SIZE}}{{cbf_blur.UNIT}}) !important;',
				),
				'condition'  => array(
					'cbf' => 'yes',
				),
			)
		);
		$this->end_popover();
		$this->end_controls_section();
		/*desc style*/

		/*paly/pause*/
		$this->start_controls_section(
			'section_autoplay_buttton_style',
			array(
				'label'     => esc_html__( 'Autoplay', 'theplus' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'tabs_autoplay'      => 'yes',
					'tabs_autoplaypause' => 'yes',
				),
			)
		);
		$this->add_responsive_control(
			'autoplay_buttton_padding',
			array(
				'label'      => esc_html__( 'Padding', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_responsive_control(
			'autoplay_buttton_size',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Icon Size', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 1,
						'max'  => 200,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 12,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap .tp-tab-play-pause i' => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap .tp-tab-play-pause svg' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}}',
				),
			)
		);
		$this->start_controls_tabs( 'autoplay_buttton_tabs' );
		$this->start_controls_tab(
			'autoplay_buttton_n',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);
		$this->add_control(
			'autoplay_buttton_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap .tp-tab-play-pause i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap .tp-tab-play-pause svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'autoplay_buttton_BG',
				'label'    => esc_html__( 'Background Type', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'autoplay_buttton_Border',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap',
			)
		);
		$this->add_responsive_control(
			'autoplay_buttton_BRadius',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'separator'  => 'after',
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'autoplay_buttton_Shadow',
				'label'    => esc_html__( 'Box Shadow', 'theplus' ),
				'selector' => '{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap',
			)
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'autoplay_buttton_h',
			array(
				'label' => esc_html__( 'Hover', 'theplus' ),
			)
		);
		$this->add_control(
			'autoplay_buttton_color_h',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap:hover .tp-tab-play-pause i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap:hover .tp-tab-play-pause svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'autoplay_buttton_BG_h',
				'label'    => esc_html__( 'Background Type', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap:hover',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'autoplay_buttton_Border_h',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap:hover',
			)
		);
		$this->add_responsive_control(
			'autoplay_buttton_BRadius_h',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'separator'  => 'after',
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'autoplay_buttton_Shadow_h',
				'label'    => esc_html__( 'Box Shadow', 'theplus' ),
				'selector' => '{{WRAPPER}} .tp-tab-playpause-button .tp-tab-play-pause-wrap:hover',
			)
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
		/*paly/pause*/

		/*slider style*/
		$this->start_controls_section(
			'section_slider_acr_styling',
			array(
				'label'     => esc_html__( 'Slider & Pagination', 'theplus' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'horizontal_accordion!' => 'yes',
					'slider_accordion'      => 'yes',
				),
			)
		);
		$this->add_responsive_control(
			'sliacc_next_prev_padding',
			array(
				'label'      => esc_html__( 'Padding', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next,{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sliacc_next_typography',
				'label'    => esc_html__( 'Typography', 'theplus' ),
				'global'   => array(
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next',
			)
		);
		$this->add_control(
			'sliacc_next_heading',
			array(
				'label' => esc_html__( 'Next', 'theplus' ),
				'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
				'label_off' => esc_html__( 'Default', 'theplus' ),
				'label_on' => esc_html__( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default' => 'yes',
			)
		);
		$this->start_popover();
		
		$this->add_responsive_control(
			'sliacc_next_icon_size',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Icon Size', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 1,
						'max'  => 250,
						'step' => 1,
					),
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}}  .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next i' => 'font-size:{{SIZE}}{{UNIT}}',
					'{{WRAPPER}}  .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next svg' => 'width: {{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}}',
				),
			)
		);
		$this->add_responsive_control(
			'sliacc_next_icon_offset',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Offset', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 5,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}}  .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next .tpas-icon' => 'margin-left:{{SIZE}}{{UNIT}}',
				),
			)
		);
		$this->start_controls_tabs( 'tabs_sliacc_next' );
		$this->start_controls_tab(
			'tab_sliacrn_n',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);
		$this->add_control(
			'sliacrn_n_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next' => 'color: {{VALUE}};',
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'sliacrn_n_bg',
				'label'    => esc_html__( 'Background', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'sliacrn_n_brd',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next',
			)
		);
		$this->add_responsive_control(
			'sliacrn_n_br',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sliacrn_n_shadow',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next',
			)
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_sliacrn_h',
			array(
				'label' => esc_html__( 'Hover', 'theplus' ),
			)
		);
		$this->add_control(
			'sliacrn_h_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next:hover svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'sliacrn_h_bg',
				'label'    => esc_html__( 'Background', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next:hover',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'sliacrn_h_brd',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next:hover',
			)
		);
		$this->add_responsive_control(
			'sliacrn_h_br',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sliacrn_h_shadow',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-next:hover',
			)
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_popover();
		$this->add_control(
			'sliacc_previous_heading',
			array(
				'label'     => esc_html__( 'Previous', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sliacc_prev_typography',
				'label'    => esc_html__( 'Typography', 'theplus' ),
				'global'   => array(
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev',
			)
		);
		$this->add_responsive_control(
			'sliacc_prev_icon_size',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Icon Size', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 1,
						'max'  => 250,
						'step' => 1,
					),
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}}  .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev i' => 'font-size:{{SIZE}}{{UNIT}}',
					'{{WRAPPER}}  .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev svg' => 'width: {{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}}',
				),
			)
		);
		$this->add_responsive_control(
			'sliacc_p_icon_offset',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Offset', 'theplus' ),
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 5,
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'{{WRAPPER}}  .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev .tpas-icon' => 'margin-right:{{SIZE}}{{UNIT}}',
				),
			)
		);
		$this->start_controls_tabs( 'tabs_sliacc_p_next' );
		$this->start_controls_tab(
			'tab_sliacrn_n_p',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);
		$this->add_control(
			'sliacrn_n_color_p',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev' => 'color: {{VALUE}};',
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'sliacrn_n_bg_p',
				'label'    => esc_html__( 'Background', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'sliacrn_n_brd_p',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev',
			)
		);
		$this->add_responsive_control(
			'sliacrn_n_br_p',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sliacrn_n_shadow_p',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev',
			)
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_sliacrn_h_p',
			array(
				'label' => esc_html__( 'Hover', 'theplus' ),
			)
		);
		$this->add_control(
			'sliacrn_h_color_p',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev:hover svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'sliacrn_h_bg_p',
				'label'    => esc_html__( 'Background', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev:hover',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'sliacrn_h_brd_p',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev:hover',
			)
		);
		$this->add_responsive_control(
			'sliacrn_h_br_p',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sliacrn_h_shadow_p',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-prev:hover',
			)
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->add_control(
			'sliacc_currentslide_heading',
			array(
				'label'     => esc_html__( 'Current Slide', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);
		$this->add_responsive_control(
			'sliacc_cs_margin',
			array(
				'label'      => esc_html__( 'Margin', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-active-slide' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sliacc_cs_typo',
				'label'    => esc_html__( 'Typography', 'theplus' ),
				'global'   => array(
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-active-slide',
			)
		);
		$this->add_control(
			'sliacc_cs_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-active-slide' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_control(
			'sliacc_totalslide_heading',
			array(
				'label'     => esc_html__( 'Total Slide', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);
		$this->add_responsive_control(
			'sliacc_ts_margin',
			array(
				'label'      => esc_html__( 'Margin', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-total-slide' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sliacc_ts_typo',
				'label'    => esc_html__( 'Typography', 'theplus' ),
				'global'   => array(
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-total-slide',
			)
		);
		$this->add_control(
			'sliacc_ts_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-total-slide' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_control(
			'sliacc_seprator_heading',
			array(
				'label'     => esc_html__( 'Seprator', 'theplus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);
		$this->add_control(
			'sliacc_seprator_f',
			array(
				'label'       => esc_html__( 'Seprator', 'theplus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '/',
				'dynamic'     => array(
					'active' => true,
				),
				'label_block' => true,
			)
		);
		$this->add_responsive_control(
			'sliacc_sep_margin',
			array(
				'label'      => esc_html__( 'Margin', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-sep-slide' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_responsive_control(
			'sliacc_sep_size',
			array(
				'label'     => esc_html__( 'Size', 'theplus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 150,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-sep-slide' => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);
		$this->add_control(
			'sliacc_sep_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.tp-accr-slider .tp-aec-slide-page .tpasp-sep-slide' => 'color: {{VALUE}};',
				),
			)
		);
		$this->end_controls_section();
		/*slider style*/
		/*Text Highlight Start*/
		$this->start_controls_section(
			'highlight_background',
			array(
				'label'     => esc_html__( 'Search Text Highlight', 'theplus' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'search_text_highlight' => 'yes',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'content_typography',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper .tp-highlight',
			)
		);

		$this->start_controls_tabs(
			'style_tabs'
		);

		$this->start_controls_tab(
			'text_highlight_tab',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);

		$this->add_control(
			'highlight_bg_color',
			array(
				'label'     => esc_html__( 'Background Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#FFFF33',
				'condition' => array(
					'search_text_highlight' => 'yes',
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .tp-highlight' => 'background-color: {{value}};',
				),
			)
		);
		$this->add_control(
			'highlight_text_color',
			array(
				'label'     => esc_html__( 'Text Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#000',
				'condition' => array(
					'search_text_highlight' => 'yes',
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .tp-highlight' => 'color: {{value}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'hover_highlight_tab',
			array(
				'label' => esc_html__( 'Hover', 'theplus' ),
			)
		);

		$this->add_control(
			'hover_bg_color',
			array(
				'label'     => esc_html__( 'Highlight Background Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#FFFF33',
				'condition' => array(
					'search_text_highlight' => 'yes',
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .tp-highlight:hover' => 'background-color: {{value}};',
				),

			)
		);
		$this->add_control(
			'hover_text_color',
			array(
				'label'     => esc_html__( 'Highlight Text Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ff0000',
				'condition' => array(
					'search_text_highlight' => 'yes',
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper .tp-highlight:hover' => 'color: {{value}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();
		$this->end_controls_section();
		/*Text Highlight End*/
		/*search*/
		$this->start_controls_section(
			'section_seach_styling',
			array(
				'label'     => esc_html__( 'Search Accordion', 'theplus' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'search_accordion' => 'yes',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sa_input_typography',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput',
			)
		);
		$this->add_responsive_control(
			'sa_input_icon_size',
			array(
				'label'     => esc_html__( 'Icon Size', 'theplus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 150,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tp-acr-search-icon' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tp-acr-search-icon svg' => 'width: {{SIZE}}{{UNIT}};height: {{SIZE}}{{UNIT}};',
				),
			)
		);
		$this->add_responsive_control(
			'sa_input_icon_offset',
			array(
				'label'     => esc_html__( 'Icon Offset', 'theplus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 150,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tp-acr-search-icon' => 'left: {{SIZE}}{{UNIT}};',
				),
			)
		);
		$this->add_control(
			'sa_input_placeholder_color',
			array(
				'label'     => esc_html__( 'Placeholder Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput::-webkit-input-placeholder' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_responsive_control(
			'sa_input_inner_padding',
			array(
				'label'      => esc_html__( 'Padding', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_responsive_control(
			'sa_input_inner_margin',
			array(
				'label'      => esc_html__( 'Margin', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'separator'  => 'after',
			)
		);
		$this->start_controls_tabs( 'tabs_sa_input_field_style' );
		$this->start_controls_tab(
			'tab_sa_input_field_normal',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);
		$this->add_control(
			'sa_input_field_color',
			array(
				'label'     => esc_html__( 'Text Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_control(
			'sa_input_icon_color',
			array(
				'label'     => esc_html__( 'Icon Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tp-acr-search-icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tp-acr-search-icon svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'sa_input_field_bg',
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'sa_input__border_n',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput',
			)
		);
		$this->add_responsive_control(
			'sa_input_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sa_input_shadow_n',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput',
			)
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_sa_input_field_focus',
			array(
				'label' => esc_html__( 'Focus', 'theplus' ),
			)
		);
		$this->add_control(
			'sa_input_field_focus_color',
			array(
				'label'     => esc_html__( 'Text Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput:focus' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_control(
			'sa_input_icon_color_h',
			array(
				'label'     => esc_html__( 'Icon Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container:hover .tp-acr-search-icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container:hover .tp-acr-search-icon svg' => 'fill: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'sa_input_field_focus_bg',
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput:focus',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'sa_input__border_f',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput:focus',
			)
		);
		$this->add_responsive_control(
			'sa_input_border_radius_f',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput:focus' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sa_input_shadow_f',
				'selector' => '{{WRAPPER}} .theplus-accordion-wrapper #accordion_search_bar_container .tpacsearchinput:focus',
			)
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
		/*search*/

		/*Expand/Collapse style*/
		$this->start_controls_section(
			'section_eca_styling',
			array(
				'label'     => esc_html__( 'Expand/Collapse Button', 'theplus' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'expand_collapse_accordion' => 'yes',
				),
			)
		);
		$this->add_responsive_control(
			'eca_padding',
			array(
				'label'      => esc_html__( 'Padding', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-aec-button .tp-toggle-accordion' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_responsive_control(
			'eca_margin',
			array(
				'label'      => esc_html__( 'Margin', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-aec-button .tp-toggle-accordion' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'separator'  => 'after',
			)
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'eca_typography',
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion',
			)
		);
		$this->start_controls_tabs( 'tabs_eca_main' );
		$this->start_controls_tab(
			'tab_eca_n',
			array(
				'label' => esc_html__( 'Normal', 'theplus' ),
			)
		);
		$this->add_control(
			'tab_eca_color_n',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tp-aec-button .tp-toggle-accordion' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'tab_eca_background_n',
				'label'    => esc_html__( 'Background', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'tab_eca_border_n',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion',
			)
		);
		$this->add_responsive_control(
			'tab_eca_br_n',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-aec-button .tp-toggle-accordion' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'tab_eca_shadow_n',
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion',
			)
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_eca_h',
			array(
				'label' => esc_html__( 'Hover', 'theplus' ),
			)
		);
		$this->add_control(
			'tab_eca_color_h',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tp-aec-button .tp-toggle-accordion:hover' => 'color: {{VALUE}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'tab_eca_background_h',
				'label'    => esc_html__( 'Background', 'theplus' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion:hover',
			)
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'tab_eca_border_h',
				'label'    => esc_html__( 'Border', 'theplus' ),
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion:hover',
			)
		);
		$this->add_responsive_control(
			'tab_eca_br_h',
			array(
				'label'      => esc_html__( 'Border Radius', 'theplus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .tp-aec-button .tp-toggle-accordion:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'tab_eca_shadow_h',
				'selector' => '{{WRAPPER}} .tp-aec-button .tp-toggle-accordion:hover',
			)
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
		/*Expand/Collapse style*/

		/*Hover Animation style*/
		$this->start_controls_section(
			'section_hover_styling',
			array(
				'label' => esc_html__( 'Hover', 'theplus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);
		$this->add_control(
			'hover_style',
			[
				'label' => esc_html__( 'Style', 'theplus' ),
				'type' => Controls_Manager::VISUAL_CHOICE,
				'label_block' => true,
				'options' => [
					'' => [
						'title' => esc_attr__( 'Normal', 'theplus' ),
						'image' => THEPLUS_URL . 'assets/images/widget-style/accordion/Normal.svg',
					],
					'hover-style-1' => [
						'title' => esc_attr__( 'Style 1', 'theplus' ),
						'image' => THEPLUS_URL . 'assets/images/widget-style/accordion/Style-1.svg',
					],
					'hover-style-2' => [
						'title' => esc_attr__( 'Style 2', 'theplus' ),
						'image' => THEPLUS_URL . 'assets/images/widget-style/accordion/Style-2.svg',
					],
				],
				'default' => '',
				'columns' => 2,
				'classes' => 'tpae-visual_choice',
			]
		);
		$this->add_control(
			'hover_color',
			array(
				'label'     => esc_html__( 'Color', 'theplus' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#8072fc',
				'selectors' => array(
					'{{WRAPPER}} .theplus-accordion-wrapper.hover-style-1 .plus-accordion-header:before,{{WRAPPER}} .theplus-accordion-wrapper.hover-style-2 .plus-accordion-header:before' => 'background: {{VALUE}};',
				),
				'condition' => array(
					'hover_style' => array( 'hover-style-1', 'hover-style-2' ),
				),
			)
		);
		$this->end_controls_section();
		/*Adv tab*/
		$this->start_controls_section(
			'section_plus_extra_adv',
			array(
				'label' => esc_html__( 'Plus Extras', 'theplus' ),
				'tab'   => Controls_Manager::TAB_ADVANCED,
			)
		);
		$this->end_controls_section();
		/*Adv tab*/

		/*--On Scroll View Animation ---*/
		include THEPLUS_PATH . 'modules/widgets/theplus-widget-animation.php';
	}

	/**
	 * Render.
	 *
	 * @since 1.0.0
	 * @version 5.4.2
	 */
	protected function render() {
		$settings  = $this->get_settings_for_display();
		$templates = Theplus_Element_Load::elementor()->templates_manager->get_source( 'local' )->get_items();

		$title_align = isset($settings['title_align']) ? $settings['title_align'] : '';

		$id_int = substr( $this->get_id_int(), 0, 3 );
		$uid    = uniqid( 'accordion' );

		$on_hover_accordion = 'yes' === $settings['on_hover_accordion'] ? 'hover' : 'accordion';

		$search_accordion = isset( $settings['search_accordion'] ) ? $settings['search_accordion'] : '';

		$search_text_highlight = ! empty( $settings['search_text_highlight'] ) ? true : false;

		$accordiannew = '';
		if ( ! empty( $search_text_highlight ) ) {
			$highlight_data = array( 'search_text_highlight' => (bool) $search_text_highlight );
			$accordiannew   = 'data-accordiannew="' . htmlspecialchars( wp_json_encode( $highlight_data, true ), ENT_QUOTES, 'UTF-8' ) . '"';
		}

		$search_accordion_length = ! empty( $settings['search_accordion_length'] ) ? $settings['search_accordion_length'] : 3;

		$accordion_stager     = isset( $settings['accordion_stager'] ) ? $settings['accordion_stager'] : '';
		$accordion_stager_gap = ! empty( $settings['accordion_stager_gap'] ) ? $settings['accordion_stager_gap'] : 500;

		$accordion_stager_visi_delay = ! empty( $settings['accordion_stager_visi_delay'] ) ? $settings['accordion_stager_visi_delay'] : 500;

		$accordion_scroll_top = isset( $settings['accordion_scroll_top'] ) ? $settings['accordion_scroll_top'] : '';

		$act_speed  = ! empty( $settings['act_speed'] ) ? $settings['act_speed'] : 500;
		$act_offset = ! empty( $settings['act_offset'] ) ? $settings['act_offset'] : 0;

		$search_accordion_placeholder = ! empty( $settings['search_accordion_placeholder'] ) ? $settings['search_accordion_placeholder'] : '';

		$slider_accordion = isset( $settings['slider_accordion'] ) ? $settings['slider_accordion'] : '';

		$slider_accordion_show = ! empty( $settings['slider_accordion_show'] ) ? $settings['slider_accordion_show'] : 3;

		$markupSch   = isset( $settings['schema_accordion'] ) ? $settings['schema_accordion'] : false;
		$mainschema  = '';
		$schemaAttr  = '';
		$schemaAttr1 = '';
		$schemaAttr2 = '';
		$schemaAttr3 = '';

		if ( isset( $markupSch ) && 'yes' === $markupSch ) {
			$mainschema  = 'itemscope itemtype="https://schema.org/FAQPage"';
			$schemaAttr  = 'itemscope itemprop="mainEntity" itemtype="https://schema.org/Question"';
			$schemaAttr1 = 'itemprop="name"';
			$schemaAttr2 = 'itemscope itemprop="acceptedAnswer" itemtype="https://schema.org/Answer"';
			$schemaAttr3 = 'itemprop="text"';
		}

		$connect_carousel = '';
		$row_bg_conn      = '';
		if ( ! empty( $settings['connection_unique_id'] ) ) {
			$connect_carousel = 'tpca_' . $settings['connection_unique_id'];

			$uid = 'tptab_' . $settings['connection_unique_id'];

			$row_bg_conn = ' data-row-bg-conn="bgcarousel' . esc_attr( $settings['connection_unique_id'] ) . '"';
		}

		$sattr       = '';
		$sattrclasss = '';
		$searchicon  = '';
		if ( 'yes' === $search_accordion && ! empty( $search_accordion_length ) ) {
			$sattr .= ' data-search-accr="' . $search_accordion . '"';
			$sattr .= ' data-search-attr-len="' . $search_accordion_length . '"';

			$sattrclasss .= ' tp-seachaccr';

			if ( ! empty( $settings['search_accordion_icon'] ) ) {
				ob_start();
				\Elementor\Icons_Manager::render_icon( $settings['search_accordion_icon'], array( 'aria-hidden' => 'true' ) );
				$searchicon = ob_get_contents();
				ob_end_clean();
			}
		}

		if ( 'yes' === $accordion_stager && ! empty( $accordion_stager_visi_delay ) && ! empty( $accordion_stager_gap ) ) {
			$sattr .= ' data-stager-vd-accr="' . $accordion_stager_visi_delay . '"';
			$sattr .= ' data-stager-gap-accr="' . $accordion_stager_gap . '"';

			$sattrclasss .= ' tp-stageraccr';
		}

		if ( 'yes' === $accordion_scroll_top && ! empty( $act_speed ) ) {
			$sattrclasss .= ' tp-scrolltopacc';

			$sattr .= ' data-scroll-top-accr="' . $accordion_scroll_top . '"';
			$sattr .= ' data-scroll-top-speed-accr="' . $act_speed . '"';
			$sattr .= ' data-scroll-top-offset-accr="' . $act_offset . '"';
		}

		if ( 'yes' === $slider_accordion && ! empty( $slider_accordion_show ) ) {
			$sattrclasss .= ' tp-accr-slider';
		}

		$horizontal_accordion = isset( $settings['horizontal_accordion'] ) ? $settings['horizontal_accordion'] : 'no';
		if ( isset( $horizontal_accordion ) && 'yes' === $horizontal_accordion && 'yes' !== $on_hover_accordion ) {
			$ha_title_width = ! empty( $settings['horizontal_accordion_title_width'] ) ? $settings['horizontal_accordion_title_width'] : '';
			$ha_open_speed  = ! empty( $settings['horizontal_accordion_open_speed'] ) ? $settings['horizontal_accordion_open_speed'] : 400;
			$ha_close_speed = ! empty( $settings['horizontal_accordion_close_speed'] ) ? $settings['horizontal_accordion_close_speed'] : 400;

			$horizontal_accordion_layout = ! empty( $settings['horizontal_accordion_layout'] ) ? $settings['horizontal_accordion_layout'] : '';

			$sattrclasss .= ' tp-acc-hori ' . esc_attr( $horizontal_accordion_layout );

			$sattr .= ' data-hori-title-width="' . esc_attr( $ha_title_width ) . '"';
			$sattr .= ' data-hori-open-speed="' . esc_attr( $ha_open_speed ) . '"';
			$sattr .= ' data-hori-close-speed="' . esc_attr( $ha_close_speed ) . '"';
		}

		if ( isset( $settings['tabs_autoplay'] ) && 'yes' === $settings['tabs_autoplay'] ) {
			$tabs_autoplay_duration = ! empty( $settings['tabs_autoplay_duration'] ) ? $settings['tabs_autoplay_duration'] : 5;
			$sattrclasss           .= ' tp-tab-playloop';

			if ( isset( $settings['tabs_autoplaypause'] ) && 'yes' === $settings['tabs_autoplaypause'] ) {
				$sattrclasss .= ' tp-tab-playpause-button';
			}

			$sattr .= ' data-tab-autoplay="yes"';
			$sattr .= ' data-tab-autoplay-duration="' . esc_attr( $tabs_autoplay_duration ) . '"';
		}

		/*--Plus Extra ---*/
		$PlusExtra_Class = 'plus-accordion-widget';
		include THEPLUS_PATH . 'modules/widgets/theplus-widgets-extra.php';

		/*--OnScroll View Animation ---*/
		include THEPLUS_PATH . 'modules/widgets/theplus-widget-animation-attr.php';

		echo $before_content; ?>

<div class="theplus-accordion-wrapper elementor-accordion <?php echo esc_attr( $sattrclasss ); ?> <?php echo esc_attr( $settings['hover_style'] ); ?> <?php echo esc_attr( $animated_class ); ?>"
    id="<?php echo esc_attr( $uid ); ?>" data-accordion-id="<?php echo esc_attr( $uid ); ?>"
    data-connection="<?php echo esc_attr( $connect_carousel ); ?>"
    data-accordion-type="<?php echo esc_attr( $on_hover_accordion ); ?>"
    data-heightlight-text<?php echo $animation_attr; ?> <?php echo $row_bg_conn; ?> <?php echo $accordiannew; ?>
    <?php echo $sattr; ?> role="tablist" <?php echo $mainschema; ?>>

    <?php if ( 'yes' === $search_accordion && ! empty( $search_accordion_length ) ) { ?>
    <div id="accordion_search_bar_container">
        <?php
				if ( ! empty( $searchicon ) ) {
					echo '<span class="tp-acr-search-icon">' . $searchicon . '</span>';
				}
				?>

        <input type="search" id="tpsb<?php echo esc_attr( $uid ); ?>" class="tpacsearchinput"
            placeholder="<?php echo esc_html( $search_accordion_placeholder ); ?>" value=""
            onkeyup="this.setAttribute('value', this.value);">
    </div>
    <?php
		}

		$expand_collapse_accordion = isset( $settings['expand_collapse_accordion'] ) ? $settings['expand_collapse_accordion'] : 'no';

		$saip = '';
		$satp = '';
		$sain = '';
		$satn = '';

		$slider_accordion       = isset( $settings['slider_accordion'] ) ? $settings['slider_accordion'] : 'no';
		$slider_accordion_align = ! empty( $settings['slider_accordion_align'] ) ? $settings['slider_accordion_align'] : '';
		if ( 'yes' === $slider_accordion ) {
			if ( ! empty( $settings['slider_accordion_icon_prev'] ) ) {
				ob_start();
				\Elementor\Icons_Manager::render_icon( $settings['slider_accordion_icon_prev'], array( 'aria-hidden' => 'true' ) );
				$saip = '<span class="tpas-icon">' . ob_get_contents() . '</span>';
				ob_end_clean();
			}

			if ( ! empty( $settings['slider_accordion_text_prev'] ) ) {
				$satp = '<span class="tpas-text">' . $settings['slider_accordion_text_prev'] . '</span>';
			}

			if ( ! empty( $settings['slider_accordion_icon_nxt'] ) ) {
				ob_start();
				\Elementor\Icons_Manager::render_icon( $settings['slider_accordion_icon_nxt'], array( 'aria-hidden' => 'true' ) );
				$sain = '<span class="tpas-icon">' . ob_get_contents() . '</span>';
				ob_end_clean();
			}

			if ( ! empty( $settings['slider_accordion_text_nxt'] ) ) {
				$satn = '<span class="tpas-text">' . $settings['slider_accordion_text_nxt'] . '</span>';
			}
		}

		$eca_position = ! empty( $settings['eca_position'] ) ? $settings['eca_position'] : 'ecabefore';
		if ( $expand_collapse_accordion === 'yes' && $eca_position === 'ecabefore' ) {
			?>
    <div class="tp-aec-button"><a href="javascript:void(0)" class="tp-toggle-accordion active"></a></div>
    <?php
		}

		$ij = 1;
		$mj = 1;

		$totalloop = 0;
		foreach ( $settings['tabs'] as $index => $item ) {
			++$totalloop;
		}

		foreach ( $settings['tabs'] as $index => $item ) :
			$tab_count = $index + 1;

			if ( $settings['active_accordion'] == $tab_count || 'all-open' === $settings['active_accordion'] ) {
				$active_default = 'active-default';
			} elseif ( $settings['active_accordion'] == 0 ) {
				$active_default = '0';
			} else {
				$active_default = 'no';
			}

			if ( ! empty( $item['tab_hashid'] ) ) {
				$tab_title_id   = trim( $item['tab_hashid'] );
				$tab_content_id = 'tab-content-' . trim( $item['tab_hashid'] );
			} else {
				$tab_title_id   = 'elementor-tab-title-' . $id_int . $tab_count;
				$tab_content_id = 'elementor-tab-content-' . $id_int . $tab_count;
			}

			$tab_title_setting_key = $this->get_repeater_setting_key( 'tab_title', 'tabs', $index );

			$tab_content_setting_key = $this->get_repeater_setting_key( 'tab_content', 'tabs', $index );

			$lz2 = function_exists( 'tp_has_lazyload' ) ? tp_bg_lazyLoad( $settings['box_background_image'], $settings['box_active_background_image'] ) : '';

			$this->add_render_attribute(
				$tab_title_setting_key,
				array(
					'id'            => $tab_title_id,
					'class'         => array( 'elementor-tab-title', 'plus-accordion-header', $active_default, $title_align, $lz2 ),
					'data-tab'      => $tab_count,
					'aria-controls' => $tab_content_id,
				// 'role' => 'tab',
				// 'tabindex' => $id_int . $tab_count,
				)
			);

			$lz3 = function_exists( 'tp_has_lazyload' ) ? tp_bg_lazyLoad( $settings['content_box_background_image'] ) : '';
			$this->add_render_attribute(
				$tab_content_setting_key,
				array(
					'id'              => $tab_content_id,
					'class'           => array( 'elementor-tab-content', 'elementor-clearfix', 'plus-accordion-content', $active_default, $lz3 ),
					'data-tab'        => $tab_count,
					'role'            => 'tabpanel',
					'aria-labelledby' => $tab_title_id,
				)
			);

			$this->add_inline_editing_attributes( $tab_content_setting_key, 'advanced' );

			$accordion_toggle_icon = '';

			$accslideactive = '';
			if ( isset( $slider_accordion ) && 'yes' === $slider_accordion && ! empty( $slider_accordion_show ) ) {
				if ( $tab_count % $slider_accordion_show == 1 ) {
					$sliderclass = '';
					if ( $ij === 1 ) {
						$sliderclass = ' tpaccractive';
					}
					?>
    <div class="tp-accr-list-slider <?php echo $sliderclass; ?>" data-tabslide="<?php echo $mj; ?>">
        <?php
					++$ij;
					++$mj;
				}
			}

			if ( ! empty( $item['tab_title'] ) ) {
				?>
        <div class="theplus-accordion-item" <?php echo $schemaAttr; ?> role="tab">
            <<?php echo theplus_validate_html_tag( $settings['title_html_tag'] ); ?> <?php
					echo $this->get_render_attribute_string( $tab_title_setting_key );
					echo $schemaAttr1;
					?>>
                <?php
					if ( 'yes' === $settings['display_icon'] ) :

						if ( 'font_awesome' === $settings['icon_style'] ) {
							$icons        = $settings['icon_fontawesome'];
							$icons_active = $settings['icon_fontawesome_active'];
						} elseif ( 'icon_mind' === $settings['icon_style'] ) {
							$icons        = $settings['icons_mind'];
							$icons_active = $settings['icons_mind_active'];
						} elseif ( ! empty( $settings['icon_style'] ) && 'font_awesome_5' === $settings['icon_style'] ) {
							ob_start();
							\Elementor\Icons_Manager::render_icon( $settings['icon_fontawesome_5'], array( 'aria-hidden' => 'true' ) );
							$icons = ob_get_contents();
							ob_end_clean();

							ob_start();
							\Elementor\Icons_Manager::render_icon( $settings['icon_fontawesome_5_active'], array( 'aria-hidden' => 'true' ) );
							$icons_active = ob_get_contents();
							ob_end_clean();
						} else {
							$icons        = '';
							$icons_active = '';
						}

						if ( ! empty( $icons ) && ! empty( $icons_active ) ) {
							$accordion_toggle_icon = '<span class="elementor-accordion-icon elementor-accordion-icon-' . esc_attr( $settings['icon_align'] ) . '" aria-hidden="true">';
							if ( ! empty( $settings['icon_style'] ) && 'font_awesome_5' === $settings['icon_style'] ) {
								$accordion_toggle_icon .= '<span class="elementor-accordion-icon-closed">' . $icons . '</span>';
								$accordion_toggle_icon .= '<span class="elementor-accordion-icon-opened">' . $icons_active . '</span>';
							} else {
								$accordion_toggle_icon .= '<i class="elementor-accordion-icon-closed ' . esc_attr( $icons ) . '"></i>';
								$accordion_toggle_icon .= '<i class="elementor-accordion-icon-opened ' . esc_attr( $icons_active ) . '"></i>';
							}

							$accordion_toggle_icon .= '</span>';
						}
						?>
                <?php
						endif;

					if ( ! empty( $settings['icon_align'] ) && 'left' === $settings['icon_align'] ) {
						echo $accordion_toggle_icon;
					}

					if ( ! empty( $item['display_icon'] ) && 'yes' === $item['display_icon'] ) :
						if ( 'font_awesome' === $item['icon_style'] ) {
							$icons_loop = $item['icon_fontawesome'];
						} elseif ( 'icon_mind' === $item['icon_style'] ) {
							$icons_loop = $item['icons_mind'];
						} elseif ( 'font_awesome_5' === $item['icon_style'] ) {
							ob_start();
							\Elementor\Icons_Manager::render_icon( $item['icon_fontawesome_5'], array( 'aria-hidden' => 'true' ) );
							$icons_loop = ob_get_contents();
							ob_end_clean();
						} else {
							$icons_loop = '';
						}

						if ( ! empty( $icons_loop ) && ! empty( $icons_loop ) ) {
							$lz1 = function_exists( 'tp_has_lazyload' ) ? tp_bg_lazyLoad( $settings['loop_icon_background_image'], $settings['loop_icon_hover_background_image'] ) : '';

							if ( ! empty( $item['icon_style'] ) && 'font_awesome_5' === $item['icon_style'] ) {
								?>
                <span class="accordion-icon-prefix <?php echo esc_attr( $lz1 ); ?>"><span
                        class="plus-icon-accordion"><?php echo $icons_loop; ?></span></span>
                <?php
							} else {
								?>
                <span class="accordion-icon-prefix <?php echo esc_attr( $lz1 ); ?>"><i
                        class="plus-icon-accordion <?php echo esc_attr( $icons_loop ); ?>"></i></span>
                <?php
							}
						} endif;
					?>

                <?php echo '<span style="width:100%">' . wp_kses_post( $item['tab_title'] ) . '</span>'; ?>
                <?php
						if ( ! empty( $settings['icon_align'] ) && 'right' === $settings['icon_align'] ) {
							echo $accordion_toggle_icon;
						}
						?>
            </<?php echo theplus_validate_html_tag( $settings['title_html_tag'] ); ?>>

            <?php 

					$content_template = isset( $item['content_template'] ) ? intval( $item['content_template'] ) : 0;
					$backend_preview = !empty( $item['backend_preview_template'] ) ? $item['backend_preview_template'] : '';

					if ( ( 'content' === $item['content_source'] && ! empty( $item['tab_content'] ) ) || ( 'page_template' === $item['content_source'] ) ) { ?>
            <div <?php
						echo $this->get_render_attribute_string( $tab_content_setting_key );
						echo $schemaAttr2;
						?>>
                <?php
										if ( 'content' === $item['content_source'] && ! empty( $item['tab_content'] ) ) {
											echo '<div class="plus-content-editor" ' . $schemaAttr3 . '>' . $this->parse_text_editor( $item['tab_content'] ) . '</div>';
										}

										if ( \Elementor\Plugin::$instance->editor->is_edit_mode() && 'page_template' === $item['content_source'] ) {
											if ( 'yes' === $backend_preview ) {

												if ( empty( $content_template ) || '0' === $content_template ) {
													echo '<div class="tab-preview-template-notice">
															<div class="preview-temp-notice-heading">' . esc_html__( 'Select Template', 'theplus' ) . '</div>
															<div class="preview-temp-notice-desc">' . esc_html__( 'Please select a template to display its content.', 'theplus' ) . '</div>
														</div>';
												} else {
													$template_status = get_post_status( $content_template );
													if ( 'publish' === $template_status ) {
														echo '<div class="plus-content-editor" ' . $schemaAttr3 . '>' . Theplus_Element_Load::elementor()->frontend->get_builder_content_for_display( $content_template ) . '</div>';
													} else {
														echo '<div class="tab-preview-template-notice"><div class="preview-temp-notice-heading">' . esc_html__( 'Unauthorized Access', 'theplus' ) . '</b></div><div class="preview-temp-notice-desc"><b>' . esc_html__( 'Note :', 'theplus' ) . '</b> ' . esc_html__( 'You need to upgrade your permissions to Editor or Administrator level to update this option.', 'theplus' ) . '</div></div>';
													}
												}
											} else {
												$get_template_name = '';
												$get_template_id   = $content_template;
												if ( ! empty( $templates ) && ! empty( $get_template_id ) ) {
													foreach ( $templates as $value ) {
														if ( $value['template_id'] == $get_template_id ) {
															$get_template_name = $value['title'];
														}
													}
												}

												echo '<div class="tab-preview-template-notice"><div class="preview-temp-notice-heading">Selected Template : <b>"' . esc_attr( $get_template_name ) . '"</b></div><div class="preview-temp-notice-desc"><b>' . esc_html__( 'Note :', 'theplus' ) . '</b> ' . esc_html__( 'We have turn off visibility of template in the backend due to performance improvements. This will be visible perfectly on the frontend.', 'theplus' ) . '</div></div>';
											}
										} elseif ( 'page_template' === $item['content_source'] ) {

											if ( empty( $content_template ) || '0' === $content_template ) {
												echo '<div class="tab-preview-template-notice">
														<div class="preview-temp-notice-heading">' . esc_html__( 'Select Template', 'theplus' ) . '</div>
														<div class="preview-temp-notice-desc">' . esc_html__( 'Please select a template to display its content.', 'theplus' ) . '</div>
													</div>';
											} else {
												if ( has_filter( 'wpml_object_id' ) ) {
													$content_template = apply_filters( 'wpml_object_id', $content_template, get_post_type( $content_template ), true);
												}
												
												$template_status = get_post_status( $content_template );
												if( 'publish' === $template_status ) {
													echo '<div class="plus-content-editor" ' . $schemaAttr3 . '>' . Theplus_Element_Load::elementor()->frontend->get_builder_content_for_display( $content_template ) . '</div>';
												} else {
													echo '<div class="tab-preview-template-notice"><div class="preview-temp-notice-heading">' . esc_html__( 'Unauthorized Access', 'theplus' ) . '</b></div><div class="preview-temp-notice-desc"><b>' . esc_html__( 'Note :', 'theplus' ) . '</b> ' . esc_html__( 'You need to upgrade your permissions to Editor or Administrator level to update this option.', 'theplus' ) . '</div></div>';
												}
											}

										} 
										?>

            </div>
            <?php } ?>
        </div>
        <?php
			}

			if ( isset( $slider_accordion ) && 'yes' === $slider_accordion && ! empty( $slider_accordion_show ) ) {

				if ( $tab_count % $slider_accordion_show == 0 || $tab_count == $totalloop ) {
					?>
    </div>
    <?php
				}
			}

		endforeach;

		if ( 'yes' === $expand_collapse_accordion && 'ecaafter' === $eca_position ) {
			?>
    <div class="tp-aec-button"><a href="javascript:void(0)" class="tp-toggle-accordion active"></a></div>
    <?php
		}

		/*playpausebutton*/
		if ( isset( $settings['tabs_autoplay'] ) && 'yes' === $settings['tabs_autoplay'] && isset( $settings['tabs_autoplaypause'] ) && 'yes' === $settings['tabs_autoplaypause'] ) {
			$iconsPlay  = '';
			$iconsPause = '';
			if ( ! empty( $settings['autopauseicon'] ) ) {
				ob_start();
				\Elementor\Icons_Manager::render_icon( $settings['autopauseicon'], array( 'aria-hidden' => 'true' ) );
				$iconsPlay = ob_get_contents();
				ob_end_clean();
			}
			if ( ! empty( $settings['autoplayicon'] ) ) {
				ob_start();
				\Elementor\Icons_Manager::render_icon( $settings['autoplayicon'], array( 'aria-hidden' => 'true' ) );
				$iconsPause = ob_get_contents();
				ob_end_clean();
			}
			?>

    <div class="tp-tab-play-pause-wrap">
        <div class="tp-tab-play-pause tpplay active"><?php echo $iconsPlay; ?></div>
        <div class="tp-tab-play-pause tppause"><?php echo wp_kses_post( $iconsPause ); ?></div>
    </div>
    <?php
		}

		if ( 'yes' === $slider_accordion && ! empty( $slider_accordion_show ) ) {
			if ( $tab_count > $slider_accordion_show ) {
				$sliacc_seprator_f = ! empty( $settings['sliacc_seprator_f'] ) ? $settings['sliacc_seprator_f'] : '';
				?>

    <div class="tp-aec-slide-page <?php echo wp_kses_post( $slider_accordion_align ); ?>">
        <span class="tpasp-prev tpas-disabled"><?php echo $saip . $satp; ?></span>
        <span class="tpasp-active-slide">1</span>

        <?php if ( ! empty( $sliacc_seprator_f ) ) { ?>
        <span class="tpasp-sep-slide"><?php echo wp_kses_post( $sliacc_seprator_f ); ?></span>
        <?php } ?>

        <span class="tpasp-total-slide">1</span>
        <span class="tpasp-next"><?php echo wp_kses_post( $satn ) . wp_kses_post( $sain ); ?></span>
    </div>
    <?php
			}
		}
		?>

</div>
<?php

		echo $after_content;
	}
}
